#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <signal.h>

#include <X11/X.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "lwm.h"

Mode mode;			/* The window manager's mode. (See "lwm.h".) */
int start_x;			/* The X position where the mode changed. */
int start_y;			/* The Y position where the mode changed. */

Display * dpy;			/* The connection to the X server. */
int screen_count;		/* The number of screens. */
ScreenInfo * screens;		/* Information about these screens. */
ScreenInfo * current_screen;

XFontStruct *font;		/* Actual titlebar font. */
XFontStruct *popup_font;	/* Actual menu font. */

Bool shape;			/* Does server have Shape Window extension? */
int shape_event;		/* ShapeEvent event type. */

/* Atoms we're interested in. See the ICCCM for more information. */
Atom wm_state;
Atom wm_change_state;
Atom wm_protocols;
Atom wm_delete;
Atom wm_take_focus;
Atom wm_colormaps;
Atom compound_text;

/** Netscape uses this to give information about the URL it's displaying. */
Atom _mozilla_url;

char *argv0;

static void initScreens(void);
static void initScreen(int);

/*ARGSUSED*/
extern int
main(int argc, char *argv[]) {
	XEvent ev;
	struct sigaction sa;

	argv0 = argv[0];

	mode = wm_initialising;

	/* Open a connection to the X server. */
	dpy = XOpenDisplay("");
	if (dpy == 0)
		panic("can't open display.");

	parseResources();

	/* Set up an error handler. */
	XSetErrorHandler(errorHandler);

	/* Set up signal handlers. */
	signal(SIGTERM, Terminate);
	signal(SIGINT, Terminate);
	signal(SIGHUP, Terminate);

	/* Ignore SIGCHLD. */
	sa.sa_handler = SIG_IGN;
#ifdef SA_NOCLDWAIT
	sa.sa_flags = SA_NOCLDWAIT;
#else
	sa.sa_flags = 0;
#endif
	sigemptyset(&sa.sa_mask);
	sigaction(SIGCHLD, &sa, 0);

	/* Internalize useful atoms. */
	wm_state = XInternAtom(dpy, "WM_STATE", False);
	wm_change_state = XInternAtom(dpy, "WM_CHANGE_STATE", False);
	wm_protocols = XInternAtom(dpy, "WM_PROTOCOLS", False);
	wm_delete = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
	wm_take_focus = XInternAtom(dpy, "WM_TAKE_FOCUS", False);
	wm_colormaps = XInternAtom(dpy, "WM_COLORMAP_WINDOWS", False);
	compound_text = XInternAtom(dpy, "COMPOUND_TEXT", False);
	
	_mozilla_url = XInternAtom(dpy, "_MOZILLA_URL", False);
	
	/*
	 * Get fonts for our titlebar and our popup window. We try to
	 * get Lucida, but if we can't we make do with fixed because everyone
	 * has that.
	 */
	font = XLoadQueryFont(dpy, font_name);
	if (font == 0)
		font = XLoadQueryFont(dpy, "fixed");
	if (font == 0)
		panic("can't find a font for the titlebars.");
	
	popup_font = XLoadQueryFont(dpy, popup_font_name);
	if (popup_font == 0)
		popup_font = XLoadQueryFont(dpy, "fixed");
	if (popup_font == 0)
		panic("can't find a font for the popup window.");
	
	initScreens();
	
	/* See if the server has the Shape Window extension. */
	shape = serverSupportsShapes();
	
	/*
	 * Initialisation is finished, but we start off not interacting with the
	 * user.
	 */
	mode = wm_idle;
	
	/*
	 * The main event loop.
	 */
	for (;;) {
		XNextEvent(dpy, &ev);
		dispatch(&ev);
	}
}

void
sendConfigureNotify(Client *c) {
	XConfigureEvent ce;

	ce.type = ConfigureNotify;
	ce.event = c->window;
	ce.window = c->window;
	ce.x = c->size.x + border;
	ce.y = c->size.y + border;
	ce.width = c->size.width - 2 * border;
	ce.height = c->size.height - 2 * border;
	ce.above = None;
	ce.border_width = c->border;
	ce.override_redirect = 0;
	XSendEvent(dpy, c->window, False, StructureNotifyMask, (XEvent *) &ce);
}

extern void
scanWindowTree(int screen) {
	unsigned int i;
	unsigned int nwins;
	Client * c;
	Window dw1;
	Window dw2;
	Window * wins;
	XWindowAttributes attr;
	
	XQueryTree(dpy, screens[screen].root, &dw1, &dw2, &wins, &nwins);
	for (i = 0; i < nwins; i++) {
		XGetWindowAttributes(dpy, wins[i], &attr);
		if (attr.override_redirect || isShaped(wins[i]) || wins[i] == screens[screen].popup)
			continue;
		c = Client_Add(wins[i], screens[screen].root);
		if (c != 0 && c->window == wins[i]) {
			c->screen = &screens[screen];
			c->size.x = attr.x - border;
			c->size.y = attr.y - border;
			c->size.width  = attr.width  + 2 * border;
			c->size.height = attr.height + 2 * border;
			c->border = attr.border_width;
			if (attr.map_state == IsViewable) {
				c->internal_state = IPendingReparenting;
				manage(c, 1);
			}
		}
	}
	XFree(wins);
}

/*ARGSUSED*/
extern void
shell(ScreenInfo * screen, int button, int x, int y) {
	char * command;
	char * sh;
	
	/* Get the command we're to execute. Give up if there isn't one. */
	command = (button == Button1) ? btn1_command : btn2_command;
	if (command == 0)
		return;
	
	sh = getenv("SHELL");
	if (sh == 0)
		sh = "/bin/sh";
	
	switch (fork()) {
	case 0:		/* Child. */
		close(ConnectionNumber(dpy));
		if (screen && screen->display_spec != 0)
			putenv(screen->display_spec);
		execl(sh, sh, "-c", command, 0);
		fprintf(stderr, "%s: can't exec \"%s -c %s\"\n", argv0, sh,
			command);
		execlp("xterm", "xterm", 0);
		exit(EXIT_FAILURE);
	case -1:	/* Error. */
		fprintf(stderr, "%s: couldn't fork\n", argv0);
		break;
	}
}

extern int
titleHeight(void) {
	return font->ascent + font->descent + 1;
}

static void
initScreens(void) {
	int screen;
	
	/* Find out how many screens we've got, and allocate space for their info. */
	screen_count = ScreenCount(dpy);
	screens = (ScreenInfo *) malloc(screen_count * sizeof(ScreenInfo));
	
	/* Go through the screens one-by-one, initialising them. */
	for (screen = 0; screen < screen_count; screen++) {
		initialiseCursors(screen);
		initScreen(screen);
		scanWindowTree(screen);
	}
}

static void
initScreen(int screen) {
	XGCValues gv;
	XSetWindowAttributes attr;
	XColor colour, exact;
	int len;
	char * display_string = DisplayString(dpy);
	char * colon = strrchr(display_string, ':');
	char * dot = strrchr(display_string, '.');
	
	/* Set the DISPLAY specification. */
	if (colon) {
		len = 9 + strlen(display_string) + ((dot == 0) ? 2 : 0) + 10;
		screens[screen].display_spec = (char *) malloc(len);
		sprintf(screens[screen].display_spec, "DISPLAY=%s", display_string);
		if (dot == 0) dot = screens[screen].display_spec + len - 3;
		else dot = strrchr(screens[screen].display_spec, '.');
		sprintf(dot, ".%i", screen);
	} else {
		screens[screen].display_spec = 0;
	}
	
	/* Find the root window. */
	screens[screen].root = RootWindow(dpy, screen);
	screens[screen].display_width = DisplayWidth(dpy, screen);
	screens[screen].display_height = DisplayHeight(dpy, screen);
	
	/* Get the pixel values of the only two colours we use. */
	screens[screen].black = BlackPixel(dpy, screen);
	screens[screen].white = WhitePixel(dpy, screen);
	XAllocNamedColor(dpy, DefaultColormap(dpy, screen), "DimGray", &colour, &exact);
	screens[screen].gray = colour.pixel;
	
	/* Set up root (frame) GC's. */
	gv.foreground = screens[screen].black ^ screens[screen].white;
	gv.background = screens[screen].white;
	gv.font = font->fid;
	gv.function = GXxor;
	gv.line_width = 1;
	gv.subwindow_mode = IncludeInferiors;
	screens[screen].gc_thin = XCreateGC(dpy, screens[screen].root,
		GCForeground | GCBackground | GCFunction | GCFont |
		GCLineWidth | GCSubwindowMode, &gv);
	
	gv.line_width = 2;
	screens[screen].gc = XCreateGC(dpy, screens[screen].root,
		GCForeground | GCBackground | GCFunction |
		GCFont | GCLineWidth | GCSubwindowMode, &gv);
	
	/* Create a window for our popup. */
	screens[screen].popup = XCreateSimpleWindow(dpy, screens[screen].root,
		0, 0, 1, 1, 1, screens[screen].black, screens[screen].white);
	attr.event_mask = ButtonMask | ButtonMotionMask | ExposureMask;
	XChangeWindowAttributes(dpy, screens[screen].popup, CWEventMask, &attr);
	
	/* Create menu GC. */
	gv.line_width = 1;
	gv.font = popup_font->fid;
	screens[screen].menu_gc = XCreateGC(dpy, screens[screen].popup,
		GCForeground | GCBackground | GCFunction | GCFont |
		GCLineWidth | GCSubwindowMode, &gv);
	
	/* Create size indicator GC. */
	gv.foreground = screens[screen].black;
	gv.function = GXcopy;
	screens[screen].size_gc = XCreateGC(dpy, screens[screen].popup,
		GCForeground | GCBackground | GCFunction | GCFont |
		GCLineWidth | GCSubwindowMode, &gv);
	
	/* Announce our interest in the root window. */
	attr.cursor = screens[screen].root_cursor;
	attr.event_mask = SubstructureRedirectMask | SubstructureNotifyMask |
		ColormapChangeMask | ButtonPressMask | PropertyChangeMask |
		EnterWindowMask;
	XChangeWindowAttributes(dpy, screens[screen].root, CWCursor |
		CWEventMask, &attr);
	
	/* Make sure all our communication to the server got through. */
	XSync(dpy, False);
}

/**
Find the screen for which root is the root window.
*/
ScreenInfo *
getScreenFromRoot(Window root) {
	int screen;
	
	for (screen = 0; screen < screen_count; screen++)
		if (screens[screen].root == root)
			return &screens[screen];
	
	return 0;
}
