/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: dotconf.c,v 1.10 2002/01/29 04:51:25 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <libhalfgnome/halfgnome.h>

#include "dotconf.h"
#include "network.h"

#include "spell.h"
#include "util.h"

/* last time the configuration file format changed in an incompatible way.
 * versions less than this version are incompatible.
 */ 
#define LASTCHANGE "" /* new lj means all known versions are ok! */
#define CONFIGNAME ".logjam"

config conf;

int parseversion(char *ver) {
	int maj, min, rev;
	unsigned int ret;

	if (strcasecmp(ver, "cvs") == 0)
		return 0;

	if (sscanf(ver, "%d.%d.%d", &maj, &min, &rev) < 3)
		return -1;

	ret = (maj << 16) | (min << 8) | (rev);
	return ret;
}

int badversion(char *ver) {
	int lastchange;
	int configversion;
	
	return FALSE; /* all known versions are good right now! */
	configversion = parseversion(ver);
	lastchange = parseversion(LASTCHANGE);

	if (configversion == 0) return FALSE; /* CVS version */
	return (configversion < lastchange);
}

static void skipspace(FILE *file) {
	int c;
	do {
		c = fgetc(file);
	} while (c == '\n' || c == ' ');
	ungetc(c, file);
}

static void nexttoken(FILE *f, guchar *tokenbuf) {
	int c;
	do {
		c = fgetc(f);
	} while (c == ' ');

	while (c != ' ' && c != '\n' && c != EOF) {
		*tokenbuf++ = c;
		c = fgetc(f);
	}
	ungetc(c, f);
	*tokenbuf = 0;
}

/* bleh.  hack to grab the rest of the line as a token */
static void nexttokenline(FILE *f, char *tokenbuf) {
	int c;
	do {
		c = fgetc(f);
	} while (c == ' ');

	while (c != '\n' && c != EOF) {
		*tokenbuf++ = c;
		c = fgetc(f);
	}
	ungetc(c, f);
	*tokenbuf = 0;
}

static int make_full_conf_path(char *file, char *buf, size_t maxlen) {
	char *homedir;

	if ((homedir = getenv("HOME")) == NULL) 
		return -1;
	snprintf(buf, maxlen, "%s/%s%s", homedir, CONFIGNAME, file ? file : "");
	return 0;
}

static int check_config() {
	char path[1024];

	if (make_full_conf_path(NULL, path, sizeof(path)) < 0)
		return -1;

	if (mkdir(path, 0700) < 0 && errno != EEXIST) {
		perror("mkdir");
		return -1;
	}
	return 0;
}

static FILE* open_config(char *filename, char *mode) {
	FILE *f;
	char path[1024];

	if (make_full_conf_path(filename, path, sizeof(path)) < 0)
		return NULL;

	if ((f = fopen(path, mode)) == NULL) {
		fprintf(stderr, "Can't open '%s' -- will create.\n",
				path);
		return NULL; 
	}

	return f;
}

gint mood_compare_alpha(mood *a, mood *b) {
	return strcasecmp(a->name, b->name);
}

static int read_moods(config *conf) {
	FILE *file;
	mood *m;
	char value[100];

	if ((file = open_config("/moods", "r")) == NULL) 
		return -1;

	while (!feof(file)) {
		skipspace(file);

		m = g_new0(mood, 1);

		nexttoken(file, value);
		m->id = atoi(value);
		if (m->id == 0) {
			g_free(m);
			continue;
		}
		nexttokenline(file, value);
		m->name = g_strdup(value);

		conf->moods = g_list_insert_sorted(conf->moods, m, 
				(GCompareFunc)mood_compare_alpha);
	}
	return 0;
}

static void
read_geom(FILE *file, geometry *geom) {
	char value[200];
	nexttoken(file, value);
	geom->x = atoi(value);
	nexttoken(file, value);
	geom->y = atoi(value);
	nexttoken(file, value);
	geom->width = atoi(value);
	nexttoken(file, value);
	geom->height = atoi(value);
}

static int read_rc(config *conf) {
	FILE *file;
	int c;
	char option[200];
	guchar value[200];
	char *curuser = NULL;

	if ((file = open_config("/rc", "r")) == NULL) 
		return -1;

	while (!feof(file)) {
		skipspace(file);
		if (c == EOF)
			break;

		nexttoken(file, option);

		if (option[0] == '#') {
			do {
				c = fgetc(file);
			} while (c != '\n' && c != EOF);
			continue;
		}

		if (strcasecmp(option, "version") == 0) {
			nexttoken(file, value);
			conf->version = g_strdup(value);
			if (badversion(conf->version)) {
				fprintf(stderr, 
					"config file version (%s) has changed in "
						"a later version (%s) of " PROGRAMNAME ".\n"
					"ignoring rest of config file; "
						"it will be recreated.\n",
					conf->version, LASTCHANGE);
				break;
			}
		} else if (strcasecmp(option, "currentuser") == 0) {
			nexttoken(file, value);
			curuser = urldecode(value);
		} else if (strcasecmp(option, "user") == 0) {
			user *u;
			u = g_new0(user, 1);

			nexttoken(file, value);
			u->username = urldecode(value);
			nexttoken(file, value);
			u->password = urldecode(value);

			nexttoken(file, value);
			if (strcasecmp(value, "usefastserver") == 0)
				u->fastserver = TRUE;

			conf->users = g_list_append(conf->users, u);
		} else if (strcasecmp(option, "geometrymain") == 0) {
			read_geom(file, &conf->mainwin_geom);
		} else if (strcasecmp(option, "geometryhistoryitem") == 0) {
			read_geom(file, &conf->hiwin_geom);
		} else if (strcasecmp(option, "geometryfriends") == 0) {
			read_geom(file, &conf->friends_geom);
		} else if (strcasecmp(option, "geometryconsole") == 0) {
			read_geom(file, &conf->console_geom);
		} else if (strcasecmp(option, "revertusejournalafterpost") == 0) {
			conf->revertusejournalafterpost = TRUE;
		} else if (strcasecmp(option, "savemetadata") == 0) {
			conf->savemetadata = TRUE;
		} else if (strcasecmp(option, "confirmdelete") == 0) {
			conf->confirmdelete = TRUE;
		} else if (strcasecmp(option, "netdump") == 0) {
			conf->netdump = TRUE;
		} else if (strcasecmp(option, "nothread") == 0) {
			conf->nothread = TRUE;
		} else if (strcasecmp(option, "ljserver") == 0) {
			int lastofs;
			nexttoken(file, value);
			conf->ljserver = g_strdup(value);
			/* make sure server url doesn't end with a / */
			lastofs = strlen(conf->ljserver)-1;
			if (conf->ljserver[lastofs] == '/') 
				conf->ljserver[lastofs] = 0;
		} else if (strcasecmp(option, "useproxy") == 0) {
			conf->useproxy = TRUE;
		} else if (strcasecmp(option, "proxyserver") == 0) {
			nexttoken(file, value);
			conf->proxyserver = g_strdup(value);
		} else if (strcasecmp(option, "proxyport") == 0) {
			nexttoken(file, value);
			conf->proxyport = atoi(value);
		} else if (strcasecmp(option, "spawn") == 0) {
			nexttokenline(file, value);
			halfgnome_spawn_set_command(value);
		} else if (strcasecmp(option, "spellcheck") == 0) {
			nexttokenline(file, value);
			g_free(conf->spellcheck);
			conf->spellcheck = g_strdup(value);
		} else if (strcasecmp(option, "usespellcheck") == 0) {
			conf->usespellcheck = TRUE;
		} else if (strcasecmp(option, "defaultsecurity") == 0) {
			nexttoken(file, value);
			if (strcasecmp(value, "public") == 0) {
				conf->defaultsecurity = SECURITY_PUBLIC;
			} else if (strcasecmp(value, "private") == 0) {
				conf->defaultsecurity = SECURITY_PRIVATE;
			} else if (strcasecmp(value, "friends") == 0) {
				conf->defaultsecurity = SECURITY_FRIENDS;
			}
		} else if (strcasecmp(option, "convertcharset") == 0) {
			nexttoken(file, value);
			if (strcasecmp(value, "none") == 0) {
				conf->convertcharset = CONVERT_NONE;
			} else if (strcasecmp(value, "koi2win") == 0) {
				conf->convertcharset = CONVERT_KOI2WIN;
			}
		}
		/*printf("'%s' -> '%s'\n", option, value);*/
	}

	if (conf->proxyserver == NULL || conf->proxyport == 0) 
		conf->useproxy = FALSE;

	if (curuser != NULL) { /* if they specified a CurrentUser */
		user *u;
		u = conf_user_by_name(conf, curuser);
		if (u) {
			conf->username = g_strdup(u->username);
			conf->password = g_strdup(u->password);
		}
		g_free(curuser);
	}

	fclose(file);

	return 0;
}

int config_read(config *conf) {
	check_config();

	memset(conf, 0, sizeof(conf));

	/* set defaults here */
	conf->ljserver = g_strdup("http://www.livejournal.com");
	conf->spellcheck = g_strdup("ispell -a -P");

	read_rc(conf);
	read_moods(conf);
	return 0;
}

static void
geom_write(FILE *file, char *name, geometry *geom) {
	if (geom->width == 0)
		return;
	fprintf(file, "%s %d %d %d %d\n",
			name,
			geom->x,     geom->y,
			geom->width, geom->height);
}

int config_write(config *conf) {
	FILE *file;
	GList *l;

	if ((file = open_config("/rc", "w")) == NULL) 
		return -1;

	fchmod(fileno(file), 0600);

	fprintf(file, 
			"# automatically generated lj configuration file.\n"
			"# do not edit -- changes will be lost on exit.\n"
			"\n"
			"Version " VERSION "\n");
	if (conf->username && (conf_user_by_name(conf, conf->username) != NULL)) {
		guchar *u;
		u = urlencode(conf->username);
		fprintf(file, "CurrentUser %s\n", u);
		g_free(u);
	}
	for (l = conf->users; l != NULL; l = l->next) {
		user *u = l->data;
		if (u->username && u->password) {
			guchar *username, *password;
			username = urlencode(u->username); 
			password = urlencode(u->password);
			fprintf(file, "User %s %s", username, password);
			if (u->fastserver) 
				fprintf(file, " UseFastServer # please note that you still must be a paid user for this to work!\n");
			else 
				fprintf(file, "\n");
			g_free(username); 
			g_free(password);
		}
	}
	if (conf->revertusejournalafterpost) fprintf(file, "RevertUseJournalAfterPost\n");
	if (conf->savemetadata) fprintf(file, "SaveMetaData\n");
	if (conf->netdump) fprintf(file, "NetDump\n");
	if (conf->nothread) fprintf(file, "NoThread\n");
	geom_write(file, "GeometryMain", &conf->mainwin_geom);
	geom_write(file, "GeometryHistoryItem", &conf->hiwin_geom);
	geom_write(file, "GeometryFriends", &conf->friends_geom);
	geom_write(file, "GeometryConsole", &conf->console_geom);
	if (conf->confirmdelete) fprintf(file, "ConfirmDelete\n");
	if (conf->ljserver) fprintf(file, "LJServer %s\n", conf->ljserver);
	if (conf->useproxy) fprintf(file, "UseProxy\n");
	if (conf->proxyserver) fprintf(file, "ProxyServer %s\n", conf->proxyserver);
	if (conf->proxyport) fprintf(file, "ProxyPort %d\n", conf->proxyport);
	if (halfgnome_spawn_get_command()) 
		fprintf(file, "Spawn %s\n", halfgnome_spawn_get_command());
	if (conf->spellcheck) fprintf(file, "SpellCheck %s\n", conf->spellcheck);
	if (conf->usespellcheck) fprintf(file, "UseSpellCheck\n");
	if (conf->defaultsecurity != SECURITY_PUBLIC) {
		char *sec = NULL;
		switch (conf->defaultsecurity) {
			case SECURITY_PRIVATE: sec = "Private"; break;
			case SECURITY_FRIENDS: sec = "Friends"; break;
			default: break;
		}
		if (sec != NULL)
			fprintf(file, "DefaultSecurity %s\n", sec);
	}
	if (conf->convertcharset != CONVERT_NONE) {
		char *convert = NULL;
		switch (conf->convertcharset) {
			case CONVERT_KOI2WIN: convert = "koi2win"; break;
			default: break;
		}
		if (convert != NULL)
			fprintf(file, "ConvertCharset %s\n", convert);
	}

	fclose(file);

	if ((file = open_config("/moods", "w")) == NULL) 
		return -1;
	for (l = conf->moods; l != NULL; l = l->next) {
		mood *m = l->data;
		fprintf(file, "%d %s\n", m->id, m->name);
	}
	fclose(file);


	return 0;
}

user* conf_user_by_name(config *conf, char *username) {
	GList *l;
	for (l = conf->users; l != NULL; l = l->next) 
		if (strcasecmp(username, ((user*)l->data)->username) == 0) 
			return l->data;
	return NULL;
}

extern int mood_get_last_cached(config *conf) {
	GList *l;
	mood *m;
	int lastid = 0;

	for (l = conf->moods; l != NULL; l = l->next) {
		m = l->data;
		if (m->id > lastid) lastid = m->id;
	}
	return lastid;
}
