# -*- coding: utf-8 -*-
# vim: ts=4
###
#
# Listen is the legal property of mehdi abaakouk <theli48@gmail.com>
# Copyright (c) 2006 Mehdi Abaakouk
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
#
###



import gobject
import urllib

from xml.etree.ElementTree import fromstring as XMLFromString


from config import config

from helper import Dispatcher
from web_threading import WebFetchThread
from library import ListenDB

from logger import Logger

import utils


class LastFmBadUser(Exception):
    pass

class LastFmNotConnected(Exception):
    pass

class LastFmPlayerWrapper(gobject.GObject,Logger):
    __gsignals__ = {
        "new-song" : (gobject.SIGNAL_RUN_LAST,
                      gobject.TYPE_NONE,
                      (object,))
                }

    def __init__(self):
        gobject.GObject.__init__(self)
        self.__debug = True
        self.download_thread = WebFetchThread(1024,self.__fail_fetch)
        self.download_thread.set_mode_queue(True)
        self.download_thread.set_mode_cache(False)

        self.connection_info = {}
        self.__metadata_lock = False
        self.__id_metadata_monitor = None
        self.song = None
        #ActionManager.register_action("lastfm_love",21,"lastfmradio",(gtk.STOCK_APPLY,self.love))
        #ActionManager.register_action("lastfm_hate",22,"lastfmradio",(gtk.STOCK_CANCEL,self.hate))
        #ActionManager.register_action("lastfm_skip",23,"lastfmradio",(gtk.STOCK_GOTO_LAST,self.skip))
        #ActionManager.register_action("lastfm_playlist",24,"lastfmradio",(gtk.STOCK_GOTO_LAST,self.get_playlist))

    def __fail_fetch(self,*args):
        self.__metadata_lock = False
        self.loginfo("fail retrieve url %s",self.download_thread.last_url_fail)
    
    def handshake(self,callback,*args,**kwargs):
        if not self.connection_info:
            self.username = config.get("audioscrobbler","username")
            self.md5_pass = config.get("audioscrobbler","password")

            self.logdebug("Handshake start")

            url = "http://ws.audioscrobbler.com/radio/handshake.php?version=" + "1.1.1" + "&platform=" + "linux" + "&username=" + self.username + "&passwordmd5=" + self.md5_pass + "&debug=" + "0" + "&partner=";
            self.download_thread.fetch_url(url,self.__handshake_cb,callback,*args,**kwargs)
        else:
            callback(True,*args,**kwargs)

    
    def get_station_name(self,uri):
        info = uri.replace("lastfm://","").split("/")
        if not info: return ""

        if info[0] == "artist" and info[2] == "similarartists":
            return _("Similar Music to")+' "'+urllib.unquote(info[1])+'"'
        elif info[0] == "artist" and info[2] == "fans":
            return urllib.unquote(info[1])+_("'s Fan")
        elif info[0] == "globaltags":
            return _("Music tagged")+' "'+urllib.unquote(info[1])+'"'
        elif info[0] == "user" and info[2] == "neighbours":
            return _("Neighbour station of user ")+' "'+urllib.unquote(info[1])+'"'
        elif info[0] == "user" and info[2] == "personal":
            return _("Station of user")+' "'+urllib.unquote(info[1])+'"'
        else:
            return ""

    def get_error(self,num_error):
        if num_error == "1" :
            return _("There is not enough content to play this station.")

        elif num_error == "2" :
            return _("This group does not have enough members for radio.")

        elif num_error == "3" :
            return _("This artist does not have enough fans for radio.")

        elif num_error == "4" :
            return _("This item is not available for streaming.")

        elif num_error == "5" :
            return _("This feature is only available to subscribers.")

        elif num_error == "6" :
            return _("There are not enough neighbours for this radio.")

        elif num_error == "7" :
            return _("This stream has stopped.")

    def made_user_radio(self,username):
        return "lastfm://user/" + urllib.quote(username) + "/personal"

    def made_neighbours_radio(self,username):
        return "lastfm://user/" + urllib.quote(username) + "/neighbours"

    def make_artist_radio(self,station):
        return "lastfm://artist/" + urllib.quote(station.encode('UTF-8')) + "/similarartists"

    def make_fan_radio(self,station):
        return "lastfm://artist/" + urllib.quote(station.encode('UTF-8')) + "/fans"

    def make_tag_radio(self,station):
        return "lastfm://globaltags/" + urllib.quote(station.encode('UTF-8'))

    def make_group_radio(self,station):
        return "lastfm://group/" + urllib.quote(station.encode('UTF-8'))

    def make_recommended_radio(self,username,percent=100):
        return "lastfm://user/" + urllib.quote(username) + "/recommended/" + str(percent)

    def __handshake_cb(self,buffer,callback,*args,**kwargs):
        self.logdebug("Handshake received")
        lines = buffer.read()
        lines = lines.splitlines()
        for line in lines:
            opts = line.split('=')
            self.connection_info[opts[0].lower()] = opts[1]

        for key in ["banned","subscriber"]:
            if self.connection_info.has_key(key):
                self.connection_info[key] = ("1" == self.connection_info[key])

        if not self.connection_info.has_key("session") or self.connection_info["session"].lower() == "failed":
            self.connection_info = {}
            self.logwarn("Authentication failed: invalid username/password or connection failed.")
            callback(False,*args,**kwargs)
        else:
            self.logdebug("Handshake Finish")
            callback(True,*args,**kwargs)

    def get_uri(self):
        print self.connection_info
        if self.connection_info:
            return self.connection_info["stream_url"]+"="+self.connection_info["session"]
        else:
            raise LastFmNotConnected()
        
    def change_station(self,song,callback,*args,**kwargs):
        self.logdebug("Change station start for %s",song.get("uri"))

        station = song.get("uri")
        self.song = song
        url = "http://" + self.connection_info["base_url"] + "/" + self.connection_info["base_path"] + "/adjust.php?session=" + self.connection_info["session"] + "&url=" + station + "&debug=0"
        self.download_thread.fetch_url(url,self.__change_station_cb,callback,*args,**kwargs)

    def __change_station_cb(self, buffer,callback,*args,**kwargs):
        self.logdebug("Change station callback")

        lines = buffer.read()
        lines = lines.splitlines()
        info = {}
        for line in lines:
            opts = line.split('=')
            info[opts[0].lower()] = opts[1]
        if info["response"].lower() == "ok":
            self.logdebug("Change station OK")
            try: gobject.source_remove(self.__id_metadata_monitor)
            except: pass
            self.__id_metadata_monitor = gobject.timeout_add(2000,self.__monitor_cb)
            callback(info,self.song,*args,**kwargs)
        else:
            if info.has_key("error"):
                ListenDB.set_property( self.song, { "album":self.get_error(info["error"]) } )
                self.logdebug("Change station Failed: %s", info["error"])
            else:
                self.logdebug("Change station Failed")
            callback([],self.song,*args,**kwargs)
    
    def __monitor_cb(self):
        if not self.__metadata_lock:
            self.__metadata_lock = True
            url = "http://" + self.connection_info["base_url"] + "/" + self.connection_info["base_path"] + "/np.php?session=" + self.connection_info["session"]
            self.download_thread.fetch_url(url,self.__monitor_metadata_cb)
        return True

    def __monitor_metadata_cb(self,buffer):
        #if self.__debug: print "I:LastFmPlayer:Receive Metadata for ",song.get("uri")
        self.__metadata_lock = False
        if not self.song: return 

        self.logdebug("Metadata callback for %s",self.song.get("uri"))

        IDS = {
            "album":"album",
            "artist":"artist",
            "track":"title",
            "trackduration":"#duration",
            "albumcover_medium":"album_cover_url",
            "track_url":"station_track_url"
        }   
#            "trackprogress":"#progress",
        lines = buffer.read()
        lines = lines.splitlines()

        old_track = self.song.get("station_track_url")
        
        mod = {}
        for line in lines:
            opts = line.split("=")
            if opts[0].lower() in ["trackduration","trackprogress"]:
                mod[IDS[opts[0].lower()]] = int(opts[1])*1000
            elif opts[0].lower() in IDS.keys():
                mod[IDS[opts[0].lower()]] = opts[1] 

        
        if mod.get("station_track_url") and mod.get("station_track_url") != old_track:
            ListenDB.del_property(self.song,IDS.values())
            ListenDB.set_property(self.song,mod)
            self.logdebug("Metadata new for %s",self.song.get("uri"))
            self.logdebug(" - %s",mod.get("station_track_url"))
            ListenDB.del_property(self.song,["#stream_offset"])
            self.emit("new-song",self.song)
        else:
            self.logdebug("Metadata not changed")

    def stop(self):
        try: gobject.source_remove(self.__id_metadata_monitor)
        except: pass
        if not self.song: return 

        ListenDB.del_property(self.song,["#stream_offset","album","artist","track_url","#duration","title"])
        self.song = None

    def web(self,*args,**kwargs):
        if self.song: utils.website(self.song.get("track_url"))

    def love(self,*args,**kwargs):
        url = "http://" + self.connection_info["base_url"] + "/" + self.connection_info["base_path"] + "/control.php?session=" + self.connection_info["session"] + "&command=love&debug=0"
        self.download_thread.fetch_url(url,self.__nothing_cb)

    def skip(self,*args,**kwargs):
        url = "http://" + self.connection_info["base_url"] + "/" + self.connection_info["base_path"] + "/control.php?session=" + self.connection_info["session"] + "&command=skip&debug=1"
        self.download_thread.fetch_url(url,self.__skip_cb)

    def hate(self,*args,**kwargs):
        url = "http://" + self.connection_info["base_url"] + "/" + self.connection_info["base_path"] + "/control.php?session=" + self.connection_info["session"] + "&command=ban&debug=0"
        self.download_thread.fetch_url(url,self.__nothing_cb)

    def __nothing_cb(self,buffer,*args,**kwargs):
        lines = buffer.read()
        if self.__debug:print "I:NothingCallback:",lines
    
    def __skip_cb(self,buffer,*args,**kwargs):
        lines = buffer.read()
        if self.__debug:print "I:SkipCallback:",lines

    def get_playlist(self,*args,**kwargs):
        url = "http://" + self.connection_info["base_url"] + "/" + self.connection_info["base_path"] + "/xspf.php?sk=" + self.connection_info["session"] + "&discovery=0&desktop=1.3.1.1"
        self.download_thread.fetch_url(url,self.__playlist_cb)

    def __playlist_cb(self,buffer,*args,**kwargs):
        songs = []
        try:
            root = XMLFromString(buffer.read())
        except:pass
        else:
            tracklist = root.find("trackList")
            for track in list(tracklist):
                tags = {}
                for item in list(track):
                    if item.tag == "location":
                        tags["uri"] = item.text
                        tags["station"] = LastFmPlayer.get_station_name(item.text)
                    elif item.tag == "album":
                        tags["album"] = item.text
                    elif item.tag == "image":
                        tags["album_cover_url"] = "item.text"
                    elif item.tag == "title":
                        tags["title"] = item.text
                songs.append(ListenDB.get_or_create_song( tags, "lastfmradio" , hidden=True))

        self.logdebug("%d songs found in playlist",len(songs))
        if songs: Dispatcher.cur_playlist_enqueue(songs)
        

LastFmPlayer= LastFmPlayerWrapper()

