package tests::TimeslotTest;

use strict;

use base qw/Lire::Test::TestCase tests::TestStoreFixture/;

use Lire::Timeslot;
use Lire::ReportSpec;
use Lire::Report::Group;
use Lire::Report::TableInfo;
use Lire::DlfQuery;

sub new {
    my $self = shift()->SUPER::new( @_ );

    $self->init();

    return $self;
}

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->set_up_test_schema();

    $self->set_up_tz( 'UTC' );

    $self->{'spec'} = new Lire::ReportSpec();
    $self->{'spec'}->superservice( 'test' );
    $self->{'spec'}->id( 'timeslot-test' );

    foreach my $p ( qw/3M 1w 4w 1d 4h 15m 15s/ ) {
        $self->{ $p . '_timeslot'} =
          new Lire::Timeslot( 'report_spec' => $self->{'spec'},
                              'unit' =>  $p,
                              'label' => 'Period',
                            );
    }

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub test_name {
    my $self = $_[0];

    $self->assert_equals( 'timeslot:time_start',
                          $self->{'1d_timeslot'}->name() );
}

sub test_create_categorical_info {
    my $self = $_[0];

    my $info = new Lire::Report::TableInfo();
    $self->{'1d_timeslot'}->create_categorical_info( $info );

    my @cols = $info->children();
    $self->assert_equals( 1, scalar @cols );

    my $col = $cols[0];
    $self->assert_equals( 'categorical', $col->class() );
    $self->assert_equals( 'timeslot:time_start', $col->name() );
    $self->assert_equals( 'timestamp', $col->type() );
    $self->assert_equals( 'Period', $col->label() );
}

sub test_build_query {
    my $self = $_[0];

    foreach my $t ( [ '3M', q{lr_timeslot("time-start",3,'M')} ],
                    [ '1w', 'lr_timeslot_week("time-start",1)' ] )
    {
        my $timeslot = $self->{ $t->[0] . '_timeslot' };
        $timeslot->{'field'} = 'time-start';
        my $e_query = new Lire::DlfQuery( 'test' );
        $e_query->add_aggr_field( '_lr_nrecords', 'count(*)' );
        $e_query->add_group_field( 'timeslot:time-start', $t->[1] );
        $e_query->set_sort_spec( 'timeslot:time-start' );

        my $query = new Lire::DlfQuery( 'test' );
        $timeslot->build_query( $query );
        $self->assert_deep_equals( $e_query, $query );
    }
}

sub test_create_entry {
    my $self = $_[0];

    my $info = new Lire::Report::TableInfo();
    $self->{'1d_timeslot'}->create_categorical_info( $info );

    foreach my $t ( [ '3M', 9, 'October', undef, 'M3', 3 ],
                    [ '1w', 42, '-W42',  'ISO', 'W41', 1 ],
                    [ '1w', 41, 'Week 41',  'U', 'W40', 1 ],
                    [ '1w', 0, 'Week 00',  'U', 'W0', 1 ],
                    [ '4w', 1, '-W01',  'ISO', 'W0', 4 ],
                    [ '4w', 5, '-W05',  'ISO', 'W1', 4 ],
                    [ '4w', 1, 'Week 01',  'U', 'W0', 4 ],
                    [ '1d', 2, 'Tuesday', undef, 'D2', 1 ],
                    [ '4h', 16, '16:00', undef,  '2-4', 4 ],
                    [ '15m', 7, '00:07', undef,  '1-0', 15 ],
                    [ '15m', 0, '00:00', undef,  '1-0', 15 ],
                    [ '15s', 9, '00:00:09',  undef, '0-0', 15 ],
                    [ '15s', 15, '00:00:15',  undef, '0-1', 15 ], )
    {
        my ( $period, $slot, $fmt_value, $style, $value, $mult ) = @$t;

        $self->{'cfg'}{'lr_week_numbering'} = $style
          if defined $style;

        my $timeslot = $self->{$period . '_timeslot'};
        my $dlf = { 'timeslot:time_start' => $slot };
        my $group = new Lire::Report::Group( bless( {}, 'Lire::Report::Entry'),
                                             $info );
        my $entry = $timeslot->create_entry( $group, $dlf );

        $self->assert_not_null( $entry, "create_entry() returned undef" );
        my @data = $entry->data();
        $self->assert_num_equals( 1, scalar @data );
        $self->assert_str_equals( $fmt_value, $data[0]->{'content'} );
        $self->assert_str_equals( $value, $data[0]->{'value'} );
        $self->assert_num_equals( $mult, $data[0]->{'range'} );
    }
}

sub test_create_entry_mc {
    my $self = $_[0];

    my $timeslot = $self->{'3M_timeslot'};
    my $info = new Lire::Report::TableInfo();
    $timeslot->create_categorical_info( $info );
    my $group = new Lire::Report::Group( bless( {}, 'Lire::Report::Entry'),
                                         $info );

    my $entry = $timeslot->create_entry( $group,
                                          { 'timeslot:time-start' => undef,
                                            '_lr_nrecords' => 24 } );
    $self->assert_null( $entry, "create_entry() should have returned undef" );
    $self->assert_equals( 24, $group->missing_cases() );
}

1;
