package tests::ChartTypeTest;

use strict;

use base qw/Lire::Test::TestCase/;

use Lire::Config::Build qw/ac_info/;
use Lire::Config::ChartSpec;
use Lire::ChartType;
use Lire::Config::Index;
use Lire::Config::ExecutableSpec;
use Lire::Config::ConfigSpec;
use Lire::PluginManager;
use Lire::Report::Subreport;
use Lire::Report::TableInfo;
use Lire::Utils qw/file_content tempdir/;

use Lire::Test::Mock;

use IO::Scalar;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'chart_cfg'} =
      new Lire::Config::ChartSpec( 'name' => 'cfg' )->instance();

    $self->{'cfg'}{'ploticus_path'} =
      new Lire::Config::ExecutableSpec( 'name' => 'ploticus_path' )->instance( 'value' => ac_info( 'PLOTICUS' ) );
    $self->{'cfg'}{'lr_image_format'} = 'png';
    $self->{'cfg'}{'lr_chart_font'} = 'Arial';
    $self->{'cfg'}{'_lr_config_spec'} = new Lire::Config::ConfigSpec();

    $self->{'_old_reg'} = \%Lire::Config::Index::REGISTRY;
    %Lire::Config::Index::REGISTRY = ();

    $self->set_up_plugin_mgr();
    $self->{'chart_type'} = new_proxy Lire::Test::Mock( 'Lire::ChartType' );
    $self->{'chart_type'}->set_result( 'name', 'mychart' );
    Lire::PluginManager->register_plugin( $self->{'chart_type'} )
;
    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    *Lire::Config::Index::REGISTRY = $self->{'_old_reg'};

    return;
}

sub set_up_table_info {
    my $self = $_[0];

    my $info = $self->{'table_info'} = new Lire::Report::TableInfo();
    $info->create_column_info( 'period', 'categorical', 'timestamp', 'Period' );
    my $group = $info->create_group_info( 'by_page' );
    $group->create_column_info( 'page', 'categorical', 'string',
                                'Requested Page' );
    $group->create_column_info( 'hits', 'numerical', 'int', 'Hits' );
    $group->create_column_info( 'size', 'numerical', 'int' );

    $info->compute_group_layout();

    return;
}

sub set_up_subreport {
    my $self = $_[0];

    $self->set_up_table_info();
    $self->{'subreport'} = new Lire::Report::Subreport( 'test', 'myreport' );
    $self->{'subreport'}->table_info( $self->{'table_info'} );
    my $data = [ [ '2003-10-12', undef,  9, 53 ],
                 [ undef, 'page1.html',  5, 50 ],
                 [ undef, 'page2.html',  4,  3 ],
                 [ '2003-10-13', undef,  5, 10 ],
                 [ undef, 'page3.html',  3,  6  ],
                 [ undef, 'page2.html',  1,  2  ],
                 [ undef, 'page4.html',  1,  2 ] ];
    my $group;
    foreach my $r ( @$data ) {
        my ( $period, $page, $hits, $size ) = @$r;
        if ( $period ) {
            my $entry = $self->{'subreport'}->create_entry();
            $entry->add_name( $period );
            $group = $entry->create_group();
            $group->set_summary_value( 'hits', 'value' => $hits );
            $group->set_summary_value( 'size', 'value' => $size );
        } else {
            my $entry = $group->create_entry();
            $entry->add_name( $page );
            $entry->add_value( 'value' => $hits );
            $entry->add_value( 'value' => $size );
        }
    }
    $self->{'subreport'}->finalize();
    return;
}

sub test__outputfile {
    my $self = $_[0];

    my $subreport = new Lire::Report::Subreport( 'test', 'top-dirs' );
    $subreport->id( 'top-dirs.0' );

    $self->assert_str_equals( '/tmp/top-dirs.0.png',
                              $self->{'chart_type'}->_outputfile( $self->{'chart_cfg'}, $subreport, { 'outputdir' => '/tmp', 'format' => 'png' } ) );

    $self->{'chart_cfg'}->basename( 'chart' );
    $self->assert_str_equals( './chart.jpg',
                              $self->{'chart_type'}->_outputfile( $self->{'chart_cfg'}, $subreport, { 'outputdir' => '.', 'format' => 'jpeg' } ) );
}

sub test_write_chart {
    my $self = $_[0];

    $self->set_up_table_info();
    my $subreport = new Lire::Report::Subreport( 'test', 'top-dirs' );
    $subreport->id( 'top-dirs.0' );
    $subreport->table_info( $self->{'table_info'} );
    $self->assert_dies( qr/'format' should be one of 'eps', 'gif', 'jpeg' or 'png': 'wawa'/,
                        sub { $self->{'chart_type'}->write_chart( $self->{'chart_cfg'}, $subreport, 'format' => 'wawa' ) } );

    $self->assert_dies( qr/invalid chart configuration/,
                        sub { $self->{'chart_type'}->write_chart( $self->{'chart_cfg'}, $subreport ) } );

    $self->{'chart_type'}->set_result( 'write_parameters' =>
                                       sub { $self->assert( Lire::Config::Index->has_index( 'variables' ) ? 1 : 0, 'missing variables indices' ) },
                                      'write_data' =>
                                       sub { print {$_[3]} "Error\n" },
                                       '_pl_cmdline' => 
                                       sub { "cat > $_[1]->{'errfile'}; exit 1" } );
    $self->{'chart_cfg'}->get( 'case_var' )->set( 'period' );
    $self->{'chart_cfg'}->get( 'type' )->set_plugin( $self->{'chart_type'}->name() );

    $self->assert_null( $self->{'chart_type'}->write_chart( $self->{'chart_cfg'}, $subreport, 'format' => 'jpeg' ), 'write_chart() should have returned undef' );
    $subreport->create_entry();

    $self->assert_dies( qr/Ploticus error: Error/,
                        sub { $self->{'chart_type'}->write_chart( $self->{'chart_cfg'}, $subreport ) } );

    $self->{'chart_type'}->set_result( 'write_parameters' =>
                                       sub { print {$_[2]} "Parameters\n" },
                                      'write_data' =>
                                       sub { print {$_[3]} "Data\n" },
                                       '_pl_cmdline' =>
                                       sub { "cat > $_[1]->{'outputfile'}" } );
    $self->{'chart_cfg'}->get( 'name' )->set( 'chart' );
    my $tmpdir = tempdir( $self->name() . "_XXXXXX", 'CLEANUP' => 1 );


    my $chart =
      $self->{'chart_type'}->write_chart( $self->{'chart_cfg'}, $subreport,
                                          'format' => 'jpeg',
                                          'outputdir' => $tmpdir );
    $self->assert_str_equals( "$tmpdir/chart.jpg", $chart );
    $self->assert( -f $chart ? 1 : 0, "file '$chart' not created" );
    $self->assert_str_equals( "Parameters\nData\n",
                              file_content( $chart ) );

}

sub test_write_chart_no_ploticus {
    my $self = $_[0];

    $self->{'cfg'}{'ploticus_path'}->set( '/no/such/file' );
    my $subreport = new Lire::Report::Subreport( 'test', 'top-dirs' );

    $self->assert_dies( qr/'ploticus_path' configuration variable isn't valid/,
                        sub { $self->{'chart_type'}->write_chart( $self->{'chart_cfg'}, $subreport, 'format' => 'jpeg' ) } );
}

sub test__pl_cmdline {
    my $self = $_[0];

    $self->{'chart_type'}->set_result( 'prefab' => '-prefab chron' );
    my $params = { 'outputfile' => 'plot.png',
                   'format' => 'png',
                   'font'   => 'Times New Roman',
                   'errfile' => '/tmp/errfile' };
    my $ploticus_path = $self->{'cfg'}{'ploticus_path'}->as_value();
    $self->assert_str_equals( "$ploticus_path -prefab chron data=- delim=tab header=yes -png -o 'plot.png' -font 'Times New Roman' 2> /tmp/errfile",
                              $self->{'chart_type'}->_pl_cmdline( $params ) );
    $params->{'font'} = '';
    $self->assert_str_equals( "$ploticus_path -prefab chron data=- delim=tab header=yes -png -o 'plot.png' -font 'Arial' 2> /tmp/errfile",
                              $self->{'chart_type'}->_pl_cmdline( $params ) );

    $params->{'font'} = '';
    $self->{'cfg'}{'lr_chart_font'} = '';
    $self->assert_str_equals( "$ploticus_path -prefab chron data=- delim=tab header=yes -png -o 'plot.png'  2> /tmp/errfile",
                              $self->{'chart_type'}->_pl_cmdline( $params ) );
}

sub test_write_parameters {
    my $self = $_[0];

    $self->set_up_subreport();
    $self->{'table_info'}->set_variables_indices();
    my $vars = [ $self->{'table_info'}->column_info_by_name( 'hits' ),
                 $self->{'table_info'}->column_info_by_name( 'size' ) ];
    $self->{'chart_type'}->set_result( 'get_vars', $vars );

    my $fh = new IO::Scalar();
    my $buf = $fh->sref();
    $self->{'chart_type'}->write_parameters( $self->{'chart_cfg'}, $fh );
    $self->assert_str_equals( <<EOF, $$buf );
#set x = 1
#set yrange 0
#set name = Hits
#set name2 = size
EOF
    $$buf = '';
    $self->{'chart_cfg'}->get( 'title' )->set( 'My title' );
    $self->{'chart_cfg'}->get( 'xlabel' )->set( 'X Axis' );
    $self->{'chart_cfg'}->get( 'ylabel' )->set( 'Y Axis' );
    $self->{'chart_type'}->write_parameters( $self->{'chart_cfg'}, $fh );
    $self->assert_str_equals( <<EOF, $$buf );
#set x = 1
#set yrange 0
#set title = My title
#set xlbl = X Axis
#set ylbl = Y Axis
#set name = Hits
#set name2 = size
EOF
}

sub test_write_data {
    my $self = $_[0];

    $self->set_up_subreport();
    $self->{'table_info'}->set_variables_indices();
    my $fh = new IO::Scalar();
    my $buf = $fh->sref();
    my $vars = [ $self->{'table_info'}->column_info_by_name( 'hits' ),
                 $self->{'table_info'}->column_info_by_name( 'size' ) ];
    $self->{'chart_type'}->set_result( 'get_vars', $vars );

    $self->{'chart_cfg'}->get( 'case_var' )->set( 'period' );
    $self->{'chart_type'}->write_data( $self->{'chart_cfg'},
                                       $self->{'subreport'}, $fh );
    $self->assert_str_equals( <<EOF, $$buf );
period\thits\tsize
2003-10-12\t9\t53
2003-10-13\t5\t10
EOF
    $self->assert_num_equals( 6, $self->{'chart_type'}->invocation_count( 'encode_value' ) );

    $$buf = '';
    $self->{'chart_cfg'}->get( 'case_var' )->set( 'page' );
    $self->{'chart_type'}->write_data( $self->{'chart_cfg'},
                                       $self->{'subreport'}, $fh );
    $self->assert_str_equals( <<EOF, $$buf );
page\thits\tsize
page1.html\t5\t50
page2.html\t4\t3
page3.html\t3\t6
page2.html\t1\t2
page4.html\t1\t2
EOF
}

sub test_encode_value {
    my $self = $_[0];

    my $chart = $self->{'chart_type'};
    $self->assert_str_equals( '2003-01-01',
                              $chart->encode_value( { 'type' => 'name',
                                                      'content' => '2003-01-01' } ) );
    $self->assert_num_equals( 1024,
                              $chart->encode_value( { 'type' => 'value',
                                                      'content' => '1k',
                                                      'value' => 1024
                                                    } ) );
}

1;
