/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <locale.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include "curse.h"
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"

static WINDOW	*Win;
static WINDOW	*Subwin;
static PANEL	*Panel;

static int	StockNum, FieldNum;
static int	SaveStockCursor;
static char	FieldBuf[256];

static int	Obese;
static int	PerShare;	// Data entry by pershare price

// Shares ...
#define	F1X	15
#define	F1W	7
#define	OF1X	82
#define	OF1W	7

// Total cost ...
#define	F2X	23
#define	F2W	8
#define	OF2X	90
#define	OF2W	10

// Gain ...
#define	F3X	31
#define	F3W	9
#define	OF3X	100
#define	OF3W	11

// %Gain ...
#define	OF4X	112
#define	OF4W	7

static void
display_cursor(int sn, int fn, int on)
{
	int	y, x;
	int	fx;
	int	fwid;

	if (fn == 0)
	{
		fx = Obese ? OF1X : F1X;
		fwid = Obese ? OF1W : F1W;
	}
	else
	{
		fx = Obese ? OF2X : F2X;
		fwid = Obese ? OF2W : F2W;
	}

	y = Stock[sn].y;

	if (on)
	{
		mvaddch(y, fx-1, '[');
		for (x = 0; x < fwid; ++x)
		{
			chtype	ch = mvinch(y, fx + x);
			ch |= RevOrBold;
			mvaddch(y, fx + x, ch);
		}
		addch(']');
	}
	else
	{
		mvaddch(y, fx-1, ' ');
		for (x = 0; x < fwid; ++x)
		{
			chtype	ch = mvinch(y, fx + x);
			ch &= ~RevOrBold;
			mvaddch(y, fx + x, ch);
		}
		addch(' ');
	}
}

static char *Help0[] =
{
"Enter the number of shares in your portfolio and the total cost of",
"those shares in the SHARES and TOTCOST fields.  Use the arrow keys,",
"tab, space, or enter to move between the SHARES and the TOTCOST",
"fields.  Enter short positions using a negative TOTCOST. To enter",
"by cost per share, type '*<cost-per-share>' in the TOTCOST field.",
"",
"Press 'q' to quit or 's' to save and quit.",
NULL
};

static char *Help1[] =
{
"Enter the number of shares in your portfolio and the cost per share",
"of those shares in the SHARES and SHRCOST fields.  Use the arrow keys,",
"tab, space, or enter to move between the SHARES and the SHRCOST",
"fields.  Enter short positions using a negative SHRCOST. To enter",
"by total cost, type '=<total-cost>' in the SHRCOST field.",
"",
"Press 'q' to quit or 's' to save and quit.",
NULL
};

int
holdings_mode(int stocknum, int obese, int pershare)
{
	int	cols;
	int	i;
	char	**help;

	Win = newwin(9, cols80(), 4+NumStock, 0);
	if (!Win)
	{
		// error(1, "Can't create holdings window\n");
		return 1;
	}

	cols = getmaxx(Win);

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	box(Win, 0, 0);
	mvwprintw(Win, 0, (cols - 19)/2, "Holdings Data Entry");

	Subwin = derwin(Win, getmaxy(Win) - 2, getmaxx(Win) - 2, 1, 1);
	if (!Subwin)
	{
		// error(1, "Can't create info subwindow\n");
		delwin(Win);
		Win = NULL;
		return 2;
	}

	Obese = obese;
	StockNum = stocknum;
	PerShare = pershare;
	FieldNum = 0;
	display_cursor(StockNum, FieldNum, TRUE);

	SaveStockCursor = StockCursor;
	StockCursor = -1;

	FieldBuf[0] = 0;

	help = PerShare ? Help1 : Help0;
	for (i = 0; help[i]; ++i)
		mvwprintw(Subwin, i, 5, help[i]);

	Panel = new_panel(Win);
	return (0);
}

void
display_fieldbuf(void)
{
	int	y = Stock[StockNum].y;
	int	x, w;

	if (FieldNum == 0)
	{
		x = Obese ? OF1X : F1X;
		w = Obese ? OF1W : F1W;
	}
	else
	{
		x = Obese ? OF2X : F2X;
		w = Obese ? OF2W : F2W;
	}

	if (strlen(FieldBuf) > w)
	{
		FieldBuf[w] = 0;
		beep();
	}

	mvwprintw(stdscr, y, x, "%-*.*s", w, w, FieldBuf);
}

void
set_fieldbuf(void)
{
	STOCK	*sp = &Stock[StockNum];
	int	y = sp->y;
	double	gain, shrcost;

	qfmt_init();
	if (FieldNum == 0)
	{
		if (FieldBuf[0])
			sp->nshares = atoi(FieldBuf);
		mvwprintw(stdscr, y,
			Obese ? OF1X : F1X, "%*d",
			Obese ? OF1W : F1W, sp->nshares);
	}
	else
	{
		if (FieldBuf[0])
		{
			if (FieldBuf[0] == '*')
				sp->cost = atof(FieldBuf+1) * sp->nshares;
			else if (FieldBuf[0] == '=')
				sp->cost = atof(FieldBuf+1);
			else if (PerShare)
				sp->cost = atof(FieldBuf) * sp->nshares;
			else
				sp->cost = atof(FieldBuf);
		}
		shrcost = sp->nshares ? (sp->cost/sp->nshares) : 0;
		mvwprintw(stdscr, y,
			Obese ? OF2X : F2X, "%*.2f",
			Obese ? OF2W : F2W,
			PerShare ? shrcost : sp->cost);
	}

	if (sp->cost >= 0)
		gain = sp->nshares * sp->cur.last - sp->cost;
	else
		gain = -sp->cost - sp->nshares * sp->cur.last;

	mvwprintw(stdscr, y, Obese ? OF3X : F3X,
			"%s", Obese ? dolfmt11(gain) : dolfmt9(gain));
	set_color(sp->y, Obese ? OF3X : F3X, Obese ? OF3W : F3W,
			gain, 0.0,
			COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
	if (Obese)
	{
		mvwprintw(stdscr, y, OF4X,
			"%*s", OF4W,
			pctfmt7(gain, sp->cost >= 0 ? sp->cost : - sp->cost));
		set_color(sp->y, OF4X, OF4W,
			gain, 0.0,
			COLOR_PAIR(1), A_NORMAL, COLOR_PAIR(2));
	}

	FieldBuf[0] = 0;
}

void
holdings_cursor(void)
{
	int	x;

	if (FieldNum == 0)
		x = Obese ? OF1X : F1X;
	else
		x = Obese ? OF2X : F2X;
	x += strlen(FieldBuf);

	move(Stock[StockNum].y, x);
}

int
holdings_command(int c, STREAMER sr)
{
	char	*p;
	MEVENT	m;

	switch (c)
	{
	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
	case '+': case '-': case '.':
	value:
		p = strchr(FieldBuf, 0);
		if (p == FieldBuf)
		{
		}
		*p++ = c;
		*p = 0;
		display_fieldbuf();
		display_cursor(StockNum, FieldNum, TRUE);
		break;
	case '*':
	case '=':
		if (FieldNum == 0)
		{
			display_cursor(StockNum, FieldNum, FALSE);
			set_fieldbuf();
			FieldNum = 1;
			display_cursor(StockNum, FieldNum, TRUE);
			goto value;
		}
		else if (FieldNum == 1 && FieldBuf[0] == 0)
			goto value;
		else
			beep();
		break;
	case '\b':
		p = strchr(FieldBuf, 0);
		if (p == FieldBuf)
		{
			beep();
			break;
		}
		--p;
		*p = 0;
		display_fieldbuf();
		display_cursor(StockNum, FieldNum, TRUE);
		break;
	case KEY_DL:
	case KEY_DC:
	case 127:
		FieldBuf[0] = 0;
		display_fieldbuf();
		display_cursor(StockNum, FieldNum, TRUE);
		break;
	case KEY_HOME:
		display_cursor(StockNum, FieldNum, FALSE);
		set_fieldbuf();
		StockNum = FieldNum = 0;
		display_cursor(StockNum, FieldNum, TRUE);
		break;
	case KEY_END:
		display_cursor(StockNum, FieldNum, FALSE);
		set_fieldbuf();
		StockNum = NumStock - 1;
		FieldNum = 0;
		display_cursor(StockNum, FieldNum, TRUE);
		break;
	case '\r':
	case 'l':
	case KEY_RIGHT:
	case CTRL('I'):
	case ' ':
		display_cursor(StockNum, FieldNum, FALSE);
		set_fieldbuf();
		if (++FieldNum == 2)
		{
			FieldNum = 0;
			if (++StockNum == NumStock)
				StockNum = 0;
		}
		display_cursor(StockNum, FieldNum, TRUE);
		break;
	case 'h':
	case KEY_LEFT:
		display_cursor(StockNum, FieldNum, FALSE);
		set_fieldbuf();
		if (--FieldNum < 0)
		{
			FieldNum = 1;
			if (--StockNum < 0)
				StockNum = NumStock - 1;
		}
		display_cursor(StockNum, FieldNum, TRUE);
		break;
	case 'j':
	case KEY_DOWN:
		display_cursor(StockNum, FieldNum, FALSE);
		set_fieldbuf();
		if (++StockNum == NumStock)
			StockNum = 0;
		display_cursor(StockNum, FieldNum, TRUE);
		break;
	case 'k':
	case KEY_UP:
		display_cursor(StockNum, FieldNum, FALSE);
		set_fieldbuf();
		if (--StockNum < 0)
			StockNum = NumStock - 1;
		display_cursor(StockNum, FieldNum, TRUE);
		break;

	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;
		// Ignore all clicks
		break;

	case KEY_F(11):
		print_rect_troff(2, 0,
				getbegy(Win)+getmaxy(Win) - 2, getmaxx(Win),
				NULL, "screen.tr");
		break;

	case 's':
		set_fieldbuf();
		display_cursor(StockNum, FieldNum, FALSE);
		StockCursor = SaveStockCursor;
		del_panel(Panel);
		delwin(Subwin);
		delwin(Win);
		Win = NULL;
		Subwin = NULL;
		// Let main code issue save confirmation
		return (1);

	case 033:
	case 'q':
	case 'x':
	case '#':
		set_fieldbuf();
		display_cursor(StockNum, FieldNum, FALSE);
		StockCursor = SaveStockCursor;
		del_panel(Panel);
		delwin(Subwin);
		delwin(Win);
		Win = NULL;
		Subwin = NULL;
		// Quit no save
		return (2);
	default:
		beep();
		break;
	}
	return (0);
}
