/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#include <sys/wait.h>
#include "curse.h"
#include "error.h"
#include "debug.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"
#include "alert.h"
#include "news.h"
#include "help.h"
#include <sys/socket.h>
#include "util.h"

//
// Crude alerts.  Mainly, a placeholder for a more sophisticated
// implementation someday.
//

static WINDOW	*Win;
static WINDOW	*Subwin;
static PANEL	*Panel;

static int	Cursor;
static STOCK	*Sp;

static int	NewsFd = -1;
static int	NewsPID = 0;

/*
 * Read/write alerts.  Hmm. Could grow without bound.  Later.
 */
#define		NALERT	1000

static ALERT	Alert[NALERT];
static int	NumAlert = 2;

int
alert_write(char *filename)
{
	char	*home = getenv("HOME");
	char	buf[16384];
	FILE	*fp;
	int	i;

	sprintf(buf, "%s/%s", home, filename);

	fp = fopen(buf, "w");
	if (!fp)
		return 1;

	fprintf(fp,"# Stock alerts\n");
	fprintf(fp,"#\n");
	fprintf(fp,"#SYM	en	var	op	val	action\n");

	for (i = 0; i < NumAlert; ++i)
	{
		ALERT	*ap = &Alert[i];

		if (!ap->sym[0])
			continue;
		if (!ap->act)
			continue;
		if (!ap->op)
			continue;
		if (!ap->var)
			continue;

		fprintf(fp, "%s	%d	%c	%c	%7.2f	%d\n",
				ap->sym, ap->alstate, ap->var, ap->op,
				ap->val, ap->act);
	}

	fclose(fp);

	return (0);
}

void
alert_save(void)
{
	alert_write("." PROGNAMESTR "/alerts");
}

int
alert_read(char *filename)
{
	char	*home = getenv("HOME");
	char	buf[16384];
	FILE	*fp;
	int	rc;
	ALERT	*ap;

	NumAlert = 0;
	sprintf(buf, "%s/%s", home, filename);

	fp = fopen(buf, "r");
	if (!fp)
		return 1;

	ap = &Alert[0];
	while (fgets(buf, sizeof(buf), fp))
	{
		if (NumAlert > NALERT)
			error(1, "Too many alerts\n");

		if (buf[0] == '#')
			continue;

		rc = sscanf(buf, "%s	%d	%c	%c	%f	%d",
				ap->sym, &ap->alstate, &ap->var, &ap->op,
				&ap->val, &ap->act);
		if (rc != 6)
			continue;

		// Convert old trendfund focus list alerts
		if (ap->op == 't') ap->op = 'f';
		else if (ap->op == 'T') ap->op = 'F';

		++ap;
		++NumAlert;
	}

	fclose(fp);

	return (0);
}

ALERT *
alert_new(char *sym)
{
	if (NumAlert >= NALERT)
		error(1, "Too many alerts\n");

	memset(&Alert[NumAlert], 0, sizeof(ALERT));
	strcpy(Alert[NumAlert].sym, sym);
	return &Alert[NumAlert++];
}

void
alert_bind1(STOCK *sp)
{
	int	i;

	sp->nalert = 0;
	memset(sp->alert, 0, sizeof(sp->alert));
	for (i = 0; i < NumAlert; ++i)
	{
		if (strcmp(sp->sym, Alert[i].sym))
			continue;

		sp->alert[sp->nalert++] = Alert + i;
	}
}

void
alert_bind(void)
{
	int	i;

	for (i = 0; i < NumStock; ++i)
		alert_bind1(&Stock[i]);
}

void
alert_delete_all(char *sym)
{
	int	i;

	for (i = 0; i < NumAlert; ++i)
		if (strcmp(Alert[i].sym, sym) == 0)
			strcpy(Alert[i].sym, "");
}

void
alert_delete_all_but_news(char *sym)
{
	int	i;

	for (i = 0; i < NumAlert; ++i)
		if (strcmp(Alert[i].sym, sym) == 0
			&& Alert[i].var != 'n')
			strcpy(Alert[i].sym, "");
}

/*
 * Convert a variable code letter to its long name
 */
static char *
var2str(ALERT *ap)
{
	switch (ap->var)
	{
	case 'a': return "ask";
	case 'b': return "bid";
	case 'l': return "last";
	case 'H': return "high";
	case 'L': return "low";
	case 'v': return "vol.";
	case 'n': return "news";
	default:  return "????";
	}
}

static void
display_alert_line(STOCK *sp, int num, int hilight)
{
	ALERT	*ap = sp->alert[num];
	char	*var;

	if (!ap)
		ap = sp->alert[num] = alert_new(sp->sym);

	if (hilight)
		wattrset(Subwin, RevOrBold);

	var = ap->var ? var2str(ap) : "";

	mvwprintw(Subwin, 4+num, 5,
			"[%4s]	[%c]	[%7.2f]	[%c]	[%c]	[%c]	[%c]",
			var,
			ap->op ? ap->op : ' ',
			ap->val,
			(ap->act & ALERT_TERM) ? 'X' : ' ',
			(ap->act & ALERT_EMAIL) ? 'X' : ' ',
			(ap->act & ALERT_EXT) ? 'X' : ' ',
			ap->alstate == ALERT_EN ? 'X' : ' '
		);
	wattrset(Subwin, A_NORMAL);
}

static void
get_pivots(void)
{
	char	buf[BUFSIZ];
	FILE	*fp;
	int	i;

	sprintf(buf, PROGNAMESTR ".pivot -a '%s'", Sp->sym);
	fp = popen(buf, "r");
	if (!fp)
		return;

	alert_delete_all_but_news(Sp->sym);
	while (fgets(buf, sizeof(buf), fp))
	{
		ALERT	*ap;
		int	rc;

		ap = alert_new(Sp->sym);
		rc = sscanf(buf, "%s	%d	%c	%c	%f	%d",
				ap->sym, &ap->alstate, &ap->var, &ap->op,
				&ap->val, &ap->act);
		if (rc != 6)
			continue;
	}
	pclose(fp);
	alert_bind1(Sp);
	Cursor = 0;
	for (i = 0; i < NUMALERT; ++i)
		display_alert_line(Sp, i, i == 0);
}

void
alert_popup(STOCK *sp)
{
	int	i, x, y, cols;

	Sp = sp;
	if (!sp)
		return;

	Win = bestwin(23);
	if (!Win)
		error(1, "Can't create alert window\n");

	cols = getmaxx(Win);

	wbkgd(Win, Reverse ? A_REVERSE : A_NORMAL);

	box(Win, 0, 0);

	wattrset(Win, A_BOLD);
	mvwprintw(Win, 0, cols/2 - 3 - strlen(sp->sym),
			" %s Alerts ", sp->sym);
	wattrset(Win, A_NORMAL);

	Subwin = derwin(Win, getmaxy(Win) - 2, cols - 2, 1, 1);
	if (!Subwin)
		error(1, "Can't create alert subwindow\n");

	Panel = new_panel(Win);

	wattrset(Subwin, A_BOLD);
	mvwprintw(Subwin, 1, 5,
		"Alert Conditions........		Alert Actions......    ");
	mvwprintw(Subwin, 2, 5,
		"Type	Op	Value		xTerm	Mail	Ext	Enabled");
	wattrset(Subwin, A_NORMAL);

	x = 0; y = 10;
	mvwprintw(Subwin, y++, x, "a	Alert on ask price");
	mvwprintw(Subwin, y++, x, "b	Alert on bid price");
	mvwprintw(Subwin, y++, x, "l	Alert on last price");
	mvwprintw(Subwin, y++, x, "H	Alert on high price");
	mvwprintw(Subwin, y++, x, "L	Alert on low price");
	mvwprintw(Subwin, y++, x, "v	Alert on volume");
	mvwprintw(Subwin, y++, x, "n	Alert on new news");
	mvwprintw(Subwin, y++, x, "<	Alert on less than value");
	mvwprintw(Subwin, y++, x, "=	Alert on equal to value");
	mvwprintw(Subwin, y++, x, ">	Alert on more than value");
	mvwprintw(Subwin, y++, x, "p	Set 4 alerts based on pivots");
	x = getmaxx(Subwin)/2; y = 10;
	mvwprintw(Subwin, y++, x, "NUM	Set alert value");
	mvwprintw(Subwin, y++, x, "+/-	Increase/Decrease value by 5%%");
	mvwprintw(Subwin, y++, x, "<--/-->\tIncrease/Decrease by 0.01");
	mvwprintw(Subwin, y++, x, "t	Display alert on terminal");
	mvwprintw(Subwin, y++, x, "m	Send email alert");
	mvwprintw(Subwin, y++, x, "x	Send External alert");
	mvwprintw(Subwin, y++, x, "d/DEL	Delete current alert");
	mvwprintw(Subwin, y++, x, "e	Enable/Disable current alert");
	mvwprintw(Subwin, y++, x, "s	Save and quit alert screen");
	mvwprintw(Subwin, y++, x, "q/Esc	Quit alert screen");

	Cursor = 0;
	for (i = 0; i < NUMALERT; ++i)
		display_alert_line(sp, i, i == 0);

	touchwin(Win);
}

static void
popdown(void)
{
	if (Sp)
		display_quote(&Sp->cur, 0);

	del_panel(Panel);
	delwin(Subwin);
	delwin(Win);
	Win = NULL;
}

int
alert_command(int c, STREAMER sr)
{
	static int	(*handler)(int c, STREAMER sr);
	int		rc;
	int		i, n;
	static char	collect = 0;
	static char	collectbuf[256];
	MEVENT		m;

	if (handler)
	{
		rc = (*handler)(c, sr);
		if (rc)
			handler = NULL;
		touchwin(Win);
		move(LINES-1, CursorX);
		update_panels(); refresh(); // doupdate();
		return 0;
	}

	if (collect)
	{
		int	eol = 0;

		if (c == '\r' || c == '\n')
		{
			eol = 1;
			CursorX = 0;
			blankrect(LINES-1, 0, LINES-1, COLS-1, 1);
		}
		else if (c == '\b')
		{
			char	*s;

			s = strchr(collectbuf, 0);
			if (s > collectbuf)
			{
				*--s = 0;
				--CursorX;
				mvaddch(LINES-1, CursorX, ' ');
			}
		}
		else
		{
			char s[2];
			s[0] = c; s[1] = 0;
			strcat(collectbuf, s);
			mvaddch(LINES-1, CursorX, c);
			++CursorX;
		}

		if (eol)
		{
			collect = 0;
			if (!collectbuf[0])
				return 0;
			Sp->alert[Cursor]->val = atof(collectbuf);
			display_alert_line(Sp, Cursor, 1);
			touchwin(Win);
		}
		return 0;
	}

	switch (c)
	{
	case '\f':
		move(LINES-1, CursorX);
		wrefresh(curscr);
		break;
	case KEY_DC:
	case 'd':
		Sp->alert[Cursor]->var = 0;
		Sp->alert[Cursor]->val = 0;
		Sp->alert[Cursor]->op = 0;
		Sp->alert[Cursor]->act = 0;
		Sp->alert[Cursor]->alstate = ALERT_DIS;
		goto redisp;
	case 'a':
		Sp->alert[Cursor]->var = c;
		Sp->alert[Cursor]->val = Sp->cur.ask;
		goto redisp;
	case 'b':
		Sp->alert[Cursor]->var = c;
		Sp->alert[Cursor]->val = Sp->cur.bid;
		goto redisp;
	case 'l':
		Sp->alert[Cursor]->var = c;
		Sp->alert[Cursor]->val = Sp->cur.last;
		goto redisp;
	case 'H':
		Sp->alert[Cursor]->var = c;
		Sp->alert[Cursor]->val = Sp->cur.high;
		goto redisp;
	case 'L':
		Sp->alert[Cursor]->var = c;
		Sp->alert[Cursor]->val = Sp->cur.low;
		goto redisp;
	case 'v':
		Sp->alert[Cursor]->var = c;
		Sp->alert[Cursor]->val = Sp->cur.volume;
		goto redisp;
	case 'n':
		Sp->alert[Cursor]->var = c;
		Sp->alert[Cursor]->op = '=';
		Sp->alert[Cursor]->val = 0.00;
		goto redisp;
	case 'f': case 'F':
	case '<': case '=': case '>':
		Sp->alert[Cursor]->op = c;
		goto redisp;
	case 'e':
		if (!Sp->alert[Cursor]->act)
			Sp->alert[Cursor]->act |=  ALERT_TERM | ALERT_EXT;
		if (Sp->alert[Cursor]->alstate == ALERT_EN)
			Sp->alert[Cursor]->alstate = ALERT_DIS;
		else
			Sp->alert[Cursor]->alstate = ALERT_EN;
		goto redisp;
	case 'm':
		if (Sp->alert[Cursor]->act & ALERT_EMAIL)
			Sp->alert[Cursor]->act &=  ~ALERT_EMAIL;
		else
			Sp->alert[Cursor]->act |=  ALERT_EMAIL;
		goto redisp;
	case 't':
		if (Sp->alert[Cursor]->act & ALERT_TERM)
			Sp->alert[Cursor]->act &=  ~ALERT_TERM;
		else
			Sp->alert[Cursor]->act |=  ALERT_TERM;
		goto redisp;
	case 'x':
		if (Sp->alert[Cursor]->act & ALERT_EXT)
			Sp->alert[Cursor]->act &=  ~ALERT_EXT;
		else
			Sp->alert[Cursor]->act |=  ALERT_EXT;
		goto redisp;
	case '-':
		Sp->alert[Cursor]->val *= 0.95;
		goto redisp;
	case '+':
		Sp->alert[Cursor]->val *= 1.05;
		goto redisp;
	case KEY_LEFT:
		if (Sp->alert[Cursor]->val)
			Sp->alert[Cursor]->val -= 0.01;
		goto redisp;
	case KEY_RIGHT:
		Sp->alert[Cursor]->val += 0.01;
		goto redisp;

	redisp:
		display_alert_line(Sp, Cursor, 1);
		touchwin(Win);
		break;

	case 'p':
		get_pivots();
		touchwin(Win);
		break;

	case '0': case '1': case '2': case '3': case '4':
	case '5': case '6': case '7': case '8': case '9':
	case '.':
		collect = '0';
		collectbuf[0] = c;
		collectbuf[1] = 0;
		CursorX = 7+1;
		mvprintw(LINES-1, 0, "Value: %c                           ", c);
		break;

	case KEY_UP: case 'k':
		if (Cursor)
		{
			display_alert_line(Sp, Cursor, 0);
			--Cursor;
			display_alert_line(Sp, Cursor, 1);
			touchwin(Win);
		}
		else
			beep();
		break;
	case KEY_DOWN: case 'j':
		if (Cursor < NUMALERT-1)
		{
			display_alert_line(Sp, Cursor, 0);
			++Cursor;
			display_alert_line(Sp, Cursor, 1);
			touchwin(Win);
		}
		else
			beep();
		break;
	default:
		beep();
		break;

	case KEY_MOUSE:
		if (getmouse(&m) != OK)
			break;

		// Ignore clicks in our window
		// TODO: set field values with clicks
		if (m.y >= getbegy(Win)
			&& m.y < getbegy(Win) + getmaxy(Win))
			break;

		// popdown and reprocess clicks in main window
		// TODO: change displayed alert!
		if (ungetmouse(&m) == OK)
			Ungetch = 1;
		popdown();
		return 2;

	case KEY_F(11):
		print_rect_troff(getbegy(Win), getbegx(Win),
				getmaxy(Win), getmaxx(Win),
				NULL, "screen.tr");
		break;

	case 's':
		// Figure out how many alerts we have now
		for (n = i = 0; i < NUMALERT; ++i)
			if (Sp->alert[i]->var)
				n = i + 1;
		Sp->nalert = n;

		alert_save();
		alert_news_startpolling();
		popdown();
		return 2;
	case 'q':
	case 033:
		popdown();
		return 2;
	}
	return 0;
}

static void
alert_email(STOCK *sp, ALERT *ap)
{
	char	buf[512];

	if (sp->alerting & 4)
	{
		sp->alerting &= ~4;

		sprintf(buf, "mutt -s 'NEWS: %s' '%s' "
			"</dev/null "
			">/dev/null "
			"2>/dev/null ",
			sp->sym,
			get_rc_value(RcFile, "alert_mail")
			);
	}
	else
		sprintf(buf, "mutt -s 'ALERT: %s %s %c %.2f' '%s' "
			"</dev/null "
			">/dev/null "
			"2>/dev/null ",
			sp->sym,
			var2str(ap),
			ap->op,
			ap->val,
			get_rc_value(RcFile, "alert_mail")
			);
	system(buf);
}

static void
alert_ext(STOCK *sp, ALERT *ap)
{
	char	buf[512];
	char	*prog = get_rc_value(RcFile, "alert_ext");
	char	*synth = get_rc_value(RcFile, "alert_synth");

	sp->alerting &= ~4;

	if (DemoTime || prog == NULL || prog[0] == 0)
		prog = "linuxtrade.audio";

	// progme symbol last volume var op val
	sprintf(buf, "%s -s'%s' '%s' %.2f %ld '%s' '%c' %.2f",
			prog,
			synth,
			sp->sym,
			sp->cur.last,
			sp->cur.volume,
			var2str(ap),
			ap->op,
			ap->val
			);
	if (sp->comment[0])
	{
		strcat(buf, " '");
		strcat(buf, sp->comment);
		strcat(buf, "'");
	}
	strcat(buf, " </dev/null >/dev/null 2>&1 &");
	system(buf);
}

void
alert_check(STOCK *sp)
{
	int	i;
	ALERT	*ap;
	float	var;
	int	bool;
	time_t	now;
	struct tm *tm;
	int	dowrite = 0;

	for (i = 0; i < NUMALERT; ++i)
	{
		ap = sp->alert[i];
		if (!ap || ap->alstate != ALERT_EN || ap->var == 'n')
			continue;

		switch (ap->var)
		{
		case 'a':	var = sp->cur.ask; break;
		case 'b':	var = sp->cur.bid; break;
		case 'l':	var = sp->cur.last; break;
		case 'L':	var = sp->cur.low; break;
		case 'H':	var = sp->cur.high; break;
		case 'v':	var = sp->cur.volume; break;
		default:	continue;
		}

		// Hack bug fix
		if (var == 0)
			continue;

		switch (ap->op)
		{
		case '<':	bool = var < ap->val; break;
		case '>':	bool = var > ap->val; break;
		case '=':	bool = var = ap->val; break;
		default:	continue;
		case 'f':
				time(&now); tm = localtime(&now);
				if (tm->tm_hour == 9)
				{
					if (var < ap->val)
					{
						ap->val = var;
						dowrite = 1;
					}
					continue;
				}
				else
					bool = var <= ap->val;
				break;
		case 'F':
				time(&now); tm = localtime(&now);
				if (tm->tm_hour == 9)
				{
					if (var > ap->val)
					{
						ap->val = var;
						dowrite = 1;
					}
					continue;
				}
				else
					bool = var >= ap->val;
				break;
		}

		if (!bool)
			continue;

		ap->alstate = ALERT_TRIG;	// Alerts are one-shot
		if (Win && sp == Sp)
		{
			display_alert_line(Sp, Cursor, i == Cursor);
			touchwin(Win);
		}

		if (ap->act & ALERT_EMAIL)
			alert_email(sp, ap);
		if (ap->act & ALERT_EXT)
			alert_ext(sp, ap);
		if (ap->act & ALERT_TERM)
			sp->alerting = 1;

		dowrite = 1;
	}

	if (dowrite)
		alert_save();
}

void
alert_comment(STOCK *sp, int width)
{
	int	i;
	ALERT	*ap;
	static char buf[128];
	int	len;
	int	left = width;
	int	blank = 0;

	for (i = 0; i < NUMALERT; ++i)
	{
		ap = sp->alert[i];
		if (!ap || ap->alstate == ALERT_DIS || ap->var == 'n')
			continue;
		if (ap->val == 0)
			continue;

		switch (ap->op)
		{
		case '<':
		case 'f':
			if (ap->alstate == ALERT_TRIG)
				attrset(A_NORMAL|COLOR_PAIR(2));
		display:
			len = sprintf(buf, "%s%c%.2f",
					blank ? " " : "", ap->op, ap->val);
			buf[left] = 0;
			printw("%s", buf);
			blank = 1;
			left -= strlen(buf);
			attrset(A_NORMAL);
			break;
		case '>':
		case 'F':
			if (ap->alstate == ALERT_TRIG)
				attrset(A_NORMAL|COLOR_PAIR(1));
			goto display;
			break;
		case '=':
			if (ap->alstate == ALERT_TRIG)
				attrset(A_BOLD);
			goto display;
			break;
		}
	}
	if (left > 0)
		printw("%*s", left, "");
}

int
alert_news_fd(void)
{
	return NewsFd;
}

void
alert_news_symbol(char *sym)
{
	int	i;
	STOCK	*sp;
	int	alerted;

	sp = find_stock(sym);
	if (!sp)
		return;

	sp->alerting |= 6;

	alerted = 0;
	for (i = 0; i < NUMALERT; ++i)
	{
		ALERT	*ap = sp->alert[i];

		if (ap && ap->var == 'n')
		{
			alerted = 1;
			if (ap->act & ALERT_EMAIL)
				alert_email(sp, ap);
			if (ap->act & ALERT_EXT)
				alert_ext(sp, ap);
			if (ap->act & ALERT_TERM)
				display_alert(sp, TRUE);
		}
	}

	if (!alerted)
		display_alert(sp, TRUE);
}

void
alert_news_data(void)
{
	char	buf[512];
	int	len;
	char	*timestr;
	char	*idstr;
	char	*headline;
	char	*sym;
	time_t	utime;

	if (NewsFd < 0)
		return;

	len = read(NewsFd, buf, sizeof(buf));
	if (len <= 0)
	{
		NewsFd = -1;
		kill(NewsPID, SIGTERM);
		NewsPID = 0;
		return;
	}

	buf[len-1] = 0;

	// fprintf(stderr, "len=%d <%s>\n", len, buf);

	sym = buf;
	timestr = strchr(buf, '\t');
	if (!timestr)
		return;
	*timestr++ = 0;

	utime = strtoul(timestr, 0, 0);

	idstr = strchr(timestr, '\t');
	if (!idstr)
		return;
	*idstr++ = 0;

	headline = strchr(idstr, '\t');
	if (!headline)
		return;
	*headline++ = 0;

	display_scrollnews(sym, "", headline, utime, "" /*src*/, 0);
	save_headline(sym, idstr, headline, utime, "", NULL, 0);
	alert_news_symbol(sym);
}

static int
poll_one(char *sym, char *buf, int bufsiz)
{
	FILE	*fp;
	char	cmd[512];
	char	*url = "http://fast.quote.com/fq/quotetracker/headlines?"
			"symbols=";

	if (1)
		sprintf(cmd, "%s \"%s%s\" 2>/dev/null "
				"| head -n1 2>/dev/null", SUCKURL, url, sym);
	else
		sprintf(buf, "%s\t11/20/01\t10:00:00\t1234\tdsds\n", sym);

	fp = popen(cmd, "r");
	if (!fp)
		return 1;

	fgets(buf, bufsiz, fp);

	pclose(fp);

	// error(0, "<%s>\n", buf);
	return (0);
}

int
alert_news_poller(int argc, char *argv[])
{
	int		n;
	int		rc;
	int		tperiod = 60 * 10;
	int		nperiod;
	char		buf[512];
	char		datestr[512];
	char		timestr[512];
	char		headline[512];
	int		id;
	static int	lastid[100];

	--argc;
	++argv;

	// Do a fast first scan
	nperiod = 1;
	for (n = -1;;)
	{
		sleep(nperiod);
		if (++n >= argc)
		{
			nperiod = tperiod / argc;
			if (nperiod == 0)
				nperiod = 1;
			n = 0;
		}

		if (getppid() <= 1)
			return (0);

		if (poll_one(argv[n], buf, sizeof(buf)))
			continue;

		//
		//INTC	12/18/2002	13:00	30603352	
		//BusinessWire	786	Intel Appoints 13 New Vice Presidents
		//
		rc = sscanf(buf,
				"%*s	%s	%s	%d	"
				"%*[^\t]	%*s	%[^\n]",
				datestr, timestr, &id, headline);
		if (rc != 4)
			continue;

		if (!lastid[n])
			lastid[n] = id;
		if (Debug >= 6)
			lastid[n] = 0;
		if (id != lastid[n])
		{
			debug(1, "NEWS: %s lastid=%d id=%d buf=<%s>\n",
					argv[n], lastid[n], id, buf);

			rc = printf("%s\t%lu\t%d\t%s\n",
					argv[n],
					datetime2unix(datestr, timestr),
					id, headline);
			if (rc <= 0)
				return (0);
			if (fflush(stdout) < 0)
				return (0);
			lastid[n] = id;
		}
	}
}

void
alert_news_startpolling(void)
{
	int		i, j;
	char		list[NUMSTOCK * (SYMLEN+1) + 1];
	static char	oldlist[NUMSTOCK * (SYMLEN+1) + 1];
	pid_t		pid;
	int		rc;
	int		fds[2];
	#define NARG	100
	int		argc;
	char		*argv[NARG];
	char		*s, *p;
	int		news_alerts;

	//
	// Get global news alerts setting
	// 0 - always off
	// 1 - on only if turned on in each per-stock alert menu
	// 2 - on for all stocks in current portfolio
	//
	news_alerts = atoi(get_rc_value(RcFile, "news_alerts"));
	if (news_alerts == 0)
		return;

	strcpy(list, "");

	for (i = 0; i < NumStock; ++i)
	{
		if (news_alerts == 2)
		{
			if (list[0])
				strcat(list, " ");
			strcat(list, Stock[i].sym);
			continue;
		}

		for (j = 0; j < NUMALERT; ++j)
		{
			ALERT	*ap = Stock[i].alert[j];

			if (news_alerts == 1)
			{
				if (!ap || ap->alstate != ALERT_EN
						|| ap->var != 'n' || !ap->act)
					continue;
			}

			if (list[0])
				strcat(list, " ");
			strcat(list, ap->sym);
		}
	}

	// See if no changes needed
	if (strcmp(oldlist, list) == 0 && NewsPID)
		return;

	// If changes needed, kill old poller
	if (strcmp(oldlist, list) && NewsPID)
	{
		int	waitpid;

		kill(NewsPID, SIGTERM);
		wait(&waitpid);

		NewsPID = 0;
		NewsFd = -1;
		strcpy(oldlist, "");
	}

	if (!list[0])
		return;

	debug(1, "alert_news: Start polling '%s'\n", list);

	strcpy(oldlist, list);

	rc = pipe(fds);
	if (rc == -1)
		return;

	// Flush stderr/stdout so we don't get stale data in the child
	fflush(stdout);
	fflush(stderr);

	pid = fork();
	if (pid < 0)
	{
		// Couldn't fork
		close(fds[0]);
		close(fds[1]);
		error(1, "Couldn't fork new poller\n");
		return;
	}

	if (pid)
	{
		close(fds[1]);
		NewsFd = fds[0];
		NewsPID = pid;
		return;
	}

	/*
	 * Child...
	 */
	close(0);
	dup2(fds[1], 1);
	for (i = 3; i < 50; ++i)
		close(i);

	argv[0] = "poller";
	for (i = 1, s = list; (p = strtok(s, " ")); ++i, s = NULL)
	{
		if (i == NARG)
			break;
		argv[i] = p;
	}
	argc = i;

	alert_news_poller(argc, argv);
	exit (0);
}

void
alert_news_stoppolling(void)
{
	if (NewsPID)
	{
		int	waitpid;

		debug(1, "alert_news: Stop polling\n");

		kill(NewsPID, SIGTERM);
		NewsPID = 0;
		NewsFd = -1;
		wait(&waitpid);
	}
}
