#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <misc.h>
#include "dnsconf.h"
#include "internal.h"
#include "../paths.h"
#include "dnsconf.m"
#include <dialog.h>
#include <netconf.h>
#include <hostinfo.h>

DNSCONF_HELP_FILE help_dnsedit ("edit");
static DNSCONF_HELP_FILE help_rename("rename");
static DNSCONF_HELP_FILE help_reverse("reverse");
static DNSCONF_HELP_FILE help_conflict("conflict");


/*
	Extract the non empty IP number from tba and place them in tbip.
	Return the number extracted
*/
int dnsrecs_tbip (
	IP_ADDRS &tba,
	const char *tbip[])
{
	int nbip = 0;
	for (int i=0; i<tba.getnb(); i++){
		SSTRING *a = tba.getitem(i);
		if (!a->is_empty()){
			tbip[nbip++] = a->get();
		}
	}
	return nbip;
}

/*
	Update all record associated with a name
	Return -1 if full was not part of any reverse domain.
*/
PRIVATE int DNS::update_one(
	SSTRING &full,
	IP_ADDRS &tba,
	SSTRINGS &tbattls,
	SSTRINGS &tbns,
	SSTRINGS &tbnsttls,
	SSTRINGS &tbmx,
	SSTRINGS &tbmxttls,
	SSTRING &cname,
	char doreverse[4])
{
	const char *tbip[tba.getnb()];
	int nbip = dnsrecs_tbip(tba,tbip);
	int matchrev;
	set (full.get(),tbip,nbip,tbattls,doreverse,matchrev);
	setns (full,tbns,tbnsttls);
	setmx (full,tbmx,tbmxttls);
	setcname (full,cname);
	return matchrev != nbip ? -1 : 0;
}
PRIVATE int DNS::update_one(
	SSTRING &full,
	IP_ADDRS &tba,
	SSTRINGS &tbattls,
	SSTRINGS &tbns,
	SSTRINGS &tbnsttls,
	SSTRINGS &tbmx,
	SSTRINGS &tbmxttls,
	SSTRING &cname)
{
	char doreverse[4];
	memset (doreverse,1,sizeof(doreverse));
	return update_one (full,tba,tbattls,tbns,tbnsttls,tbmx,tbmxttls,cname,doreverse);
}

/*
	Collect all IP address in use in this DNS
*/
PUBLIC void DNS::getalladr (IP_ADDRS &adrs)
{
	primarys.getalladr(adrs);
	primarys_rev.getalladr(adrs);
}

static struct{
	char ip[10];
	char ip_ttl[10];
	char ns[10];
	char ns_ttl[10];
	char mx[10];
	char mx_ttl[10];
} regkeys;

PUBLIC void DNS::setip_in_dia(
	DIALOG &dia,
	int level,
	IP_ADDRS &tba,
	SSTRINGS &ttls,
	const char *title)
{
	IPMAPS maps;
	IP_ADDRS adrs;
	getalladr (adrs);
	maps.setuse (adrs);
	dia.auto_newline(false);
	dia.newf_title (title,level,"",title);
	const char *t1 = MSG_U(F_IP,"IP number");
	const char *t2 = MSG_R(F_TTL);
	if (dialog_mode == DIALOG_GUI){
		// Put a heading
		dia.newf_info (NULL,t1);
		dia.newf_info (NULL,t2);
		dia.newline();
		t1 = NULL;
		t2 = NULL;
	}
	for (int i=0; i<tba.getnb(); i++){
		IP_ADDR *a = tba.getitem(i);
		FIELD *f;
		if (maps.getnb()==0){
			f = dia.newf_str (t1,*a);
		}else{
			FIELD_COMBO *comb = dia.newf_combo (t1,*a);
			maps.setcombo(comb);
			f = comb;
		}
		f->set_registry_key (&regkeys.ip[i]);
		f = dia.newf_str (t2,*ttls.getitem(i),13);
		f->set_registry_key (&regkeys.ip_ttl[i]);
		dia.newline();
	}
	dia.auto_newline(true);
}

PUBLIC void DNS::setip_in_dia(
	DIALOG &dia,
	IP_ADDRS &tba,
	SSTRINGS &ttls)
{
	setip_in_dia (dia,1,tba,ttls,MSG_U(F_IPADRS,"IP addrs."));
}
/*
	Edit the reverse mapping of a host not part of any domain on this
	server
*/
PRIVATE void DNS::editone_rev(SSTRING &full)
{
	IP_ADDRS tba;
	SSTRING absfull (full);
	dns_cnv2abs (absfull);
	SSTRINGS ttls;
	for (int i=0; i<primarys_rev.getnb(); i++){
		PRIMARY *pri = primarys_rev.getitem(i);
		RECORDS recs;
		pri->locate_rev (absfull.get(),tba);
	}
	while (tba.getnb()<4){
		tba.add (new IP_ADDR);
		ttls.add (new TIMESTR);
	}
	DIALOG dia;
	dia.newf_str (MSG_U(F_HOST,"host name"),full);
	setip_in_dia (dia,tba,ttls);
	int nof = 0;
	while (1){
		if (dia.edit(MSG_U(T_IPREVERSE,"Host address")
			,MSG_U(I_IPREVERSE,"You can enter multiple IP numbers\n"
				"for one host. These IP numbers will be used to lookup\n"
				"a host name from its IP number. Another DNS should\n"
				"contain the official definition of this host.")
			,help_reverse
			,nof)==MENU_ACCEPT){
			SSTRINGS tbempty;
			SSTRING empty;
			if (update_one (full,tba,ttls,tbempty,tbempty,tbempty,tbempty,empty)==-1){
				xconf_error (MSG_U(E_NOMATCH
					,"The IP numbers you have entered was not part\n"
					 "of any reverse mapping pseudo-domain"));
			}else{
				write();
				break;
			}
		}else{
			break;
		}
	}
}

/*
	Check if a name is lexically acceptable
*/
void dns_lexcheck (
	SSTRING &s,
	int &nof,
	int new_nof,
	int &err,
	bool allow_space)
{
	if (!err && !s.is_empty()){
		const char *start = s.get();
		const char *pt = start;
		if (*pt == '*') pt++;	// Accept the *, needed for wild card MX
		char space = allow_space ? ' ' : '\0';
		while (*pt != '\0'){
			if (!isdigit(*pt)
				&& !isalpha(*pt)
				&& *pt != '-'
				&& *pt != '.'
				&& *pt != space){
				xconf_error (MSG_U(E_LEXDNS
					,"Invalid character in name\n"
					 "\t%s\n"
					 "only digits, letter, - and . are allowed\n"
					 "May also start with a *")
					,start);
				err = 1;
				nof = new_nof;
				break;
			}
			pt++;
		}
	}
}

/*
	Check if a set of names are lexically acceptable
*/
void dns_lexcheck (
	SSTRINGS &tb,
	int &nof,
	int new_nof,
	int &err,
	bool allow_space)
{
	for (int i=0; i<tb.getnb(); i++){
		dns_lexcheck (*tb.getitem(i),nof,new_nof++,err,allow_space);
	}
}

/*
	Check if there is duplicate usage of some IPs
	Return -1 if the user do not wish to continue.
*/
PRIVATE int DNS::check_dup(
	SSTRING &full,
	IP_ADDRS &tba,
	char doreverse[])
{
	int ret = 0;
	/* #Specification: DNS / domain / IP conflict
		We never do the reverse mapping on domain name so the
		IP allocation conflict detection is not used when
		updating the IP associated with a domain.

		If there is a conflict, we popup a dialog (several name
		pointing to the same IP) so the user can tell which one
		should get the reverse pointer.

		Before poping the dialog, we check if we indeed have a reverse
		zone handling this IP number.
	*/
	if (locate_domain(full.get())==NULL){
		const char *tbip[tba.getnb()];
		int nbip = dnsrecs_tbip(tba,tbip);
		SSTRINGS tbhosts[nbip];
		bool conflict = false;
		for (int i=0; i<nbip; i++){
			doreverse[i] = 1;
			SSTRINGS *hosts = tbhosts + i;
			locate_ip (tbip[i],*hosts);
			// We remove blindly this host from the list found.
			// If the list is not empty, then there is another host
			// using this IP
			hosts->remove_del (hosts->lookup(full.get()));
			// We also remove domains which may have a default IP
			// and this is normal
			for (int h=0; h<hosts->getnb(); h++){
				const char *hh = hosts->getitem(h)->get();
				if (locate_domain(hh)!=NULL){
					// This is a domain
					hosts->remove_del (h);
					h--;
				}
			}			
			if (hosts->getnb() > 0){
				// Ok, we have a conflict, but we have to check if there
				// is a reverse zone handling this IP. If not, no need
				// to care.
				IP_ADDR ipa;
				ipa.setfrom(tbip[i]);
				char revdom[40];
				ipa.setrev(revdom);
				FQHOST rfq (revdom);
				char left[100];
				if (primarys_rev.getitem(rfq,left) != NULL){
					doreverse[i] = 0;
					conflict = true;
				}
			}
		}
		if (conflict){
			/* #Specification: DNS management / Two host sharing one IP /strategy
				Linuxconf detect if one is trying to use an IP already
				allocated on this DNS. This is done for each of the four
				possible IPs.

				When there is a conflict, linuxconf pops a dialog allowing one
				to select which one of the IP will be the "main" A record
				and will be used to generate the reverse mapping

				IP per IP, the user may select the proper behavior.

				In general, it is a bad practice to have multiple A record
				pointing to the same IP. There is an exception (managed
				by linuxconf) for the defaults IPs of a domain.
			*/	
			DIALOG dia;
			for (int i=0; i<nbip; i++){
				dia.newf_chk (tbip[i],doreverse[i],i>0 ? "" : MSG_U(I_DOREV,"Do reverse mapping on this IP"));
				SSTRINGS *hosts = tbhosts + i;
				for (int h=0; h<hosts->getnb(); h++){
					SSTRING *ho = hosts->getitem(h);
					dia.newf_str (h==0 ? MSG_U(F_ALLOCTO,"Allocated to")
									   : ""
						,*ho);
					dia.set_lastreadonly();
				}
			}
			MENU_STATUS code = dia.edit (MSG_U(T_IPCONFLICT,"IP allocation conflict resolution")
				,MSG_U(I_IPCONFLICT,"The IPs you have allocated are already\n"
					"in use in this DNS.\n"
					"You should pick either different IPs\n"
					"    (linuxconf supply available ones)\n"
					"Or you must specify which IP will be the \"main\" IP\n"
					"and will receive the reverse mapping\n")
					,help_conflict);
			if (code == MENU_ESCAPE || code == MENU_CANCEL){
				ret = -1;
			}
		}
	}
	return ret;
}


/*
	Check if the current record conflict with the basic host definition
	Return -1 if the user do not wish to continue.
*/
static int check_basicconflict(
	SSTRING &full,
	IP_ADDRS &tba)
{
	int ret = 0;
	HOSTINFO info;
	if (netconf_loadinfos(info)!=-1){
		bool hostname_same = info.hostname.cmp(full)==0;
		const int nbdev = info.nbdev;
		const char *tbip[nbdev];
		int nbip = 0;
		for (int i=0; i<nbdev; i++){
			INTER_INFO *itf = &info.a[i];
			if (!itf->ipaddr.is_empty()){
				const char *ip = itf->ipaddr.get();
				if ((itf->name.is_empty() && hostname_same)
					|| itf->name.cmp(full)==0){
					tbip[nbip++] = ip;
				}
			}
		}
		if (nbip > 0){
			bool conflict = false;
			const char *tbaip[tba.getnb()];
			int nba = dnsrecs_tbip (tba,tbaip);
			if (nbip != nba){
				conflict = true;
			}else{
				int found = 0;
				for (int i=0; i<nba; i++){
					const char *ip = tbaip[i];
					for (int j=0; j<nbip; j++){
						if (strcmp(ip,tbip[j])==0){
							found++;
							break;
						}
					}
				}
				if (found != nbip) conflict = true;
			}
			char buf[2000];
			snprintf (buf,sizeof(buf)-nbip*30
				,MSG_U(I_BASICCONFLICT
					,"The current record is conflicting with\n"
					 "with the basic host information screen.\n"
					 "Either fix this screen or the basic host information\n"
					 "\n"
					 "The basic host information screen associates the following\n"
					 "IP numbers to the host %s\n")
				,full.get());
			for (int i=0; i<nbip; i++){
				strcat (buf,"\n    ");
				strcat (buf,tbip[i]);
			}
			strcat (buf,"\n\n");
			strcat (buf,MSG_U(Q_UPDATEANYWAY,"Do you still want to update the DNS ?"));
			if (conflict
				&& dialog_yesno(MSG_U(Q_CONFIRM,"Please confirm")
					,buf
					,help_nil)!=MENU_YES) ret = -1;
		}
	}
	return ret;
}

/*
	Define the field for NS or MX records
*/
static void dnsrecs_setnsdia (
	DIALOG &dia,
	const char *ftitle,
	SSTRINGS &tb,
	SSTRINGS &ttls,
	const char *reg_key,		// For the virtual registry
	const char *reg_ttl_key)
{
	const char *fttl = MSG_R(F_TTL);
	if (dialog_mode == DIALOG_GUI){
		dia.newf_info (NULL,ftitle);
		dia.newf_info (NULL,fttl);
		dia.newline();
		ftitle = NULL;
		fttl = NULL;
	}
	for (int i=0; i<tb.getnb(); i++){
		FIELD *f = dia.newf_str (ftitle,*(tb.getitem(i)));
		f->set_registry_key (reg_key+i);
		f = dia.newf_str (fttl,*(ttls.getitem(i)),13);
		f->set_registry_key (reg_ttl_key+i);
		dia.newline();
	}
}

/*
	Edit a host or sub-domain information.
	 -A and PTR record) globally.
	 -NS record
	 -CNAME
	 -MX

	All in a single dialog
*/
PUBLIC void DNS::editone(const char *name)
{
	FQHOST fq (name);
	SSTRING full;
	fq.formatfull(full);
	PRIMARY *pri = primarys.getitem(fq,NULL);
	if (pri == NULL){
		editone_rev (full);
	}else{
		SSTRING original_name (full);
		DIALOG dia;
		dia.newf_str (MSG_U(F_HOST_DOM,"host or sub-domain"),full);
		SSTRINGS tbns,tbnsttls;
		pri->getns(full,tbns,tbnsttls);
		primary_addblank (tbns,tbnsttls,3);
		SSTRINGS tbmx,tbmxttls;
		pri->getmx(full,tbmx,tbmxttls);
		primary_addblank (tbmx,tbmxttls,3);
		IP_ADDRS tba;
		SSTRINGS tbattls;
		pri->geta(full,tba,tbattls);
		while (tba.getnb()<4){
			tba.add (new IP_ADDR);
			tbattls.add (new TIMESTR);
		}
		SSTRING cname;
		pri->getcname (full,cname);
		int start_cn = dia.getnb();
		dia.newf_str (MSG_U(F_NICKNAME,"nick name for (CNAME)"),cname);

		setip_in_dia(dia,tba,tbattls);
		dia.auto_newline(false);
		dia.newf_title (MSG_U(F_DNSADV,"Name servers (NS)"),1,"",MSG_R(F_DNSADV));
		int start_ns = dia.getnb();
		dnsrecs_setnsdia (dia,MSG_U(F_SERVER,"Server"),tbns,tbnsttls,regkeys.ns,regkeys.ns_ttl);

		dia.newf_title (MSG_U(F_EMAILADV,"Mail servers (MX)"),1,""
			,MSG_R(F_EMAILADV));
		int start_mx = dia.getnb();
		dnsrecs_setnsdia (dia,MSG_R(F_SERVER),tbmx,tbmxttls,regkeys.mx,regkeys.mx_ttl);
		int nofield = 0;
		while (1){
			MENU_STATUS code = dia.edit(MSG_U(T_HOSTINFO,"Host information")
				,MSG_U(I_HOSTINFO,"Enter a host name + domain\n"
				 "and its IP numbers (at least one)\n"
				 "and all DNS's table will be updated\n")
				,help_dnsedit
				,nofield
				,MENUBUT_ACCEPT|MENUBUT_CANCEL|MENUBUT_DEL);
			if (code == MENU_CANCEL || code == MENU_ESCAPE){
				break;
			}else if (code == MENU_DEL){
				if (dialog_yesno(MSG_R(Q_CONFIRM)
					,MSG_U(I_CONFIRM,"Are you sure you want to\n"
					 "delete this entry")
					,help_dnsedit)
					== MENU_YES){
					SSTRINGS tbempty;
					SSTRING  empty;
					IP_ADDRS noip;
					update_one (full,noip,tbempty,tbempty,tbempty,tbempty,tbempty,empty);
					write();
					break;
				}
			}else{
				int err = 0;
				if (!cname.is_empty()){
					for (int i=0; i<tba.getnb(); i++){
						if (!tba.getitem(i)->is_empty()){
							xconf_error (MSG_U(E_NOIPWCNAME,"No IP number allowed with a nickname"));
							nofield = start_cn;
							err = 1;
							break;
						}
					}
				}
				dns_lexcheck (full,nofield,0,err,false);
				dns_lexcheck (cname,nofield,start_cn,err,false);
				dns_lexcheck (tbns,nofield,start_ns,err,false);
				dns_lexcheck (tbmx,nofield,start_mx,err,true);
				if (!err){
					if (full.cmp(original_name)!=0){
						char msg[2000];
						sprintf (msg
							,MSG_U(Q_RENAME
								,"You are renaming the host or domain\n"
								 "    %s\n"
								 "to\n"
								 "    %s\n"
								 "\n"
								 "Do you want to delete the informations\n"
								 "associated with the original name ?")
							,original_name.get(),full.get());
						if (dialog_yesno(MSG_U(T_RENAME,"Renaming")
							,msg,help_rename) == MENU_YES){
							SSTRINGS tbempty;
							SSTRING  empty;
							IP_ADDRS noip;
							update_one (original_name,noip,tbempty,tbempty,tbempty,tbempty,tbempty
								,empty);
						}
					}
					char doreverse[tba.getnb()];
					if (check_basicconflict(full,tba) != -1
						&& check_dup(full,tba,doreverse)!=-1){
						update_one (full,tba,tbattls,tbns,tbnsttls,tbmx,tbmxttls,cname,doreverse);
						write();
						break;
					}
				}
			}
		}
	}
}

/*
	Edit the host information (A, PTR, NS and MX records) globally.
*/
PUBLIC void DNS::editrecs(const char *preset)
{
	/* #Specification: dnsconf / host editition
		The user is prompted to enter a host or domain name.
		The system query all configured information about
		this host. All this is displayed in a form. The user
		can correct and add information.
	*/
	while (1){
		SSTRING name (preset);
		MENU_STATUS code;
		{
			// We want the dialog to go away once the user
			// has input the value. It goes away when out of scope.
			DIALOG dia;
			dia.newf_str ("",name);
			code = dia.edit(MSG_U(Q_HOSTNAME,"Host or domain specification")
				,MSG_U(I_HOSTNAME,"Enter the name of a host or the name of a\n"
					 "sub-domain.")
				,help_dnsedit);
		}
		if (code==MENU_ACCEPT){
			const char *pt = name.get();
			if (pt[0] != '\0'){
				if (strcmp(pt,preset)==0){
					xconf_error (MSG_U(E_MUSTINSERT
						,"Insert a name before\n"
						 "the domain name"));
				}else{
					editone (pt);
				}
			}
		}else{
			break;
		}
	}
}

static void dnsrecs_locate (const char *host, bool setting)
{
	DNS *dns = dns_load();
	if (dns != NULL){
		dns->editone (host);
	}
	dns_unload();
}

#include <modregister.h>
static REGISTER_VARIABLE_LOOKUP_MSG dnsrecs_var_list1[]={
	{"host",NULL,P_MSG_R(F_HOST_DOM),NULL,dnsrecs_locate},
	{"nickname",NULL,P_MSG_R(F_NICKNAME),NULL,dnsrecs_locate},
	{ NULL, NULL, NULL, NULL }
};
static REGISTER_VARIABLES dnsrecs_registry1("dnsconf",dnsrecs_var_list1);

static REGISTER_VARIABLE_LOOKUP dnsrecs_var_list2[]={
	{"ip1",NULL,&regkeys.ip[0],NULL,dnsrecs_locate},
	{"ip2",NULL,&regkeys.ip[1],NULL,dnsrecs_locate},
	{"ip3",NULL,&regkeys.ip[2],NULL,dnsrecs_locate},
	{"ip4",NULL,&regkeys.ip[3],NULL,dnsrecs_locate},
	{"ipttl1",NULL,&regkeys.ip_ttl[0],NULL,dnsrecs_locate},
	{"ipttl2",NULL,&regkeys.ip_ttl[1],NULL,dnsrecs_locate},
	{"ipttl3",NULL,&regkeys.ip_ttl[2],NULL,dnsrecs_locate},
	{"ipttl4",NULL,&regkeys.ip_ttl[3],NULL,dnsrecs_locate},
	{"ns1",NULL,&regkeys.ns[0],NULL,dnsrecs_locate},
	{"ns2",NULL,&regkeys.ns[1],NULL,dnsrecs_locate},
	{"ns3",NULL,&regkeys.ns[2],NULL,dnsrecs_locate},
	{"ns4",NULL,&regkeys.ns[3],NULL,dnsrecs_locate},
	{"nsttl1",NULL,&regkeys.ns_ttl[0],NULL,dnsrecs_locate},
	{"nsttl2",NULL,&regkeys.ns_ttl[1],NULL,dnsrecs_locate},
	{"nsttl3",NULL,&regkeys.ns_ttl[2],NULL,dnsrecs_locate},
	{"nsttl4",NULL,&regkeys.ns_ttl[3],NULL,dnsrecs_locate},
	{"mx1",NULL,&regkeys.mx[0],NULL,dnsrecs_locate},
	{"mx2",NULL,&regkeys.mx[1],NULL,dnsrecs_locate},
	{"mx3",NULL,&regkeys.mx[2],NULL,dnsrecs_locate},
	{"mx4",NULL,&regkeys.mx[3],NULL,dnsrecs_locate},
	{"mxttl1",NULL,&regkeys.mx_ttl[0],NULL,dnsrecs_locate},
	{"mxttl2",NULL,&regkeys.mx_ttl[1],NULL,dnsrecs_locate},
	{"mxttl3",NULL,&regkeys.mx_ttl[2],NULL,dnsrecs_locate},
	{"mxttl4",NULL,&regkeys.mx_ttl[3],NULL,dnsrecs_locate},
	{ NULL, NULL, NULL, NULL }
};
static REGISTER_VARIABLES dnsrecs_registry2("dnsconf",dnsrecs_var_list2);

