#include "dialog.h"
#include "diadef.h"
#include "../diajava/proto.h"


class FIELD_TEXTAREA: public FIELD{
	int width;
	SSTRING &text;
	SSTRING backup;
	char *buf;
	char *yank;
protected:
	int save_x;
	enum cursor_dir {
		CURSOR_UP,
		CURSOR_DOWN,
		CURSOR_LEFT,
		CURSOR_RIGHT,
		CURSOR_EOL,
		CURSOR_BOL
	};
	struct {
		int input_x;
		int input_y;
		int real_x;
		int real_y;
		int hscroll;
		int vscroll;
		int buf;
	} pos;

	struct {
		int size;
		int len;
	} buf_info;


	/*~PROTOBEG~ FIELD_TEXTAREA */
public:
	FIELD_TEXTAREA (const char *prompt,
		 SSTRING&_text,
		 int _width,
		 int height);
protected:
	void clearall (WINDOW *dialog);
public:
	MENU_STATUS dokey (WINDOW *dialog,
		 int key,
		 FIELD_MSG&msg,
		 bool&grab);
	void drawtxt (WINDOW *dialog, int);
	const char *get_registry_value (void);
	char getidprefix (void);
	void gui_draw (int nofield, SSTRINGS&);
	MENU_STATUS gui_get (int nofield,
		 const char *,
		 const char *);
	void html_draw (int nofield);
	int html_validate (int);
protected:
	void insert_char (char c);
	int insert_point (void);
	void insert_string (char *str);
	int line_length (int line);
	bool move_cursor (cursor_dir dir);
public:
	void reload (const char *dianame, int nof);
	void restore (void);
	void save (void);
protected:
	void sendlines (void);
public:
	void set_registry_value (const char *v);
private:
	void setbuf (const char *s);
protected:
	void setcursor (WINDOW *dialog, int);
public:
	/*~PROTOEND~ FIELD_TEXTAREA */
};

PUBLIC FIELD_TEXTAREA::FIELD_TEXTAREA(
	const char *prompt,
	SSTRING &_text,
	int _width,		// How many visible character horizontally
				// (a hint)
	int height)		// How many lines to show
	: FIELD (prompt), text(_text)
{
	width = _width;
	vsize = height;
	box.width = _width;
	backup.setfrom (text);
	buf = NULL;
	setbuf (text.get());
	pos.hscroll = pos.input_x = pos.real_x = 0;
	pos.vscroll = pos.input_y = pos.real_y = 0;
	save_x = 0;
	yank = NULL;
}

PRIVATE void FIELD_TEXTAREA::setbuf(const char *s)
{
	free (buf);
	buf = strdup(s);
	buf_info.len = strlen(buf);
	buf_info.size = buf_info.len;
}

PUBLIC void FIELD_TEXTAREA::set_registry_value(const char *v)
{
	setbuf (v);
}
PUBLIC const char *FIELD_TEXTAREA::get_registry_value()
{
	return buf;
}


PROTECTED void FIELD_TEXTAREA::setcursor(WINDOW *dialog, int)
{
	wmove (dialog, box.y+pos.input_y, box.x+pos.input_x);
}

PUBLIC void FIELD_TEXTAREA::drawtxt(WINDOW *dialog, int)
{
	const char *s = buf;
	int posx, posy;
	posx = posy = 0;
	bool writechar = false;
	wattrset (dialog, inputbox_attr);
	for (int i=0; *s!=0 && i<pos.vscroll; s++){
		if (*s == '\n') i++;
	}
	for (int i=0; *s!='\n' && *s!=0 && i<pos.hscroll; i++,s++){
		if (*s == '\t'){
			i++;
			while (i%8 != 0 && i < pos.hscroll){
				i++;
			}
			if (i == pos.hscroll){
				while (i%8 != 0){
					posx++;
					i++;
				}
			}
			i--;
		}
	}
	while (*s != 0){
		switch (*s){
			case '\r':
				break;
			case '\n':
				posx = 0;
				posy++;
				if (posy == vsize) return;
				s++;
				for (int i=0; *s!='\n' && *s!=0  && i<pos.hscroll; i++,s++){
					if (*s == '\t'){
						i++;
						while (i%8 != 0 && i < pos.hscroll){
							i++;
						}
						if (i == pos.hscroll){
							while (i%8 != 0){
								posx++;
								i++;
							}
						}
						i--;
					}
				}
				s--;
				break;
			case '\t':
				posx++;
				while ((posx+pos.hscroll)%8 != 0){
					posx++;
				}
				break;
			default:
				writechar = true;
				break;
		}
		if (writechar){
			if (posx < box.width){
				wmove (dialog, posy+box.y, posx+box.x);
				waddch (dialog, *s);
			}
			posx++;
			writechar = false;
		}
		s++;
	}
}

PROTECTED int FIELD_TEXTAREA::line_length(int line)
{
	const char *s = buf;
	for (int i=0; i < line; s++){
		if (*s == '\n') i++;
		else if (*s == 0) return -1;
	}
	int len = 0;
	while (*s != '\n' && *s != 0){
		s++;
		len++;
	}
	return len;
}

PROTECTED int FIELD_TEXTAREA::insert_point(void)
{
	char *s = buf;
	for (int x=0,y=0; x!=pos.real_x || y!=pos.real_y; s++,x++){
		if (*s == '\n'){
			x = -1; 
			y++;
		}
	}
	return s-buf;
}

PROTECTED void FIELD_TEXTAREA::clearall(WINDOW *dialog)
{
	wattrset(dialog,inputbox_attr);
	for (int i=0; i < box.width; i++){
		for (int j=0; j < vsize; j++){
			wmove (dialog,box.y+j,box.x+i);
			waddch (dialog,' ');
		}
	}
}

PROTECTED bool FIELD_TEXTAREA::move_cursor(cursor_dir dir)
{
	bool redraw = false;
	static int x_backup = 0;
	bool goto_x_backup = false;

	switch (dir){
		case CURSOR_UP:
			pos.real_y--;
			goto_x_backup = true;
			break;
		case CURSOR_DOWN:
			pos.real_y++;
			goto_x_backup = true;
			break;
		case CURSOR_LEFT:
			if (pos.input_x == 0){
				if (pos.hscroll > 0){
					pos.real_x--;
				}else if (pos.real_y > 0){
					pos.real_y--;
					int len = line_length(pos.real_y);
					pos.real_x = len==0?len:len+1;
				}
			}else{
				pos.real_x--;
			}
			break;
		case CURSOR_RIGHT:
			{
			int len = line_length(pos.real_y);
			if (pos.real_x < len){
				pos.real_x++;
			}else{
				len = line_length(pos.real_y+1);
				if (len != -1){
					pos.real_y++;
					pos.real_x = 0;
				}
			}
			break;
			}
		case CURSOR_BOL:
			pos.hscroll = pos.input_x = pos.real_x = save_x = 0;
			redraw = true;
			break;
		case CURSOR_EOL:
			pos.real_x = line_length(pos.real_y);
			redraw = true;	
			break;
	}

	char *s = buf;

	// Verify if real_y is valid and find the current buffer position.
	char *last_y = s;
	if (pos.real_y < 0){
		pos.real_y = 0;
	}else{
		for (int y=0; y<pos.real_y; s++){
			if (*s == '\n'){
				y++;
				last_y = s+1;
			}else if (*s == 0){
				pos.real_y = y;
				s = last_y;
				break;
			}
		}
	}
	
	// Handle vertical scrolling
	if (pos.real_y-pos.vscroll > vsize-1){
		pos.vscroll = pos.real_y-vsize+1;
		pos.input_y = pos.real_y-pos.vscroll;
		redraw = true;
	}else if (pos.real_y < pos.vscroll){
		pos.vscroll = pos.real_y;
		pos.input_y = 0;
		redraw = true;
	}else{
		pos.input_y = pos.real_y-pos.vscroll;
	}
	
	// Handle line length
	int len = 0;
	for (char *l = s; *l != '\n' && *l != 0; l++, len++);

	// Find the correct horizontal input position
	pos.input_x = 0;
	if (goto_x_backup){
		int x = 0;
		for (; x<len && pos.input_x<x_backup; x++,s++){
			pos.input_x++;
			if (*s == '\t'){
				while(pos.input_x%8 != 0){
					pos.input_x++;
				}
			}
		}
		pos.real_x = x;
		redraw = true;
	}else{
		if (pos.real_x > len){
			pos.real_x = len;
			redraw = true;
		}
	
		for (int x=0; x<pos.real_x; x++,s++){
			pos.input_x++;
			if (*s == '\t'){
				while(pos.input_x%8 != 0){
					pos.input_x++;
				}
			}
		}
		x_backup = pos.input_x;
	}
	
	// Handle horizontal scrolling
	if (pos.input_x-pos.hscroll > box.width-1){
		pos.hscroll = pos.input_x-box.width+1;
		pos.input_x -= pos.hscroll;
		redraw = true;
	}else if (pos.input_x < pos.hscroll){
		pos.hscroll = pos.input_x;
		pos.input_x = 0;
		redraw = true;
	}else{
		pos.input_x -= pos.hscroll;
	}

	return redraw;
}

PROTECTED void FIELD_TEXTAREA::insert_string(char *str)
{
	int len = strlen(str);
	if (buf_info.len+len > buf_info.size){
		buf_info.size = (buf_info.size*2)+256+len;
		buf = (char*) realloc(buf,buf_info.size);
	}
	int ip = insert_point();
	char *s = buf+ip;
	memmove(s+len,s,buf_info.len-ip+1);
	buf_info.len+=len;
	memcpy(s,str,len);
	for (int i=0; i<len; i++){
		move_cursor (CURSOR_RIGHT);
	}
}

PROTECTED void FIELD_TEXTAREA::insert_char(char c)
{
	if (buf_info.len+1 > buf_info.size){
		buf_info.size = (buf_info.size*2)+256;
		buf = (char*) realloc(buf,buf_info.size);
	}
	int ip = insert_point();
	char *s = buf+ip;
	memmove(s+1,s,buf_info.len-ip+1);
	buf_info.len++;
	*s = c;
	if (c == '\n'){
		move_cursor (CURSOR_DOWN);
		move_cursor (CURSOR_BOL);
	}else{
		move_cursor (CURSOR_RIGHT);
	}
}

PUBLIC MENU_STATUS FIELD_TEXTAREA::dokey(
	WINDOW *dialog,
	int key,
	FIELD_MSG &msg,
	bool &grab)
{
	bool redraw = false;
	if (readonly) return MENU_NULL;
	if (grab == false){
		if (key == ' '){
			grab = true;
		}
		return MENU_NULL;
	}
	static bool killing = false;
	switch (key){
		case 16: // ^P
		case KEY_UP:
			redraw = move_cursor(CURSOR_UP);
			killing = false;
			break;
		case 14: // ^N
		case KEY_DOWN:
			redraw = move_cursor(CURSOR_DOWN);
			killing = false;
			break;
		case 2: // ^B
		case KEY_LEFT:
			redraw = move_cursor(CURSOR_LEFT);
			killing = false;
			break;
		case 6: // ^F
		case KEY_RIGHT:
			redraw = move_cursor(CURSOR_RIGHT);
			killing = false;
			break;
		case 1: // ^A
		case KEY_HOME:
			redraw = move_cursor(CURSOR_BOL);
			killing = false;
			break;
		case 5: // ^E:
		case KEY_END:
			redraw = move_cursor(CURSOR_EOL);
			killing = false;
			break;
		case KEY_BACKSPACE:
		{
			int ip = insert_point();
			if (ip > 0){
				move_cursor(CURSOR_LEFT);
				memmove(buf+ip-1,buf+ip,buf_info.len-ip+1);
				buf_info.len--;
				redraw = true;
			}
			killing = false;
			break;
		}
		case 4: // ^D
		case DEL:
		{
			int ip = insert_point();
			if (ip < buf_info.len){
				memmove(buf+ip,buf+ip+1,buf_info.len-ip+1);
				buf_info.len--;
				redraw = true;
			}
			killing = false;
			break;
		}
		case 11: // ^K
		{
			int ip = insert_point();
			char *s = buf+ip;
			int len = 0;
			int yank_len = 0;
			if (*s != 0){
				if (*s != '\n'){
					for (;*s!='\n' && *s!=0;s++);
					len = s-(buf+ip);
				}else{
					len = 1;
				}
				if (killing && yank != NULL){
					yank_len = (yank != NULL) ? strlen(yank) : 0;
					yank = (char*) realloc(yank,yank_len+len+1);
				}else{
					free(yank);
					yank = NULL;
					if (len > 0){
						yank = (char*) malloc(len+1);
					}
				}
				memcpy (yank+yank_len,buf+ip,len);
				*(yank+yank_len+len) = 0;
				memmove (buf+ip,buf+ip+len,buf_info.len-(ip+len)+1);
				buf_info.len -= len;
				redraw = true;
				killing = true;
			}
			break;
		}
		case 25: // ^Y
			insert_string (yank);
			redraw = true;
			killing = false;
			break;
		default:
			insert_char(key);
			redraw = true;
			killing = false;
			break;
	}
	if (redraw){
		clearall (dialog);
		drawtxt (dialog,0);
	}
	return MENU_NULL;
}

PUBLIC void FIELD_TEXTAREA::save()
{
	text.setfrom(buf);
}

PUBLIC void FIELD_TEXTAREA::restore()
{
	text.setfrom(backup);
}

PUBLIC void FIELD_TEXTAREA::reload(const char *dianame, int nof)
{
	free(buf);
	buf = strdup(text.get());
	buf_info.len = strlen(buf);
	buf_info.size = buf_info.len;
	if (dianame != NULL){
		char tmp1[1000];
		diagui_sendcmd (P_Setval,"%s T%d reset\n"
			,formatpath(tmp1,dianame),nof);
		sendlines();
	}
}
/*
	Send the text to the GUI front-end line by line
*/
PROTECTED void FIELD_TEXTAREA::sendlines()
{
	const char *pt = buf;
	while (*pt != '\0'){
		char line[1000];
		char *dst = line;
		while (*pt != '\n' && *pt != '\0'
			&& (unsigned)(dst-line)<sizeof(line)) *dst++ = *pt++;
		*dst = '\0';
		char tmp[1000];
		diagui_sendcmd (P_Str,"%s\n",diagui_quote(line,tmp));
		if (*pt == '\n') pt++;
	}
	diagui_sendcmd (P_End,"\n");
}

PUBLIC void FIELD_TEXTAREA::html_draw(int nofield)
{
	html_printf ("<tr><td>%s<td>",prompt);
	if (readonly){
		html_printf ("%s\n",buf);
	}else{
		char key[100];
		format_htmlkey (key,nofield);
		html_printf ("<textarea name=%s rows=%d cols=%d>\n",key,vsize
			,width);
		html_printf ("%s\n",buf);
		html_printf ("</textarea>\n");
	}
}

PUBLIC void FIELD_TEXTAREA::gui_draw(int nofield, SSTRINGS &)
{
	guisendprompt();
	if (readonly){
		diagui_send_Label (buf);
	}else{
		diagui_sendcmd (P_Textarea,"T%d %d %d\n",nofield,width,vsize);
		sendlines();
	}
}

PUBLIC MENU_STATUS FIELD_TEXTAREA::gui_get(int nofield, const char *, const char *)
{
	SSTRINGS tb;
	int nb = diagui_getvals('T',nofield,tb);
	int len = 0;
	for (int i=0; i<nb; i++){
		len += tb.getitem(i)->getlen()+1;
	}
	char tmp[len+1];
	char *acc = tmp;
	for (int i=0; i<nb; i++){
		tb.getitem(i)->copy (acc);
		acc += strlen(acc);
		*acc ++ = '\n';
	}
	*acc = '\0';
	free(buf);
	buf = strdup(tmp);
	buf_info.len = strlen(buf);
	buf_info.size = buf_info.len;
	return MENU_NULL;
}

PUBLIC char FIELD_TEXTAREA::getidprefix ()
{
	return 'T';
}

PUBLIC int FIELD_TEXTAREA::html_validate(int)
{
	return 0;
}

PUBLIC void DIALOG::newf_textarea (
	const char *prompt,
	SSTRING &text,
	int width,		// How many visible character horizontally
					// (a hint)
	int height)		// How many lines to show
{
	add (new FIELD_TEXTAREA(prompt,text,width,height));
}

