/*
 * $Id: types.h,v 1.6 2001/12/15 05:13:08 antona Exp $
 *
 * types.h - Misc type definitions not related to on-disk structure. Part of
 *	     the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef NTFS_TYPES_H
#define NTFS_TYPES_H

#include "layout.h"
#include "list.h"
#include "unistr.h"

/*
 * run_list
 * ========
 * 
 * The last vcn (in fact the last vcn + 1) is reached when length == 0.
 * 
 * When lcn == -1 this means that the count vcns starting at vcn are not 
 * physically allocated (i.e. this is a hole / data is sparse).
 *
 * For attribute extents which have the previous_extents flag set, the first
 * entry is a sparse entry (lcn == -1). It is used to get the vcn count up to
 * lowest_vcn for that extent.
 */
typedef struct {	/* In memory vcn to lcn mapping structure element. */
	VCN vcn;	/* vcn = Starting virtual cluster number. */
	LCN lcn;	/* lcn = Starting logical cluster number. */
	__u64 length;	/* Run length in clusters. */
} run_list_element;

typedef run_list_element run_list; /* In memory vcn to lcn mapping array. */

/*
 * Union of all known attribute values. For convenience. Used in the attr
 * structure.
 */
typedef union {
	__u8;	/* Unnamed __u8 to serve as default when just using
		   a_val without specifying any of the below. */
	STANDARD_INFORMATION std_inf;
	ATTR_LIST_ENTRY al_entry;
	FILE_NAME_ATTR filename;
	OBJECT_ID_ATTR obj_id;
	SECURITY_DESCRIPTOR_ATTR sec_desc;
	VOLUME_NAME vol_name;
	VOLUME_INFORMATION vol_inf;
	DATA_ATTR data;
	INDEX_ROOT index_root;
	INDEX_BLOCK index_blk;
	BITMAP_ATTR bmp;
	REPARSE_POINT reparse;
	EA_INFORMATION ea_inf;
	EA_ATTR ea;
	PROPERTY_SET property_set;
	LOGGED_UTILITY_STREAM logged_util_stream;
	EFS_ATTR efs;
} attr_val;
 
/*
 * Necessary forward definitions and typedefs.
 */ 
struct _ntfs_attr;
typedef struct _ntfs_attr attr;

struct _mft_entry;
typedef struct _mft_entry mft_entry;

struct _ntfs_file;
typedef struct _ntfs_file ntfs_file;

struct _ntfs_volume;
typedef struct _ntfs_volume ntfs_volume;

/**
 * attr - structure describing a mapped attribute
 * @a_rec:		attribute record of the attribute
 * @a_flags:		flags describing the attribute
 * @a_val:		attribute value
 * @a_run_list:		run list of the attribute value
 * @a_count:		usage count of the attribute
 * @a_dirty_list:	anchor for the @a_vol->dirty_attrs linked list
 * @a_vol:		volume the attribute belongs to
 * @a_m_entry:		mft entry @a_rec is contained in
 * @a_base_m_entry:	base mft entry holding the attribute list attribute
 * @a_attr_list_entry:	attribute list entry of this attribute
 *
 * @a_rec is the attribute record (ATTR_REC *) of the attribute (and is always
 * valid).
 * 
 * The following flags are defined for @a_flags:
 * 	dirty		- Attribute value is dirty. Needs to be written to disk.
 * 			  Only for non_resident attributes. For resident ones
 * 			  the mft entry is set dirty. Note: when the meta data
 * 			  part of the attribute is dirtied the mft record is
 * 			  marked dirty directly. No flag reflects this in the
 * 			  attr structure as it is unnecessary.
 *	val_mapped	- Attribute value is mapped. If not mapped @a_val and
 *		  	  @a_dirty_list are not defined.
 *	error		- There was some kind of error on I/O so it is possible
 *			  that one or more of the other fields are corrupt.
 *	non_resident	- Attribute is not resident. If resident, @a_run_list
 *			  and @a_dirty_list are not defined.
 *	in_attr_list	- Attribute is described by the attribute list attribute
 *			  @a_base_m_entry->m_attr_list, the entry in this is
 *			  @a_attr_list_entry.
 *	in_ext_record	- Attribute is described by the attribute list
 *			  attribute and it is stored in an extension record
 *			  (pointed to by @a_m_entry) rather than in the base mft
 *			  record (pointed to by @a_base_m_entry). This only
 *			  makes sense if in_attr_list is set.
 *	previous_extents - Attribute is not the first extent. Only makes sense
 *			  if in_attr_list is set. Also, if previous_extents is
 *			  set, this implies in_ext_record is also set.
 *	last_extent	- Attribute is the last extent. So if there are no
 *			  previous_extents and this is the last_extent, we know
 *			  there are no extents at all. Only make sense if
 *			  in_attr_list is set. Note, we don't keep pointers to
 *			  the prev/next extents as their attributes are not
 *			  necessarily mapped.
 *
 * @a_val points to the attribute value of the attribute and will either point
 * into @a_rec (resident attribute) or it will point to a especially for it
 * allocated memory which has to be free()d when the attribute is unmapped.
 * @a_val is only valid when the val_mapped flag is set in @a_flags.
 *
 * It is important to note that @a_val contains only the mapped portion of this
 * extent of the attribute value which is indicated by a lcn >= 0 in the run
 * list.
 *
 * If lcn is -1, @a_val does contain this portion of the value, but it is all
 * zero, and there is no storage allocated for it (it is a sparse run). When
 * writing it to disk, it is checked for being zero again and if it is, the
 * (zero) data is not written to disk, but a sparse entry is saved in the
 * mapping pairs array. If it is not zero, storage is allocated on disk and the
 * data is written. The run list and mapping pairs array is then updated to
 * reflect this and lcn is no longer -1.
 *
 * @a_run_list is only valid if the non_resident flag is set. It will only hold
 * the part of the run list belonging to this extent. If previous_extents is
 * zero and last_extent is set, this is the full run_list as there are no more
 * extents. Should the vcn being looked for not be present in @a_run_list, it
 * should be checked for other extents before returning out of bounds error
 * (check previous_extents/last_extent flags first).
 *
 * Note: If previous_extents is set then the first entry of @a_run_list has a
 * vcn equal to lowest_vcn of this extent. We do NOT have a virtual sparse
 * entry (vcn = 0, lcn = -1, and length = lowest_vcn) at the beginning, unlike
 * the WinNT/2k NTFS drivers. IMO this only makes things more complex instead
 * of easier. If you are familiar with Win NTFS driver this is something to
 * watch out for... (AIA)
 *
 * Note, that @a_run_list may be NULL which means that the run list has not
 * been extracted from @a_rec (yet). This can only happen when the attribute
 * value is not mapped. If val_mapped is set, @a_run_list must be valid.
 *
 * @a_count is the number of times the attribute value has been mapped. When it
 * reaches zero the value may be safely unmapped (after being synced if dirty
 * and hence also non_resident). This can be done at any point in time. Remember
 * we only free() the memory if the attribute is non resident! Once the value is
 * unmapped, it is also safe to free() @a_run_list AND set it to NULL.
 *
 * @a_dirty_list anchors the attribute entry into the dirty_attrs list of
 * @a_vol. Used when syncing the volume to be able to quickly walk all dirty
 * attributes and sync them as well. Only applies to non_resident attributes
 * as resident ones are covered by virtue of being contained inside their
 * mft entries (@a_m_entry) which are dirty and synced to disk, already.
 *
 * @a_vol is the volume the attribute belongs to. This is just so lookups are
 * quicker. Otherwise it could be obtained from @m_entry->m_vol.
 * 
 * @a_m_entry points to the mft entry in which the attribute is held. If the
 * attribute is in the base mft record, @a_m_entry is also the base mft record
 * and @a_base_m_entry equals @m_entry (in_ext_record flag is zero). If the
 * attribute is located in an extension mft record, @a_base_m_entry points to
 * the mft entry of the base mft record and the in_ext_record flag is set in
 * @a_flags (implies in_attr_list flag is also set).
 *
 * If in_attr_list is set, @a_base_m_entry points to the mft entry containing
 * the attribute list attribute describing this attribute. The attribute list is
 * obtained from @a_base_m_entry->m_attr_list.
 *
 * If in_attr_list is set, @a_attr_list_entry points to the attribute list entry
 * within the attribute list attribute which describes this attribute.
 *
 * Note, we don't store the file this attribute belongs to, as this information
 * is irrelevant for all intents and purposes and can be obtained from
 * @m_entry->m_file should it be needed at some point.
 */
struct _ntfs_attr {
	ATTR_RECORD *a_rec;
	unsigned long a_flags;
	attr_val *a_val;
	run_list *a_run_list;
	unsigned int a_count;
	struct list_head a_dirty_list;
	ntfs_volume *a_vol;
	mft_entry *a_m_entry;
	mft_entry *a_base_m_entry;
	ATTR_LIST_ENTRY *a_attr_list_entry;
};

/**
 * mft_entry - structure describing a loaded mft record
 * @m_rec:		mft record of this mft entry
 * @m_flags:		flags describing the mft entry
 * @m_ref:		mft reference of this mft entry (includes sequence no.)
 * @m_count:		usage count of the mft entry
 * @m_mapped_attrs:	array of all mapped attributes of this mft entry
 * @nr_mapped_attrs:	number of mapped attributes in the @m_mapped_attrs array
 * @m_file:		file this mft entry belongs to (or NULL if none)
 * @m_list:		anchor for the @m_vol->mft_entries linked list
 * @m_dirty_list:	anchor for the @m_vol->dirty_mft_entries linked list
 * @m_vol:		volume the mft entry belongs to
 * @m_attr_list:	the attribute list value of this (base) mft entry
 *
 * Each loaded mft record is represented by a mft_entry structure.
 * 
 * @m_rec is a pointer to the actual MFT_RECORD. Only valid if mapped flag is
 * set.
 * 
 * @m_flags is a set of flags describing the mft entry / record. The
 * following flags are defined:
 *	mapped		- Is the MFT_RECORD actually in memory? (i.e. Is @m_rec
 *			  valid?)
 *	dirty		- Does the record need to be written to disk? If not,
 *			  it is safe to discard @m_rec, but only if @m_count is
 *			  zero.
 * 	error		- When reading the entry from disk it was detected that
 * 			  it is	corrupt. For example, it could be that a multi
 * 			  sector transfer error was detected.
 *	is_base_record	- Is this mft record a base record?
 *	attr_list_present	- Set if an attribute list is present in this
 *				  mft record. This implies that is_base_record
 *				  is set.
 *	attr_list_mapped	- Set if the attribute list is mapped. Implies
 *				  that is_base_record and attr_list_present are
 *				  set.
 *
 * @m_ref is the mft reference of the mft record described by this entry. This
 * includes the sequence number for consistency checking purposes.
 *
 * @m_count is the usage count. When zero this record is not referenced anywhere
 * and can be thrown away safely (writing it to disk first if the dirty flag is
 * set).
 * 
 * @m_mapped_attrs is an array of pointers to all mapped attributes of this mft
 * entry.
 * 
 * @nr_mapped_attrs is the number of mapped attributes and hence the number of
 * elements in @m_mapped_attrs.
 * 
 * @m_file is the ntfs_file this mft entry belongs to. If this is NULL, the mft
 * entry does not have a file associated with it. This can happen when a file is
 * closed but the associated mft entries are left in the cache or when mft
 * entries are directly accessed at low level for whatever reason.
 *
 * @m_list is an anchor for the @m_vol->mft_entries linked list.
 *
 * @m_dirty_list is an anchor for the @m_vol->dirty_mft_enties linked list.
 * Only valid if the dirty flag is set.
 * 
 * @m_vol is the ntfs_volume this entry belongs to.
 *
 * @m_attr_list is a pointer to the attribute list attribute value. Only if
 * @m_flags has the attr_list_mapped flag set (which implies that is_base_record
 * and attr_list_present flags are set, too). This will effectively just contain
 * the same memory reference as the @a_val field of the mapped attribute list
 * attribute.
 */
struct _mft_entry {
	MFT_RECORD *m_rec;
	unsigned long m_flags;
	MFT_REF m_ref;
	unsigned int m_count;
	attr **m_mapped_attrs;
	unsigned int nr_mapped_attrs;
	ntfs_file *m_file;
	struct list_head m_list;
	struct list_head m_dirty_list;
	ntfs_volume *m_vol;
	ATTR_LIST_ENTRY *m_attr_list;
};

/*
 * NOTE: I have completely ignored the existence of things like directories
 * and directory references for the moment! These will just be added later on.
 * The directories themselves are of course covered by being base mft records
 * and hence follow the same semantics like every other mft record.
 */

/**
 * ntfs_file - structure describing an open file
 * @f_m_refs:		array of mft references belonging to this file
 * @f_m_entries:	array mft entries belonging to the @f_m_refs
 * @nr_m_refs:		number of entries in above two arrays
 * @f_count:		usage count of the file
 * @f_vol:		volume the file belongs to
 * @f_list:		anchor for the @f_vol->open_files list
 *
 * Each open file is described by an ntfs_file structure.
 *
 * @f_m_refs is the mft references of the file (i.e. the on disk inode numbers).
 * This is an array of MFT_REFs. The first reference is the base mft record and
 * if any others exist they are the associated loaded extension mft records.
 * Used for attribute list purposes.
 *
 * @f_m_entries is the loaded mft records for the file. This is an array of
 * pointers to mft_entries, corresponding to the f_m_refs. This is not strictly
 * necessary but results in a big speed up as we don't need to search the
 * @f_vol->mft_entries list every time.
 *
 * When an entry is referenced in the above lists the usage count for the
 * corresponding entry is increased by one.
 * 
 * @nr_m_refs is the number of records in the @m_refs and @m_entries arrays. If
 * it is greater 1, we know there must be an attribute list attribute (but if
 * equals 1, we	cannot assume there isn't one). This is of course the same as a
 * possible nr_m_entries, but refs is shorter to write. (-8
 *
 * @f_count it the usage count. When zero, this file is no longer open by anyone
 * and is thrown away. No need to concern oneself with dirty state as this is
 * done on a mft_entry level, i.e. a level below the idea of files.
 *
 * @f_vol is the volume the file belongs to.
 *
 * @f_list is an anchor into @f_vol->files list. When @f_count reaches 0, the
 * file is unlinked from the list and disposed off.
 */
struct _ntfs_file {
	MFT_REF *f_m_refs;
	mft_entry **f_m_entries;
	unsigned int nr_m_refs;
	unsigned int f_count;
	ntfs_volume *f_vol;
	struct list_head f_list;
};

/*
 * FIXME: For RAID stuff will need the fd and dev_name variables to be arrays
 * or lists or something... Also will need to know the number of sectors (and
 * sector size if not 512) of each device making up the RAID array. Obviously,
 * will need RAID information as well like which RAID is used together with
 * state information, etc. But that is the far future.
 */

/*
 * ntfs_volume - structure describing an open volume in memory
 * FIXME: put in the remaining fields...
 * @open_files:			linked list of all opened files
 * @nr_open_files:		number of @open_files
 * @max_open_files:		maximum number of @open_files
 * @mft_entries:		linked list of all mft entries cached in memory
 * @nr_mft_entries:		number of @mft_entries
 * @dirty_mft_entries:		linked list of all dirty mft entries
 * @nr_dirty_mft_entries:	number of @dirty_mft_entries
 * @dirty_attrs:		linked list of all dirty, mapped attributes
 * @nr_dirty_attrs:		number of @dirty_attrs
 * @v_list:			anchor for list of mounted volumes
 * 
 * @open_files is a linked list of ntfs_file struct pointers, keeping track of
 * of all open files. When a file is opened it is added to the back of this list
 * as we do not expect to be closing the file again that soon.
 *
 * @nr_open_files is the number of @open_files.
 *
 * @max_open_files is the maximum number of @open_files allowed.
 * 
 * @mft_entries is a linked list of all loaded mft records (disk inodes) in
 * order of ascending mft record number (disk inode number, i.e. disregarding
 * the sequence number). Note, the list members are not the actual MFT_RECORDS,
 * but in memory mft_entry structs.
 *
 * @nr_mft_entries is the number of @mft_entries.
 *
 * @dirty_mft_entries is a linked list of all loaded mft records which are
 * dirty. Used for flushing the dirty mft entries - when one is marked dirty,
 * the entry is added to the end of this list. When it is flushed to disk, it is
 * marked clean and removed from this list. Effectively a FIFO.
 *
 * @nr_dirty_mft_entries is the number of @dirty_mft_entries.
 *
 * @dirty_attrs is a linked list of all dirty, non-resident, mapped attributes.
 * 
 * @nr_dirty_attrs is the number of @dirty_attrs.
 * 
 * @v_list is an anchor for the linked list of mounted volume.
 */
struct _ntfs_volume {
	int fd;			/* File descriptor associated with volume. */
	char *dev_name;		/* Name of the device/file the volume is in. */
	char *vol_name;		/* Name of the volume from the volume name
				   attribute. */
	/* Version nos. and flags are from the volume information attribute. */
        __u8 major_ver;		/* Ntfs major version of volume. */
        __u8 minor_ver;		/* Ntfs minor version of volume. */
        __u16 flags;		/* Bit array of VOLUME_* flags. */
	__s64 number_of_clusters;/* Volume size in clusters, hence also the
				    number of bits in lcn_bitmap. */
	__u32 cluster_size;	/* Byte size of a cluster. */
	__u32 mft_record_size;	/* Byte size of a mft record. */
	__s64 mft_lcn;		/* Logical cluster number of the data attribute
				   for FILE_$Mft. */
       	__s64 mftmirr_lcn;	/* Logical cluster number of the data attribute
				   for FILE_$MftMirr. */
	__u8 cluster_size_bits;	/* Log(2) of the byte size of a cluster. */
	__u8 mft_record_size_bits;/* Log(2) of the byte size of a mft record. */
	__u8 *lcn_bitmap;	/* Value of data attribute of FILE_$Bitmap. Each
				   bit represents a cluster on the volume, bit 0
				   representing lcn 0 and so on. A set bit means
				   that the cluster is in use and a zero bit
				   means the cluster is free. */
	__u8 *mft_bitmap;	/* Value of bitmap attribute of FILE_$Mft. Has
				   same interpretation as the lcn_bitmap above
				   but except that it keeps track of the usage
				   of the mft records rather than the lcns. */
	run_list *mft_runlist;	/* Decompressed run list of the data attribute
				   of FILE_$Mft. */
	__s64 number_of_mft_records;	/* Number of records in the mft, equals
					   the number of bits in mft_bitmap. */
	uchar_t *upcase;		/* Upper case equivalents of all 65536 
					   2-byte Unicode characters. Obtained
					   from FILE_$UpCase. */
	__u32 upcase_len;		/* Length in Unicode characters of the
					   upcase table. */
 	struct list_head open_files;
 	int nr_open_files;
	int max_open_files;
 	struct list_head mft_entries;
	int nr_mft_entries;
 	struct list_head dirty_mft_entries;
 	int nr_dirty_mft_entries;
	struct list_head dirty_attrs;
	int nr_dirty_attrs;
	struct list_head v_list;
};

#endif /* defined NTFS_TYPES_H */

