#include <syslog.h>
#include <string.h>
#include "global_vars.h"
#include "con.h"
#include "pinger.h"
#include "debug.h"
#include "times.h"

int lines_up()	// 0, 1, 2, 3, ... how many lines are established?
		// CST_UP_SERVER || CST_UP_USER || CST_LOCKED_UP_*
{
    struct line_t *line = (struct line_t*)lstlines.first;
    int up = 0;
    while ( line )
    {
	if ( (line->con_stat == CST_UP_SERVER)
	    || (line->con_stat == CST_UP_USER)
	    || (line->con_stat == CST_LOCKED_UP_CLT)
	    || (line->con_stat == CST_LOCKED_UP_TIMED) ) up++;
	line = (struct line_t*)line->next;
    }
    return(up);
}

int lines_active_srv()
// how many lines are neither CST_DOWN nor CST_UP_USER nor CST_LOCKED_*?
{
    struct line_t *line = (struct line_t*)lstlines.first;
    int up = 0;
    while ( line )
    {
	if ( (line->con_stat != CST_DOWN)
	    && (line->con_stat != CST_UP_USER)
	    && (line->con_stat != CST_LOCKED_UP_CLT)
	    && (line->con_stat != CST_LOCKED_UP_TIMED)
	    && (line->con_stat != CST_LOCKED_DOWN)) up++;
	line = (struct line_t*)line->next;
    }
    return(up);
}

void lines_shutdown_log()
{
    struct line_t *line = (struct line_t*)lstlines.first;
    time_t now;
    while ( line )
    {
	switch ( line->con_stat )
	{
	    case CST_UP_USER:
	        now = time(NULL);
	        fprintf(server->logfile, "manually: %s '%s' ",
			line->linename, cor_ctime(&line->con_up_time));
	        fprintf(server->logfile, "'%s' %d %u\n", cor_ctime(&now),
			(int)(now-line->con_up_time), line->bytes_recvd);
		fflush(server->logfile);
		break;
	    case CST_UP_SERVER:
	    case CST_LOCKED_UP_TIMED:
	    case CST_LOCKED_UP_CLT:
	        now = time(NULL);
	        fprintf(server->logfile, "line: %s '%s' ",
			line->linename, cor_ctime(&line->con_up_time));
	        fprintf(server->logfile, "'%s' %d %u\n", cor_ctime(&now),
			(int)(now-line->con_up_time), line->bytes_recvd);
		fflush(server->logfile);
		break;
	}
	line = (struct line_t*)line->next;
    }
}

void lines_check_close()
{ // closes lines on which there are no clients anymore
    struct line_t *line = (struct line_t*)lstlines.first;
    // recalculate how many clients are online on which line
    while ( line )
    {
		struct client_t *who = (struct client_t*)cltlist.first;
		line->online = 0;
		while ( who )
		{
		    if ( (who->line == line)
					 && (who->status == CLT_ONLINE) ) line->online++;
			    who = (struct client_t*)who->next;
		}
		line = (struct line_t*)line->next;
    }
    // check which lines to close
    // close it +- at the same time!
    line = (struct line_t*)lstlines.first;
    while ( line )
    {
		if ( ((line->con_stat != CST_UP_SERVER) || line->online)
		     && (line->con_stat != CST_LOCKED_UP_TIMED))
		{ // line down or someone has status online for this line or locked it
		    line = (struct line_t*)line->next;
		    continue;
		}
		// close the line
		call_con_dn(NULL, line);
		// ok, go on with the next line
		line = (struct line_t*)line->next;
    }
}

void lines_check_pinger()
{ // will send a ping if required (time, ...)
    struct line_t *line = (struct line_t*)lstlines.first;
    while ( line )
    {
	if ( (!line->pinger)
	    || ((line->con_stat != CST_UP_SERVER)
		&& (line->con_stat != CST_UP_USER)
		&& (line->con_stat != CST_LOCKED_UP_CLT)
		&& (line->con_stat != CST_LOCKED_UP_TIMED))
	    || (line->pinger_interval + line->pinger_prev > time(NULL)) )
	{ // don't ping anyone!
	    line = (struct line_t*)line->next;
	    continue;
	}
	// ping!
	pinger_send(line);
	line = (struct line_t*)line->next;
    }
}

int lines_same_group(struct line_t *a, struct line_t *b)
// returns 1 if they are in the same group (else 0)
// (means that the status gets detected by the same interface, device or file)
{
    if ( a->con_type != b->con_type ) return(0);
    switch ( a->con_type )
    {
	case CT_NETDEV:
	    if ( strcmp(a->interface, b->interface) ) return(0);
	    break;
	case CT_ISDN:
	    if ( strcmp(a->isdn_id, b->isdn_id) ) return(0);
	    break;
	case CT_FILE:
	    if ( strcmp(a->con_status_file, b->con_status_file) ) return(0);
	    break;
	default:
	    syslog(LOG_ERR, "Bad programmer: update lines.c:same_group()! Please notify the author.");
	    return(0);
    }
    return(1);
}

void lines_lock_group(struct line_t *nolock)
{
    struct line_t *cl = (struct line_t*)lstlines.first;
    if ( nolock->has_locked ) return; // already done the job
    for (; cl; cl = (struct line_t*)cl->next )
    {
	if ( cl == nolock ) continue;
	if ( lines_same_group(nolock, cl) )
	    switch ( cl->con_stat )
	    {
		case CST_DOWN: // only good case...
		    set_line_status(cl, CST_LOCKED_DOWN);
		    break;
		default:
		    syslog(LOG_ERR, "Bad line locking 1. Please notify the author.");
	    }
    }
    nolock->has_locked = 1;
}

void lines_unlock_group(struct line_t *nolock)
{
    struct line_t *cl = (struct line_t*)lstlines.first;
    if ( !nolock->has_locked ) return; // already done the job
    for (; cl; cl = (struct line_t*)cl->next )
    {
	if ( cl == nolock ) continue;
	if ( lines_same_group(nolock, cl) )
	    switch ( cl->con_stat )
	    {
		case CST_LOCKED_DOWN: // only good case...
		    set_line_status(cl, CST_DOWN);
		    break;
		default:
		    syslog(LOG_ERR, "Bad line locking 2. Please notify the author.");
	    }
    }
    nolock->has_locked = 0;
}

void lines_check_locks(struct line_t *line)
{
    static char norecurse = 0;
    if ( norecurse ) return;
    norecurse++;
    switch ( line->con_stat )
    {
	// lock other lines in group
	case CST_UP_SERVER:
	case CST_UP_USER:
	case CST_DIALING:
	case CST_CLOSE_ERROR:
	case CST_CLOSING:
	case CST_LOCKED_UP_CLT:
	case CST_LOCKED_UP_TIMED:
	    if ( !line->has_locked ) lines_lock_group(line);
	    break;
	// unlock other lines in group
	case CST_DOWN:
	    if ( line->has_locked ) lines_unlock_group(line);
	    break;
	// don't do anything!
	case CST_LOCKED_DOWN:
	    break;
	// programmers (update) error:
	default:
	    syslog(LOG_ERR, "Bad programmer: update lines.c:lines_check_locks().");
    }
    norecurse--;
}

int lines_unlock_estab()
{
    struct line_t *cl = (struct line_t*)lstlines.first;
    int num = 0;
    while ( cl )
    {
	if (   (cl->con_stat == CST_LOCKED_UP_CLT)
	    || (cl->con_stat == CST_LOCKED_UP_TIMED) )
	{
	    set_line_status(cl, CST_UP_SERVER);
	    num++;
	}
	cl = (struct line_t*)cl->next;
    }
    return(num);
}
