#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <unistd.h>
#include <stdlib.h>
#include "../common.h"
#include "srv_def.h"
#include "global_vars.h"
#include "lists.h"
#include "client.h"

/*
 the life of a command (CMD_*):
 1. cmd_q makes an entry in ordq
 2. cmd_q looks if we're waiting for acks from this client
 if not:
   3.1. cmd_q calls cmd_send, which sends the command
   3.2. cmd_send sends & calls ack_q which moves the ordq entry to the ackq
 else:
   3.1. cmd_q returns
   3.2. we get an ack: ack_proc finds command for this client waiting in ordq
   3.3. ack_proc calls cmd_send, which sends the command from the ordq
   3.4. cmd_send calls ack_q which moves the ordq entry to the ackq
 4. cmd now waits as an ackq entry for an ack
 5. cmd_rsend eventually resends cmd from ackq

 the life of broadcasts (CBR_*):
 1. cmd_broadcast get's called... that's it. Neither ordq nor ackq entries
    will be made!
*/

/*-----------*/
int cmd_send(struct ordq_item_t *cmdr);
int ack_send(struct sockaddr_in *trg, struct p_order_t *cmdpack);
int cmd_q(struct client_t *who, int cmd, char *msg, int msglen);
int cmd_direct(int cmd, int sock, struct sockaddr_in *who, char *msg, int mlen, char type);
int cmd_direct_udp(int cmd, struct sockaddr_in *who, char *msg, int mlen);
int cmd_direct_tcp(int, int, struct sockaddr_in *, char *, int);
int ack_proc(int ip, short port);
int ack_q(struct ordq_item_t *which);
int ackq_check();
int cmd_broadcast(int cbr, char *msg, int mlen);
int cmd_rsend(struct ackq_item_t *cmdr);
int ack_remove(struct p_order_t *pack, struct sockaddr_in *from);
int secure_write(int sock, void *buf, int len);
//int secure_write(int sock, int ip, unsigned short port, void *buf, int len);
/*-----------*/

int cmd_q(struct client_t *who, int cmd, char *msg, int msglen)
{
 struct ordq_item_t *item;
 int efflen = LC_MAX_MSG_LEN;
 if ( msglen < efflen ) efflen = msglen;
 if ( (item = (struct ordq_item_t*)list_add(&ordq, sizeof(struct ordq_item_t))) == NULL )
 {
  syslog(LOG_WARNING, "cmd_q(): malloc(%d) returned NULL", sizeof(struct ordq_item_t));
  return(-2);
 }
 item->ip = who->ip;
 item->who = who;
 item->cmd = cmd;
 item->len = efflen;
 if ( msg != NULL ) memcpy(item->msg, msg, efflen);
 // look for waiting ackq-entries of this client
 if ( list_find(&ackq, (int)who) == NULL ) return(cmd_send(item));
 return(0);
}

int cmd_send(struct ordq_item_t *cmdr)
{
 int slen, len;
 struct sockaddr_in other;
 struct p_order_t pack;
 pack.magic = LC_MAGIC_NR;
 pack.version = LC_PACKET_VERSION;
 pack.id = cmdr->who;
 pack.cmd = cmdr->cmd;
 pack.interval = cmdr->who->interval;
 pack.dum = 0;
 pack.msglen = cmdr->len;
 memcpy(pack.msg, cmdr->msg, cmdr->len);
 // ok, packet is ready
 other.sin_family = AF_INET;
 other.sin_port = cmdr->who->port;
 other.sin_addr.s_addr = cmdr->who->ip;
 len = sizeof(struct p_order_t) - LC_MAX_MSG_LEN + pack.msglen - 1;
 switch ( cmdr->who->type )
 {
     case LCS_CLT_LC:
         slen = secure_write(cmdr->who->tcp_sock, &pack, len);
//         slen = secure_write(cmdr->who->tcp_sock, cmdr->who->ip, cmdr->who->port, &pack, len);
         break;
     case LCS_CLT_DC:
         slen = sendto(server->udp_socket, &pack, len+1, 0,
	               (struct sockaddr*)&other, sizeof(struct sockaddr_in));
		       // len+1 because they ran well with that till now...
	 len++;
	 break;
     default: slen = len;
 }
 if ( slen != len )
 {
  syslog(LOG_WARNING, "cmd_send(): %s:%d error sending packet.", inet_ntoa(other.sin_addr), ntohs(other.sin_port));
  return(-1);
 }
 else ack_q(cmdr);
 return(0);
}

int ack_send(struct sockaddr_in *trg, struct p_order_t *cmdpack)
{
 struct p_ack_t ack;
 int slen, type;
 struct client_t *who;
 ack.magic = LC_MAGIC_NR;
 ack.version = LC_PACKET_VERSION;
 ack.id = cmdpack->id;
 ack.cmd = CMD_ACK;
 ack.ackcmd = cmdpack->cmd;
 // LCC or DCC ?? --> UDP or TCP?
 who = get_client(trg->sin_addr.s_addr, trg->sin_port);
 if ( !who ) type = SOCK_TYPE_UDP;
 else switch ( who->type )
 {
     case LCS_CLT_LC: type = SOCK_TYPE_TCP; break;
     case LCS_CLT_DC: type = SOCK_TYPE_UDP; break;
     default: type = 0;
 }
 switch ( type )
 {
     case SOCK_TYPE_TCP:
         slen = secure_write(who->tcp_sock, &ack, sizeof(struct p_ack_t));
//         slen = secure_write(who->tcp_sock, who->ip, who->port, &ack, sizeof(struct p_ack_t));
         break;
     case SOCK_TYPE_UDP:
         slen = sendto(server->udp_socket, &ack, sizeof(struct p_ack_t), 0,
	               (struct sockaddr*)trg, sizeof(struct sockaddr_in));
         break;
     default: slen = 0;
 }
 if ( slen != sizeof(struct p_ack_t) ) return(-1);
 return(0);
}

int cmd_rsend(struct ackq_item_t *cmdr)
{
 int slen, len;
 struct sockaddr_in other;
 struct p_order_t pack;
 pack.magic = LC_MAGIC_NR;
 pack.version = LC_PACKET_VERSION;
 pack.id = cmdr->who;
 pack.cmd = cmdr->cmd;
 pack.interval = cmdr->who->interval;
 pack.dum = 0;
 pack.msglen = cmdr->len;
 if ( cmdr->len != 0 ) memcpy(pack.msg, cmdr->msg, cmdr->len);
 // ok, packet is ready
 len = sizeof(struct p_order_t) - LC_MAX_MSG_LEN + pack.msglen - 1;
 other.sin_family = AF_INET;
 other.sin_port = cmdr->who->port;
 other.sin_addr.s_addr = cmdr->who->ip;
 switch ( cmdr->who->type )
 {
     case LCS_CLT_LC:
         slen = secure_write(cmdr->who->tcp_sock, &pack, len);
//         slen = secure_write(cmdr->who->tcp_sock, cmdr->who->ip, cmdr->who->port, &pack, len);
         break;
     case LCS_CLT_DC:
         slen = sendto(server->udp_socket, &pack, len+1, 0,
	              (struct sockaddr*)&other, sizeof(struct sockaddr_in));
		      // send the 'old-DC-style' junk-byte to
	 len++;
         break;
     default:
     syslog(LOG_ERR, "cmd_rsend(): bad client type");
     slen = 0;
 }
 if ( slen != len )
 {
  syslog(LOG_WARNING, "cmd_rsend(): %s:%d error sending packet.", inet_ntoa(other.sin_addr), ntohs(other.sin_port));
  return(-1);
 }
 cmdr->prev_try = time(NULL);
 cmdr->cnt--;
 return(0);
}

int cmd_direct(int cmd, int sock, struct sockaddr_in *who, char *msg, int mlen, char type)
{
    int res = -1;
    switch ( type )
    {
	case SOCK_TYPE_TCP:
	    res = cmd_direct_tcp(cmd, sock, who, msg, mlen);
	    break;
	case SOCK_TYPE_UDP:
	    res = cmd_direct_udp(cmd, who, msg, mlen);
	    break;
    }
    return(res);
}

int cmd_direct_udp(int cmd, struct sockaddr_in *who, char *msg, int mlen)
{
 struct p_order_t pack;
 int slen, len;
 if ( (len = mlen) > LC_MAX_MSG_LEN ) len = LC_MAX_MSG_LEN;
 pack.magic = LC_MAGIC_NR;
 pack.version = LC_PACKET_VERSION;
 pack.cmd = cmd;
 pack.id = 0;		// client has no id!
 pack.interval = 0;	// this client has no record...
 pack.dum = 0;
 pack.msglen = len;
 if ( msg != NULL ) memcpy(pack.msg, msg, len);
 len = sizeof(struct p_order_t) - LC_MAX_MSG_LEN + pack.msglen;
 slen = sendto(server->udp_socket, &pack, len, 0, (struct sockaddr*)who, sizeof(struct sockaddr_in));
 if ( slen != len )
 {
     syslog(LOG_WARNING, "cmd_direct_udp(): %s:%d error sending packet.", inet_ntoa(who->sin_addr), ntohs(who->sin_port));
     return(-1);
 }
 return(0);
}

int cmd_direct_tcp(int cmd, int sock, struct sockaddr_in *who, char *msg, int mlen)
{
 struct p_order_t pack;
 int slen, len;
 if ( (len = mlen) > LC_MAX_MSG_LEN ) len = LC_MAX_MSG_LEN;
 pack.magic = LC_MAGIC_NR;
 pack.version = LC_PACKET_VERSION;
 pack.cmd = cmd;
 pack.id = 0;		// client has no id!
 pack.interval = 0;	// this client has no record...
 pack.dum = 0;
 pack.msglen = len;
 if ( msg != NULL ) memcpy(pack.msg, msg, len);
 len = sizeof(struct p_order_t) - LC_MAX_MSG_LEN + pack.msglen - 1;
 slen = secure_write(sock, &pack, len);
// slen = secure_write(sock, who->sin_addr.s_addr, who->sin_port, &pack, len);
 if ( slen != len )
 {
     syslog(LOG_WARNING, "cmd_direct_tcp(): %s:%d error sending packet.", inet_ntoa(who->sin_addr), ntohs(who->sin_port));
     return(-1);
 }
 return(0);
}

int ack_proc(int ip, short port)
{
 struct ackq_item_t *aq;
 struct ordq_item_t *oq;
 struct client_t *who;
 if ( (who = get_client(ip, port)) == NULL ) return(-1);
 /* the ack got removed from ackq by ack_remove() */
 oq = (struct ordq_item_t *)ordq.first;
 while ( oq != NULL )
 {
     if ( oq->who == who )
     {
         aq = (struct ackq_item_t *)ackq.first;
         while ( aq != NULL )
         {
          if ( aq->who == who )
          {
	      syslog(LOG_WARNING, "ack_proc(): client deadlock!");
	      return(1);
	  } // still waiting for an ack (this can not happen!)
          aq = (struct ackq_item_t *)aq->next; // but you never know...
         }
         return(cmd_send(oq));
     }
     oq = (struct ordq_item_t *)oq->next;
 }
 return(0);
}

int ack_q(struct ordq_item_t *which) // moves cmd from ordq to ackq
{
 struct ackq_item_t *item;
 if ( which->cmd >= CBR_INFO ) return(0); // we shouldn't get acks for cmds
                                          // >= CBR_INFO, so block here.
 if ( (item = (struct ackq_item_t*)list_add(&ackq, sizeof(struct ackq_item_t))) == NULL )
 {
  syslog(LOG_WARNING, "ack_q(): malloc(%d) returned NULL", sizeof(struct ackq_item_t));
  return(-2);
 }
 item->ip = which->ip;
 item->who = which->who;
 item->prev_try = time(NULL); // these few msecs...
                 // I assume you're not using something slower than a i386...
 item->cnt = LC_ACK_CNT;
 item->cmd = which->cmd;
 item->len = which->len;
 if ( item->len > 0 ) memcpy(item->msg, which->msg, which->len);
 list_del(&ordq, (struct list_hdr_t*)which);
 return(0);
}

int ackq_check() // looks for timed out acks (resend / kick client)
{
 struct ackq_item_t *aq;
 struct ackq_item_t *dum;
 aq = (struct ackq_item_t*)ackq.first;
 while ( aq )
 {
    dum = (struct ackq_item_t*)aq->next;
    // I don't know why there are entries from a LCP3 client in this list...
    // anyway, let's kill them; the LCP3 client worked well :)
    // 2000113: sometimes the netinput mechanism doesn't interpret
    // the packets well... so it gets strange random numbers... :(
    // But don't ask me then why the packet goes through... I'm the
    // opinion it doesn't!
    if ( (aq->who->type != LCS_CLT_LC) && (aq->who->type != LCS_CLT_DC) )
    {
	list_del(&ackq, (struct list_hdr_t*)aq);
	aq = dum;
	continue;
    }
    if ( aq->prev_try+LC_ACK_WAITQ <= time(NULL) )
    {
	if ( aq->cnt > 0 )	// send cmd again
	    cmd_rsend(aq);
	else			// kick client
	{
	    // now take the first entry with ip != aq->ip for dum!
	    // so we have the next record to check when aq->who got kicked.
	    while ( (dum != NULL) && (dum->who == aq->who) )
		dum = (struct ackq_item_t *)dum->next;
	    syslog(LOG_INFO, "kick-reason: command not acked");
    	    kick_client(aq->who, 0);
	    dum = list_secure(&ackq, (struct list_hdr_t*)dum);
	}
    }
    aq = dum;
    //if ( aq == NULL ) aq = (struct ackq_item_t *)ackq.first;
 }
 return(0);
}

int cmd_broadcast(int cbr, char *msg, int mlen)
/* it's not a true broadcast. we just send it to each client... */
{
 int slen, len;
 struct p_order_t pack;
 struct sockaddr_in trg;
 struct client_t *who;
 pack.magic = LC_MAGIC_NR;
 pack.version = LC_PACKET_VERSION;
 pack.dum = 0;
 pack.cmd = cbr;
 if ( (len = mlen) > LC_MAX_MSG_LEN ) len = LC_MAX_MSG_LEN;
 pack.msglen = len;
 if ( msg ) memcpy(pack.msg, msg, len);
 len = sizeof(struct p_order_t) - LC_MAX_MSG_LEN - 1 + pack.msglen;
 clean_clients(NULL);
 who = (struct client_t *)cltlist.first;
 while ( who )
 {
     if ( (who->type != LCS_CLT_DC) && (who->type != LCS_CLT_LC) )
     {
         who = (struct client_t*)who->next;
	 continue;
     }
     pack.id = who;
     pack.interval = who->interval;
     trg.sin_family = AF_INET;
     trg.sin_port = who->port;
     trg.sin_addr.s_addr = who->ip;
     switch ( who->type )
     {
         case LCS_CLT_LC:
	     if ( who->status == CLT_NOAUTH ) break; // not regged yet...
             slen = secure_write(who->tcp_sock, &pack, len);
//             slen = secure_write(who->tcp_sock, who->ip, who->port, &pack, len);
	     break;
	 case LCS_CLT_DC:
	     len++;
             slen = sendto(server->udp_socket, &pack, len, 0,
	                  (struct sockaddr *)&trg, sizeof(struct sockaddr_in));
	     len--;
	     break;
     }
     // one could slen==/!=len handle here... but we have to try every clt!!
     who = (struct client_t*)who->next;
 }
 return(0);
}

int ack_remove(struct p_order_t *pack, struct sockaddr_in *from)
{
 struct ackq_item_t *aq = (struct ackq_item_t*)ackq.first;
 struct client_t *who = get_client(from->sin_addr.s_addr, from->sin_port);
 while ( (aq != NULL) && who )
 {
     if ( (aq->cmd == ((struct p_ack_t*)pack)->ackcmd)
          && (aq->who == who) )
     {
         list_del(&ackq, (struct list_hdr_t*)aq);
	 return(0);
     }
     aq = (struct ackq_item_t*)aq->next;
 }
 return(-1);
}

int secure_write(int sock, void *buf, int len)
{
    fd_set wfd;
    struct timeval tv;
    int slen = -1;
    FD_ZERO(&wfd);
    FD_SET(sock, &wfd);
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    if ( select(sock+1, NULL, &wfd, NULL, &tv) == 1 )
    {
//	slen = write(sock, buf, len);
	slen = send(sock, buf, len, 0);
    }
    return(slen);
}
/*
int secure_write(int sock, int ip, unsigned short port, void *buf, int len)
{
    fd_set wfd;
    struct timeval tv;
    int slen = -1;
    struct sockaddr_in to;
    FD_ZERO(&wfd);
    FD_SET(sock, &wfd);
    tv.tv_sec = 0;
    tv.tv_usec = 0;
    to.sin_family = AF_INET;
    to.sin_port = port;
    to.sin_addr.s_addr = ip;
    if ( select(sock+1, NULL, &wfd, NULL, &tv) == 1 )
    {
	slen = sendto(sock, buf, len, 0, (struct sockaddr*)&to, sizeof(struct sockaddr_in));
    }
    return(slen);
}
*/
