/***********************************************************************************

	Copyright (C) 2009 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include <gtkmm/table.h>

#include "dialogexport.hpp"
#include "dialogpassword.hpp"


using namespace LIFEO;

// DIALOGSAVEDIARY
DialogSaveDiary::DialogSaveDiary( )
	:	Gtk::FileChooserDialog(
		// TRANSLATORS: this is the title of filechooser dialog for saving a new diary
		_( "Where to Save The Diary?" ), Gtk::FILE_CHOOSER_ACTION_SAVE )
{
	add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	add_button( Gtk::Stock::OK, Gtk::RESPONSE_ACCEPT );
}


// DIALOGEXPORT
DialogExport::DialogExport( Database *database )
	:	Gtk::Dialog( _( "Save a Copy of the Diary" ) ),
	m_checkbutton_overwrite( _("Overwrite" ) ),
	m_ptr2widget_options( NULL ), m_ptr2database( database )
{
	set_default_size( 450, -1 );

	add_saver( new SaverEncrypted( database ) );
	add_saver( new SaverUnencrypted( database ) );
	add_saver( new ExporterText( database ) );

	m_table.set_spacings( 5 );
	m_table.set_border_width( 5 );

	Gtk::Label *label_file = Gtk::manage( new Gtk::Label( _( "File:" ) ) );
	// TRANSLATORS: type of file export operation will produce
	Gtk::Label *label_type = Gtk::manage( new Gtk::Label( _( "Type:" ) ) );

	m_entry_path.set_icon_from_icon_name( "folder" );
	m_entry_path.set_icon_tooltip_text( _( "Browse..." ) );

	m_table.attach( *label_file, 0, 1, 0, 1, Gtk::SHRINK, Gtk::SHRINK );
	m_table.attach( m_entry_path, 1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );
	m_table.attach( m_checkbutton_overwrite, 1, 2, 1, 2, Gtk::FILL, Gtk::SHRINK );
	m_table.attach( *label_type, 0, 1, 2, 3, Gtk::SHRINK, Gtk::SHRINK );
	m_table.attach( m_combobox, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );

	get_vbox()->pack_start( m_table );

	add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	m_button_save = add_button( Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT );

	m_combobox.signal_changed().connect(
			sigc::mem_fun( *this, &DialogExport::handle_savetype_changed ) );

	m_entry_path.signal_icon_release().connect(
			sigc::mem_fun( *this, &DialogExport::handle_browse ) );

	m_entry_path.signal_changed().connect(
			sigc::mem_fun( *this, &DialogExport::handle_path_changed ) );

	m_checkbutton_overwrite.signal_toggled().connect(
			sigc::mem_fun( *this, &DialogExport::handle_overwrite_toggled ) );

	std::string path = database->get_path();
	path += ".backup";
	m_entry_path.set_text( path );
	m_combobox.set_active( 0 );  // defaults to encrypted diary

	show_all();
}

void
DialogExport::add_saver( Databasesaver *saver )
{
	m_savers.push_back( saver );
	m_combobox.append_text( saver->get_name() );
	saver->signal_updated().connect(
		sigc::mem_fun( *this, &DialogExport::handle_saver_updated ) );
}

void
DialogExport::handle_saver_updated( bool ready )
{
	m_flag_saverready = ready;
	m_button_save->set_sensitive( is_ready( ) );
}

void
DialogExport::handle_savetype_changed( void )
{
	int i = m_combobox.get_active_row_number();
	
	if( m_ptr2widget_options )
	{
		m_table.remove( *m_ptr2widget_options );
		delete m_ptr2widget_options;
	}
	m_ptr2widget_options = m_savers[ i ]->draw_options();
	if( m_ptr2widget_options )
		m_table.attach( *m_ptr2widget_options, 1, 2, 3, 4, Gtk::FILL, Gtk::FILL );
	
	m_flag_saverready = m_savers[ i ]->is_ready();
	
	m_button_save->set_sensitive( is_ready() );
}

void
DialogExport::handle_browse( Gtk::EntryIconPosition, const GdkEventButton* )
{
	DialogSaveDiary *dialog_save = new DialogSaveDiary();

	dialog_save->set_filename( m_entry_path.get_text() );

	if( dialog_save->run() == Gtk::RESPONSE_ACCEPT )
		m_entry_path.set_text( dialog_save->get_filename() );

	delete dialog_save;
}

void
DialogExport::handle_path_changed( void )
{
	m_checkbutton_overwrite.set_active( false );

	if( m_entry_path.get_text().empty() )
	{
		m_flag_fileready = false;
		m_checkbutton_overwrite.set_sensitive( false );
		m_button_save->set_sensitive( false );
	}
	else
	if( Glib::file_test( m_entry_path.get_text(), Glib::FILE_TEST_EXISTS ) )
	{
		m_flag_fileready = false;
		m_checkbutton_overwrite.set_sensitive( true );
		m_button_save->set_sensitive( false );
	}
	else
	{
		m_flag_fileready = true;
		m_checkbutton_overwrite.set_sensitive( false );
		m_button_save->set_sensitive( is_ready() );
	}
}

void
DialogExport::handle_overwrite_toggled( void )
{
	m_flag_fileready = m_checkbutton_overwrite.get_active();
	m_button_save->set_sensitive( is_ready() );
}

inline bool
DialogExport::is_ready( void )
{
//	if( m_flag_fileready == false )
//		return false;
//
//	if( m_flag_saverready == false )
//		return false;

//	return true;
	return( m_flag_fileready && m_flag_saverready );
}

void
DialogExport::on_response( int response )
{
	if( response == Gtk::RESPONSE_ACCEPT )
	{
		m_savers[ m_combobox.get_active_row_number() ]->save(
				m_entry_path.get_text() );
	}
}

// SAVERS & EXPORTERS
Databasesaver::Databasesaver( const Glib::ustring& name, Database *database )
	: m_name( name ), m_ptr2database( database )
{
}


SaverEncrypted::SaverEncrypted( Database *database )
	:	Databasesaver( _( "Encrypted Diary File" ), database ),
	m_passphrase( "" )
{
}

Gtk::Widget*
SaverEncrypted::draw_options( void )
{
	m_entry_passphrase = Gtk::manage( new Gtk::Entry );
	m_entry_confirm = Gtk::manage( new Gtk::Entry );
	Gtk::Table *table = new Gtk::Table;

	m_entry_passphrase->set_visibility( false );
	m_entry_confirm->set_visibility( false );

	table->set_spacings( 5 );
	table->set_border_width( 5 );
	table->attach(	* Gtk::manage( new Gtk::Label( _( "Password:" ) ) ),
					0, 1, 0, 1, Gtk::SHRINK );
	table->attach( *m_entry_passphrase,
					1, 2, 0, 1, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );
	// TRANSLATORS: confirm here means re-enter password
	table->attach(	* Gtk::manage( new Gtk::Label( _( "Confirm:" ) ) ),
					0, 1, 1, 2, Gtk::SHRINK );
	table->attach( *m_entry_confirm,
					1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );

	// TODO: re-use password entry table along with its match indicator icon...
	// ...in DialogPassword

	m_entry_passphrase->signal_changed().connect(
			sigc::mem_fun( *this, &SaverEncrypted::handle_passphrase_changed ) );

	m_entry_confirm->signal_changed().connect(
			sigc::mem_fun( *this, &SaverEncrypted::handle_passphrase_changed ) );

	table->show_all();

	return table;
}

bool
SaverEncrypted::is_ready( void )
{
	return( m_passphrase.size() >= LIFEO::PASSPHRASE_MIN_SIZE );
}

void
SaverEncrypted::handle_passphrase_changed( void )
{
	m_passphrase = m_entry_passphrase->get_text();
	if( m_passphrase != m_entry_confirm->get_text() )
		m_passphrase.clear();
	else
	if( m_passphrase.size() < LIFEO::PASSPHRASE_MIN_SIZE )
		m_passphrase.clear();
	
	m_signal_updated.emit( m_passphrase.size() >= LIFEO::PASSPHRASE_MIN_SIZE );
}

Result
SaverEncrypted::save( const std::string &path )
{
	if( m_passphrase.size() < LIFEO::PASSPHRASE_MIN_SIZE )
		return FAILURE;
	else
		return m_ptr2database->write_copy( path, m_passphrase );
}


SaverUnencrypted::SaverUnencrypted( Database *database )
	: Databasesaver( _( "Unencrypted Diary File" ), database )
{
}

Result
SaverUnencrypted::save( const std::string &path )
{
	return m_ptr2database->write_copy( path, "" );
}


ExporterText::ExporterText( Database *database )
	: Databasesaver( _( "Plain Text File" ), database )
{

}

Result
ExporterText::save( const std::string &path )
{
	return m_ptr2database->write_txt( path );
}

