/***************************************************************************
                          slot.c  -  description
                             -------------------
    begin                : Sat Jun 23 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <SDL.h>

#ifdef WITH_SOUND
#include <SDL_mixer.h>
#include "audio.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <dirent.h>
#include "sdl.h"
#include "dynlist.h"
#include "tools.h"
#include "config.h"
#include "date.h"
#include "nation.h"
#include "unit.h"
#include "player.h"
#include "theme.h"
#include "map.h"
#include "scenario.h"
#include "campaign.h"
#include "gui.h"
#include "ai_action.h"
#include "engine_tools.h"
#include "engine.h"
#include "slot.h"

extern Config config;
extern Map map;
extern Scen scen;
extern Camp camp;

/* true if this is a saved game; return slot index 0-9 */
int is_saved_game( char *file_name, int *i )
{
    if ( strncmp( file_name, "lg_save_", 8 ) ) return 0;
    (*i) = 0;
    (*i) = (char)file_name[8] - 48;
    return 1;
}

/* check the save directory for saved games and add them to the slot list else
setup a new entry: '_index_ <empty>' */
void init_slots( Engine *engine )
{
    int i;
    DIR *dir = 0;
    struct dirent *entry = 0;
    FILE *file = 0;

    if ( ( dir = opendir( config.dir_name ) ) == 0 ) {
        fprintf( stderr, "init_slots: can't open directory '%s' to read saved games\n",
                 config.dir_name );
        return;
    }

    /* set all slots empty */
    for ( i = 0; i < SLOT_COUNT; i++ ) {
        strcpy( engine->slots[i].name, "<emtpy>" );
        if ( i + 1 < 10 )
            sprintf( engine->slots[i].slot_name, "%i.  %s", i + 1, engine->slots[i].name );
        else
            sprintf( engine->slots[i].slot_name, "%i. %s", i + 1, engine->slots[i].name );
        engine->slots[i].file_name[0] = 0;
        engine->slots[i].valid = 0;
    }

    /* read all directory entries */
    while ( ( entry = readdir( dir ) ) != 0 ) {
        if ( is_saved_game( entry->d_name, &i ) ) {
            sprintf( engine->slots[i].file_name, "%s/%s", config.dir_name, entry->d_name );
            if ( ( file = fopen( engine->slots[i].file_name, "r" ) ) == 0 ) {
                fprintf( stderr, "file '%s' not accessable\n", engine->slots[i].file_name );
                break;
            }
            /* read slot::slot_name and slot::name saved there */
            read_slot_name( &engine->slots[i], file );
            fclose( file );
            engine->slots[i].valid = 1;
        }
    }

}

/* get the slot names of the saved games and return them as a list;
must be 10 entries */
Text* get_slot_names( Engine *engine )
{
    Text *text = 0;
    int i;

    text = calloc( 1, sizeof( Text ) );
    text->count = SLOT_COUNT;
    text->lines = calloc( text->count, sizeof( char*) );

    for ( i = 0; i < SLOT_COUNT; i++ )
        text->lines[i] = strdup( engine->slots[i].slot_name );

    return text;
}

/* read slot::name and slot::slot_name from file */
void write_slot_name( Save_Slot *slot, FILE *file )
{
    fwrite( slot->name, sizeof( slot->name ), 1, file );
    fwrite( slot->slot_name, sizeof( slot->slot_name ), 1, file );
}

/* read slot::name and slot::slot_name from file */
void read_slot_name( Save_Slot *slot, FILE *file )
{
    fread( slot->name, sizeof( slot->name ), 1, file );
    fread( slot->slot_name, sizeof( slot->slot_name ), 1, file );
}

/* set slot::name to name and update the slot::slot_name string */
void update_slot_name( Engine *engine, int slot_id, char *name )
{
    strcpy( engine->slots[slot_id].name, name );
    if ( slot_id + 1 < 10 )
        sprintf( engine->slots[slot_id].slot_name, "%i.  %s", slot_id + 1, name );
    else
        sprintf( engine->slots[slot_id].slot_name, "%i. %s", slot_id + 1, name );
}

/*
====================================================================
Save/load a single integer
====================================================================
*/
void save_int( FILE *file, int i )
{
    fwrite( &i, sizeof( int ), 1, file );
}
int load_int( FILE *file )
{
    int i;
    fread( &i, sizeof( int ), 1, file );
    return i;
}
/*
====================================================================
Save/load a string to/from file.
====================================================================
*/
void save_string( FILE *file, char *str )
{
    int length;
    /* save length and then string itself */
    length = strlen( str );
    fwrite( &length, sizeof( int ), 1, file );
    fwrite( str, sizeof( char ), length, file );
}
char* load_string( FILE *file )
{
    char *str = 0;
    int length;

    fread( &length, sizeof( int ), 1, file );
    str = calloc( length + 1, sizeof( char ) );
    fread( str, sizeof( char ), length, file );
    str[length] = 0;
    return str;
}
/*
====================================================================
Save/load a unit from/to file.
The only pointers are sel_prop which can turned into an indices
and unit::cur_prop::attack which will be saved right after
the unit struct itself. However, prop and tran_prop have some
pointers that will be reloaded so we need to keep their indices.
As the unit itself must not changed (we're in the game right?)
the struct is doubled.
====================================================================
*/
void save_unit( FILE *file, Unit *unit )
{
    int sel_prop_index;

    /* layout:
        unit struct
        sel_prop index
        prop entry_name
        tran_prop entry_name (optional)
    */

    /* save unit struct */
    fwrite( unit, sizeof( Unit ), 1, file );

    /* sel_prop into index */
    if ( unit->sel_prop == &unit->prop )
        sel_prop_index = 0; /* points at prop */
    else
        sel_prop_index = 1; /* points at tran_prop */
    fwrite( &sel_prop_index, sizeof( int ), 1, file );

    /* remember indices of prop and tran_prop by saving their entry names */
    /* if there is no tran_prop this is already saved in tran_prop_set */
    save_string( file, unit->prop.entry_name );
    if ( unit->tran_prop_set )
        save_string( file, unit->tran_prop.entry_name );
}
Unit* load_unit( FILE *file )
{
    Unit_Lib_Entry *lib_entry;
    Unit *unit = 0;
    char *prop_entry_name, *tran_prop_entry_name = 0;
    int sel_prop_index;
    int i;

    unit = calloc( 1, sizeof( Unit ) );

    /* load unit */
    fread( unit, sizeof( Unit ), 1, file );

    /* load sel_prop index */
    fread( &sel_prop_index, sizeof( int ), 1, file );
    if ( sel_prop_index == 0 )
        unit->sel_prop = &unit->prop;
    else
        unit->sel_prop = &unit->tran_prop;

    /* load prop entry names */
    prop_entry_name = load_string( file );
    if ( unit->tran_prop_set )
        tran_prop_entry_name = load_string( file );

    /* repair the broken pointers in prop and tran_prop */
    for ( i = 0; i < scen.unit_lib.count; i++ ) {
        lib_entry = (Unit_Lib_Entry*)dl_get( &scen.unit_lib, i );
        if ( equal_str( prop_entry_name, lib_entry->entry_name ) ) {
            unit->prop.entry_name = lib_entry->entry_name;
            unit->prop.cap = lib_entry->cap;
            unit->prop.unit_def = lib_entry->unit_def;
            unit->prop.pic = lib_entry->pic;
            unit->prop.small_pic = lib_entry->small_pic;
#ifdef WITH_SOUND
            unit->prop.move_sound = lib_entry->move_sound;
#endif
        }
    }
    if ( unit->tran_prop_set ) {
        for ( i = 0; i < scen.unit_lib.count; i++ ) {
            lib_entry = (Unit_Lib_Entry*)dl_get( &scen.unit_lib, i );
            if ( equal_str( tran_prop_entry_name, lib_entry->entry_name ) ) {
                unit->tran_prop.entry_name = lib_entry->entry_name;
                unit->tran_prop.cap = lib_entry->cap;
                unit->tran_prop.unit_def = lib_entry->unit_def;
                unit->tran_prop.pic = lib_entry->pic;
                unit->tran_prop.small_pic = lib_entry->small_pic;
#ifdef WITH_SOUND
                unit->tran_prop.move_sound = lib_entry->move_sound;
#endif
            }
        }
    }
    /* current unit properties */
    unit->cur_prop.entry_name = unit->sel_prop->entry_name;
    unit->cur_prop.cap = unit->sel_prop->cap;
    unit->cur_prop.unit_def = unit->sel_prop->unit_def;
    unit->cur_prop.pic = unit->sel_prop->pic;
    unit->cur_prop.small_pic = unit->sel_prop->small_pic;
#ifdef WITH_SOUND
    unit->cur_prop.move_sound = unit->sel_prop->move_sound;
#endif

    /* free auxiliaries */
    free( prop_entry_name );
    if ( unit->tran_prop_set )
       free( tran_prop_entry_name );

    return unit;
}
/*
====================================================================
Save/load player structs.
====================================================================
*/
void save_player( FILE *file, Player *player )
{
    int i;

    /* layout:
        struct
        entry_name
        name
        diplomacy
        reinf count
        reinforcements
    */

    /* struct itself */
    fwrite( player, sizeof( Player ), 1, file );
    /* strings */
    save_string( file, player->entry_name );
    save_string( file, player->name );
    /* diplomacy */
    fwrite( player->dipl, sizeof( int ), scen.unit_def->type_count, file );
    /* reinforcements */
    save_int( file, player->avail_reinf.count );
    for ( i = 0; i < player->avail_reinf.count; i++ )
        save_unit( file, (Unit*)dl_get( &player->avail_reinf, i ) );
}
Player* load_player( FILE *file )
{
    Player *player;
    int i, reinf_count;

    player = calloc( 1, sizeof( Player ) );

    fread( player, sizeof( Player ), 1, file );
    player->entry_name = load_string( file );
    player->name = load_string( file );
    player->dipl = calloc( scen.unit_def->type_count, sizeof( int ) );
    fread( player->dipl, scen.unit_def->type_count, sizeof( int ), file );

    reinf_count = load_int( file );
    dl_init( &player->avail_reinf, AUTO_DELETE, delete_unit );
    for ( i = 0; i < reinf_count; i++ )
        dl_add( &player->avail_reinf, load_unit( file ) );

    return player;
}
/*
====================================================================
Save indices in scen::units of ground and air unit on this tile.
====================================================================
*/
void save_map_tile_units( FILE *file, Map_Tile *tile )
{
    int i;
    int index;
    /* layout:
        ground unit index
        air unit index
    */

    index = -1;
    if ( tile->unit )
        for ( i = 0; i < scen.units.count; i++ )
            if ( tile->unit == (Unit*)dl_get( &scen.units, i ) ) {
                index = i;
                break;
            }
    save_int( file, index );

    index = -1;
    if ( tile->air_unit )
        for ( i = 0; i < scen.units.count; i++ )
            if ( tile->air_unit == (Unit*)dl_get( &scen.units, i ) ) {
                index = i;
                break;
            }
    save_int( file, index );
}
/* load map tile units assuming that scen::units is set to the correct units */
void load_map_tile_units( FILE *file, Unit **unit, Unit **air_unit )
{
    int index;
    index = load_int( file );
    if ( index == -1 ) *unit = 0;
    else
        *unit = (Unit*)dl_get( &scen.units, index );
    index = load_int( file );
    if ( index == -1 ) *air_unit = 0;
    else
        *air_unit = (Unit*)dl_get( &scen.units, index );
}
/*
====================================================================
Save map flags.
====================================================================
*/
void save_map_tile_flag( FILE *file, Map_Tile *tile )
{
    /* layout:
        nation_id
        player_id
    */
    save_int( file, tile->nation_id );
    save_int( file, tile->player_id );
}
void load_map_tile_flag( FILE *file, int *nation_id, int *player_id )
{
    *nation_id = load_int( file );
    *player_id = load_int( file );
}

/* save game to a slot; filename: lg_slot_index */
int save_game( Engine *engine, int slot_id )
{
    FILE *file = 0;
    char file_name[512];
    int i, j;

    /* layout:
        slot_name
        campaign loaded
        campaign name (optional)
        campaign scenario id (optional)
        scenario file name
        current turns
        remaining turns
        current player_id
        player count
        player structs
        unit count
        units
        reinf count
        reinf
        map width
        map height
        map tile units
        map tile flags
    */

    /* get file name */
    sprintf( file_name, "%s/lg_save_%i", config.dir_name, slot_id );
    /* open file */
    if ( ( file = fopen( file_name, "w" ) ) == 0 ) {
        fprintf( stderr, "can't open file '%s' to save game\n", file_name );
        return 0;
    }
    printf( "saving '%s'...\n", engine->slots[slot_id].slot_name );

    /* write slot identification */
    write_slot_name( &engine->slots[slot_id], file );

    /* if campaing is set some campaign info follows */
    fwrite( &camp.loaded, sizeof( int ), 1, file );
    if ( camp.loaded ) {
        save_string( file, camp.file_name );
        save_int( file, camp.scen_id );
    }

    /* basic data */
    save_string( file, scen.file_name );
    save_int( file, scen.cur_turn );
    save_int( file, scen.rem_turns );
    save_int( file, engine->player_id );

    /* players */
    save_int( file, scen.player_count );
    for ( i = 0; i < scen.player_count; i++ )
        save_player( file, scen.players[i] );

    /* units */
    save_int( file, scen.units.count );
    for ( i = 0; i < scen.units.count; i++ )
        save_unit( file, (Unit*)dl_get( &scen.units, i ) );
    /* reinforcements */
    save_int( file, scen.reinf.count );
    for ( i = 0; i < scen.reinf.count; i++ )
        save_unit( file, (Unit*)dl_get( &scen.reinf, i ) );

    /* map stuff */
    save_int( file, map.width );
    save_int( file, map.height );
    for ( i = 0; i < map.width; i++ )
        for ( j = 0; j < map.height; j++ )
            save_map_tile_units( file, map_tile( i, j ) );
    for ( i = 0; i < map.width; i++ )
        for ( j = 0; j < map.height; j++ )
            save_map_tile_flag( file, map_tile( i, j ) );

    fclose( file );
    return 1;
}

/* load game from slot */
int load_game( Engine *engine, int slot_id )
{
    FILE *file = 0;
    char file_name[512];
    int camp_saved;
    int i, j;
    Unit_Lib_Entry *def_air_tran;
    Unit_Lib_Entry *def_sea_tran;
    char *scen_file_name = 0;
    int unit_count;

    /* get file name */
    sprintf( file_name, "%s/lg_save_%i", config.dir_name, slot_id );
    /* open file */
    if ( ( file = fopen( file_name, "r" ) ) == 0 ) {
        fprintf( stderr, "can't open file '%s' to load saved game\n", file_name );
        return 0;
    }
    printf( "loading '%s'...\n", engine->slots[slot_id].slot_name );

    /* read slot identification -- won't change anything but the file handle needs to move */
    read_slot_name( &engine->slots[slot_id], file );

    /* if campaing is set some campaign info follows */
    clear_camp(); /* if no campaign is loaded the current one must be cleared anyway */
    fread( &camp_saved, sizeof( int ), 1, file );
    if ( camp_saved ) {
        /* reload campaign and set to current scenario id */
        load_camp( load_string( file ) );
        camp.scen_id = load_int( file );
    }

    /* the scenario that is loaded now is the one that belongs to the scenario id of the campaign above */

    /* read scenario file name */
    scen_file_name = load_string( file );

    /* shutdown game and reinit */
    clear_engine( engine );
    init_engine( engine, scen_file_name, 0 );
    engine->scen_loaded = 1;

    /* basic data */
    scen.cur_turn = load_int( file );
    scen.rem_turns = load_int( file );
    engine->player_id = load_int( file ); /* player pointer is set after loading players */

    /* players */
    scen.player_count = load_int( file );
    for ( i = 0; i < scen.player_count; i++ ) {
        /* backup default transporters */
        def_air_tran = scen.players[i]->def_air_tran;
        def_sea_tran = scen.players[i]->def_sea_tran;
        /* get rid of old struct */
        free( scen.players[i] );
        /* get saved data */
        scen.players[i] = load_player( file );
        /* restore transporters */
        scen.players[i]->def_air_tran = def_air_tran;
        scen.players[i]->def_sea_tran = def_sea_tran;
    }
    engine->player = scen.players[engine->player_id];

    /* unit stuff */
    dl_clear( &scen.units );
    unit_count = load_int( file );
    for ( i = 0; i < unit_count; i++ )
        dl_add( &scen.units, load_unit( file ) );
    dl_clear( &scen.reinf );
    unit_count = load_int( file );
    for ( i = 0; i < unit_count; i++ )
        dl_add( &scen.reinf, load_unit( file ) );

    /* map stuff */
    map.width = load_int( file );
    map.height = load_int( file );
    for ( i = 0; i < map.width; i++ )
        for ( j = 0; j < map.height; j++ )
            load_map_tile_units( file, &map_tile( i, j )->unit, &map_tile( i, j )->air_unit );
    for ( i = 0; i < map.width; i++ )
        for ( j = 0; j < map.height; j++ )
            load_map_tile_flag( file, &map_tile( i, j )->nation_id, &map_tile( i, j )->player_id );

    /* initate current player */
    set_player( engine, engine->player_id );

    /* free auxilaries */
    free( scen_file_name );

    fclose( file );

    return 1;
}
