/***************************************************************************
                          scenario.c  -  description
                             -------------------
    begin                : Fri Jan 19 2001
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <SDL.h>

#ifdef WITH_SOUND
#include <SDL_mixer.h>
#include "audio.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "sdl.h"
#include "dynlist.h"
#include "date.h"
#include "config.h"
#include "nation.h"
#include "unit.h"
#include "player.h"
#include "tools.h"
#include "file.h"
#include "map.h"
#include "scenario.h"

extern Config config;
extern Map map;
extern int cur_weather; /* used in unit.c to compute weather influence on units; set by end_turn() */

Scen scen;

/* open scenario file and return file handle (adds the correct path)*/
FILE* open_scen_file( char *scen_name )
{
    char full_path[512];

    sprintf( full_path, "%s/scenarios/%s", SRC_DIR, scen_name );
    printf( "parsing scenarios/%s\n", scen_name );

    return file_open( full_path );
}

/* load scenario by reading information from file and returning a pointer */
/* if type is SHALLOW only entries from this scenario file are loaded; if type is DEEP then links will
be resolved (graphics, maps, units) */
int load_scen( char *scen_name, int type )
{
    FILE *file = 0;
    int i = 0, j = 0, x = 0, y = 0;
    char *cur_arg = 0; /* current argument entry */
    char **args = 0;
    int arg_count = 0; /* argument list for get_arg_cluster */
    int weather_type = 0, weather_period = 0;
    int cur_pos = 0;
    int player_id = 0, exp_level = 0, delay = 0, entr = 0, dir = 0;
    char unit_name[256];
    Unit_Lib_Entry *unit_poi = 0, *tran_poi = 0;
    Unit *cur_unit = 0;
    int *unit_ref_count = 0; /* used to assign a unit name automatically */
    int *first_team_id = 0, *second_team_id = 0;
    char *map_name = 0;
    int ok = 1;
    Player **att_team;
    int att_team_count;
    int player_cont;

    printf( "----\n" );

    /* get file handle */
    if ( ( file = open_scen_file( scen_name ) ) == 0 )
        return 0;

    /* clear scen struct */
    clear_scen();

    /* init unit lib */
    init_unit_lib( &scen.unit_lib );

    /* init units */
    dl_init( &scen.units, AUTO_DELETE, delete_unit );
    dl_init( &scen.reinf, AUTO_DELETE, delete_unit );

    /* save it's own file name */
    scen.file_name = strdup( scen_name );

    /* title and description */
    if ( ( scen.title = get_arg( file, "title", RESET_FILE_POS ) ) == 0 ) goto failure;
    if ( ( scen.desc = get_arg( file, "desc", RESET_FILE_POS ) ) == 0 ) goto failure;

    /* load authors */
    if ( ( scen.authors = get_arg( file, "authors", RESET_FILE_POS ) ) == 0 )
        scen.authors = strdup( "unknown" );

    /* date */
    if ( ( cur_arg = get_arg( file, "date", RESET_FILE_POS ) ) == 0 ) goto failure;
    str_to_date( &scen.date, cur_arg );
    FREE( cur_arg );

    /* glorious victory turn limit */
    if ( ( cur_arg = get_arg( file, "major_turn_limit", RESET_FILE_POS ) ) == 0 ) goto failure;
    scen.major_turn_limit = atoi( cur_arg );
    FREE( cur_arg );

    /* victory turn limit */
    if ( ( cur_arg = get_arg( file, "minor_turn_limit", RESET_FILE_POS ) ) == 0 ) goto failure;
    scen.minor_turn_limit = atoi( cur_arg );
    FREE( cur_arg );

    /* nation definitions */
    if ( ( cur_arg = get_arg( file, "nation_lib", RESET_FILE_POS ) ) == 0 ) goto failure;
    scen.nations = read_nations( cur_arg, &scen.nation_count );
    FREE( cur_arg );
    if ( !scen.nations ) goto failure;

    /* map */
    if ( type == DEEP ) {
        if ( ( map_name = get_arg( file, "map", RESET_FILE_POS ) ) == 0 ) goto failure;
        ok = read_map( map_name );
        FREE( map_name );
        if ( !ok ) goto failure;
    }

    /* weather */
    if ( ( args = get_arg_cluster( file, "weather", &arg_count, RESET_FILE_POS, NO_WARNING ) ) == 0 ) goto failure;
    scen.weather = calloc( scen.minor_turn_limit, sizeof(int) );
    cur_pos = 0;
    for ( i = 0; i < arg_count; i++ ) {
        cur_arg = args[i];
        switch ( cur_arg[0] ) {

            case 'f': weather_type = FAIR; break;
            case 'c': weather_type = CLOUDS; break;
            case 'r': weather_type = RAIN; break;
            case 's': weather_type = SNOW; break;

        }
        cur_arg = cur_arg + 1;
        weather_period = atoi( cur_arg );
        for ( j = 0; j < weather_period; j++ )
            if ( cur_pos < scen.minor_turn_limit )
                scen.weather[cur_pos++] = weather_type;
    }
    delete_arg_cluster( args, arg_count );

    /* player count */
    scen.player_count = count_arg( file, "player" );
    /* create player struct */
    scen.players = calloc( scen.player_count, sizeof( Player* ) );

    /* read player information */
    if ( !find_token( file, "player", RESET_FILE_POS, WARNING ) ) goto failure;
    for ( j = 0; j < scen.player_count; j++ ) {
        args = get_arg_cluster( file, "player", &arg_count, FROM_CURRENT_FILE_POS, WARNING );
        /* control of player */
        player_cont = CPU;
        if ( equal_str( "human", args[2] ) )
            player_cont = HUMAN;
        /* create player */
        scen.players[j] = create_player( args[0], args[1], player_cont );
        /* diplomacy is not created by create_player() so it's done here */
        scen.players[j]->dipl = calloc( scen.player_count, sizeof( int ) );
        delete_arg_cluster( args, arg_count );
    }

    /* read diplomatic relations */
    /* all players in one team are allies; both teams are enemies */
    args = get_arg_cluster( file, "first_team", &arg_count, FROM_CURRENT_FILE_POS, WARNING );
    if ( args == 0 ) goto failure;
    /* setup first team */
    scen.first_team_count = 0;
    scen.first_team = calloc( arg_count, sizeof( Player* ) );
    first_team_id = calloc( arg_count, sizeof( int ) );
    for ( i = 0; i < arg_count; i++ )
        for ( j = 0; j < scen.player_count; j++ )
            if ( equal_str( scen.players[j]->entry_name, args[i] ) ) {
                first_team_id[scen.first_team_count] = j;
                scen.first_team[scen.first_team_count++] = scen.players[j];
            }
    /* second team */
    scen.second_team_count = 0;
    scen.second_team = calloc( scen.player_count - scen.first_team_count, sizeof( Player* ) );
    second_team_id = calloc( scen.player_count - scen.first_team_count, sizeof( int ) );
    for ( i = 0; i < scen.player_count; i++ ) {
        for ( j = 0; j < scen.first_team_count; j++ )
            if ( scen.players[i] == scen.first_team[j] ) break; /* in first team */
        if ( j == scen.first_team_count ) {
            second_team_id[scen.second_team_count] = i;
            scen.second_team[scen.second_team_count++] = scen.players[i];
        }
    }
    if ( scen.first_team_count == 0 || scen.second_team_count == 0 ) {
        fprintf( stderr, "load_scen: both teams need players\n" );
        /* free stuff */
        FREE( first_team_id );
        FREE( second_team_id );
        delete_arg_cluster( args, arg_count );
        goto failure;
    }
    /* ally player in each team */
    for ( i = 0; i < scen.first_team_count; i++ )
        for ( j = 0; j < scen.first_team_count; j++ )
            scen.first_team[i]->dipl[first_team_id[j]] = ALLIES;
    for ( i = 0; i < scen.second_team_count; i++ )
        for ( j = 0; j < scen.second_team_count; j++ )
            scen.second_team[i]->dipl[second_team_id[j]] = ALLIES;
    /* players in different teams are enemies */
    for ( i = 0; i < scen.first_team_count; i++ )
        for ( j = 0; j < scen.second_team_count; j++ )
            scen.first_team[i]->dipl[second_team_id[j]] = ENEMIES;
    for ( i = 0; i < scen.second_team_count; i++ )
        for ( j = 0; j < scen.first_team_count; j++ )
            scen.second_team[i]->dipl[first_team_id[j]] = ENEMIES;
    /* free stuff */
    FREE( first_team_id );
    FREE( second_team_id );
    delete_arg_cluster( args, arg_count );

    /* first assume player defends */
    scen.human_is_att = 0;
    /* which team attacks? */
    if ( ( cur_arg = get_arg( file, "attacker", RESET_FILE_POS ) ) == 0 ) goto failure;
    if ( equal_str( cur_arg, "first_team" ) ) {
        scen.att_team = FIRST_TEAM;
        for ( i = 0; i < scen.first_team_count; i++ )
            if ( scen.first_team[i]->cont == HUMAN ) {
                scen.human_is_att = 1;
                break;
            }
    }
    else {
        scen.att_team = SECOND_TEAM;
        for ( i = 0; i < scen.second_team_count; i++ )
            if ( scen.second_team[i]->cont == HUMAN ) {
                scen.human_is_att = 1;
                break;
            }
    }
    FREE( cur_arg );

    /* determine default cpu strategy */
    /* do this for human players to as this simplifies overwriting default controls */
    for ( i = 0; i < scen.player_count; i++ ) {
        att_team = get_att_team( &att_team_count );
        for ( j = 0; j < att_team_count; j++ )
            if ( scen.players[i] == att_team[j] ) {
                scen.players[i]->cpu_strat = AGGR;
                break;
            }
        if ( j == att_team_count )
            scen.players[i]->cpu_strat = DEF;
    }

    /* map flags */
    if ( type == DEEP ) {
        if ( !find_token( file, "flag", RESET_FILE_POS, WARNING ) ) goto failure;
        while ( ( args = get_arg_cluster( file, "flag", &arg_count, FROM_CURRENT_FILE_POS, NO_WARNING ) ) != 0 ) {
            /* set flags on map */
            /* position */
            get_coord( args[0], &x, &y );
            /* nation id */
            cur_arg = args[1];
            for ( i = 0; i < scen.nation_count; i++)
                if ( equal_str( scen.nations[i]->entry_name, cur_arg ) ) {
                    map.tiles[x][y].nation_id = i;
                    /* player name */
                    for ( j = 0; j < scen.player_count; j++ )
                        if ( equal_str( scen.players[j]->entry_name, args[2] ) ) {
                            map.tiles[x][y].player_id = j;
                            /* military target ? */
                            if ( arg_count >= 4 ) {
                                if ( equal_str( "obj", args[3] ) )
                                    map.tiles[x][y].obj = 1;
                            }
                        }
                    break;
                }
            delete_arg_cluster( args, arg_count );
        }

        /* read victory conditions */
        args = get_arg_cluster( file, "victory_conditions", &arg_count, RESET_FILE_POS, WARNING );
        if ( args == 0 ) goto failure;
        /* fullfill one or all objectives? */
        if ( equal_str( args[0], "all" ) )
            scen.victory_cond.type = ALL_COND;
        else
            scen.victory_cond.type = ONE_COND;
        /* required military targets */
        scen.victory_cond.req_obj_count = atoi( args[1] );
        /* primary objectives that must be aquired */
        scen.victory_cond.prim_obj_count = 0;
        scen.victory_cond.prim_obj = calloc( arg_count - 2, sizeof( Prim_Obj ) );
        for ( i = 2; i < arg_count; i++ ) {
            get_coord( args[i], &x, &y );
            /* check if this is a mil_obj */
            if ( !map.tiles[x][y].obj ) {
                fprintf( stderr, "load_scen: can't find military objective at %i,%i\n", x, y );
                continue;
            }
            /* assign to prim_obj_list */
            scen.victory_cond.prim_obj[scen.victory_cond.prim_obj_count].x = x;
            scen.victory_cond.prim_obj[scen.victory_cond.prim_obj_count++].y = y;
        }
        delete_arg_cluster( args, arg_count );

        /* unit classes/type definitions */
        if ( ( cur_arg = get_arg( file, "unit_class_lib", RESET_FILE_POS ) ) == 0 ) goto failure;
        scen.unit_def = read_unit_def( cur_arg );
        FREE( cur_arg );
        if ( scen.unit_def == 0 ) goto failure;

        /* read used unit_libs */
        if ( ( args = get_arg_cluster( file, "unit_libs", &arg_count, RESET_FILE_POS, WARNING ) ) == 0 )
            goto failure;
        for ( i = 0; i < arg_count; i++ )
            read_units_from_file( args[i], &scen.unit_lib, scen.unit_def, scen.nations, scen.nation_count );
        delete_arg_cluster( args, arg_count );

        /* read sea and air transporter definitions for each player */
        if ( !find_token( file, "transporters", RESET_FILE_POS, WARNING ) ) goto failure;
        for ( i = 0; i < scen.player_count; i++ ) {
            args = get_arg_cluster( file, "transporters", &arg_count, FROM_CURRENT_FILE_POS, WARNING );
            if ( args == 0 ) goto failure;
            /* which player is meant */
            for ( i = 0; i < scen.player_count; i++ )
                if ( equal_str( scen.players[i]->entry_name, args[0] ) ) {
                    /* standard air transporter */
                    scen.players[i]->def_air_tran = find_unit_lib_entry( &scen.unit_lib, args[1] );
                    /* number of unsued air transporters */
                    scen.players[i]->air_tran_count = atoi( args[2] );
                    /* standard sea transporter */
                    scen.players[i]->def_sea_tran = find_unit_lib_entry( &scen.unit_lib, args[3] );
                    /* number of unsued sea transporters */
                    scen.players[i]->sea_tran_count = atoi( args[4] );
                    break;
                }
            delete_arg_cluster( args, arg_count );
        }

        /* init unit reference counters */
        unit_ref_count = calloc( scen.player_count, sizeof( int ) );
        for ( i = 0; i < scen.player_count; i++ )
            unit_ref_count[i] = 1;

        /* read units */
        if ( !find_token( file, "unit", RESET_FILE_POS, WARNING ) ) goto failure;
        while ( ( args = get_arg_cluster( file, "unit", &arg_count, FROM_CURRENT_FILE_POS, NO_WARNING ) ) != 0 ) {
            /* read delay */
            delay = atoi( args[0] );
            /* find player */
            if ( arg_count >= 8 ) /* just for safety */
                for ( i = 0; i < scen.player_count; i++ )
                    if ( equal_str( scen.players[i]->entry_name, args[1] ) ) {
                        /* player_id */
                        player_id = i ;
                        /* unit */
                        unit_poi = find_unit_lib_entry( &scen.unit_lib, args[2] );
                        if ( unit_poi ) {
                            /* unit name */
                            strcpy( unit_name, args[3] );
                            if ( equal_str( "none", unit_name ) ) {
                                /* determine type */
                                switch ( unit_ref_count[player_id] ) {
                                    case 1:
                                        sprintf( unit_name, "1st %s", scen.unit_def->classes[unit_poi->class].name );
                                        unit_ref_count[player_id]++;
                                        break;
                                    case 2:
                                        sprintf( unit_name, "2nd %s", scen.unit_def->classes[unit_poi->class].name );
                                        unit_ref_count[player_id]++;
                                        break;
                                    case 3:
                                        sprintf( unit_name, "3rd %s", scen.unit_def->classes[unit_poi->class].name );
                                        unit_ref_count[player_id]++;
                                        break;
                                    default:
                                        sprintf( unit_name, "%ith %s", unit_ref_count[player_id], scen.unit_def->classes[unit_poi->class].name );
                                        unit_ref_count[player_id]++;
                                        break;
                                }
                        }
                        /* direction */
                        dir = RIGHT;
                        if ( equal_str( "left", args[4] ) )
                            dir = LEFT;
                        /* ALLDIR's not supported yet */
                        /* position */
                        get_coord( args[5], &x, &y );
                        /* entrenchment */
                        entr = atoi( args[6] );
                        /* experience */
                        exp_level = atoi( args[7] );
                        /* transporter */
                        tran_poi = 0;
                        if ( arg_count > 8 && !equal_str( "none", args[8] ) )
                            tran_poi = find_unit_lib_entry( &scen.unit_lib, args[8] );
                        /* and now please create this unit */
                        cur_unit = create_unit( delay, player_id, unit_poi, unit_name, dir, x, y, entr, exp_level, tran_poi );
                        /* if unit's delayed add it to scen::reinf */
                        if ( delay > 0 )
                            dl_add( &scen.reinf, cur_unit );
                        else
                            if ( cur_unit ) {
                                dl_add( &scen.units, cur_unit );
                                /* and entry in map */
                                if ( cur_unit->sel_prop->flags & FLYING )
                                    map.tiles[x][y].air_unit = cur_unit;
                                else
                                    map.tiles[x][y].unit = cur_unit;
                                /* check if air transporters are used and increase current value and limit */
                                if ( cur_unit->embark == AIR_EMBARK ) {
                                    scen.players[player_id]->air_tran_used++;
                                    scen.players[player_id]->air_tran_count++;
                                }
                                else
                                    if ( cur_unit->embark == SEA_EMBARK ) {
                                        scen.players[player_id]->sea_tran_used++;
                                        scen.players[player_id]->sea_tran_count++;
                                    }
                            }
                        break;
                    }
                }

            /* get next entry */
            delete_arg_cluster( args, arg_count );
        }

        /* free reference counter */
        FREE( unit_ref_count );
    }

    fclose( file );

    return 1;

failure:
    if ( file ) fclose( file );
    clear_scen();
    return 0;
}

/* frees memory allocated by scen */
void clear_scen( )
{
    int i;

    if ( scen.title ) FREE( scen.title );
    if ( scen.desc ) FREE( scen.desc );
    if ( scen.authors ) FREE( scen.authors );

    if ( scen.nations ) {
        for ( i = 0; i < scen.nation_count; i++ )
            delete_nation( scen.nations[i] );
        FREE( scen.nations );
    }

    if ( scen.weather ) FREE( scen.weather );

    if ( scen.players ) {
        for ( i = 0; i < scen.player_count; i++ )
            delete_player( scen.players[i] );
        FREE( scen.players );
    }
    if ( scen.first_team ) FREE( scen.first_team );
    if ( scen.second_team ) FREE( scen.second_team );
    if ( scen.victory_cond.prim_obj ) FREE( scen.victory_cond.prim_obj );

    if ( scen.unit_def ) delete_unit_def( scen.unit_def );
    dl_clear( &scen.unit_lib );
    dl_clear( &scen.units );
    dl_clear( &scen.reinf );

    /* scenario initiated struct map so it has to delete it as well */
    clear_map( );

    memset( &scen, 0, sizeof( Scen ) );
}

/* prepare unit for turn */
void prep_unit_for_turn( Unit *unit, int turn, int flags )
{
    int max_entr, min_entr;

    /* set supply level */
    comp_unit_supply_level( unit );

    /* if unit didn't take an action last turn supply automatically */
    if ( unit->no_action_yet && unit->supply_level > 0  ) {
        supply_unit( unit );
        /* no additional fuel cost for this turn */
        unit->add_fuel_cost = 0;
    }

    if ( !( flags & NO_FUEL_CHECK ) ) {
        /* maybe flying and dying */
        if ( unit->prop.flags & FLYING ) {
            /* decrease fuel by remaining movement points */
            if ( config.supply )
                unit->cur_prop.fuel -= unit->add_fuel_cost;
            if ( unit->kill_next_turn && unit->cur_prop.fuel == 0 )
                unit->killed = 1; /* will explode at the very beginning of your turn */
            else
                if ( unit->cur_prop.fuel <= 0  ) {
                    unit->cur_prop.fuel = 0;
                    unit->kill_next_turn = 1;
                }
                else
                    unit->kill_next_turn = 0;
        }
    }

    /* no action yet */
    unit->no_action_yet = 1;

    /* no additional cost right now */
    unit->add_fuel_cost = 0;

    /* set move points */
    unit->cur_prop.mov = unit->sel_prop->mov;
    /* force tran mov points when ground transporter */
    if ( unit->ground_tran )
        unit->cur_prop.mov = unit->tran_prop.mov;
    /* maybe not enough fuel ? */
    if ( use_fuel( unit ) && unit->cur_prop.fuel < unit->cur_prop.mov )
        unit->cur_prop.mov = unit->cur_prop.fuel;

    /* flying units have full move costs */
    if ( unit->sel_prop->flags & FLYING )
        unit->add_fuel_cost = unit->cur_prop.mov;

    /* unmount unit */
    if ( unit->ground_tran && unit->embark == GROUND_EMBARK )
        unmount_unit( unit );

    /* set attacks */
    unit->cur_prop.attack_count = unit->sel_prop->attack_count;

    if ( !( flags & NO_ENTR_CHECK ) && !( unit->sel_prop->flags & FLYING ) ) {

        /* check if unit is infantry and flag INF_ENTR_BONUS is set */
        min_entr = map_tile( unit->x, unit->y )->prop->min_entr;
        max_entr = map_tile( unit->x,unit->y )->prop->max_entr;
        if ( unit->sel_prop->class == INFANTRY )
            if ( map_tile( unit->x, unit->y )->prop->flags & INF_ENTR_BONUS ) {
                min_entr += 2;
                max_entr += 4;
            }
        /* increase entrenchment */
        if ( unit->entr < min_entr )
            unit->entr = min_entr;
        else
            if ( unit->entr >= max_entr )
                unit->entr = max_entr;
            else
                unit->entr++;

    }
    /* update all unit props */
    update_cur_unit_prop( unit );
}

/* get attacking team */
Player **get_att_team( int *count )
{
    /* get attacking team */
    if ( scen.att_team == FIRST_TEAM ) {
        *count = scen.first_team_count;
        return scen.first_team;
    }
    else {
        *count = scen.second_team_count;
        return scen.second_team;
    }
}

/* check if player belongs to attacking team */
int is_att( Player *player )
{
    int count;
    Player **team;
    int i;

    team = get_att_team( &count );
    /* check players */
    for ( i = 0; i < count; i++ )
        if ( team[i] == player )
            return 1;
    return 0;
}

/* check if victory conditions are fullfilled and set scen::att_team_won */
void check_victory_cond( )
{
    int count;
    Player **team;
    int x, y, i, j;
    int obj_count = 0; /* how many objectives are hold by the att team? */
    int prim_obj_count = 0;

    scen.att_team_won = 0;

    /* get attacker team */
    team = get_att_team( &count );
    /* check flags on map */
    for ( x = 0; x < map.width; x++ )
        for ( y = 0; y < map.height; y++ )
            if ( map.tiles[x][y].player_id != -1 )
               for ( i = 0; i < count; i++ )
                    if ( team[i] == scen.players[map.tiles[x][y].player_id] ) {
                        if ( map_tile( x, y )->obj ) {
                            obj_count++; /* belongs to att_team */
                            /* check if this is a primary obj */
                            for ( j = 0; j < scen.victory_cond.prim_obj_count; j++ )
                                if ( scen.victory_cond.prim_obj[j].x == x &&
                                     scen.victory_cond.prim_obj[j].y == y
                                ) {
                                    prim_obj_count++;
                                    break;
                                }
                            break;
                        }
                    }
    /* check result */
    if ( scen.victory_cond.type == ONE_COND ) {
        if ( obj_count >= scen.victory_cond.req_obj_count || prim_obj_count )
            scen.att_team_won = 1;
    }
    else {
        if ( scen.victory_cond.prim_obj_count == 0 ) {
            if ( obj_count >= scen.victory_cond.req_obj_count )
                scen.att_team_won = 1;
        }
        else {
            if ( obj_count >= scen.victory_cond.req_obj_count )
                if ( prim_obj_count >= scen.victory_cond.prim_obj_count )
                    scen.att_team_won = 1;
        }
    }
}


/*
====================================================================
Check if map tile at x,y belongs to player and is of type flag.
====================================================================
*/
int allied_depot( Unit *unit, int x, int y, int type_flag )
{
    if ( map_tile( x, y )->player_id != unit->player_id ) return 0;
    if ( !(map_tile( x, y )->prop->flags & type_flag) ) return 0;
    return 1;
}

/*
====================================================================
Check if unit is near a harbour or an airfield.
====================================================================
*/
int is_close_to_harbor( Unit *unit )
{
    int x = unit->x, y = unit->y;
    if ( allied_depot( unit, x, y, HARBOR ) ) return 1;
    if ( allied_depot( unit, x + 1, y, HARBOR ) ) return 1;
    if ( allied_depot( unit, x - 1, y, HARBOR ) ) return 1;
    if ( allied_depot( unit, x, y + 1, HARBOR ) ) return 1;
    if ( allied_depot( unit, x, y - 1, HARBOR ) ) return 1;
    if ( x & 1 ) {
        if ( allied_depot( unit, x + 1, y + 1, HARBOR ) ) return 1;
        if ( allied_depot( unit, x - 1, y + 1, HARBOR ) ) return 1;
    }
    else {
        if ( allied_depot( unit, x + 1, y - 1, HARBOR ) ) return 1;
        if ( allied_depot( unit, x - 1, y - 1, HARBOR ) ) return 1;
    }
    return 0;
}
int is_close_to_airfield( Unit *unit )
{
    int x = unit->x, y = unit->y;
    if ( map_tile( x, y )->prop->flags & AIRFIELD ) return 1;
    if ( map_tile( x + 1, y )->prop->flags & AIRFIELD ) return 1;
    if ( map_tile( x - 1, y )->prop->flags & AIRFIELD ) return 1;
    if ( map_tile( x, y + 1)->prop->flags & AIRFIELD ) return 1;
    if ( map_tile( x, y - 1)->prop->flags & AIRFIELD ) return 1;
    if ( x & 1 ) {
        if ( map_tile( x + 1, y + 1 )->prop->flags & AIRFIELD )return 1;
        if ( map_tile( x - 1, y + 1 )->prop->flags & AIRFIELD )return 1;
    }
    else {
        if ( map_tile( x + 1, y - 1 )->prop->flags & AIRFIELD )return 1;
        if ( map_tile( x - 1, y - 1 )->prop->flags & AIRFIELD )return 1;
    }
    return 0;
}
/*
====================================================================
Compute a units supply level. Each times it supplies it gets this
percentage of its maximum value in fuel and ammo.
====================================================================
*/
void comp_unit_supply_level( Unit *unit )
{
    int x, y, w, h, i, j;
    int flag_supply_level;
    /* flying and swimming units get a 100% supply if near an airfield or a harbour */
    if ( unit->sel_prop->flags & SWIMMING ) {
        unit->supply_level = 0;
        if ( is_close_to_harbor( unit ) ) unit->supply_level = 100;
    }
    else
    if ( unit->sel_prop->flags & FLYING ) {
        unit->supply_level = 0;
        if ( is_close_to_airfield( unit ) ) unit->supply_level = 100;
    }
    else {
        /* ground units get a 100% close to a flag and looses about 10% for each title it gets away */
        /* test all flags within a region x-10,y-10,20,20 about their distance */
        /* get region first */
        x = unit->x - 10; y = unit->y - 10; w = 20; h = 20;
        if ( x < 0 ) { w += x; x = 0; }
        if ( y < 0 ) { h += y; y = 0; }
        if ( x + w > map.width ) w = map.width - x;
        if ( y + h > map.height ) h = map.height - y;
        /* now check flags */
        unit->supply_level = 0;
        for ( i = x; i < x + w; i++ )
            for ( j = y; j < y + h; j++ )
                if ( map_tile( i, j )->player_id == unit->player_id ) {
                    flag_supply_level = get_dist( unit->x, unit->y, i, j );
                    if ( flag_supply_level < 2 ) flag_supply_level = 100;
                    else {
                        flag_supply_level = 100 - ( flag_supply_level - 1 ) * 10;
                        if ( flag_supply_level < 0 ) flag_supply_level = 0;
                    }
                    if ( flag_supply_level > unit->supply_level )
                        unit->supply_level = flag_supply_level;
                }
    }
    /* if hostile influence is 1 supply's cut in half if influence >1 no supply possible */
    if ( unit->sel_prop->flags & FLYING ) {
        if ( mask_tile( unit->x, unit->y )->air_infl == 1 )
            unit->supply_level /= 2;
        else
            if ( mask_tile( unit->x, unit->y )->air_infl > 1 )
                unit->supply_level = 0;
    }
    else {
        if ( mask_tile( unit->x, unit->y )->infl == 1 )
            unit->supply_level /= 2;
        else
            if ( mask_tile( unit->x, unit->y )->infl > 1 )
                unit->supply_level = 0;
    }
}

