#! /bin/bash

#
# This script tests ledcontrol's features.
# This script can also be executed by the user. It may require root access.
#

set -e
exec 1>&2   # Everything to STDERR
exec 5>&2   # Give startup.sh a channel through which to talk to.

if test -e /var/run/ledd.pid; then
    echo >&2
    echo "/var/run/ledd.pid exists, suggesting existing ledd daemon (PID `head -n1 /var/run/ledd.pid`)." >&2
    echo "Remove the existing process before starting." >&2
    echo >&2
    exit 1
fi

LEDD=./ledd/ledd


### Functions etc...

waitforfile () {
    COUNT=0
    while test "$COUNT" -lt "$2"; do
	if test -e /tmp/$1 ; then
	    break
	fi
	if test -z "`pidof ledd`"; then
	    echo "ledd seems to have crashed!" >&2
	    echo "Aborting..." >&2
	    atexit
	    exit 1
	fi
	sleep 1
	COUNT=$[ $COUNT + 1 ]
    done
    if test "$COUNT" -ge "$2"; then
	echo "Didn't receive notice for next phase!" >&2
	echo "Aborting..." >&2
	atexit
	exit 1
    fi
}

waiting () {
    COUNT=$1
    while test "$COUNT" -gt 0; do
	echo -ne "Waiting $COUNT...   \r"
	sleep 1
	COUNT=$[ $COUNT - 1 ]
    done
    echo "Waiting 0...   "
}

atexit () {
    echo "Killing ledd (not checking...)"
    killall ledd
}


### Main piece starts here...

test -e ./test-ledd.conf
test -x ./test-startup.sh
test -x ./ledcontrol
test -x $LEDD

if ! test -r /dev/console; then
    echo "" >&2
    echo "\"make test\" requires read access to /dev/console. The easiest" >&2
    echo "way to achieve this is run it as root (the only files it creates" >&2
    echo "are /tmp/ledd-test-* and /var/run/ledd.pid, all of which should" >&2
    echo "be removed on exit or abort)." >&2
    echo "" >&2
    exit 1
fi


echo
echo This script will test features in ledd / ledcontrol. You should be
echo able to stop it at any time by pressing CTRL-C. It will tell you
echo what it is doing at each time.
echo

waiting 5

rm -f /tmp/ledd-test-*

# Startup procedures...
echo -n "Testing ledd version..."
$LEDD --version

echo -n "Starting ledd..."
$LEDD -c ./test-ledd.conf
trap "atexit" SIGINT
echo OK

# First phase...
echo "Waiting for startup script..."
waitforfile ledd-test-first 5
echo "OK, I hear you loud and clear... go ahead, startup!"
rm -f /tmp/ledd-test-first


# Give it room and wait for phase two... (shouldn't take longer than 3 minutes)
waitforfile ledd-test-second 180

echo "OK, I'll take it from here..."
rm -f /tmp/ledd-test-second

echo
echo "Testing the ledcontrol script with commands:"
echo "    ./ledcontrol -c ./test-ledd.conf set s9 blink 100"
echo "    ./ledcontrol -c ./test-ledd.conf set c9 blink 500"
echo "    ./ledcontrol -c ./test-ledd.conf set n9 blink 1000"
echo "Your LEDs should start flashing frantically..."
echo

./ledcontrol -c ./test-ledd.conf set s9 blink 100
./ledcontrol -c ./test-ledd.conf set c9 blink 500
./ledcontrol -c ./test-ledd.conf set n9 blink 1000

waiting 10

echo -n "Killing ledd..."

trap SIGINT

if ! grep -qi "^ *pidfile  *[^ ][^ ]* *$" test-ledd.conf; then
    PIDFILE=""
else
    PIDFILE="`grep -i "^ *pidfile  *[^ ][^ ]* *$" test-ledd.conf | cut -d " " -f 2`"
fi


if test -f "$PIDFILE"; then
    test -n "$PIDFILE" && PID="`cat "$PIDFILE"`"
    test -z "$PIDFILE" && PID="`pidof ledd`"
    if test -n "$PID"; then
	if grep -q ledd /proc/$PID/cmdline 2>/dev/null ; then
	    kill $PID
	    sleep 1
	    if grep -q ledd /proc/$PID/cmdline 2>/dev/null ; then
		echo -n "not dying, have to use -9..."
		kill -9 $PID
	    fi
	fi
    fi
    rm -f "$PIDFILE"
fi

if grep -q ledd /proc/$PID/cmdline 2>/dev/null ; then
    echo "NOT DYING (pid $PID)!"
    echo "Abort..."
    exit 1
else
    echo OK
fi

echo
echo "That's all folks!"
echo

if test -n "$DISPLAY"; then
    echo "You seem to be in X, so some of your LEDs may be incorrectly lit."
    echo "Press some Lock key twice to set them back to normal."
    echo
fi

