
#ifndef LEDD_H
#define LEDD_H

#include <config.h>
#include <glib.h>
#include <sys/kd.h>
#include <sys/types.h>

#include <errno.h>

#define STRERROR g_strerror(errno)

#define PRIORITY_MAX 9 /* Priority levels lowest=0 ... PRIORITY_MAX=highest */
#define SLEEPTIME 10   /* Check stuff 100 times per second (milliseconds). */


/* No one (space-separated) space is allowed to be longer than
 * MAXPIECELENGTH chars. This does not apply to morse code strings.
 * The only things that could be longer than this are the commands
 * ("set", "anim", etc.) which they aren't, user-given numbers (would
 * most likely cause an overflow anyway...) or user-given setting strings
 * (and to set every led state possible 60 chars is sufficient...).
 */
#define MAXPIECELENGTH 64


/*
 * Make a function equivalent to g_error, except that it aborts execution
 * only after the message is sent. This avoids recursion (glib should accept
 * one level of recursion in the message system!!!).
 * It sends the message actually on G_LOG_LEVEL_WARNING.
 * TODO: This may not work on non-gcc compliers!
 */
#define G_ERROR(format, args...) G_STMT_START { \
	g_log(G_LOG_DOMAIN, \
	      G_LOG_LEVEL_WARNING, \
	      format, ##args); \
        abort(); \
	} G_STMT_END

/*** ENUMS: ***/

/* Define all the leds */
/* You have to change also led_flags in led.c to match this. */
typedef enum {
	LED_S = 0,
	LED_C,
	LED_N,
	LED_MAX
} LedName;


/* Define different types of lightings for leds.
 * ANIM_xxx must have same values! */
typedef enum {
	LEDTYPE_NORMAL=0,   /* Don't care what the light is. */
	LEDTYPE_OFF,        /* Led light off */
	LEDTYPE_ON,         /* Led light on */
	LEDTYPE_BLINK       /* Use blinking info */
} LedType;


/* Define animation piece types.
 * Has to be in same order (normal,off,on) as LEDTYPE_xxx! */
typedef enum {
	ANIM_NORMAL=0,
	ANIM_OFF,
	ANIM_ON,
	ANIM_WAIT,
	ANIM_LOOP
} AnimType;

/* Types for the File structure ->type. This includes all sorts of stuff,
 * from config.c CMDLINE or CONFIG, to whether in ->pipes is a script
 * or true pipe. */
typedef enum {
	TYPE_NONE=0,
	/* config.c */
	TYPE_DEFAULT,     /* A default setting */
	TYPE_CONFIG,      /* Setting from a config file */
	TYPE_CMDLINE,     /* Setting from command line */
	/* pipe.c */
	TYPE_PIPE,        /* It's a real pipe */
	TYPE_SCRIPT,      /* It's a script */
	/* led.c */
	TYPE_GRAPHICS,    /* Graphics tty (X) */
	TYPE_TEXT,        /* Normal text tty */
	/* log.c */
	TYPE_LOG_SYSLOG,  /* Log via syslog */
	TYPE_LOG_STDERR,  /* Log to stderr */
	TYPE_LOG_FILE,    /* Log to file (not implemented yet) */
	TYPE_LOG_NONE,    /* Don't log anywhere */

	TYPE_MAX
} FileType;


/*** STRUCTS: ***/

/* Control of one led */
typedef struct {
	LedType type;        /* What light type */
	GSList *blink;       /* Blink sequence, ->data is type (int) */
	                     /* DO NOT gslist_free_all() THIS!!! */
	gint mode;           /* FALSE=now off, TRUE=now (should be) on */
	GTimer *timer;       /* Timer */
} LedControl;


/* Holds one action in animation. */
typedef struct {
	AnimType type;
	gint data;
} AnimAction;
/* I should make some standard method for these... */
#define ANIM(x) ((AnimAction *)(x))


/* File contains the name of the file and some associated stuff.
 * For opts->pipes:
 *     name  = name of pipe or startup script
 *     type  = type of pipe (config.c)
 *     fd    = file descriptor
 * For opts->startup
 *     name  = name of startup script
 *     type  = type of script (config.c)
 *     fd    = STDERR of script (for error handling)
 *     pid   = pid of child process
 * For opts->ttys
 *     name  = name of tty device
 *     type  = cmdline/config (config.c) or graphics/text (led.c)
 *     fd    = file descriptor, if open
 *     timer = timer, how long descriptor has been open
 */

typedef struct {
	gchar *name;
	FileType type; /* File type, as in TYPE_xxx above. */
	int fd;        /* Possible file descriptor. fd<0 if none. */
	pid_t pid;     /* Possible pid. pid==0 if none. */
	GTimer *timer; /* Possible timeout stuff (tty re-opening) */
} File;


/* Holds all options */
typedef struct {
	/* Files: */
	GSList *pipes;       /* Pipes (type File) */
	GSList *startup;     /* Startup files (type File) */
	GSList *ttys;        /* TTYs to use (type File, not yet) */
	/* tty-change */
	GSList *fixttys;     /* TTYs to fix on exit (type File, not yet) */
	File *pidfile;       /* PID lock file */

	File *logging;       /* Where do we log to? */

	LedControl *leds[LED_MAX][PRIORITY_MAX+1];
	                     /* Leds with priority */
	GSList *anim;        /* List is AnimActions, NULL if no anim. */
	gint anim_and;       /* Previous animation and flag */
	gint anim_or;        /* Previous animation or flag */
	GTimer *anim_timer;  /* For counting a wait. */
	gboolean anim_loop;  /* Is it looping? */

	gint daemon;         /* Fork to background? Yes, if > 0 */
} options;


/*** FUNCTIONS: ***/

/* parse.c */
extern void parse_pipe_command(options *,gchar *,File *);
extern gchar *parse_jumpspace(gchar *);
extern gchar *parse_getpiece(gchar *,gchar *);
extern gboolean parse_eol(gchar *);

/* signal.c */
extern void signal_set_handler(options *);
extern void signal_ignore_abort(void);

/* led.c */
extern const gint led_flags[LED_MAX];
extern const gint led_flags_all;
extern gint led_set(File *,gint,gint);
extern gboolean led_do_your_thing(options *);
extern void led_flag_set(gint *,gint *,LedName,LedType); /* AND, OR, ... */
extern void led_free(LedControl *);
extern void led_do_anim(options *opts);
extern gint led_set_all_normal(options *opts);
extern gint led_set_normal(gchar *name);

/* pipe.c */
extern void pipe_open(File *);
extern void pipe_open_all(options *);
extern void pipe_create(int fd[2]);
extern void pipe_close(File *);
extern void pipe_close_all(options *);
extern void pipe_check_new(options *,File *);
extern gboolean pipe_check_all(options *);
extern gchar *pipe_gets(int);

/* config.c */
extern void config_cmdline(options *,int,char **);

/* startup.c */
extern void startup_exec_all(options *);
extern void startup_fork(void);
extern gboolean startup_check_all(options *);
extern gint startup_pidfile_check(options *);
extern void startup_pidfile_write(options *);
extern void startup_pidfile_remove(options *);

/* gslist.c */
extern GSList *gslist_rotate(GSList *);
extern GSList *gslist_next_free(GSList *);
extern void gslist_free_all(GSList *);
extern void gslist_free_all_files(GSList *);

/* log.c */
extern void log_set_facility(File *);
extern void log_close(File *);


#endif

