/*
 * $Id: kl_error.c,v 1.1 2004/12/21 23:26:20 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>
#include <stdarg.h>

/* for entire libklib output and error messages
 */
FILE *kl_stdout = NULL;
FILE *kl_stderr = NULL;

/* A trace message is only printed iff its trace level is less than or equal
 * this threshold. Minimum threshold is 1.
 */
static uint32_t kl_trace_threshold = 0;

/* Hold selected debug components (stored as a bit field).
 */
static uint64_t kl_dbg_component = 0;

/* set debug components
 */
void
kl_set_dbg_component(uint64_t component)
{
	kl_dbg_component = component;
	return;
}

/* get debug components
 */
uint64_t
kl_get_dbg_component(void)
{
	return(kl_dbg_component);
}

/* Set trace threshold (currently allowd values: 0-4).
 * A value of 0 turns off all trace messages.
 */
int
kl_set_trace_threshold(uint32_t threshold)
{
	if(threshold > KLE_TRACELEVELMAX){
		return(-1);
	}
	kl_trace_threshold = threshold;
	return(0);
}

/* get trace threshold
 */
uint32_t
kl_get_trace_threshold(void)
{
	return(kl_trace_threshold);
}

/*
 * libklib's vehicle to print regular, error and trace messages.
 * Trace messages are printed if their trace level is less than or equal to
 * the trace threshold. Trace messages always go to kl_stderr and are preceded
 * by an origin (file, line, function name).
 * All output is flushed at the end.
 */
void
kl_msg(uint64_t component, uint32_t flags, const char *function,
       const char *file, int line, const char *argfmt, ...)
{
	uint32_t trace;
	FILE *out;
	char fmt[32];
	va_list arglist;

	fmt[0] = 0;

	if(component && !(component & kl_dbg_component)){
		return;
	}

	trace = KLE_GETTRACELEVEL(flags);
	if(trace > kl_trace_threshold){
		return;
	}

	va_start(arglist, argfmt);


	if(flags & KLE_F_ERRORMSG) {
		flags |= KLE_F_NOORIGIN;
		out = kl_stderr;
	}

	if(trace){
		/* write message to stderr */
		out = kl_stderr;
	} else {
		/* no error/trace message -> write to ofp resp. stdout */
		out = kl_stdout;
	}

	fflush(kl_stdout);
	if(!(flags & KLE_F_NOORIGIN) && trace && function && file && line){
		fprintf(out, "%s:%d:%s(): ", file, line, function);
	}
	vfprintf(out, argfmt, arglist);
	fflush(out);
	va_end(arglist);
	return;
}

/*  change out stream of libklib
 */
void
kl_set_stdout(FILE *ofp)
{
	kl_stdout = ofp ? ofp : stdout;
	return;
}

/*  change error stream of libklib
 */
void
kl_set_stderr(FILE *ofp)
{
	kl_stderr = ofp ? ofp : stderr;
	return;
}
