/*
 * $Id: cmd_vertex.c,v 1.1 2004/12/21 23:26:19 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 *
 * Copyright (C) 2003 - 2004 Silicon Graphics, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#include <lcrash.h>
#include <kl_libhwconfig.h>

/* Function prototypes
 */
int print_vertex(kaddr_t, void *, int, FILE *);
void vertex_banner(FILE *, int);

/*
 * vertex_cmd() -- Run the 'vertex' command.
 */
int
vertex_cmd(command_t *cmd)
{
	int i, mode, vertex_cnt = 0, first_time = 1;
	int DENTRY_SZ;
        kaddr_t vertex;
	uint64_t value;
	void *dentry;
	vertex_t *vp;

	DENTRY_SZ = kl_struct_len("dentry");
	if (cmd->nargs == 0) {
		if (kl_init_hwconfig()) {
			return(1);
		}
		vp = hwgraph_root;
		while(vp) {
			if (first_time || (cmd->flags & (C_FULL|C_NEXT))) {
                                vertex_banner(cmd->ofp, BANNER|SMAJOR);
                                first_time = FALSE;
                        }
			print_vertex(vp->vertex, 
				vp->dentry, cmd->flags, cmd->ofp);
			vp = (vertex_t *)kl_next_htnode((htnode_t *)vp);
		}
	} else {
		dentry = kl_alloc_block(DENTRY_SZ, K_TEMP);
		if (KL_ERROR) {
			return(1);
		}
		for (i = 0; i < cmd->nargs; i++) {
			kl_get_value(cmd->args[i], &mode, 0, &value);
                        if (KL_ERROR || (mode != 2)) {
                                continue;
                        }
                        vertex = (kaddr_t)value;
                        if (first_time || (cmd->flags & (C_FULL|C_NEXT))) {
                                vertex_banner(cmd->ofp, BANNER|SMAJOR);
                                first_time = FALSE;
                        }
			GET_BLOCK(vertex, DENTRY_SZ, dentry);
			if (KL_ERROR) {
				/* XXX -- print error */;
			} else {
				print_vertex(vertex, dentry, 
					cmd->flags, cmd->ofp);
				vertex_cnt++;
			}
		}
		kl_free_block(dentry);
	}
	return(0);
}

#define _VERTEX_USAGE "[-a] [-f] [-n] [-w outfile] [vertex_list]"

/*
 * vertex_usage() -- Print the usage string for the 'vertex' command.
 */
void
vertex_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _VERTEX_USAGE);
}

/*
 * vertex_help() -- Print the help information for the 'vertex' command.
 */
void
vertex_help(command_t *cmd)
{
	CMD_HELP(cmd, _VERTEX_USAGE,
        	"Display relevant information for each entry in "
		"vertex_list. If no entries are specified, display all "
		"active entries in the hwgraph. Entries in vertex_list "
		"can take the form of a vertex handle or virtual address. "
		"If the -a option is specified, display all entries in the "
		"hwgraph (in use or free); if the -f option is specified, "
		"display additional information about each vertex (connect "
		"point and index values); if the -n option is specified, "
                "display all edge and info information related to a given "
		"vertex.");
}

/*
 * vertex_parse() -- Parse the command line arguments for 'vertex'.
 */
int
vertex_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_FULL|C_WRITE|C_NEXT|C_ALL), 0)) {
		return(1);
	}
	return(0);
}

/*
 * vertex_complete() -- Complete arguments of 'vertex' command.
 */
char *
vertex_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	vertex_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
