/*
 * $Id: cmd_deftask.c,v 1.1 2004/12/21 23:26:18 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

/*
 * deftask_cmd() -- Run the 'deftask' command.
 */
int
deftask_cmd(command_t *cmd)
{
	int mode;
	kaddr_t task;
	uint64_t value;

	if (cmd->nargs == 0) {
		if (!deftask) {
			fprintf(cmd->efp, "No default task set\n");
		} else {
			fprintf(cmd->ofp, "Default task is ");
			print_kaddr(deftask, cmd->ofp, 0);
			fprintf(cmd->ofp, "\n");
		}
		return(0);
	} else {
		kl_get_value(cmd->args[0], &mode, 0, &value);
		if (KL_ERROR || (task && (mode != 2))) {
			fprintf(cmd->efp, "Error setting deftask to %s "
				"(error=%"FMT64"d)\n", cmd->args[0], KL_ERROR);
		} else {
			task = (kaddr_t)value;
			if (kl_set_deftask(task)) {
				fprintf(cmd->efp, "Error setting deftask to "
					"%s (error=%"FMT64"d)\n", 
					cmd->args[0], KL_ERROR);
			}
		}
		fprintf(cmd->ofp, "Default task is ");
		print_kaddr(deftask, cmd->ofp, 0);
		fprintf(cmd->ofp, "\n");
	}
	return(0);
}

#define _DEFTASK_USAGE "[-w outfile] [task]"

/*
 * deftask_usage() -- Print the usage string for the 'deftask' command.
 */
void
deftask_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _DEFTASK_USAGE);
}

/*
 * deftask_help() -- Print the help information for the 'deftask' command.
 */
void
deftask_help(command_t *cmd)
{
	CMD_HELP(cmd, _DEFTASK_USAGE,
        "Set the default task if one is indicated.  Otherwise, "
        "display the current value of deftask. When 'lcrash' is run "
        "on a system core dump, deftask gets set automatically "
        "to the task that was active when the system PANIC occurred."
        "When 'lcrash' is run on a live system, deftask is not set "
        "by default.\n\n"

        "The deftask value is used by 'lcrash' in a number of ways. "
        "The trace command will display a trace for the default "
        "task if one is set. Also, the translation of certain "
        "virtual addresses (user space) depends upon deftask being "
	"set.");
}

/*
 * deftask_parse() -- Parse the command line arguments for 'deftask'.
 */
int
deftask_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_WRITE), 0)) {
		return(1);
	}
	return(0);
}

/*
 * deftask_complete() -- Complete arguments of 'deftask' command.
 */
char *
deftask_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	deftask_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
