/*
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 */

/*!
    \file   lcmaps_return_poolindex.h
    \brief  This interface of LCMAPS takes the user gsi credential as input and
            returns the uid, gids and poolindex, i.e. leaseid.
    \author Martijn Steenbakkers for EGEE.

    This header contains the following LCMAPS library functions:
    -# lcmaps_return_poolindex
        Returns the poolindex based on fixed arguments:
                security context, buffer and buffer length
    -# lcmaps_return_poolindex_from_gss_cred
        Returns the poolindex based on fixed arguments:
                gss credential, buffer and buffer length

    In general the behaviour of LCMAPS can be influenced by setting the following
    environment variables:
        Env variable                        Description
        (default value)
        ---------------                     -----------
        LCMAPS_LOG_FILE                     The logfile to which lcmaps log
        ("/var/log/lcmaps.log")             messages will be written

        LCMAPS_DB_FILE                      the location (name) of the lcmaps policy file
        (LCMAPS_ETC_DIR"/lcmaps.db")

        LCMAPS_POLICY_NAME                  the name of the lcmaps policy (in the
        ("acquisition_policy")              policy file) that should be evaluated

        LCMAPS_LOG_STRING                   A string that will be prepended to all
        ("poolindex :")                     LCMAPS logging messages.

        LCMAPS_DEBUG_LEVEL                  The debug level influences the logging
        ("0")                               verbosity. values 0-5.

        LCMAPS_MODULES_DIR                  The directory in which modules can be found.
        ("")                                Note: may be overwritten by what is in
                                            the policy file (the convential method)

        LCMAPS_ETC_DIR                      The directory of the lcmaps policy files
        ("/opt/glite/etc/lcmaps")           and other config files.

        GRIDMAPDIR                          The directory in which the poolaccount leases
        ("")                                are registered.
                                            Note: may be overwritten by what is in
                                            the policy file (the convential method)

    \ingroup LcmapsGlobusInterface
*/

#ifndef LCMAPS_RETURN_POOLINDEX_H
#define LCMAPS_RETURN_POOLINDEX_H


/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_globus.h"

/******************************************************************************
 *                 Module definition
 *****************************************************************************/

/**
    \brief LCMAPS returns the poolindex
    If LCMAPS is invoked via this interface it will return the poolindex,
    alternatively named leaseid, to the calling application.

    \param context_handle
           The GSS Security context established during invocation of the calling
           service.
    \param poolindex_buffer
           A pointer to a buffer. This buffer will contain the poolindex (NUL
           terminated string) upon successful return.
    \param buffer_length
           The length of the above mentioned buffer.

    \retval 0 success.
    \retval 1 failure.
*/
typedef int
lcmaps_return_poolindex_t(
    gss_ctx_id_t context_handle,
    char *       poolindex_buffer,
    unsigned int buffer_length
);

/**
    \brief LCMAPS returns the poolindex
    If LCMAPS is invoked via this interface it will return the poolindex,
    alternatively named leaseid, to the calling application.
    Instead of using the security context as an input it uses a gss credential
    (type gss_cred_id_t).

    \param credential_handle
           The GSS credential (of type gss_cred_id_t) of the user.
    \param poolindex_buffer
           A pointer to a buffer. This buffer will contain the poolindex (NUL
           terminated string) upon successful return.
    \param buffer_length
           The length of the above mentioned buffer.

    \retval 0 success.
    \retval 1 failure.
*/
typedef int
lcmaps_return_poolindex_from_gss_cred_t(
    gss_cred_id_t credential_handle,
    char *        poolindex_buffer,
    unsigned int  buffer_length
);

#include "_lcmaps_return_poolindex.h"

#ifndef LCMAPS_USE_DLOPEN
lcmaps_return_poolindex_t lcmaps_return_poolindex;
lcmaps_return_poolindex_from_gss_cred_t lcmaps_return_poolindex_from_gss_cred;
#endif /* LCMAPS_USE_DLOPEN */

/**
   \brief Opaque handle type for the interface

   The opaque handle type that the API user should declare a single instance of.
   This must be passed to nearly all macros (in OO terms, this would be a derived
   type of LCMAPS_HANDLE).
   Usage: 
   LCMAPS_HANDLE_RETURN_POOLINDEX handle;
 */
typedef _lcmaps_handle_return_poolindex_t LCMAPS_HANDLE_RETURN_POOLINDEX;
/* typedef void * LCMAPS_HANDLE_RETURN_POOLINDEX; */

/**
   The declarator can be used to declare and init the handle in one go.
 */
#define LCMAPS_DECL_HANDLE_RETURN_POOLINDEX(l) \
    _LCMAPS_DECL_HANDLE_RETURN_POOLINDEX(l)

/**
   The initializer of the handle object. This macro must be called before any
   other macro using an LCMAPS_HANDLE.
   \param l An instance of LCMAPS_HANDLE_RETURN_POOLINDEX
   \retval 1 always.
 */
#define LCMAPS_INIT_HANDLE_RETURN_POOLINDEX(l) \
    _LCMAPS_INIT_HANDLE_RETURN_POOLINDEX(l)

/**
   This macro loads and prepares LCMAPS; must be called before any 
   use of the interface functions, but after LCMAPS_INIT_HANDLE_RETURN_POOLINDEX.
   \param l an initialized handle
   \retval 0 in case of failure, inspect LCMAPS_ERRMSG(l)
   \retval 1 on success
 */
#define LCMAPS_LOAD_RETURN_POOLINDEX(l) \
    _LCMAPS_LOAD_RETURN_POOLINDEX(l)

/**
    \brief close the LCMAPS_HANDLE

    This macro closes any opened interface libraries and clears the handle.
    After this call the handle must be considered invalid. The LCMAPS_CLOSE_HANDLE
    can be found in lcmaps.h.
*/
#define LCMAPS_CLOSE_HANDLE_RETURN_POOLINDEX(l)	\
    LCMAPS_CLOSE_HANDLE(l)

#endif /* LCMAPS_RETURN_POOLINDEX_H */
