/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: testsome.c,v 6.4 1999/12/31 21:27:39 jsquyres Exp $
 *
 *	Function:	- check if at least one request completed
 *			- return info on all completed requests
 *	Accepts:	- # of requests
 *			- array of requests
 *			- output count (out)
 *			- array of indices (out)
 *			- array of status structures (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <rpisys.h>


/*@
    MPI_Testsome - Tests for some given communications to complete

Input Parameters:
+ count - length of array_of_requests (integer) 
- reqs - array of requests (array of handles) 

Output Parameters:
+ outcount - number of completed requests (integer) 
. indices - array of indices of operations that 
completed (array of integers) 
- stats - array of status objects for operations that completed (array
of Status), or the MPI constant 'MPI_STATUSES_IGNORE'

.N waitstatus

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COUNT
.N MPI_ERR_ARG
.N MPI_ERR_IN_STATUS

.N ACK
@*/
int MPI_Testsome(int count, MPI_Request *reqs, 
		 int *outcount, int *indices, MPI_Status *stats)
{
	int		i;			/* favourite index */
	int		fl_done;		/* request done flag */
	int		fl_err;			/* error flag */
	int		err;			/* error code */
	int		fl_trace;		/* do tracing? */
	MPI_Status	*p;			/* ptr status */

	lam_initerr_m();
	lam_setfunc_m(BLKMPITESTSOME);
/*
 * Check the arguments.
 */
	if (count < 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPITESTSOME,
				lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if ((count > 0) && ((reqs == 0) || (indices == 0))) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPITESTSOME,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if (outcount == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPITESTSOME,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Generate a start wrapper trace.
 */
	fl_trace = ((_kio.ki_rtf & RTF_TRON) == RTF_TRON);

	if (fl_trace) {
		lam_tr_wrapstart(BLKMPITESTSOME);
	}
/*
 * Initialize all status members.
 */
	if (stats != MPI_STATUSES_IGNORE) {
		for (i = 0, p = stats; i < count; ++i, ++p) {
			lam_emptystat(p);
		}
	}
/*
 * Loop testing all requests.
 * Skip inactive requests.
 */
	*outcount = MPI_UNDEFINED;
	fl_err = 0;

	for (i = 0; i < count; ++i, ++reqs) {

		if ((*reqs == MPI_REQUEST_NULL) ||
			((*reqs)->rq_state == LAM_RQSINIT)) continue;

		if (*outcount == MPI_UNDEFINED) {
			*outcount = 0;
		}

		err = MPI_Test(reqs, &fl_done, stats);

		if ((err != MPI_SUCCESS) || fl_done) {

			++(*outcount);
			*indices++ = i;

			if (err != MPI_SUCCESS) {
				if (stats != MPI_STATUSES_IGNORE) {
					stats->MPI_ERROR = err;
				}

				fl_err = 1;
			}

			if (stats != MPI_STATUSES_IGNORE) {
				++stats;
			}
		}
	}
/*
 * Generate an end wrapper trace.
 */
	if (fl_trace) {
		lam_tr_wrapend(BLKMPITESTSOME);
	}

	if (fl_err) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPITESTSOME,
				lam_mkerr(MPI_ERR_IN_STATUS, 0)));
	}

	lam_resetfunc_m(BLKMPITESTSOME);
	return(MPI_SUCCESS);
}
