/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: packsize.c,v 6.3 1999/12/31 21:27:34 jsquyres Exp $
 *
 *	Function:	- get size needed to pack data
 *	Accepts:	- count
 *			- datatype
 *			- communicator
 *			- ptr to size
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Pack_size - Returns the upper bound on the amount of space needed
to pack a message

Input Parameters:
+ count - count argument to packing call (integer) 
. dtype - datatype argument to packing call (handle) 
- comm - communicator argument to packing call (handle) 

Output Parameter:
. psize - upper bound on size of packed message, in bytes (integer) 

Notes: 

The MPI standard document describes this in terms of 'MPI_Pack', but
it applies to both the 'MPI_Pack' and 'MPI_Unpack' functions.  That
is, the value 'size' is the maximum that is needed by either
'MPI_Pack' or 'MPI_Unpack'.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_COUNT
.N MPI_ERR_TYPE
.N MPI_ERR_ARG

.seealso MPI_Pack, MPI_Unpack

.N ACK
@*/
int MPI_Pack_size(int count, MPI_Datatype dtype, 
		  MPI_Comm comm, int *psize)
{
	lam_initerr();
	lam_setfunc(BLKMPIPACKSIZE);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIPACKSIZE, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (count < 0) {
		return(lam_errfunc(comm,
			BLKMPIPACKSIZE, lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if (dtype == MPI_DATATYPE_NULL) {
		return(lam_errfunc(comm,
			BLKMPIPACKSIZE, lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (psize == 0) {
		return(lam_errfunc(comm,
			BLKMPIPACKSIZE, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Return the size.
 */
	*psize = count * dtype->dt_size;

	lam_resetfunc(BLKMPIPACKSIZE);
	return(MPI_SUCCESS);
}
