/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: pack.c,v 6.4 1999/12/31 21:27:34 jsquyres Exp $
 *
 *	Function:	- pack from data buffer
 *	Accepts:	- data buffer
 *			- data count
 *			- data type
 *			- packing buffer
 *			- packing buffer size
 *			- packing buffer position (in-out)
 *			- communicator
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@
    MPI_Pack - Packs a datatype into contiguous memory

Input Parameters:
+ buf - input buffer start (choice) 
. count - number of input data items (integer) 
. dtype - datatype of each input data item (handle) 
. packsize - output buffer size, in bytes (integer) 
- comm - communicator for packed message (handle) 

Input/Output Parameter:
. packpos - current position in buffer, in bytes (integer) 

Output Parameter:
. packbuf - output buffer start (choice) 

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_TYPE
.N MPI_ERR_COUNT
.N MPI_ERR_ARG
.N MPI_ERR_BUFFER
.N MPI_ERR_TRUNCATE

.seealso: MPI_Pack_size, MPI_Unpack

.N ACK
@*/
int MPI_Pack(void *buf, int count, MPI_Datatype dtype, 
	     void *packbuf, int packsize, int *packpos, 
	     MPI_Comm comm)
{
	int		size;
	char		*p;

	lam_initerr();
	lam_setfunc(BLKMPIPACK);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIPACK, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (dtype == MPI_DATATYPE_NULL || (!dtype->dt_commit)) {
		return(lam_errfunc(comm,
			BLKMPIPACK, lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	if (count < 0) {
		return(lam_errfunc(comm,
			BLKMPIPACK, lam_mkerr(MPI_ERR_COUNT, 0)));
	}

	if ((packsize < 0) || (packpos == 0)) {
		return(lam_errfunc(comm,
			BLKMPIPACK, lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if ((*packpos < 0) || ((*packpos >= packsize) && (packsize > 0))) {
		return(lam_errfunc(comm,
			BLKMPIPACK, lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if ((packsize > 0) && (packbuf == 0)) {
		return(lam_errfunc(comm,
			BLKMPIPACK, lam_mkerr(MPI_ERR_BUFFER, 0)));
	}
/*
 * Pack the data.
 */
	p = ((char *) packbuf) + *packpos;
	size = packsize - *packpos;

	size = lam_pack((char *) buf, count, dtype, p, size);

	if (size < 0) {
		return(lam_errfunc(comm,
			BLKMPIPACK, lam_mkerr(MPI_ERR_TRUNCATE, 0)));
	}

	*packpos += size;

	lam_resetfunc(BLKMPIPACK);
	return(MPI_SUCCESS);
}
