/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: csetname.c,v 1.3 2001/02/03 20:09:54 jsquyres Exp $
 *
 *	Function:	- set communicator name
 *	Accepts:	- communicator
 *			- communicator name
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <string.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>


/*@
   MPI_Comm_set_name - Set a name on a communicator

Input Parameters:
+ comm - Communicator
- name - Pointer to the name of communicator

Notes:

This function associates a name with a communicator, which can be
useful for debugging purposes.  The function 'MPI_Comm_get_name' can
be used to retrieve name that was previously set with
'MPI_Comm_set_name'.

The 'MPI_COMM_WORLD', 'MPI_COMM_SELF', and parent communicator (if an
application was launched via 'MPI_Comm_spawn' or
'MPI_Comm_spawn_multiple') have the predefined names `MPI_COMM_WORLD`,
`MPI_COMM_SELF`, and `MPI_COMM_PARENT`, respectively.  It is legal to
change any of these names with calls to 'MPI_Comm_set_name'.

Names passed in C must include the null terminator, and can be of
maximum length 'MPI_MAX_OBJECT_NAME' (including the null terminator).
Names passed in Fortran can also have a maximum length of
'MPI_MAX_OBJECT_NAME'.

Setting a name on a communicator that already has a name associated
with it replaces the old name.

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_ARG

.seealso MPI_Comm_get_name, MPI_Type_get_name, MPI_Type_set_name, MPI_Win_get_name, MPI_Win_set_name

.N WEB
@*/
int
MPI_Comm_set_name(MPI_Comm comm, char *name)
{
	lam_initerr_m();
	lam_setfunc_m(BLKMPICOMMSETNAME);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPICOMMSETNAME, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (name == 0) {
		return(lam_errfunc(comm,
			BLKMPICOMMSETNAME, lam_mkerr(MPI_ERR_ARG, 0)));
	}

	strncpy(comm->c_name, name, MPI_MAX_OBJECT_NAME);
	comm->c_name[MPI_MAX_OBJECT_NAME - 1] = 0;
	if (LAM_TRACE_TOP())
	  lam_tr_commname(comm->c_name, comm->c_contextid, MPI_Wtime());

	lam_resetfunc_m(BLKMPICOMMSETNAME);
	return(MPI_SUCCESS);
}
