/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	GDB
 *
 *	$Id: pqdoom.c,v 6.4 1999/06/12 17:11:33 kmeyer1 Exp $
 *
 *	Function:	- kenyad
 *			- delivers a signal to local processes
 */

#include <kio.h>
#include <preq.h>
#include <t_types.h>
#include <terror.h>
#include <typical.h>

/*
 * public functions
 */
void			pqdoom();
void			pdoom();

/*
 * external variables
 */
extern struct pprocess	ptable[];	/* kenya process table */
extern struct preq	pq;		/* kenya request */
extern struct preply	pr;		/* kenya reply */

/*
 * external functions
 */
extern void		lam_cleanup_objects();
extern void		pdetachindex();
extern void		psendr();
extern int		pmatch();

/*
 *	pqdoom
 *
 *	Function:	- signal delivery server
 */
void
pqdoom()

{
	pdoom(&pq, &pr);
/*
 * Send reply of doom operation back to client.
 */
	psendr((char *) &pr, sizeof(pr), 0);
}

/*
 *	pdoom
 *
 *	Function:	- delivers signals to matching processes
 *			- also called internally in lamd
 *			- checks for a match on each used process descriptor
 *			- signals each matched process using kdoom()
 *			- if no process is matched and a pid is supplied,
 *			  attempts to doom process with the kernel only
 */
void
pdoom(request, reply)

struct preq		*request;
struct preply		*reply;

{
	int		i;
	int		found_match;	/* did anything match? */

	found_match = FALSE;
	reply->pr_reply = 0;

	for (i = 0; (i < PMAX) && !pr.pr_reply; ++i) {

		if (!pmatch(ptable + i, request->pq_sel_method,
				request->pq_sel_value)) continue;

		found_match = TRUE;
/*
 * Signal process.
 */
		if (kdoom(ptable[i].p_pid, request->pq_signal)) {

			if (errno != ESERVICE) {

				if (!(ptable[i].p_rtf & RTF_KENYA)) {
					pdetachindex(i);
					found_match = FALSE;
				} else {
					reply->pr_reply = ltot(errno);
				}
			}
		}
	}
/*
 * Mop up any persistent RPI state when killing all user processes.
 */
	if (request->pq_sel_method == SELECT_APPL
			&& request->pq_signal == SIGUDIE) {
		lam_cleanup_objects();
	}

	if (!found_match && ((request->pq_sel_method == SELECT_PID) ||
			(request->pq_sel_method == SELECT_INDEX))) {
		reply->pr_reply = ltot(ENOTPROCESS);
	}
}
