/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: getrent.c,v 6.3 1999/05/26 18:29:02 kmeyer1 Exp $
 *
 *	Function:	- obtains routing information
 *			- based on Trollius 2.0 Copyright 1990
 *			  The Ohio State University and Cornell Research
 *			  Foundation
 */

#include <all_hash.h>
#include <events.h>
#include <kio.h>
#include <net.h>
#include <rreq.h>
#include <typical.h>

#include <string.h>
#include <unistd.h>

/*
 * external variables
 */
extern int		errno;
extern struct kio_t	_kio;

/*
 *	getrent
 *
 *	Functions:	- obtains one route entry from the router
 *			- nodeid must be set in route entry
 *	Accepts:	- route entry ptr
 *	Returns:	- 0 or ERROR
 */
int
getrent(rent)

struct route		*rent;

{
	struct rreq	*request;	/* router request */
	struct rreply	*reply;		/* router reply */
	struct nmsg	nhreq;		/* request message */
	struct nmsg	nhreply;	/* reply message */
/*
 * Set up the router request.
 */
	request = (struct rreq *) nhreq.nh_data;
	request->rq_src_event = -getpid();
	request->rq_request = RQGETROUTE;
	request->rq_nodeid = rent->r_nodeid;

	nhreq.nh_dl_event = EVROUTER;
	nhreq.nh_node = LOCAL;
	nhreq.nh_event = EVROUTER;
	nhreq.nh_type = RSMART;
	nhreq.nh_flags = 0;
	nhreq.nh_length = 0;
	nhreq.nh_msg = 0;

	nhreply.nh_event = -getpid();
	nhreply.nh_type = 0;
	nhreply.nh_flags = 0;
	nhreply.nh_length = 0;
	nhreply.nh_msg = 0;

	if (dsfr(&nhreq, &nhreply)) return(LAMERROR);

	reply = (struct rreply *) nhreply.nh_data;

	if (reply->rr_reply) {
		errno = reply->rr_reply;
		return(LAMERROR);
	} else {
		rent->r_nodetype = reply->rr_nodetype;
		rent->r_event = reply->rr_event;
		rent->r_link = reply->rr_link;
		rent->r_event2 = reply->rr_event2;
		rent->r_link2 = reply->rr_link2;
		return(0);
	}
}

/*
 *	getrentc
 *
 *	Function:	- obtains and caches a route table entry
 *			- nodeid must be set in route entry
 *			- uses getrent() to consult the router
 *	Accepts:	- route entry ptr
 *	Returns:	- 0 or ERROR
 */
int
getrentc(rent)

struct route		*rent;

{
	struct route	*phit;
	SHASH		ahsd;
/*
 * Initialize route cache.
 */
	if (ahs_init((int4) MAXRCACHE, (int4) sizeof(struct route),
			(int4) NOTNODEID, AHNOINIT,
			(void *) _kio.ki_route, (int4 *) 0, &ahsd) == 0) {
		return(LAMERROR);
	}
/*
 * Check the route cache.
 */
	phit = (struct route *) ahs_find(&ahsd, rent->r_nodeid);

	if (phit) {
		memcpy((char *) rent, (char *) phit, sizeof(struct route));
	}
/*
 * Get the route table entry from the router.
 */
	else {
		if (getrent(rent)) return(LAMERROR);

		if (rent->r_nodetype != NOTNODETYPE) {
/*
 * It is not the dumb route so insert the entry in the route cache.
 */
			if (ahs_kick(&ahsd, (void *) rent)) return(LAMERROR);
		}
	}

	return(0);
}

/*
 *	getroute
 *
 *	Function:	- fills routing info in network message header
 *			- destination nodeid/event must be set
 *			- fills forwarding event and link info
 *	Accepts:	- ptr network message header
 *	Returns:	- 0 or ERROR
 */
int
getroute(nhead)

struct nmsg		*nhead;

{
	struct route	rent;		/* route table entry */
/*
 * Local node, forward on the given event.
 */
	if (nhead->nh_node == LOCAL) {
		nhead->nh_dl_event = nhead->nh_event;
	}
/*
 * Get a route table entry.
 */
	else {
		rent.r_nodeid = nhead->nh_node;

		if (getrentc(&rent)) return(LAMERROR);
/*
 * Is this my own node.  Forward on the given event.
 */
		if (rent.r_event == NOTEVENT) {
			nhead->nh_dl_event = nhead->nh_event;
		}
/*
 * Set the forwarding event and link from the route table entry.
 */
		else {
			nhead->nh_dl_event = rent.r_event;
			nhead->nh_dl_link = rent.r_link;
		}
	}

	return(0);
}

/*
 *	getrtype
 *
 *	Function:	- gets the type of a given node
 *	Accepts:	- node identifier
 *	Returns:	- nodetype or NOTNODETYPE
 */
int4
getrtype(nodeid)

int4			nodeid;

{
	struct route	rent;		/* route table entry */

	rent.r_nodeid = nodeid;

	if (getrentc(&rent)) {
		return(NOTNODETYPE);
	} else {
/*
 * Note: Even if getrentc() is successful, the nodetype returned 
 * in rent might still be NOTNODETYPE when a dumb route is returned.
 */
		errno = 0;
		return(rent.r_nodetype);
	}
}
