// LabPlot : Plot3D.cc

#include <iostream>
#include <klocale.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include "Plot3D.h"

using namespace std;

//! general 3D Plot class
Plot3D::Plot3D(Worksheet *p)
	: Plot(p)
{
	QFont font = p->getMainWin()->defaultFont();
	font.setPointSize((int)(0.7*font.pointSize()));	// for axes label
	title = new Label(i18n("Title 3D"),font,QColor(Qt::black));
	title->setPosition(0.4,0.04);

	// set default axis settings for all axes
	for(int i=0;i<12;i++)
		readAxisSettings(&axis[i],P3D,i);

	axis[0].setLabel(new Label(i18n("x-Axis"),font));
	axis[1].setLabel(new Label(i18n("y-Axis"),font));
	axis[2].setLabel(new Label(i18n("z-Axis"),font));
	axis[3].setLabel(new Label(i18n("x2-Axis"),font));
	axis[4].setLabel(new Label(i18n("y2-Axis"),font));
	axis[5].setLabel(new Label(i18n("z2-Axis"),font));
	axis[6].setLabel(new Label(i18n("x3-Axis"),font));
	axis[7].setLabel(new Label(i18n("y3-Axis"),font));
	axis[8].setLabel(new Label(i18n("z3-Axis"),font));
	axis[9].setLabel(new Label(i18n("x4-Axis"),font));
	axis[10].setLabel(new Label(i18n("y4-Axis"),font));
	axis[11].setLabel(new Label(i18n("z4-Axis"),font));

	// grid & border
	font.setPointSize((int)(0.7*font.pointSize()));	// for tic label

	for (int i=0;i<12;i++) {
		axis[i].setTickLabelFont(font);
	}

	renderstyle=0;
}

Plot3D::~Plot3D() {
	delete graphlist;

	delete title;
	delete markx;
	delete marky;
	delete region;
}

QStringList Plot3D::Info() {
	kdDebug()<<"Plot3D::Info()"<<endl;
	QStringList s;
	s<<"3D";
	s<<QString::number(position.X())+QString(" , ")+QString::number(position.Y());
	s<<QString::number(size.X())+QString(" X ")+QString::number(size.Y());
	if (transparent)
		s<<QString("yes");
	else
		s<<QString("no");
	s<<bgcolor.color().name();
	s<<gbgcolor.color().name();

	return s;
}

void Plot3D::setBorder(int item, bool on) {
	kdDebug()<<"Plot3D::setBorder() : item="<<item<<endl;
	const int unit = 5, numunit = 40, numunit2 = 20;
	int w = worksheet->width(), h = worksheet->height();

	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	if(item == 3) {
		if (on) {
			if (axis[3].getLabel()->Title().length() > 0) ymax -= axis[3].getLabel()->Font().pointSize();
			if (axis[3].MajorTicksEnabled())   ymax -= unit+numunit2;
			if (axis[3].MinorTicksEnabled())   ymax -= unit;
		}
		else {
			if (axis[3].getLabel()->Title().length() > 0) ymax += axis[3].getLabel()->Font().pointSize();
			if (axis[3].MajorTicksEnabled())   ymax += unit+numunit2;
			if (axis[3].MinorTicksEnabled())   ymax += unit;

		}
	}
	if(item == 5) {
		if (on) {
			if (axis[5].getLabel()->Title().length() > 0) xmin += axis[5].getLabel()->Font().pointSize();
			if (axis[5].MajorTicksEnabled())   xmin += unit+numunit;
			if (axis[5].MinorTicksEnabled())   xmin += unit;
		}
		else {
			if (axis[5].getLabel()->Title().length() > 0) xmin -= axis[5].getLabel()->Font().pointSize();
			if (axis[5].MajorTicksEnabled())   xmin -= unit+numunit;
			if (axis[5].MinorTicksEnabled())   xmin -= unit;
		}
	}
	if(item == 9) {
		if (on) {
			if (axis[9].getLabel()->Title().length() > 0) ymin += axis[9].getLabel()->Font().pointSize();
			if (axis[9].MajorTicksEnabled())   ymin += unit+numunit2;
			if (axis[9].MinorTicksEnabled())   ymin += unit;
		}
		else {
			if (axis[9].getLabel()->Title().length() > 0) ymin += axis[9].getLabel()->Font().pointSize();
			if (axis[9].MajorTicksEnabled())   ymin += unit+numunit2;
			if (axis[9].MinorTicksEnabled())   ymin += unit;
		}
	}
	if(item == 11) {
		if (on) {
			if (axis[11].getLabel()->Title().length() > 0) xmax -= axis[11].getLabel()->Font().pointSize();
			if (axis[11].MajorTicksEnabled())   xmax -= unit+numunit;
			if (axis[11].MinorTicksEnabled())   xmax -= unit;
		}
		else {
			if (axis[11].getLabel()->Title().length() > 0) xmax += axis[11].getLabel()->Font().pointSize();
			if (axis[11].MajorTicksEnabled())   xmax += unit+numunit;
			if (axis[11].MinorTicksEnabled())   xmax += unit;
		}
	}

	setXMin(xmin,w);
	setXMax(xmax,w);
	setYMin(ymin,h);
	setYMax(ymax,h);
}

void Plot3D::draw(QPainter *p,int w, int h) {
	kdDebug()<<"Plot3D::draw() w/h : "<<w<<' '<<h<<endl;

	if(aspect_enabled) {	// set aspect ratio to 1
		int wsize = (int) fmin(w,h);
		w=h=wsize;
	}

	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	// 3d perspective
	xmax -= (xmax-xmin)/3;
	ymin += (ymax-ymin)/3;

	kdDebug()<<"XMIN/MXAX/YMIN/YMAX = "<<xmin<<','<<xmax<<','<<ymin<<','<<ymax<<endl;
	kdDebug()<<"p1 = "<<p1.X()<<'/'<<p1.Y()<<" p2 = "<<p2.X()<<'/'<<p2.Y()<<endl;

	if (!transparent) {
		// background color
		p->setBrush(bgcolor);
		p->setPen(Qt::NoPen);
		p->drawRect((int)(w*position.X()),(int)(h*position.Y()),(int)(w*size.X()),(int)(h*size.Y()));

		// graph background color
		p->setBrush(gbgcolor);
		p->setPen(Qt::NoPen);
	}

	QPointArray a;
	a.setPoints( 6, xmin,ymax,xmax,ymax,
		xmax+(xmax-xmin)/2,(ymax+ymin)/2,xmax+(xmax-xmin)/2,ymin-(ymax-ymin)/2,
		(xmax+xmin)/2,ymin-(ymax-ymin)/2,xmin,ymin);
	p->drawPolygon(a);

	// TODO : draw y baseline here (before every axis ?) -> see Plot2D.cc
	// TODO : draw x baseline here (before every axis ?) -> see Plot2D.cc

	// TODO
	kdDebug()<<"PLOT3D : title->draw() pos:"<<position.X()<<' '<<position.Y()<<endl;
	kdDebug()<<" 			size:"<<size.X()<<' '<<size.Y()<<endl;
	title->draw(worksheet,p,position,size,w,h,0);

	drawAxes(p,w,h,0);	// all axes & borders
	drawBorder(p,w,h,0);

	drawCurves(p,w,h);

	drawAxes(p,w,h,1);	// only front axes & borders
	drawBorder(p,w,h,1);

	if(legend.Enabled())
		legend.draw(p,type,graphlist,position,size,w,h);

	p->setPen(Qt::NoPen);
}

void Plot3D::drawBorder(QPainter *p,int w, int h,bool hide) {
	kdDebug()<<"Plot3D::drawBorder()"<<endl;
	// TODO : borderwidth
	int start=0;
	if (hide) start=3;

	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	// 3d perspective
	xmax -= (xmax-xmin)/3;
	ymin += (ymax-ymin)/3;

	int x1[12]={(xmax+xmin)/2,xmin,(xmax+xmin)/2,xmin,xmax,xmin,
		xmin,xmax,xmax,(xmax+xmin)/2,xmin,(3*xmax-xmin)/2};
	int y1[12]={(ymin+ymax)/2,ymax,(3*ymin-ymax)/2,ymax,ymax,ymin,
		ymin,ymin,ymin,(3*ymin-ymax)/2,ymin,(3*ymin-ymax)/2};
	int x2[12]={(3*xmax-xmin)/2,(xmin+xmax)/2,(xmax+xmin)/2,xmax,(3*xmax-xmin)/2,xmin,
		xmax,(3*xmax-xmin)/2,xmax,(3*xmax-xmin)/2,(xmax+xmin)/2,(3*xmax-xmin)/2};
	int y2[12]={(ymin+ymax)/2,(ymax+ymin)/2,(ymin+ymax)/2,ymax,(ymin+ymax)/2,ymax,
		ymin,(3*ymin-ymax)/2,ymax,(3*ymin-ymax)/2,(3*ymin-ymax)/2,(ymin+ymax)/2};

	for (int i=start;i<12;i++) {
		if (axis[i].BorderEnabled()) {
			p->setPen(QPen(axis[i].BorderColor(),axis[i].borderWidth()));
			p->drawLine(x1[i],y1[i],x2[i],y2[i]);
		}
	}
}

void Plot3D::drawCurves(QPainter *p, int w, int h) {
	kdDebug()<<"Plot3D::drawCurves()"<<endl;
	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	// 3d perspective
	xmax -= (xmax-xmin)/3;
	ymin += (ymax-ymin)/3;

	kdDebug()<<"xmin/xmax ymin/ymax : "<<xmin<<'/'<<xmax<<' '<<ymin<<'/'<<ymax<<endl;
	kdDebug()<<"width/height : "<<w<<'/'<<h<<endl;

	for (unsigned int i=0; i < graphlist->Number() ; i++) {
		kdDebug()<<"draw Graph "<<i<<endl;
		if(graphlist->getGraph(i)->isShown() == false) continue;

		double oldx=0, oldy=0;
		int N, NY=0, NX=0;
		Style *style=0;
		Symbol *symbol=0;
		GRAPHType gtype = graphlist->getType(i);
		Graph *graph = graphlist->getGraph(i);
		Point3D *d=0;
		if (gtype == GRAPH3D) {
			kdDebug()<<"GRAPH3D "<<endl;
			Graph3D *g = graphlist->getGraph3D(i);

			d = g->Data();
			NX=g->NX();
			NY=g->NY();
			kdDebug()<<"NX/NY "<<NX<<' '<<NY<<endl;
			style = g->getStyle();
			symbol = g->getSymbol();
		}
		else if (gtype == GRAPHM) {
			GraphM *g = graphlist->getGraphM(i);

			double *data = g->Data();
			NX=g->NX();
			NY=g->NY();
			kdDebug()<<"GRAPHM "<<NX<<'x'<<NY<<endl;
			// conversion to 3d Point data
			d = new Point3D[NX*NY];
			for (int j=0;j<NY;j++) {
				for (int k=0;k<NX;k++) {
					d[k+j*NX].setPoint(k+1,j+1,data[k+j*NX]);
				}
			}
			style =  g->getStyle();
			symbol = g->getSymbol();
		}
		if (NY==0)
			N=NX;
		else
			N=NX*NY;

		// calculate position matrix
		Point *pos = new Point[NX*NY];
		double x,y;
		for(int index=0;index<NX*NY;index++) {
			x = xmin + (d[index].X() - actrange[0].rMin()) *
				(xmax-xmin)/(actrange[0].rMax() - actrange[0].rMin())
				+ (xmax-xmin)/(2*(actrange[1].rMax() -
				actrange[1].rMin()))*(d[index].Y()-actrange[1].rMin());
			y = ymax - (d[index].Y() - actrange[1].rMin()) *
				(ymax-ymin)/(actrange[1].rMax() - actrange[1].rMin())
				+ (ymax-ymin)/(2*(actrange[1].rMax() -
				actrange[1].rMin()))*(d[index].Y()-actrange[1].rMin())
				- (d[index].Z()-actrange[2].rMin()) * (ymax-ymin)/(actrange[2].rMax()
					- actrange[2].rMin());
			pos[index].setPoint(x,y);
		}

		// do delaunay triangulation
		if (style && style->Type() == LINESTYLE && N>3) {
			kdDebug()<<"Doing Delaunay triangulation"<<endl;
			QPen pen( style->Color(), style->Width(),(Qt::PenStyle) style->PenStyle() );
			p->setPen(pen);
			QPointArray a(100);	// maximum number of points for one region
			FILE *data = fopen("/tmp/labplot-qhull","w+");
			if(data == NULL) {
				kdDebug()<<"Can't open temp file /tmp/labplot-qhull! Giving up calculation and drawing."<<endl;
				break;
			}

			if(renderstyle == TRUE) {	// 3d	: use if dz>0 and selected
				fprintf(data,"3\n%d\n",NX*NY);
				for (int l=0;l<NX*NY;l++)
					fprintf(data,"%g %g %g\n",d[l].X(),d[l].Y(),d[l].Z());
			}
			else {	// 2d 	: use if dz=0, also very nice for surfaces (projection)
				fprintf(data,"2\n%d\n",NX*NY);
				for (int l=0;l<NX*NY;l++)
					fprintf(data,"%g %g\n",d[l].X(),d[l].Y());
			}
			fclose(data);

			// call qdelaunay; filenames are very strict! dont use d, n, etc.
			KProcess *proc = new KProcess;
			*proc << "qdelaunay";
			*proc << "s Qt Fv TI /tmp/labplot-qhull TO /tmp/labplot-qhull-out";

			if( proc->start(KProcess::Block) == false) {
				kdDebug()<<"COULD NOT FIND qdelaunay! Disable line style."<<endl;
				// don't explain every time
				// KMessageBox::error(worksheet->getMainWin(),i18n("Could not find qdelaunay!"));
			}
			else {
				FILE *pdata = fopen("/tmp/labplot-qhull-out","r+");
				int nr;	// number of regions
				fscanf(pdata,"%d",&nr);
				for(int i=0;i<nr;i++) {
					int dim=0;
					fscanf(pdata,"%d",&dim);
					// printf("REGION HAS %d POINTS\n",dim);
					if (dim>100) continue;		// should not happen

					int nr_points=0, np;
					for(int j=0;j<dim;j++) {
						fscanf(pdata,"%d",&np);
						a.setPoint(nr_points++,(int)pos[np].X(),(int)pos[np].Y());
					}
					a.resize(nr_points);
					if (style->isFilled()) {	//filled
						p->setBrush(QBrush(style->FillColor(),(Qt::BrushStyle)style->Brush()));
						p->drawPolygon(a);
					}
					else
						p->drawPolyline(a);
				}
			}
			// remove tmp file names !
			unlink("/tmp/labplot-qhull");
			unlink("/tmp/labplot-qhull-out");
		}

		// other styles
		for (int k=NY-1; k>=0;k--) {
			if(worksheet->getMainWin()->speedMode()) {
				int mod = (int) (NY/sqrt((double) worksheet->getMainWin()->speedModeValue()));
				if(mod==0) mod=1;
				if(k%mod != 0) continue;	// speed mode
			}
			for(int j=NX-1 ;j>=0;j--) {
				int index = j+k*NX;
				if (d[index].Masked()) continue;

				x = pos[index].X();
				y = pos[index].Y();

				if(worksheet->getMainWin()->speedMode()) {
					int mod = (int) (NX/sqrt((double) worksheet->getMainWin()->speedModeValue()));
					if(mod==0) mod=1;
					if(j%mod != 0) continue;	// speed mode
				}
				// TODO : scales (log10,log2,ln,sqrt)

				int xlen = xmax-xmin;
				int ylen = ymax-ymin;
				if (NX>1) xlen = (xmax-xmin)/(NX-1);
				if (NY>1) ylen = (ymax-ymin)/(NY-1);

				int dropy = (int)(y+(d[index].Z()-actrange[2].rMin()) *
						(ymax-ymin)/(actrange[2].rMax() - actrange[2].rMin()));

				if (x >= xmin-1 && x <= xmax+(xmax-xmin)/2+1
						&& y >= ymin-(ymax-ymin)/2-1 && y <= ymax+1) {
					QPen pen( style->Color(), style->Width(),(Qt::PenStyle) style->PenStyle() );
					p->setPen(pen);

					if (style->isFilled()) {
						QPointArray a;
						p->setBrush(style->FillColor());

						if (style->Type() == STEPSSTYLE || style->Type() == BOXESSTYLE) {
							a.setPoints( 4,(int)(x-3*xlen/4),(int)(y+ylen/4),(int)(x+xlen/4),(int)(y+ylen/4),
									(int)(x+3*xlen/4),(int)(y-ylen/4),(int)(x-xlen/4),(int)(y-ylen/4));
							p->drawPolygon(a);
							if (style->Type() == BOXESSTYLE) {
								a.setPoints(4,(int)(x-3*xlen/4),(int)(y+ylen/4),(int)(x-3*xlen/4),
									(int)(dropy+ylen/4),(int)(x+xlen/4),(int)(dropy+ylen/4),
									(int)(x+xlen/4),(int)(y+ylen/4));
								p->drawPolygon(a);
								a.setPoints(4,(int)(x+xlen/4),(int)(y+ylen/4),(int)(x+xlen/4),
									(int)(dropy+ylen/4),(int)(x+3*xlen/4),(int)(dropy-ylen/4),
									(int)(x+3*xlen/4),(int)(y-ylen/4));
								p->drawPolygon(a);
							}
						}
					}
					else {
						if (style->Type() == STEPSSTYLE || style->Type() == BOXESSTYLE) {
							p->drawLine((int)(x-3*xlen/4),(int)(y+ylen/4),(int)(x+xlen/4),(int)(y+ylen/4));
							p->drawLine((int)(x+xlen/4),(int)(y+ylen/4),(int)(x+3*xlen/4),(int)(y-ylen/4));
							p->drawLine((int)(x+3*xlen/4),(int)(y-ylen/4),(int)(x-xlen/4),(int)(y-ylen/4));
							p->drawLine((int)(x-xlen/4),(int)(y-ylen/4),(int)(x-3*xlen/4),(int)(y+ylen/4));

							if (style->Type() == BOXESSTYLE) { // drop step lines;
								p->drawLine((int)(x-3*xlen/4),(int)(y+ylen/4), (int)(x-3*xlen/4),(int)(dropy+ylen/4));
								p->drawLine((int)(x+xlen/4),(int)(y+ylen/4), (int)(x+xlen/4),(int)(dropy+ylen/4));
								p->drawLine((int)(x+3*xlen/4),(int)(y-ylen/4), (int)(x+3*xlen/4),(int)(dropy-ylen/4));
								p->drawLine((int)(x-xlen/4),(int)(y-ylen/4), (int)(x-xlen/4),(int)(dropy-ylen/4));

								// base step
								p->drawLine((int)(x-3*xlen/4),(int)(dropy+ylen/4), (int)(x+xlen/4),(int)(dropy+ylen/4));
								p->drawLine((int)(x+xlen/4),(int)(dropy+ylen/4), (int)(x+3*xlen/4),(int)(dropy-ylen/4));
								p->drawLine((int)(x+3*xlen/4),(int)(dropy-ylen/4), (int)(x-xlen/4),(int)(dropy-ylen/4));
								p->drawLine((int)(x-xlen/4),(int)(dropy-ylen/4), (int)(x-3*xlen/4),(int)(dropy+ylen/4));
							}
						}
					}

					if (style->Type() == IMPULSESSTYLE)
						p->drawLine((int)x,(int)y,(int)x,(int)dropy);

					symbol->draw(p,(int)x,(int)y);
					graph->getAnnotateValues().draw(p,(int)x,(int)y,d[index].X(),d[index].Y(),d[index].Z());
				}
				oldx = x;oldy = y;
			}
		}
	}
	kdDebug()<<"Plot3D::drawCurves() DONE"<<endl;
}

//! draw the axes. hide=1 : only front axes (others hidden by curves)
void Plot3D::drawAxes(QPainter *p, int w, int h, bool hide) {
	kdDebug()<<"Plot3D::drawAxes()"<<endl;
	const int unit = 5, gap = 10, numunit = 40, numunit2 = 20;

	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));

	// 3d perspective
	xmax -= (xmax-xmin)/3;
	ymin += (ymax-ymin)/3;

	kdDebug()<<"xmin/xmax ymin/ymax : "<<xmin<<'/'<<xmax<<' '<<ymin<<'/'<<ymax<<endl;
	kdDebug()<<"width/height : "<<w<<'/'<<h<<endl;

	int start=0;
	if (hide == 1) start=1;

	// axes label
	// x,x2,x3,x4
	double xx[4] = {xmax,xmin+(xmax-xmin)/2,xmin+(xmax-xmin)/2,xmax};
	double yx[4] = {(ymax+ymin)/2+(2*unit+numunit2)*axis[0].MajorTicksEnabled(),
		ymax+(2*unit+numunit2)*axis[3].MajorTicksEnabled(),ymin-(2*unit+numunit2)*axis[6].MajorTicksEnabled()-3*gap,
		ymin-(ymax-ymin)/2-(2*unit+numunit2)*axis[9].MajorTicksEnabled()-3*gap};
	for (int i=start;i<4;i++) {
		Label *label = axis[3*i].getLabel();
		if (label->X()==0)	// default
			label->setPosition((xx[i]+gap)/(double)w,yx[i]/(double)h);
		if (axis[3*i].Enabled())
			label->draw(worksheet,p,position,size,w,h,0);
	}

	// z,z2,z3,z4
	double xz[4] = {xmax-2*(xmax-xmin)/3-2*gap+axis[2].getLabel()->Font().pointSize()-gap,
			gap+axis[5].getLabel()->Font().pointSize()-gap, xmax+(2*unit+numunit)*axis[8].MajorTicksEnabled(),
			xmax+(xmax-xmin)/2+(2*unit+numunit)*axis[11].MajorTicksEnabled()};
	double yz[4] = {ymin,ymin+(ymax-ymin)/2,ymin+(ymax-ymin)/2,ymin};
	for (int i=start;i<4;i++) {
		Label *label = axis[3*i+2].getLabel();
		p->save();
		if (label->X()==0)	// default
			label->setPosition(xz[i]/(double)w,(yz[i]-gap)/(double)h);
		if (axis[3*i+2].Enabled())
			label->draw(worksheet,p,position,size,w,h,270);
		p->restore();
	}

	// y,y,2,y3,y4
	double dx = (xmax-xmin)/2, dy = (ymax-ymin)/2;
	double phi = 180.0/M_PI*atan(dy/dx);
	double xy[4] = {xmin+(xmax+xmin)/4-(2*unit+numunit)*axis[1].MajorTicksEnabled(),
		xmax+(xmax+xmin)/4+(2*unit+numunit)*axis[4].MajorTicksEnabled()-4*gap,
		xmax+(xmax+xmin)/4+(2*unit+numunit)*axis[7].MajorTicksEnabled()-4*gap,
		xmin+(xmax+xmin)/4-(2*unit+numunit)*axis[10].MajorTicksEnabled()
		};
	double yy[4] = {ymax-(ymax-ymin)/4-4*gap,ymax-(ymax-ymin)/4+2*gap,
		ymin-(ymax-ymin)/4+2*gap,ymin-(ymax-ymin)/4-4*gap };
	for (int i=start;i<4;i++) {
		Label *label = axis[3*i+1].getLabel();
		p->save();
		if (label->X()==0)	// default
			label->setPosition((xy[i]-2*gap)/(double)w,yy[i]/(double)h);
		if (axis[3*i+1].Enabled())
			label->draw(worksheet,p,position,size,w,h,360-phi);
		p->restore();
	}

	// Ticks and Grid
	kdDebug()<<"Drawing Ticks and Grid"<<endl;
	// TODO : scales (log10,log2,ln,sqrt)
	// x,x2,x3,x4
	for (int k=start;k<4;k++) {
		int xi[4]={(xmin+xmax)/2,xmin,xmin,(xmin+xmax)/2};
		int yi1[4], yi2[4], ykna[4], yknb[4];
		switch (axis[3*k].TickPos()) {	// major/minor tics
		case 0:
			yi1[0]=ymax-(ymax-ymin)/2+10; yi1[1]=ymax; yi1[2]=ymin-10; yi1[3]=ymin-(ymax-ymin)/2-10;
			yi2[0]=ymax-(ymax-ymin)/2; yi2[1]=ymax+10; yi2[2]=ymin; yi2[3]=ymin-(ymax-ymin)/2;
			ykna[0]=ymax-(ymax-ymin)/2+5; ykna[1]=ymax; ykna[2]=ymin-5; ykna[3]=ymin-(ymax-ymin)/2-5;
			yknb[0]=ymax-(ymax-ymin)/2; yknb[1]=ymax+5; yknb[2]=ymin; yknb[3]=ymin-(ymax-ymin)/2;
			break;
		case 1:
			yi1[0]=ymax-(ymax-ymin)/2-10; yi1[1]=ymax-10; yi1[2]=ymin; yi1[3]=ymin-(ymax-ymin)/2;
			yi2[0]=ymax-(ymax-ymin)/2; yi2[1]=ymax; yi2[2]=ymin+10; yi2[3]=ymin-(ymax-ymin)/2+10;
			ykna[0]=ymax-(ymax-ymin)/2-5; ykna[1]=ymax-5; ykna[2]=ymin; ykna[3]=ymin-(ymax-ymin)/2;
			yknb[0]=ymax-(ymax-ymin)/2; yknb[1]=ymax; yknb[2]=ymin+5; yknb[3]=ymin-(ymax-ymin)/2+5;
			break;
		case 2:
			yi1[0]=ymax-(ymax-ymin)/2+10; yi1[1]=ymax-10; yi1[2]=ymin-10; yi1[3]=ymin-(ymax-ymin)/2-10;
			yi2[0]=ymax-(ymax-ymin)/2-10; yi2[1]=ymax+10; yi2[2]=ymin+10; yi2[3]=ymin-(ymax-ymin)/2+10;
			ykna[0]=ymax-(ymax-ymin)/2+5; ykna[1]=ymax-5; ykna[2]=ymin-5; ykna[3]=ymin-(ymax-ymin)/2-5;
			yknb[0]=ymax-(ymax-ymin)/2-5; yknb[1]=ymax-5; yknb[2]=ymin+5; yknb[3]=ymin-(ymax-ymin)/2+5;
			break;
		case 3:
			break;
		}
		int yia[4]={ymin-(ymax-ymin)/2,ymax-(ymax-ymin)/2,ymin,ymin-(ymax-ymin)/2};
		int yib[4]={ymax-(ymax-ymin)/2,ymax,ymax,ymin};
		int yina[4]={ymin-(ymax-ymin)/2,ymax-(ymax-ymin)/2,ymin,ymin-(ymax-ymin)/2};
		int yinb[4]={ymax-(ymax-ymin)/2,ymax,ymax,ymin};
		if (axis[3*k].MajorTicksEnabled() && axis[3*k].Enabled()) {
			double min = actrange[0].rMin(), max = actrange[0].rMax();
			TScale scale = axis[k].Scale();
			int t=-1;
			switch (scale) {
			case LINEAR: case SX2: t = (int) axis[3*k].MajorTicks(); break;
			case LOG10: t = (int) log10(max/min)+2; break;
			case LOG2: t = (int) log2(max/min)+2; break;
			case LN: t = (int) log(max/min)+2; break;
			case SQRT: t = (int) (pow(max,2)-pow(min,2))+1; break;
			}
			if(t==-1)
				t=autoTicks(min,max);
			if(t==0) t=-1;

			for (int i=0;i <= t;i++) {
				int x1=0, x2=0;
				switch(scale) {
				case LINEAR:
					x1 = xi[k]+i*(xmax-xmin)/t;
					x2 = xi[k]+(i+1)*(xmax-xmin)/t;
					break;
				case LOG10: {
					double gap = 1.0-log10(pow(10,ceil(log10(min)))/min);	// fragment of decade to shift left
					double decade = (xmax-xmin)/(log10(max/min));		// width of decade
					x1 = xi[k]+(int)((i-gap)*decade);
					x2 = (int) (x1+decade+ceil(fabs(log10(max))));
					} break;
				case LOG2: {
					double gap = 1.0-log2(pow(2,ceil(log2(min)))/min);	// fragment of decade to shift left
					double decade = (xmax-xmin)/(log2(max/min));		// width of decade
					x1 = xi[k]+(int)((i-gap)*decade);
					x2 = (int) (x1+decade+ceil(fabs(log2(max))));
					} break;
				case LN: {
					double gap = 1.0-log(pow(M_E,ceil(log(min)))/min);	// fragment of decade to shift left
					double decade = (xmax-xmin)/(log(max/min));		// width of decade
					x1 = xi[k]+(int)((i-gap)*decade);
					x2 = (int) (x1+decade+ceil(fabs(log(max))));
					} break;
				case SQRT: {
					x1 = xi[k]+(int)((sqrt(min*min+i)-min)/(max-min)*(xmax-xmin));
					x2 = xi[k]+(int)((sqrt(min*min+i+1)-min)/(max-min)*(xmax-xmin));
					} break;
				default: break;
				}

				int xia[4]={x1,x1+(xmax-xmin)/2,x1,x1};
				int xib[4]={x1,x1,x1,x1-(xmax-xmin)/2};

				// major ticks
				if(axis[3*k].TickPos()!=3) {
//					kdDebug()<<"DRAWING MAJOR TICKS"<<endl;
					p->setPen(QPen(axis[3*k].TickColor(),axis[3*k].majorTickWidth()));
					p->drawLine(x1,yi1[k],x1,yi2[k]);
				}

				if (axis[3*k].MajorGridEnabled()) {
					p->setPen(QPen(axis[3*k].majorGridColor(),
						axis[3*k].majorGridWidth(),axis[3*k].MajorGridType()));
					p->drawLine(xia[k],yia[k],xib[k],yib[k]);
					p->setPen(Qt::SolidLine);
				}
				if (graphlist->Number() > 0) {
					QColor c = axis[3*k].TickLabelColor();
					QFont f = axis[3*k].TickLabelFont();
					int atlf = axis[3*k].TickLabelFormat();

					double dx = max-min;
					double value=0;
					switch(scale) {
						case LINEAR: value = min + i*dx/t; break;
						case LOG10: value = pow(10,ceil(log10(min)))*pow(10.0,i-1); break;
						case LOG2: value = pow(2,ceil(log2(min)))*pow(2.0,i-1); break;
						case LN: value = pow(M_E,ceil(log(min)))*pow(M_E,i-1); break;
						case SQRT: value = min + i*dx/t; break;
						default: break;
					}

					// scale and shift value
					value = value*axis[3*k].Scaling()+axis[3*k].Shift();

					QString label = TicLabel(atlf,axis[3*k].TickLabelPrecision(),
						axis[3*k].DateTimeFormat(),value);

					// apply prefix & suffix
					label.prepend(axis[3*k].TickLabelPrefix());
					label.append(axis[3*k].TickLabelSuffix());

					// draw tic label
					QFontMetrics fm(f);
					p->save();
					int tlg = axis[3*k].TickLabelPosition();
					int yilb[4]={ymax-(ymax-ymin)/2+fm.ascent()+tlg,ymax+fm.ascent()+tlg,
						ymin-tlg,ymin-(ymax-ymin)/2-tlg};
					p->translate(x1,yilb[k]);
					p->rotate(axis[3*k].TickLabelRotation());
					if (atlf == AUTO || atlf == NORMAL || atlf == SCIENTIFIC) {
						p->setPen(c);
						f.setPointSize((int)(f.pointSize()*size.X()));	// resize tic label
						p->setFont(f);

						p->drawText(-fm.width(label)/2,fm.ascent()/2,label);
					}
					else {	// rich text label
						QSimpleRichText *richtext = new QSimpleRichText(label,f);
						QColorGroup cg;
						cg.setColor(QColorGroup::Text,c);
						richtext->draw(p,-richtext->width()/4, -fm.ascent()/2,QRect(),cg);
					}
					p->restore();
				}
				if (axis[3*k].MinorTicksEnabled() && i != t ) {
					for (int j=1;j <= axis[3*k].MinorTicks()+1;j++) {
						int x = x1+j*(x2-x1)/(axis[3*k].MinorTicks()+1);
						int xina[4]={x,x+(xmax-xmin)/2,x,x};
						int xinb[4]={x,x,x,x-(xmax-xmin)/2};
						if(axis[3*k].TickPos()!=3) {
							p->setPen(QPen(axis[3*k].TickColor(),axis[3*k].minorTickWidth()));
							p->drawLine(x,ykna[k],x,yknb[k]);
						}
						if (axis[3*k].MinorGridEnabled() && j != axis[3*k].MinorTicks()+1 ) {
							p->setPen(QPen(axis[3*k].minorGridColor(),
								axis[3*k].minorGridWidth(),axis[3*k].MinorGridType()));
							p->drawLine(xina[k],yina[k],xinb[k],yinb[k]);
							p->setPen(Qt::SolidLine);
						}
					}
				}
			}
		}
	}

	// z,z2,z3,z4
	for (int k=start;k<4;k++) {
		int yi[4]={ymin-(ymax-ymin)/2,ymin,ymin,ymin-(ymax-ymin)/2};
		int xi1[4], xi2[4], xkna[4], xknb[4];
		switch (axis[3*k+2].TickPos()) {
		case 0:
			//major tics
			xi1[0]=(xmax+xmin)/2; xi1[1]=xmin-10; xi1[2]=xmax; xi1[3]=xmax+(xmax-xmin)/2;
			xi2[0]=(xmax+xmin)/2-10; xi2[1]=xmin; xi2[2]=xmax+10; xi2[3]=xmax+(xmax-xmin)/2+10;
			//minor tics
			xkna[0]=(xmax+xmin)/2; xkna[1]=xmin-5; xkna[2]=xmax; xkna[3]=xmax+(xmax-xmin)/2;
			xknb[0]=(xmax+xmin)/2-5; xknb[1]=xmin; xknb[2]=xmax+5; xknb[3]=xmax+(xmax-xmin)/2+5;
			break;
		case 1:
			//major tics
			xi1[0]=(xmax+xmin)/2; xi1[1]=xmin+10; xi1[2]=xmax; xi1[3]=xmax+(xmax-xmin)/2;
			xi2[0]=(xmax+xmin)/2+10; xi2[1]=xmin; xi2[2]=xmax-10; xi2[3]=xmax+(xmax-xmin)/2-10;
			//minor tics
			xkna[0]=(xmax+xmin)/2; xkna[1]=xmin+5; xkna[2]=xmax; xkna[3]=xmax+(xmax-xmin)/2;
			xknb[0]=(xmax+xmin)/2+5; xknb[1]=xmin; xknb[2]=xmax-5; xknb[3]=xmax+(xmax-xmin)/2-5;
			break;
		case 2:
			//major tics
			xi1[0]=(xmax+xmin)/2+10; xi1[1]=xmin-10; xi1[2]=xmax-10; xi1[3]=xmax+(xmax-xmin)/2-10;
			xi2[0]=(xmax+xmin)/2-10; xi2[1]=xmin+10; xi2[2]=xmax+10; xi2[3]=xmax+(xmax-xmin)/2+10;
			//minor tics
			xkna[0]=(xmax+xmin)/2+5; xkna[1]=xmin-5; xkna[2]=xmax-5; xkna[3]=xmax+(xmax-xmin)/2-5;
			xknb[0]=(xmax+xmin)/2-5; xknb[1]=xmin+5; xknb[2]=xmax+5; xknb[3]=xmax+(xmax-xmin)/2+5;
			break;
		}
		int xia[4]={xmin,xmin,xmax+(xmax-xmin)/2,(xmax+xmin)/2};
		int xib[4]={(xmax+xmin)/2,xmax,xmax,xmax+(xmax-xmin)/2};
		int xina[4]={xmin,xmin,xmax,(xmax+xmin)/2};
		int xinb[4]={(xmax+xmin)/2,xmax,xmax+(xmax-xmin)/2,xmax+(xmax-xmin)/2};

		if (axis[3*k+2].MajorTicksEnabled() && axis[3*k+2].Enabled()) {
			double min = actrange[2].rMin(), max = actrange[2].rMax();
			TScale scale = axis[k].Scale();
			int t=-1;
			switch (scale) {
			case LINEAR: t = (int) axis[3*k+2].MajorTicks(); break;
			case LOG10: t = (int) log10(max/min)+2; break;
			case LOG2: t = (int) log2(max/min)+2; break;
			case LN: t = (int) log(max/min)+2; break;
			case SQRT: t = (int) (pow(max,2)-pow(min,2))+1; break;
			default: break;
			}
			if(t==-1)
				t=autoTicks(min,max);
			if(t==0) t=-1;

			for (int i=0;i <= t;i++) {
				// TODO : use scale (see x)
				int y1 = yi[k]+i*(ymax-ymin)/t;
				int y2 = yi[k]+(i+1)*(ymax-ymin)/t;
				int yia[4] = {ymin+i*(ymax-ymin)/t,y1,y1+(ymin-ymax)/2,y1};

				if (axis[3*k+2].TickPos()!=3) {
					p->setPen(QPen(axis[3*k+2].TickColor(),axis[3*k+2].majorTickWidth()));
					p->drawLine(xi1[k],y1,xi2[k],y1);
				}

				if (axis[3*k+2].MajorGridEnabled()) {
					p->setPen(QPen(axis[3*k+2].majorGridColor(),
						axis[3*k+2].majorGridWidth(),axis[3*k+2].MajorGridType()));
					p->drawLine(xia[k],yia[k],xib[k],y1);
					p->setPen(Qt::SolidLine);
				}
				if (graphlist->Number() > 0) {
					QColor c = axis[3*k+2].TickLabelColor();
					QFont f = axis[3*k+2].TickLabelFont();
					double dz = max-min;
					int atlf = axis[3*k+2].TickLabelFormat();

					double value=0;
					switch(scale) {
						case LINEAR: value = min + (t-i)*dz/t; break;
						case LOG10: value = pow(10,ceil(log10(min)))*pow(10.0,i-1); break;
						case LOG2: value = pow(2,ceil(log2(min)))*pow(2.0,i-1); break;
						case LN: value = pow(M_E,ceil(log(min)))*pow(M_E,i-1); break;
						case SQRT: value = min + (t-i)*dz/t; break;
						default: break;
					}
					// scale and shift value
					value = value*axis[3*k+2].Scaling()+axis[3*k+2].Shift();

					QString label = TicLabel(atlf,axis[3*k+2].TickLabelPrecision(),
							axis[3*k+2].DateTimeFormat(),value);

					// apply prefix & suffix
					label.prepend(axis[3*k+2].TickLabelPrefix());
					label.append(axis[3*k+2].TickLabelSuffix());

					// draw tic label
					QFontMetrics fm(f);
					p->save();
					int tlg = axis[3*k+2].TickLabelPosition();
					if (atlf == AUTO || atlf == NORMAL || atlf == SCIENTIFIC) {
						int xk[4]={(xmax+xmin)/2-fm.width(label)/2-tlg,xmin-fm.width(label)/2-tlg,
							xmax+tlg,xmax+(xmax-xmin)/2+tlg};
						p->translate(xk[k],y1);
						p->rotate(axis[3*k+2].TickLabelRotation());
						p->setPen(c);
						f.setPointSize((int)(f.pointSize()*size.X()));	// resize tic label
						p->setFont(f);

						p->drawText(-fm.width(label)/2,fm.ascent()/2,label);
					}
					else {	// rich text label
						QSimpleRichText *richtext = new QSimpleRichText(label,f);
						int xk[4]={(xmax+xmin)/2-richtext->width()/2-tlg,xmin-richtext->width()-tlg,
							xmax+tlg,xmax+(xmax-xmin)/2+tlg};
						p->translate(xk[k],y1);
						p->rotate(axis[3*k+2].TickLabelRotation());
						QColorGroup cg;
						cg.setColor(QColorGroup::Text,c);
						richtext->draw(p,-richtext->width()/2, -fm.ascent()/2,QRect(),cg);
					}
					p->restore();
				}
				if (axis[3*k+2].MinorTicksEnabled() && i != t ) {
					for (int j=1;j <= axis[3*k+2].MinorTicks()+1;j++) {
						int y = y1+j*(y2-y1)/(axis[3*k+2].MinorTicks()+1);
						int yina[4]={y+(ymax-ymin)/2,y,y,y};
						int yinb[4]={y,y,y+(ymin-ymax)/2,y};
						if(axis[3*k+2].TickPos()!=3) {
							p->setPen(QPen(axis[3*k+2].TickColor(),axis[3*k+2].minorTickWidth()));
							p->drawLine(xkna[k],y,xknb[k],y);
						}
						if (axis[3*k+2].MinorGridEnabled() && j != axis[3*k+2].MinorTicks()+1 ) {
							p->setPen(QPen(axis[3*k+2].minorGridColor(),
								axis[3*k+2].minorGridWidth(),axis[3*k+2].MinorGridType()));
							p->drawLine(xina[k],yina[k],xinb[k],yinb[k]);
							p->setPen(Qt::SolidLine);
						}
					}
				}
			}
		}
	}

	// y,y2,y3,y4
	for (int k=start;k<4;k++) {
		int xi[4]={xmin,xmax,xmax,xmin};
		int yi[4]={ymax,ymax,ymin,ymin};
		if (axis[3*k+1].MajorTicksEnabled() && axis[3*k+1].Enabled()) { 		// y
			double min = actrange[1].rMin(), max = actrange[1].rMax();
			TScale scale = axis[k].Scale();
			int t=-1;
			switch (scale) {
			case LINEAR: t =(int) axis[3*k+1].MajorTicks(); break;
			case LOG10: t = (int) log10(max/min)+2; break;
			case LOG2: t = (int) log2(max/min)+2; break;
			case LN: t = (int) log(max/min)+2; break;
			case SQRT: t = (int) (pow(max,2)-pow(min,2))+1; break;
			default: break;
			}
			if(t==-1)
				t=autoTicks(min,max);
			if (t==0) t=-1;

			for (int i=0;i <= t;i++) {
				// TODO : use scale (see x)
				int y1 = yi[k]-i*(ymax-ymin)/(2*t);
				int y2 = yi[k]-(i+1)*(ymax-ymin)/(2*t);
				int x1 = xi[k]+i*(xmax-xmin)/(2*t);
				int x2 = xi[k]+(i+1)*(xmax-xmin)/(2*t);

				int xi1[4],xi2[4];
				switch(axis[3*k+1].TickPos()) {
				case 0:
					xi1[0]=xi1[1]=xi1[2]=xi1[3]=x1;
					xi2[0]=xi2[3]=x1-10;xi2[1]=xi2[2]=x1+10;
					break;
				case 1:
					xi1[0]=xi1[1]=xi1[2]=xi1[3]=x1;
					xi2[0]=xi2[3]=x1+10; xi2[1]=xi2[2]=x1-10;
					break;
				case 2:
					xi1[0]=xi1[1]=xi1[2]=xi1[3]=x1-10;
					xi2[0]=xi2[1]=xi2[2]=xi2[3]=x1+10;
					break;
				}
				int xia[4]={x1,x1,x1-(xmax-xmin),x1};
				int xib[4]={x1+xmax-xmin,x1,x1,x1};
				int yib[4]={y1,y1-ymax+ymin,y1,y1+ymax-ymin};

				if (axis[3*k+1].TickPos()!=3) {
					p->setPen(QPen(axis[3*k+1].TickColor(),axis[3*k+1].majorTickWidth()));
					p->drawLine(xi1[k],y1,xi2[k],y1);
				}

				if (axis[3*k+1].MajorGridEnabled()) {
					p->setPen(QPen(axis[3*k+1].majorGridColor(),
						axis[3*k+1].majorGridWidth(),axis[3*k+1].MajorGridType()));
					p->drawLine(xia[k],y1,xib[k],yib[k]);
					p->setPen(Qt::SolidLine);
				}
				if (graphlist->Number() > 0) {
					QColor c = axis[3*k+1].TickLabelColor();
					QFont f = axis[3*k+1].TickLabelFont();
					double dy = max-min;
					int yy[4]={ymax,ymax,ymin,ymin};
					// TODO : use scale (see x)
					// TODO : fix rounding bug
					double value = min+2*dy*(yy[k]-y1)/(ymax-ymin);
					int atlf = axis[3*k+1].TickLabelFormat();

					// scale and shift value
					value = value*axis[3*k+1].Scaling()+axis[3*k+1].Shift();

					QString label = TicLabel(atlf,axis[3*k+1].TickLabelPrecision(),
							axis[3*k+1].DateTimeFormat(),value);

					// apply prefix & suffix
					label.prepend(axis[3*k+1].TickLabelPrefix());
					label.append(axis[3*k+1].TickLabelSuffix());

					// draw tic label
					QFontMetrics fm(f);
					p->save();
					int tlg = axis[3*k+1].TickLabelPosition();
					if (atlf == AUTO || atlf == NORMAL || atlf == SCIENTIFIC) {
						int xilb[4]={x1-fm.width(label)/2-tlg,x1+tlg,x1+tlg,x1-fm.width(label)/2-tlg};
						p->translate(xilb[k],y1);
						p->rotate(axis[3*k+1].TickLabelRotation());
						p->setPen(c);
						f.setPointSize((int)(f.pointSize()*size.X()));	// resize tic label
						p->setFont(f);

						p->drawText(-fm.width(label)/2,fm.ascent()/2-1,label);
					}
					else {	// rich text label
						QSimpleRichText *richtext = new QSimpleRichText(label,f);
						int xilb[4]={x1-richtext->width()/2-tlg,x1+tlg,x1+tlg,x1-richtext->width()/2-tlg};
						p->translate(xilb[k],y1);
						p->rotate(axis[3*k+1].TickLabelRotation());
						QColorGroup cg;
						cg.setColor(QColorGroup::Text,c);
						richtext->draw(p,-richtext->width()/2, -fm.ascent()/2,QRect(),cg);
					}
					p->restore();
				}
				if (axis[3*k+1].MinorTicksEnabled() && i != t ) {
					for (int j=1;j <= axis[3*k+1].MinorTicks()+1;j++) {
						int y = y1+j*(y2-y1)/(axis[3*k+1].MinorTicks()+1);
						int x = x1+j*(x2-x1)/(axis[3*k+1].MinorTicks()+1);
						p->setPen(QPen(axis[3*k+1].TickColor(),axis[3*k+1].minorTickWidth()));
						int xina[4]={x,x,x-(xmax-xmin),x};
						int xinb[4]={x+xmax-xmin,x,x,x};
						int yinb[4]={y,y-ymax+ymin,y,y+ymax-ymin};

						int xkna[4],xknb[4];
						switch(axis[3*k+1].TickPos()) {
						case 0:
							xkna[0]=xkna[1]=xkna[2]=xkna[3]=x;
							xknb[0]=xknb[3]=x-5;xknb[1]=xknb[2]=x+5;
							break;
						case 1:
							xkna[0]=xkna[1]=xkna[2]=xkna[3]=x;
							xknb[0]=xknb[3]=x+5;xknb[1]=xknb[2]=x-5;
							break;
						case 2:
							xkna[0]=xkna[1]=xkna[2]=xkna[3]=x-5;
							xknb[0]=xknb[1]=xknb[2]=xknb[3]=x+5;
							break;
						}

						if (axis[3*k+1].TickPos()!=3) {
							p->drawLine(xkna[k],y,xknb[k],y);
						}
						if (axis[3*k+1].MinorGridEnabled() && j != axis[3*k+1].MinorTicks()+1 ) {
							p->setPen(QPen(axis[3*k+1].minorGridColor(),
								axis[3*k+1].minorGridWidth(),axis[3*k+1].MinorGridType()));
 								p->drawLine(xina[k],y,xinb[k],yinb[k]);
							p->setPen(Qt::SolidLine);
						}
					}
				}
			}
		}
	}
}

void Plot3D::saveAxes(QTextStream *t) {
	for (int i = 0; i < 12; i++) {
		saveAxis(t,&axis[i]);
	}
}

void Plot3D::openAxes(QTextStream *t, int version) {
	for(int i = 0;i<12;i++) {
		openAxis(t,version, &axis[i]);
	}
}

void Plot3D::saveXML(QDomDocument doc,QDomElement plottag) {
	QDomElement tag;

	for (int i = 0; i < 12; i++) {
		tag = axis[i].saveXML(doc,i);
   		plottag.appendChild( tag );
	}
}

void Plot3D::openXML(QDomElement e) {
	if(e.tagName() == "Axis")
		axis[e.attribute("id").toInt()].openXML(e.firstChild());
}
