// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptSplitter"

#include <qsplitter.h>

#include "kvi_error.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_splitter.h"

/*
	@class: splitter
	@short:
		A splitter widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $setOrientation(&lt;orientation&gt;)
		Sets the orientation for this splitter widget.<br>
		&lt;orientation&gt; can be "horizontal" or "vertical".<br>

		!fn: $orientation()
		Returns the orientation of this splitter widget.<br>

	@description:
		A splitter widget object class.<br>
		It can manage the layout of one or more widgets, separating them by
		draggable bars.<br>
		It can layout its children widgets horizontally or vertically.<br>
		Basically you create a splitter as a child of any other widget
		and then create children widgets.<br>
		The children will be automatically added to the splitter and it will
		manage their geometries.<br>
	@examples:

	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * SPLITTER class
 */
void KviScriptSplitter::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("orientation", (scriptObjectFunction)&KviScriptSplitter::builtinFunction_ORIENTATION);
	d->addBuiltinFunction("setOrientation", (scriptObjectFunction)&KviScriptSplitter::builtinFunction_SETORIENTATION);
}
KviScriptSplitter::KviScriptSplitter(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptSplitter::~KviScriptSplitter()
{
	// Nothing here
}

bool KviScriptSplitter::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new QSplitter(((KviScriptWidget *) parent())->m_pWidget, name());
			((QSplitter *) m_pWidget)->setOpaqueResize();
		}
	}
	if( !m_pWidget ) return false;

	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	return true;
}

int KviScriptSplitter::builtinFunction_SETORIENTATION(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			if( kvi_strEqualCI(pS->ptr(), "vertical") ) {
				((QSplitter *) m_pWidget)->setOrientation(QSplitter::Vertical);
				return KVI_ERROR_Success;
			} else if( kvi_strEqualCI(pS->ptr(), "horizontal") ) {
				((QSplitter *) m_pWidget)->setOrientation(QSplitter::Horizontal);
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSplitter::builtinFunction_ORIENTATION(QPtrList<KviStr> *, KviStr &buffer)
{
	switch( ((QSplitter *) m_pWidget)->orientation() ) {
		case QSplitter::Vertical:
			buffer.append("vertical");
			break;
		case QSplitter::Horizontal:
			buffer.append("horizontal");
			break;
	}
	return KVI_ERROR_Success;
}

#include "m_kvi_script_splitter.moc"
