// =============================================================================
//
//      --- kvi_process_qt.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviProcess"

#include <errno.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <unistd.h>

#include <qfileinfo.h>
#include <qsocketnotifier.h>

#include "kvi_locale.h"
#include "kvi_malloc.h"
#include "kvi_process_qt.h"
#include "kvi_processcontroller_qt.h"
#include "kvi_string.h"

/**
 *  Simple versions of the KProcess and KProcessController classes
 *  original code by (C) Christian Czezatke
 *  e9025461@student.tuwien.ac.at
 *  Really good work :)
 */

QString KviProcess::m_szShellName("");

// From kvi_processcontroller_qt.cpp:
extern KviProcessController *g_pProcessController;

void kill_process_controller()
{
	// Global handler to remove the g_pProcessController
	if( g_pProcessController ) {
		delete g_pProcessController;
		g_pProcessController = 0;
	}
}

KviProcess::KviProcess()
{
	if( !g_pProcessController ) {
		g_pProcessController = new KviProcessController();
		qAddPostRoutine(kill_process_controller);
	}

	m_iExitStatus = 0;
	m_pid = 0;
	m_bIsRunning = false;
	clearSockVariables();
	m_pStdoutNotifier = 0;
	m_pStderrNotifier = 0;
	g_pProcessController->addProcess(this);
}

KviProcess::~KviProcess()
{
	// First remove all sockets
	killSockets();
	// If is running, and killOnClose is requested, well, KILL it
	if( m_bKillOnClose && m_bIsRunning ) sendSignal(SIGKILL);
	// And tell the process controller to forget about us.
	if( g_pProcessController ) g_pProcessController->removeProcess(this);
}

void KviProcess::clearSockVariables()
{
	m_stdinSock[0]  = -1;
	m_stdinSock[1]  = -1;
	m_stdoutSock[0] = -1;
	m_stdoutSock[1] = -1;
	m_stderrSock[0] = -1;
	m_stderrSock[1] = -1;
}

bool KviProcess::setupSockets()
{
	int error = socketpair(AF_UNIX,SOCK_STREAM, 0, m_stdinSock);
	if( error != 0 ) return false;
	error = socketpair(AF_UNIX,SOCK_STREAM, 0, m_stderrSock);
	if( error != 0 ) {
		close(m_stdinSock[0]);
		close(m_stdinSock[1]);
		clearSockVariables();
		return false;
	}
	error = socketpair(AF_UNIX,SOCK_STREAM, 0, m_stdoutSock);
	if( error != 0 ) {
		close(m_stdinSock[0]);
		close(m_stdinSock[1]);
		close(m_stdoutSock[0]);
		close(m_stdoutSock[1]);
		clearSockVariables();
		return false;
	}
	return true;
}

void KviProcess::killSockets()
{
	if( m_stdinSock[0]  != -1 ) close(m_stdinSock[0]);
	if( m_stdinSock[1]  != -1 ) close(m_stdinSock[1]);
	if( m_stdoutSock[0] != -1 ) close(m_stdoutSock[0]);
	if( m_stdoutSock[1] != -1 ) close(m_stdoutSock[1]);
	if( m_stderrSock[0] != -1 ) close(m_stderrSock[0]);
	if( m_stderrSock[1] != -1 ) close(m_stderrSock[1]);
	clearSockVariables();
	if( m_pStdoutNotifier != 0 ) { delete m_pStdoutNotifier; m_pStdoutNotifier = 0; }
	if( m_pStderrNotifier != 0 ) { delete m_pStderrNotifier; m_pStderrNotifier = 0; }
	m_pStdoutNotifier = 0;
	m_pStderrNotifier = 0;
}

bool KviProcess::run(const char *commandline, bool bCommunicate, bool bKillOnClose, bool bExecInSubshell)
{
	if( m_bIsRunning ) return false;
	if( !commandline || !(*commandline) ) return false;

	m_bKillOnClose = bKillOnClose;
	// Get the argument list
	QPtrList<KviStr> argList;
	argList.setAutoDelete(true);

	KviStr arg;

	if( bExecInSubshell ) {
		argList.append(new KviStr(findShell()));
		argList.append(new KviStr("-c"));
		argList.append(new KviStr(commandline));
	} else {
		while( *commandline ) {
			commandline = kvi_extractToken(arg, commandline);
			if( arg.hasData() ) argList.append(new KviStr(arg));
		}
	}

	if( argList.isEmpty() )return false;
	if( bCommunicate ) {
		if( !setupSockets() ) return false;
	}

	char **pArgs = (char **) kvi_malloc((argList.count() + 1) * sizeof(char*));
	uint cur = 0;
	for( KviStr *pA = argList.first(); pA && (cur < argList.count() ); pA = argList.next() ) {
		pArgs[cur] = (char *) pA->ptr();
		cur++;
	}
	pArgs[cur] = 0; // Null terminator

	m_pid = fork();
	switch( m_pid ) {
		case 0: { // CHILD Process : Success
			if( bCommunicate ) {
				if( !child_setupSockets() ) {
					killSockets();
					debug(_i18n_("Could not setup child communication"));
				}
			} else setpgrp();
			if( execvp(pArgs[0], pArgs) == -1 ) {
				debug(_i18n_("execvp failed for file %s: %s"), pArgs[0], strerror(errno));
				exit(-1);
			}
			return true;
		} break;
		case -1: { // Parent process: failed
			killSockets();
			kvi_free(pArgs);
			return false;
		} break;
		default: { // Parent process: success
			if( bCommunicate ) {
				if( !parent_setupSockets() ) {
					killSockets();
					debug(_i18n_("Could not setup parent communication"));
				}
			}
			m_bIsRunning = true;
			kvi_free(pArgs);
			return true;
		} break;
	}
	// Never Here...
}

bool KviProcess::isActive()
{
	return m_bIsRunning;
}

pid_t KviProcess::pid()
{
	return m_pid;
}

bool KviProcess::parent_setupSockets()
{
	// Remove the child socket side...
	close(m_stdinSock [PROC_SOCKET_CHILD]);
	close(m_stdoutSock[PROC_SOCKET_CHILD]);
	close(m_stderrSock[PROC_SOCKET_CHILD]);
	m_stdinSock [PROC_SOCKET_CHILD] = -1;
	m_stdoutSock[PROC_SOCKET_CHILD] = -1;
	m_stderrSock[PROC_SOCKET_CHILD] = -1;

	if( fcntl(m_stdinSock [PROC_SOCKET_PARENT], F_SETFL, O_NONBLOCK) == -1 ) return false;
	if( fcntl(m_stdoutSock[PROC_SOCKET_PARENT], F_SETFL, O_NONBLOCK) == -1 ) return false;
	if( fcntl(m_stderrSock[PROC_SOCKET_PARENT], F_SETFL, O_NONBLOCK) == -1 ) return false;

	m_pStdoutNotifier = new QSocketNotifier(m_stdoutSock[PROC_SOCKET_PARENT], QSocketNotifier::Read, this);
	QObject::connect(m_pStdoutNotifier, SIGNAL(activated(int)), this, SLOT(receivedStdout(int)));
	m_pStdoutNotifier->setEnabled(true);

	m_pStderrNotifier = new QSocketNotifier(m_stderrSock[PROC_SOCKET_PARENT], QSocketNotifier::Read, this);
	QObject::connect(m_pStderrNotifier, SIGNAL(activated(int)), this, SLOT(receivedStderr(int)));
	m_pStderrNotifier->setEnabled(true);

	return true;
}

bool KviProcess::child_setupSockets()
{
	// Remove the parent socket side...
	close(m_stdinSock [PROC_SOCKET_PARENT]);
	close(m_stdoutSock[PROC_SOCKET_PARENT]);
	close(m_stderrSock[PROC_SOCKET_PARENT]);
	m_stdinSock [PROC_SOCKET_PARENT] = -1;
	m_stdoutSock[PROC_SOCKET_PARENT] = -1;
	m_stderrSock[PROC_SOCKET_PARENT] = -1;

	if( dup2(m_stdinSock [PROC_SOCKET_CHILD],  STDIN_FILENO) == -1 ) return false;
	if( dup2(m_stdoutSock[PROC_SOCKET_CHILD], STDOUT_FILENO) == -1 ) return false;
	if( dup2(m_stderrSock[PROC_SOCKET_CHILD], STDERR_FILENO) == -1 ) return false;

	// Linger a while, to send all the output...
	struct linger l;
	l.l_onoff  = 1;
	l.l_linger = 100; // Linger 1 sec

	if( setsockopt(m_stdoutSock[PROC_SOCKET_CHILD], SOL_SOCKET, SO_LINGER, (char *) &l, sizeof(l)) == -1 ) return false;
	if( setsockopt(m_stderrSock[PROC_SOCKET_CHILD], SOL_SOCKET, SO_LINGER, (char *) &l, sizeof(l)) == -1 ) return false;
	return true;
}

void KviProcess::receivedStdout(int fd)
{
	char buffer[1024];
	int len;
	do {
		len = read(fd, buffer, 1024);
		if( len > 0 )
			emit receivedStdout(this, buffer, len);
	} while( len == 1024 );
}

void KviProcess::receivedStderr(int fd)
{
	char buffer[1024];
	int len;
	do {
		len = read(fd, buffer, 1024);
		if( len > 0 )
			emit receivedStderr(this, buffer, len);
	} while( len == 1024 );
}

bool KviProcess::sendSignal(int sig)
{
	if( m_bIsRunning ) {
		if( kill(m_pid, sig) == -1 ) return false;
	}
	return true;
}

void KviProcess::processHasExited(int status)
{
	// Read the last data in the buffer - sometimes the SIGCHLD arrives before the in_socket notifier fires
	receivedStderr(m_stderrSock[PROC_SOCKET_PARENT]);
	receivedStdout(m_stdoutSock[PROC_SOCKET_PARENT]);
	killSockets();
	m_bIsRunning   = false;
	m_bKillOnClose = false;
	m_iExitStatus  = status;
	emit processExited(this);
}

int KviProcess::exitStatus()
{
	return m_iExitStatus;
}

const QString KviProcess::findShell()
{
	if( !m_szShellName.isEmpty() )
		return m_szShellName;

	QString tmp = getenv("SHELL");
	if( !tmp.isEmpty() ) {
		QFileInfo f(tmp);
		if( f.isExecutable() ) {
			m_szShellName = tmp;
			return m_szShellName;
		}
	}
	m_szShellName = "/bin/sh";
	return m_szShellName;
}

bool KviProcess::writeStdin(const char *buffer, int len)
{
	if( len == -1 )
		len = strlen(buffer);
	return (write(m_stdinSock[PROC_SOCKET_PARENT], buffer, len) == len);
}

#include "m_kvi_process_qt.moc"
