//=============================================================================
//
//   File : kvi_up_cmd.cpp
//   Creation date : Mon Jul 03 2000 12:14:22 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2000-2005 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_debug.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_window.h"
#include "kvi_out.h"
#include "kvi_iconmanager.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_app.h"
#include "kvi_error.h"
#include "kvi_dns.h"
#include "kvi_console.h"
#include "kvi_query.h"
#include "kvi_scriptobject.h"
#include "kvi_fileutils.h"
#include "kvi_ircserverdb.h"
//#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_sharedfiles.h"
#include "kvi_kvs_aliasmanager.h"
#include "kvi_filedialog.h"
//#include "kvi_process.h"
#include "kvi_mediatype.h"
#include "kvi_ircurl.h"
#include "kvi_cmdformatter.h"
#include "kvi_time.h"
#include "kvi_irctoolbar.h"
#include "kvi_ircview.h"
#include "kvi_datacontainer.h"
#include "kvi_channel.h"
#include "kvi_menubar.h"
#include "kvi_ircconnection.h"
#include "kvi_ircconnectionuserinfo.h"
#include "kvi_ircconnectionasyncwhoisdata.h"
#include "kvi_ircconnectionserverinfo.h"
#include "kvi_asynchronousconnectiondata.h"
#include "kvi_kvs_eventmanager.h"
#include "kvi_kvs_eventhandler.h"
#include "kvi_kvs_processmanager.h"
#include "kvi_scriptbutton.h"

#ifdef COMPILE_IPV6_SUPPORT
	#include "kvi_netutils.h" // for kvi_isValidStringIp_V6()
#endif


#include <stdlib.h> // for srand() now

#include <qcursor.h>
#include <qpixmap.h>
#include <qregexp.h>
#include <qhbox.h>
#include <qprocess.h>

#ifdef COMPILE_NEW_KVS
	#include "kvi_kvs_coresimplecommands.h"
	#include "kvi_kvs_corecallbackcommands.h"
	#include "kvi_kvs_object.h"
	#include "kvi_kvs_kernel.h"
	#include "kvi_kvs_object_controller.h"
	#include "kvi_kvs_treenode_command.h"
#endif

// FIXME: #warning "$frameCaption"
// FIXME: #warning "backward goto"
// FIXME: #warning "IDLE (processEvents())"
// FIXME: #warning "LISTTIMER"


// KVS MISSING: SETMENU,DELPOPUPITEM,SOUND,BUTTONCTL


// kvi_app.cpp
extern KviSharedFilesManager   * g_pSharedFilesManager;
//extern KviMediaManager         * g_pMediaManager;
//extern KviTimerManager         * g_pTimerManager;



/*
	@doc: command_rebinding
	@type:
		language
	@keyterms:
		Rebinding commands to another window
	@title:
		Standard rebinding switch
	@short:
		Standard rebinding switch
	@syntax:
		<command> -r=<window_id> <parameters>
	@body:
		The -r switch is standardized along all the commands. It rebinds a command
		to the windows specified by <window_id>. It is useful to launch commands
		in windows that are not the current one. For example, you might want to
		say something in a specific channel while processing an event bound to
		a console, or say something in all the channels bound to the current irc context.
		The examples below will make everything clear.
	@examples:
		[example]
		[comment]# Run a command in the console of the current IRC context[/comment]
		[cmd]echo[/cmd] -r=$console This command is executed in the console ($window.caption)
		[comment]# Say something to all the channels of the current IRC context[/comment]
		[cmd]foreach[/cmd](%w,[fnc]$window.list[/fnc](channel))[cmd]say[/cmd] -r=%w Hi ppl on [fnc]$chan.name[/fnc]
		[/example]
*/

	/*
		@doc: window_naming_conventions
		@type:
			language
		@title:
			Window naming conventions
		@keyterms:
			IRC context,window ID,frame window,connection ID
		@short:
			KVIrc window structure and the window naming conventions
		@body:
			[big]Introduction[/big][br]
			Starting from the release 3.0.0 KVIrc window structure has
			grown in complexity. Older releases allowed one connetion
			per "frame window" and thus had a dedicated command parser
			for each connection. Finding a window in that scenario
			was quite easy: it was enough to designate it by "name"
			(that was exactly the text displayed in the window caption).
			It was sufficient to have an "unique" name for ever window;
			condition that was granted by the underlying IRC protocol
			and by the KVIrc core design.[br]
			In this version, the unique window names are impossible to be granted.[br]
			[big]Scenario[/big][br]
			The command parser is now "global" to the application.
			There can be two or more consoles in each frame and the user
			is able to join the same channel with two different nicknames
			using two separate connections.
			[ul]
			[li]
			Application (Unique command parser)
				[ul]
				[li]
				Frame X
					[ul]
					[li]
					Console M (IRC context)
						[ul]
						[li]Channel windows[/li]
						[li]Query windows[/li]
						[li]Other connection related windows[/li]
						[/ul]
					[/li]
					[li]
					Console N (IRC context)
						[ul]
						[li]Channel windows[/li]
						[li]Query windows[/li]
						[li]Other connection related windows[/li]
						[/ul]
					[/li]
					[li]
					Other windows
					[/li]
					[li]
					...
					[/li]
					[/ul]
				[/li]
				[li]
				Frame Y
					[ul]
					[li]
					Console O (IRC context)
						[ul]
						[li]Channel windows[/li]
						[li]Query windows[/li]
						[li]Other connection related windows[/li]
						[/ul]
					[/li]
					[li]
					Console P (IRC context)
						[ul]
						[li]Channel windows[/li]
						[li]Query windows[/li]
						[li]Other connection related windows[/li]
						[/ul]
					[/li]
					[li]
					Other windows
					[/li]
					[li]
					...
					[/li]
					[/ul]
				[/li]
				[li]
				...
				[/li]
				[/ul]
			[/li]
			[/ul]
			[br]
			A naming convention has becomed necessary to resolve ambiguities.[br]
			[big]Basic assumptions[/big]
			Every KVIrc window has four main properties:[br]
			-[b]an unique numeric identifier[/b][br]
			-[b]the logical name[/b][br]
			-[b]the type identifier[/b][br]
			-[b]the caption text[/b][br]
			The [b]numeric identifier[/b] is unique to the whole application,
			and is the one returned by the [fnc]$window[/fnc] function.[br]
			The identifier is assigned by KVIrc when the window is created
			and is not changed until the window is destroyed.
			This identifier will be referred as [b]window ID[/b].[br]
			The [b]logical name[/b] is a property of some kind of windows.
			It usually corresponds to the first part of the window caption.
			For example, for channel windows it is the channel name, for
			queries it is the list of the targets. For some other windows 
			the logical name corresponds to the caption text. This will be discussed later.[br]
			The [b]type identifier[/b] describes the properties of a certain window.
			For channel windows the type identifier is "channel" , for query windows is "query" ,
			for console windows it is "console", etc..[br]

			[big]Irc contexts[/big][br]
			The KVIrc frame windows are numbered starting from 0 and named
			"frame_<number>". Each frame can contain an unlimited number of consoles.[br]
			Each console is bound to an [b]IRC context[/b]. (The part "is bound to" could
			be substituted by "defines" or "is contained in").[br]
			[i]An [b]IRC context[/b] is a set of resources that can deal with a single
			IRC connection.[/i][br]
			The association between an [b]IRC context[/b]
			and a console is bijective: each [b]IRC context[/b] is associated
			to a single console window.[br]
			An [b]IRC context[/b] can be in connected or not-connected state.
			When in connected state, it contains a set of windows beside the console:
			mainly channels and query windows.
			The channels and query windows can exist ONLY if the associated
			[b]IRC context[/b] exists.[br]
			Channels and queries have unique names inside a connection so
			there is no way to confuse it. (Theoretically there can
			be more than one query window with the same name, but in fact
			all the windows refer to the same target so they are instances
			of the same resource).
			All this creates a sort of namespace: the channels and queries can be identified
			as "bound" to a specific [b]IRC context[/b].[br]
			An [b]IRC context[/b] can "contain" other windows, such as the "sockets"
			window or the "list" window. KVIrc takes care of making them
			unique inside the [b]IRC context[/b] namespace.[br]
			Each [b]IRC context[/b] has its own unique [b]IRC context ID[/b] (see [fnc]$context[/fnc]).[br]
			Since to a single [b]IRC context[/b] may correspond only a single irc connection,
			when in connected state, the [b]IRC context[/b] may be referred also as [b]connection[/b]
			or [b]connection context[/b], and the associated [b]IRC context Id[/b] can be
			referred as [b]connection ID[/b] or [b]connection context ID[/b].[br]
			There are classes of windows that are not bound to any [b]IRC context[/b]:
			this includes user created windows, DCC windows, browsers etc.[br]
			KVIrc will try to keep that windows with unique logical names.[br]
			[big]How to identify a window[/big][br]
			So what we have until now is:[br]
			[ul]
			[li]Each window has its own unique [b]window ID[/b]: we
			will refer windows always using this identifier.[/li]
			[li]Each window has a set of properties including:
			window type, logical name.[/li]
			[li]Subsets of windows are bound to a single [b]IRC context[/b][/li]
			[/ul]
			The simplest (but also the less significant) method of looking for
			a window is to finding it by caption.[br]
			The [fnc]$window[/fnc] function finds the first KVIrc window matching
			the "caption text" and returns its [b]window ID[/b].[br]
			This method will likely fail when there are more windows with the same
			caption text; for this reason several specific functions
			have been added to allow finding the correct window.[br]
			The [fnc]$console[/fnc] finds a console window bound to a specified
			[b]IRC context[/b].[br]
			The [fnc]$channel[/fnc] finds a channel window matching the specified
			name and bound to a specified [b]IRC context[/b].[br]
			The [fnc]$query[/fnc] finds a query window that has a specified target
			and is bound to a specified [b]IRC context[/b].[br]
	*/


	/*
		@doc: connection_dependant_commands
		@type:
			language
		@title:
			Connection dependant commands
		@keyterms:
			IRC context, connection dependant commands
		@body:
			Many KVIrc commands are connection dependant:
			you need an IRC connection to succesfully execute them;
			usually because some data needs to be sent to the server.
			This includes commands like [cmd]whois[/cmd],[cmd]raw[/cmd],[cmd]query[/cmd],
			[cmd]msg[/cmd],[cmd]notice[/cmd],[cmd]op[/cmd],[cmd]ctcp[/cmd]...[br]
			These commands must be executed in a window that is bound to a
			[b]connected [doc:window_naming_conventions]IRC context[/doc][/b].
			You will obviously get an error message if you try to use them in a window
			that has no associated IRC connection.[br]
			For instance: [cmd]whois[/cmd] will work only if you execute it
			in a console , channel or query window.[br]
			If you want to use these commands in a window that is not associated to
			any IRC context you may use the [doc:command_rebinding]standard -r switch[/doc].
			You can use the same switch to execute a command in an [b]IRC context[/b] that is
			not the current one.
	*/


/*
	@doc: aliases
	@type:
		language
	@keyterms:
		aliases
	@title:
		Aliases
	@short:
		Aliases : user definable command sequences
	@body:
		An alias is an user defined command.  It can be used to rename the builtin kvirc commands or functions,
		to automatize complex tasks or as structured programming mean.
		Aliases can be created or destroyed by using the scriptcenter (graphic interface)
		or from the commandline (or script) by using the [cmd]alias[/cmd] command.
		Once created, an alias remains stored permanently in the KVIrc configuration files
		until it is explicitly deleted.
		A couple of examples will make the things clear.
		join is a really commonly used command. It might be a good idea to rename it to
		simply "j" .. just to type it faster.
		Nothing easier in KVirc: just try this commandline:
		[example]
			[cmd]alias[/cmd](j){ [cmd]join[/cmd] $0-; };
		[/example]

		This will create the alias "j". From this moment you can use /j as it was a normal command.
		[example]
		j #kvirc
		[/example]
		You may have notices the strange $0- function in the alias body: it stands for
		"all parameters passed to the alias". This means that when you call
		[example]
			j #kvirc testpassword
		[/example]
		then both the parameters (#kvirc and testpassword) are passed to the join command.
		The $N functions are special functions that return the positional parameters passed
		to the current script context. In an alias the script context is the script body and
		it is the alias caller that generates the parameters.
		$N (where N is a digit) returns the (N-1)-th positional parameter passed by the caller.
		It returns the parameter numbered N-1 and not N since the parameters are indexed starting
		from zero ($0 is the first parameter!).
		$N-M returns the parameters from (N-1)-th to the (M-1)-th (a parameter range) and $N- returns
		all the parameters from (N-1)-th to the last one. In the example above $0- stands for
		all the parameters starting from the first one.
		[/p]
		[p]
		To remove an alias use again the alias command with an empty body:
		[example]
			[cmd]alias[/cmd](j){}
		[/example]
		This will remove the alias "j" defined above.
		[/p]
		[p]
		A common task in channel management is the kick & ban action.
		You first ban an user from the channel and then eventually kick him
		(obviously assuming that he is actually on the channel).
		This involves using two commands: ban and then kick.
		It could be a nice idea to have a single "kb" command to perform this action.
		Well...easy:
		[example]
			[cmd]alias[/cmd](kb){ [cmd]ban[/cmd] $0; [cmd]kick[/cmd] $0-; };
		[/example]
		This adds the "kb" alias: it can be called as a normal command:
		[example]
			kb spammer You're not welcome here!
		[/example]
		This will first execute "ban spammer" and then "kick spammer You're not welcome here".
		Our kb is a really simple example... it doesn't check for the validity of the parameters:
		the server will warn us if the parameters passed to kb were empty.
		[/p]
		[p]
		The alias can be modified at any time by re-using the alias command.
		Let's make our "kb" a bit more intelligent and add a check for the parameters.
		TIP: It is a good idea to write the following examples in a text file and then use /parse <filename> to execute it.
		[example]
			[cmd]alias[/cmd](kb)
			{
				[cmd]if[/cmd]("$0" == "")
				{
					[cmd]echo[/cmd] "Usage: /kb <nickname> <kick reason>"
					[cmd]return[/cmd]
				}
				[cmd]ban[/cmd] $0
				%reason = $1-
				[cmd]if[/cmd]("%reason" == "")%reason = "You're not welcome here!"
				[cmd]kick[/cmd] $0 %reason
			}
		[/example]
		The example above will first check the validity of the <nickname> passed to kb: 
		if no nickname was passed , it will warn the user and stop.
		The next step will be the "ban <nickname>" call. Another enchancement is the "default reason":
		we first assign the remaining parameters ($1- means "from $1 to the end") to a temporary variable,
		if the variable is empty , a default kick reason is assigned.
		Finally the "kick <nickname> <reason>" will be executed.
		Get used to looking at the single command documentation pages, they will give
		you the hints necessary to fully understand the above piece of code.
		[/p]
		[p]
		Aliases can be used as a mean for structured programming.
		In large scripts you will SURELY have "common tasks" to perform (like having specially
		colored output or calculating a value from a set of other values)...
		Aliases are the way of writing the common tasks: they are equivalent to the "procedures"
		or "functions" in many high-level programming languages.
		The alias as a procedure (subroutine or sub-task) has been shown in the "kb" example above:
		it might be commonly called from complexier scripts or other aliases in case that a
		kick & ban action is needed.
		[/p]
		[p]
		The aliases can be used also as functions.
		Assume that you need really often to calculate the sum of three numbers: a function-alias is the way.
		[example]
			[cmd]alias[/cmd](sum3){ [cmd]return[/cmd] $($0 + $1 + $2); };
		[/example]
		This will add the alias "sum3" and make it available both as a command and a function.
		The "return" command sets the return value of a sequence of commands
		(an alias is a sequence of commands...remember ?) and terminates the execution (by returning
		the control to the caller).
		So return $($0 + $1 + $2); will set the return value of the alias to the value
		computed by $($0 + $1 + $2) that actually is the sum of the first three parameters passed.
		You will then use it in the following way:
		[example]
			...
			%myfirstsum = $sum3(%somevalue,%someothervalue,4)
			%anothersum = $sum3(12,%somevalue,%anothervalue)
			...
		[/example]
		Ops.. I've used some variables without actually explaining them... hehe.. please forgive me and read on.
		This example is again really simple , but you might have complexier function-aliases.
		The function-aliases are also normal aliases.... you can use it as a command:
		[example]
			/sum3 1 2 3
		[/example]
		Is a perfectly valid call.... it's just that it will have no visible results
		(just because a command call implies ignoring the return value.
		In fact there is no difference al all between function-aliases and normal-aliases: 
		the caller makes the difference: by calling an alias as a command the return value 
		just disappears in hyperspace, by calling an alias as a function , the return value
		is propagated (and in fact "used").
		(There are some "nice" exceptions to this rule...but you don't need to care about it, for now).
		If return is not called inside an alias body , the return value will be just a null value.
		[/p]
		[p]
		Aliases can accept switches just like any other command. The [fnc]$sw[/fnc] is there
		exactly for that purpose. Check it out.
		[/p]
*/

/*
	@doc: datatypes
	@type:
		language
	@keyterms:
		variable,variables,array,arrays,dictionary,dictionaries,percent sign,global variables,local variables,
		global variable,local variable,variable creation and destruction,variable evaluation,dictionary keys,
		associative arrays,hash,hashes,scalar,scalars,command scope,data types,extended scope,array reference,
		dictionary reference
	@title:
		Data types
	@short:
		KVIrc built-in data types
	@body:
		[big]Variables[/big]
		All the variable names start with a percent '%' sign and follow with
		a name made up of alphanumeric characters and underscores.
		The variables can be either global to the application or local
		to the command scope. The variables that start with an uppercase letter
		are assumed to be global while the other are assumed to be local.
		This behaviour can be modified with the use of the [cmd]global[/cmd]
		and [cmd]local[/cmd] keywords (their usage is actually encouraged as
		experiment: we want to know if this method is better or worse than 
		using the case of the first letter as discriminant).
		Variable names are case insensitive. There is a third kind of variables
		named "extended scope variables": they are explained below.

		[big]Global variables[/big][br]
		A global variable name is formed by a "percent" sign (%),
		followed by an uppercase letter from A to Z, followed
		by a sequence of characters in range ('a' to 'z','0' to '9','.','_').[br]
		"%INDEX","%My_nickname","%Foo","%Bar1" and "%Foo.BAR" are examples of valid
		global variable names.[br]
		The [cmd]global[/cmd] keyword can be used to force a variable
		starting with a lowercase letter to be global while the [cmd]local[/cmd]
		keyword can invert this behaviour for local variables.
		A global variable is [b]global to the entire application[/b], not
		to the current frame or irc context; be sure to remember this.[br]
		You can type[br]
		[example]
			/%Hello = "Hello world!"
		[/example]
		in the commandline of any KVIrc window and then execute[br]
		[example]
			echo %Hello
		[/example]
		in the commandline of any other KVIrc window. You will see "Hello world!"
		printed in the second window view.

		[big]Local variables[/big][br]
		A local variable name is formed by a "percent" sign (%),
		followed by an lowercase letter from a to z, followed
		by a sequence of characters in range ('a' to 'z','0' to '9','.','_').[br]
		"%index","%my_nickname","%foo","%bAR1" and "%foo.BAR" are examples of valid
		local variable names.[br]
		The [cmd]global[/cmd] keyword can be used to force a variable
		starting with a lowercase letter to be global while the [cmd]local[/cmd]
		keyword can invert this behaviour for local variables.
		A local variable exists only in the current command scope.
		The exact command scope definition is rather tricky and depends
		on the internal KVIrc implementation. Just be aware that:[br]
		- An alias body is a command scope.[br]
		- An event body is a command scope.[br]
		- Any sequence of commands executed [b]at once[/b] in a window commandline
		is a command scope.[br]
		You will notice that finding out the current command scope is rather intuitive.[br]
		When you type[br]
		[example]
			%text = "very important text";
		[/example]
		in the commandline of any KVIrc window and then try to execute[br]
		[example]
			echo %text
		[/example]
		you will see nothing printed. The variable %text was local to the "command scope"
		and disappeared immediately after the execution of the assignment.[br]
		But if you execute[br]
		[example]
			%text = "hello"; echo %text wold!
		[/example]
		you will see  "hello world!" printed in the current window.[br]

		[big]Extended scope variables[/big]
		Variables that start with a ':' character are "extended scope" variables.
		"%:index" , "%:Hello" , "%:something.else" are all valid special scope variable names.[br]
		They're actually used in popups and in timers (but later I might find other usages as well :).[br]
		"Extended scope" means that these variables are somewhere in the middle between
		global and local variables. They normally act as local , but in some cases their [b]lifetime[/b] and [b]visibility[/b]
		may be extended.[br]
		For example , in the popups , all the special scope variables
		are visible during all the "lifetime" of a popup (so from the prologue code call to
		the moment when the user selects an item and the corresponding code is executed).[br]
		This allows you to pre-calculate some data or conditions in the popup prologue
		and use this data in the popup item conditions and item handlers.[br]

		[big]Variable creation and destruction[/big]
		Usually you don't need to declare variables (however you CAN do it with the [cmd]global[/cmd]
		and [cmd]local[/cmd] keyword). A variable starts to exist at the time that you assing
		something to it.[br]
		[example]
			%MyVar = some funky text
			%X = 2445833
			%SevenSpaces = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;"
		[/example]
		It terminates its existence at the time that you assingn it an empty string:
		[example]
			%MyVar = ""
			%X = %MyVar
			%SevenSpaces =
		[/example]
		The example is based on global variables, but it is valid for local ones as well.[br]
		A non existing variable is equivalent to an empty one: you will
		never get a warning about non existing variables. This convention
		allows KVIrc to automatically manage the variable creation and destruction.[br]

		[big]Data types[/big]
		KVirc has four basic built-in data types: scalars, arrays of
		scalars, associative arrays scalars (also known as hashes or dictionaries) and objects.[br]
		Scalars are typed internally to integers, strings, real numbers and booleans
		but KVIrc will usually make all the necessary conversions automatically
		so you basically don't need to care about it: you can just think the
		variable as being a [b]variant[/b] (or just a string, if this makes the thing simplier for you).
		Unlike in perl, there is a single namespace for all the datatypes:
		[example]
			%a = "10"
			%a = $array(10,20)
		[/example]
		Both statements refer to [b]the same variable[/b].
		This also means that a variable can change its type "on the fly" from a simple scalar to
		an array of scalars or a hash (or even an object, as you will see later).
		KVIrc is able to implicitly convert scalars to arrays (with a single element) and the
		inverse where needed.
		
		[big]Variable evaluation[/big]
		A variable can appear in every place where a parameter
		is expected: so after the command name, after a switch or inside
		an identifier parameters.
		KVirc will try to extract the longest possible variable name after a literal percent '%'
		sign everywhere in the parameter string. So the command sequence[br]
		[example]
			%number = 1st; echo this is my %number variable test
		[/example]
		will first assign "1st" to the variable "%number" and then execute
		"echo this is my 1st variable test".
		The following example will NOT work as expected.[br]
		[example]
			%number = 1; echo this is my %numberst variable test
		[/example]
		KVirc will assign "1" to %number in this case but the next variable
		name extracted will be "%numberst" that is actually empty; so finally
		"echo this is my variable test" will be executed.
		To avoid this problem you can use the backslash escape character:[br]
		[example]
			%number = 1; echo this is my %number\st variable test
		[/example]

		[big]Arrays[/big]
		Arrays are collections of items indexed by numbers.[br]
		The general syntax for an array is:[br]
		[b]%<name>[<index>][/b][br]
		<name> is the name of the array and follows the rules
		valid for the simple variables: the names starting with an uppercase letter
		designate a global array, the others designate local ones.[br]
		Extended scope arrays can be created as well: normally they act as local,
		but may have extended lifetime in some scopes.[br]
		<index> must be a subscript that evaluates to a positive integer and it selects an item in the array.
		The first index of the array is 0 and the last is equal to size-1.[br]
		You can obtain the size of the array by evaluating [fnc]$length[/fnc](%<name>) or
		by using the subscript %<name>[]#.[br]
		You don't need to declare the size of the array: it is automatically
		handled. When you assign a non-empty string to an item, the array
		is automatically enlarged to contain the index that you are assigning to.
		If the array was not existing before the assignment, it is created.[br]
		If the first assignment index is greater than 0 the items below that index
		will be empty: will just behave as empty/unset variables.[br]
		[example]
			[comment]# Create an array with 21 elements[/comment]
			%Names[20]=Pragma
		[/example]
		To remove an item from the array you assign it an empty string.[br]
		When you remove the highest indexed item the array is automatically shrunk to
		the next highest non-empty item. If there are no other non-empty items the array is destroyed.[br]
		Please note that the memory occupation of the array depends on the <index>.[br]
		By assigning a value to the 10000'th item (index 9999) you allocate 10000 entries! (in fact
		at least ((10000 * 4) + value size) bytes).[br]
		An array is destroyed when it contains no more items or when it goes out of scope (a local array obviously).[br]
		[example]
			[comment]# Creating an array with 800 entries[/comment]
			%Array[799]=test
			echo %Array[]#
			%Array[299]=test
			echo %Array[]#
			%Array[799]=
			echo %Array[]#
			[comment]# Now it contains 300 elements[/comment]
		[/example]
		[br]
		You can reference the whole array by using its name without the square parentheses or
		by using tye subscript %<name>[] which is also an assertion (that the variable %<name> is
		in fact an array). Using %<name>[] will thus throw a warning if %<name> is not an array while
		%<name> will not. You can assign arrays to each other:
		[example]
			%A = $array(10,20,30)
			%B[]=%A[]
			%B=%A
			echo %B[200]
		[/example]
		When you pass an array reference to a command or function, it is evaluated
		as a comma separated list of entries.[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			echo %Array
		[/example]
		By assigning a string to an array you assign it to all the array entries:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			echo %Array
			%Array[]=undefined
			echo %Array
		[/example]
		This is useful when you want to unset an array: just assign an empty string to all its entries:[br]
		[example]
			%Array[200]=Test
			echo %Array[]#
			%Array[]=
			echo %Array[]#
		[/example]
		You can loop through all the array items by using the [cmd]foreach[/cmd] command:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Pluto
			%Array[2]=Paperino
			[cmd]foreach[/cmd](%item,%Array[])[cmd]echo[/cmd] %item
		[/example]
		Obviously also the traditional [cmd]for[/cmd] and [cmd]while[/cmd] indexed-looping methods
		are available:[br]
		[example]
			%Array[0]=Pippo
			%Array[1]=Never show this
			%Array[2]=Pluto
			%Array[5]=Hidden again
			%Array[8]=Paperino
			for(%i=0;%i < %Array[]#;%i+=2)echo Entry %i: \"%Array[%i]\";
		[/example]
		[br]
		[big]Dictionaries[/big]
		Dictionaries are associative arrays of strings. They look close
		to the perl hashes. The general syntax for a dictionary name is:[br]
		[b]%<name>{<key>}[/b][br]
		<name> is the name of the dictionary and follows the same rule
		as for the variables: the names starting with an uppercase letter
		designate a global dictionary, the others designate local ones.[br]
		Again , the dictionaries have its own namespace: you can safely use
		%Names , %Names[] and %Names{} as different entities in your script.[br]
		Extended scope dictionaries can be created as well: normally they act as local,
		but may have extended lifetime in some scopes.[br]
		A dictionary associates a "data string" to each "key string" used.
		The key can be [b]any string[/b] not containing an "unescaped" '}' character
		and is [b]case insensitive[/b]: "key" is equivalent to "KEY" or "KeY".[br]
		[example]
			%Ages{Pragma} = 24
		[/example]
		This assignment associates the key "Pragma" to the number "24" in
		the global dictionary "%Ages". If the array was not existing yet,
		it is created first. If the key "Pragma" was already existing,
		the value associated is replaced with the new value.[br]
		To remove an association you simply assign the empty string to it:[br]
		[example]
			%Ages{pragma} =
		[/example]
		The dictionary is automatically destroyed when there are no more associations
		in it (eg. when you assign the empty string to the last key).[br]
		Dictionaries can be used easily to simulate arrays:[br]
		[example]
			%Links{21} = "http://www.kvirc.net"
		[/example]
		Even multidimensional ones:[br]
		[example]
			%Pixels{324.312} = 1
		[/example]
		Remember that in the example above the key "324.312" is a single string.[br]
		Obviously the key can contain variables and functions just as any other parameter.[br]
		An empty key performs operations on the whole dictionary (just like for the arrays):[br]
		You can assign dictionaries:[br]
		[example]
			%Test2{}=%Test1{}
		[/example]
		Assign a scalar to all the items
		[example]
			%Data{} = "unspecified"
		[/example]
		The example above associates the string "unspecified" to all the existing
		keys in the dictionary %Data. By using this method you can easily destroy
		a dictionary: simply assign the empty string to it.[br]
		[example]
			%Data{} =
		[/example]
		Other [doc]operators[/doc] have similar semantic when working on "empty keys".[br]

		[big]Dictionary evaluation[/big]
		A dictionary can appear in every place where a variable can appear.[br]
		[example]
			[cmd]echo[/cmd] My age is %Ages{[fnc]$mynick[/fnc]}
		[/example]
		If you pass an empty key, the dictionary evaluates to the comma separated
		list of values stored in it. The values have no defined order.
		The list is interpreted as single "string" in contexts where a "string" is required,
		and as a list of "strings" in context where lists of strings are expected. (<-- hehe :)[br]
		The special syntax %<name>{}# returns the number of keys in the
		dictionary. The same value can be obtained by using [fnc]$length[/fnc].
		The special syntax %<name>{}@ returns an array of
		of keys in the dictionary. The same array is returned by [fnc]$keys[/fnc](). The keys have no
		defined order (well, you may be only sure that the order
		of the keys is exactly equal to the values order (%name{})).[br]
		[example]
			%Songs{Jimi Hendrix} = Voodo child
			%Songs{Shawn Lane} = Gray piano's flying
			%Songs{Mina} = Brava
			%Songs{Greg Howe} = "Full Throttle"
			# This is a "single string" evaluation context
			[cmd]echo[/cmd] %Songs{}
			# This is a "list of strings" evaluation context
			[cmd]foreach[/cmd](%var,%Songs)[cmd]echo[/cmd] %var
			[cmd]foreach[/cmd](%var,$keys(%Songs))[cmd]echo [/cmd] %var : %Songs{%var}
		[/example]
		
		[big]Objects[/big]
		Objects are described in [doc:objects]this document[/doc]
*/


// kvi_app.cpp
extern KviIrcServerDataBase * g_pIrcServerDataBase;

#define cmdRegSpec(_keyIdx,_proc1,_proc2) \
	p = new KviCommandParseProc; \
	p->proc = KVI_PTR2MEMBER(KviUserParser::_proc1); \
	m_pCommandExecDict->insert(cmdNamesTable[_keyIdx],p); \
	p = new KviCommandParseProc; \
	p->proc = KVI_PTR2MEMBER(KviUserParser::_proc2); \
	m_pCommandSkipDict->insert(cmdNamesTable[_keyIdx],p)

#define cmdRegNorm(_keyIdx,_proc) cmdRegSpec(_keyIdx,_proc,skipNormalCommand)


// FIXME: #warning "KILL,REHASH,DIE,PING,PONG,RESTART,SUMMON,WALLOPS,USERHOST,ISON"
// FIXME: #warning "An echo that outputs to the lower channel view.."
// FIXME: #warning "PRIVMSG as alias to MSG"

// FIXME: #warning "SWITCH"

static char * cmdNamesTable[]=
{
	""            , ""            , "QUERY"       , "RAW"         , "QUOTE"       , //<--4
	"NICK"        , "PRIVMSG"     , "MSG"         , "NOTICE"      , "JOIN"        , //<--9
	"PART"        , "LEAVE"       , "TOPIC"       , "OP"          , "DEOP"        , //<--14
	"VOICE"       , "DEVOICE"     , "MODE"        , "DELETE"      , "DESTROY"     , //<--19
	"PARSE"       , "CLASS"       , "PRIVATEIMPL" , "SETRETURN"   , "IF"          , //<--24
	"SRAND"       , "SERVER"      , "UNUSED"      , "UNUSED"      , "POPUP"       , //<--29
	"DEFPOPUP"    , "EVENT"       , "EVENTCTL"    , "BAN"         , "UNBAN"       , //<--34
	"ME"          , "TIMER"       , "CTCP"        , "WHOIS"       , "WHOWAS"      , //<--39
	"UNUSED2"     , "AWHOIS"      , "OPTION"      , "ALIAS"       , "HALT"        , //<--44
	"KICK"        , "WHO"         , "OPER"        , "SERVICE"     , "SQUIT"       , //<--49
	"LIST"        , "INVITE"      , "MOTD"        , "LUSERS"      , "VERSION"     , //<--54
	"STATS"       , "LINKS"       , "TIME"        , "TRACE"       , "ADMIN"       , //<--59
	"INFO"        , "SQUERY"      , "AWAY"        , "BACK"        , "RUN"         , //<--64
	"PLAY"        , "QUIT"        , "OPENURL"     , "HELP"        , "KILLTIMER"   , //<--69
	"WHILE"       , "BREAK"       , "RAISE"       , "DELPOPUPITEM", ""      , //<--74
	"SWITCH"      , "BUTTON"      , "UNUSED2"     , "FOREACH"     , "BUTTONCTL"   , //<--79
	"SOUND"       , "PASS"        , "SETMENU"     , "BEEP"        , "RETURN"      , //<--84
	"UNUSED"	  , "FOR"         , "EVAL"        , "SAY"         , "EXEC"        , //<--89
	""            , ""            , "REBIND"      , "CODEPAGE"    , ""      //<--94
};


void KviUserParser::initCommandDict()
{
	// case sensitive (faster) and do NOT copy keys
	m_pCommandExecDict = new QAsciiDict<KviCommandParseProc>(109,true,false);
	m_pCommandExecDict->setAutoDelete(true);
	m_pCommandSkipDict = new QAsciiDict<KviCommandParseProc>(109,true,false);
	m_pCommandSkipDict->setAutoDelete(true);

	KviCommandParseProc * p;


//	cmdRegNorm(80,parseCmd_SOUND);


	// FIXME: #warning "SPLITQUERY,JOINQUERY (?)"
}

// FIXME: #warning "DO{} while()"

// FIXME: #warning "SET and UNSET, that can be REBOUND TO ANOTHER WINDOW"



bool KviUserParser::parseCmd_SOUND(KviCommand *c)
{
	/*
		@doc: sound
		@type:
			command
		@title:
			sound
		@syntax:
			sound [-t=<timeout>] <target> <filename>
		@short:
			Plays a file
		@description:
			Plays a multimedia file by using the media type matching and
			sends a CTCP SOUND to the <target>.[br]
			(target may be a channel name or a nickname).[br]
			The <filename> has the same meaning as in [cmd]play[/cmd].[br]
			In fact , this CTCP should be something as "MULTIMEDIA" or "MM",
			but "SOUND" is used for other clients compatibility.[br]
			KVIrc accepts also "MULTIMEDIA" or "MM" as aliases to this CTCP.[br]
			The CTCP is sent thru a NOTICE and the <filename>
			is added to the public offer list for <timeout> seconds (or a default timeout if the -t switch is not used).[br]
	*/

	ENTER_STACK_FRAME(c,"sound");

	KviStr filename;
	KviStr target;
	if(!parseCmdSingleToken(c,target))return false;
	if(!parseCmdFinalPart(c,filename))return false;

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->connection())return c->notConnectedToServer();

	if(kvi_strEqualCIN(filename.ptr(),"file:",5))filename.cutLeft(5);

	KviStr buffer;
	if(g_pApp->findUserFile(buffer,filename.ptr()))filename = buffer;

	if(!KviFileUtils::fileExists(filename.ptr()))
	{
		c->warning(__tr2qs("Can't find the multimedia file %s"),filename.ptr());
		return c->leaveStackFrame();
	}

	KviStr error;
	if(!g_pApp->playFile(filename.ptr(),error,c->window()))c->warning(error.ptr());

	// Now notify the MUTLIMEDIA file to <target> and add the offer

	if(target.isEmpty())
	{
		c->warning(__tr2qs("Missing target, no action taken"));
		return c->leaveStackFrame();
	}

	bool bTargetIsChan = (target.contains('#') || target.contains('&') || target.contains('!'));

	int iTimeout = (int)KVI_OPTION_UINT(KviOption_uintSoundOfferTimeoutInSecs);
	if(c->hasSwitch('t'))
	{
		KviStr szTimeout;
		if(c->getSwitchValue('t',szTimeout))
		{
			if(szTimeout.isUnsignedNum())iTimeout = szTimeout.toInt();
			else c->warning(__tr2qs("Invalid timeout specified , using default"));
		}
	}

	KviStr absPath = filename;
	if(filename.contains(KVI_PATH_SEPARATOR_CHAR))filename.cutToLast(KVI_PATH_SEPARATOR_CHAR);

	if(filename.hasData())
	{
		KviSharedFile * o = 0;

		if(bTargetIsChan)o = g_pSharedFilesManager->lookupSharedFile(filename.ptr(),0);
		else {
			KviIrcMask u(target.ptr());
			o = g_pSharedFilesManager->lookupSharedFile(filename.ptr(),&u);
		}

		if(!o)
		{
			KviStr szUserMask(KviStr::Format,"%s!*@*",bTargetIsChan ? "*" : target.ptr());

			o = g_pSharedFilesManager->addSharedFile(filename.ptr(),absPath.ptr(),szUserMask.ptr(),iTimeout);
			if(!o)
			{
				c->warning(__tr2qs("Can't add a file offer for file %s (huh ? file not readable ?)"),absPath.ptr());
				return c->leaveStackFrame();
			}

			if(_OUTPUT_VERBOSE)
			{
				c->window()->output(KVI_OUT_SYSTEMMESSAGE,__tr2qs("Added %d secs file offer for file %s (%s) and receiver %s"),
						iTimeout,o->absFilePath().utf8().data(),filename.ptr(),o->userMask().utf8().data());
			}
		}
	}

	c->window()->connection()->sendFmtData("NOTICE %s :%cSOUND %s%c",c->window()->connection()->encodeText(target.ptr()).data(),0x01,
				c->window()->connection()->encodeText(filename.ptr()).data(),0x01);

	if(c->hasSwitch('q'))return c->leaveStackFrame();

	// Lookup the window
	KviWindow * w = 0;
	if(bTargetIsChan)w = c->window()->console()->connection()->findChannel(target.ptr());
	else w = c->window()->console()->connection()->findQuery(target.ptr());	

	if(w)
	{
		w->output(KVI_OUT_MULTIMEDIA,__tr2qs("%s plays '%s'"),
			c->window()->connection()->currentNickName().utf8().data(),
			filename.ptr());
	} else {
		c->window()->output(KVI_OUT_MULTIMEDIA,__tr2qs("%s plays '%s' to %s"),
			c->window()->connection()->currentNickName().utf8().data(),
			filename.ptr(),target.ptr());
	}

	return c->leaveStackFrame();
}


















