//
//   File : kvi_taskbar.cpp (/usr/build/NEW_kvirc/kvirc/kvirc/kvi_taskbar.cpp)
//   Last major modification : Thu Jan 7 1999 03:59:43 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_TASKBAR_CPP_

//#define _KVI_DEBUG_CLASS_NAME_ "KviTaskBar"
//#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_taskbar.h"
#include "kvi_frame.h"
#include "kvi_window.h"
#include "kvi_debug.h"
#include "kvi_defines.h"

#include "kvi_app.h"
#include "kvi_wmdock.h"

#include <qtooltip.h>
#include <qnamespace.h>
#include <qpopupmenu.h>

#include "kvi_xutils.h"
#include "kvi_locale.h"

#include "kvi_style.h"
#include "kvi_mdi.h"

#ifdef COMPILE_USE_AA_FONTS
	extern XftFont        * g_pXftFont;
	extern XftDraw        * g_pXftDraw;
	extern int qt_use_xft (void); // qpainter_x11.cpp
	extern void *qt_ft_font (const QFont *f); // qfont_x11.cpp
	extern XftDraw * qt_lookup_ft_draw (Drawable draw, bool paintEventClipOn, QRegion *crgn);
#endif

/*
	@quickhelp: KviTaskBar
	@widget: Taskbar
		This window lists the currently open windows.<br>
		Each button corresponds to a single MDI (child) window.<br>
		The button is enabled (clickable) when the window is docked , and can be
		pressed to bring it to the top of the other docked windows.<br>
		The button text becomes red when new output is shown in the window and it is not the active one.<br>
		The button text becomes blue when new highlighted output is shown in the window and it is not the active one.<br>
		By right-clicking on the handle you will be able to dock it in
		one of the four sides of the main window.
*/

//####################################################################
//
// KviTaskBarButton
//
//####################################################################

KviTaskBarButton::KviTaskBarButton(QWidget *parent,KviTaskBar *pTaskBar,KviWindow *win_ptr,QPixmap *mem_buf)
:QWidget(parent)
{
	m_pWindow      = win_ptr;
	m_pMemBuffer   = mem_buf;
	m_pTaskBar     = pTaskBar;
	m_bIsOn        = false;
	m_bAttached   = false;
	m_bHighlighted = false;
	m_bAltColor    = false;
	m_iProgress    = 0;

	setBackgroundMode(NoBackground);
	QToolTip::add(this,win_ptr->caption());
	KviStr tmp(KviStr::Format,"taskbar_button_%s",win_ptr->caption());
	setName(tmp.ptr());
	setFocusPolicy(QWidget::NoFocus);
}

KviTaskBarButton::~KviTaskBarButton()
{
}

void KviTaskBarButton::paintEvent(QPaintEvent *e)
{
	if(!isVisible())
	{
#ifdef _KVI_DEBUG_CHECK_RANGE_
		if(e)debug("KviTaskBarButton::paintEvent() while not visible!");
#endif
		return;
	}
	Display *dpy          = x11Display();
	HANDLE hMemBuffer     = m_pMemBuffer->handle();
	GC gc_aux             = XCreateGC(dpy,hMemBuffer,0,0);
	QColorGroup cg        = colorGroup();
	QFontMetrics fm(font());
//	XSetLineAttributes(dpy,gc_aux,1,LineSolid,CapButt,JoinMiter);
#ifdef COMPILE_USE_AA_FONTS
	if(qt_use_xft())
	{
		g_pXftFont = (XftFont *)qt_ft_font(&(font()));
		g_pXftDraw = qt_lookup_ft_draw (hMemBuffer,false,0);
		if(!g_pXftDraw)
		{
			XSetFont(dpy,gc_aux,font().handle());
			g_pXftFont = 0;
		}
	} else {
#endif
		XSetFont(dpy,gc_aux,font().handle());
#ifdef COMPILE_USE_AA_FONTS
		g_pXftFont = 0;
		g_pXftDraw = 0;
	}
#endif
	const QPixmap * pix = m_pTaskBar->backgroundPixmap();
	if(pix){
		if(!pix->isNull()){
			XSetTile(dpy,gc_aux,pix->handle());
			XSetFillStyle(dpy,gc_aux,FillTiled);
			QPoint p = m_pTaskBar->mapFromGlobal(mapToGlobal(QPoint(0,0)));
			
		    XSetTSOrigin(dpy,gc_aux,-p.x(),-p.y());
		} else pix = 0;
	}
	if(!pix){
		XSetFillStyle(dpy,gc_aux,FillSolid);
		XSetForeground(dpy,gc_aux,cg.button().pixel());
		XSetBackground(dpy,gc_aux,cg.button().pixel());
	}
	XFillRectangle(dpy,hMemBuffer,gc_aux,0,0,width(),height());
	XSetFillStyle(dpy,gc_aux,FillSolid);
	/*if(m_bAttached) */
	kvi_xDraw3dRect(dpy,gc_aux,hMemBuffer,cg,0,0,width()-1,height()-1,m_bIsOn);
	int leftPos = (height() - 16)/2;
	if(m_pWindow->myIconPtr()){
		XCopyArea(dpy,m_pWindow->myIconPtr()->handle(),hMemBuffer,gc_aux,0,0,16,16,leftPos,leftPos);
	}
	if(!m_bAttached)kvi_xDraw3dRect(dpy,gc_aux,hMemBuffer,cg,leftPos+16,1,width()-2,height()-2,false);
	leftPos += leftPos+16+KVI_TASKBAR_BORDER;
	int maxWidth = width()-(leftPos + KVI_TASKBAR_BORDER);

	if(m_iProgress > 0){
		//Need to paint the progress bar...
		XSetForeground(dpy,gc_aux,cg.mid().pixel());
		int rectW=maxWidth * m_iProgress / 100;
		XFillRectangle(dpy,hMemBuffer,gc_aux,leftPos-1,KVI_TASKBAR_BORDER,rectW,height()-KVI_TASKBAR_BUTTON_DOUBLE_BORDER);
	}
	if(m_bIsOn){
		m_bHighlighted = false; //Clear highlight if on
		m_bAltColor = false;
	}
	QColor * clr;
	if(m_bHighlighted){
		XSetForeground(dpy,gc_aux,m_bAltColor ? blue.pixel() : red.pixel());
		clr = m_bAltColor ? &((QColor)blue) : &((QColor)red);
	} else {
		XSetForeground(dpy,gc_aux,cg.buttonText().pixel());
		clr = &((QColor)(cg.buttonText()));
	}
	//Calculate the length of the string to paint
	int len=0;
	int strWidth=0;
	if(maxWidth > 10){

		XRectangle r[1];
		r[0].x=leftPos;
		r[0].y=KVI_TASKBAR_BORDER;
		r[0].width=maxWidth;
		r[0].height=height()-(KVI_TASKBAR_BORDER * 2);
		XSetClipRectangles(dpy,gc_aux,0,0,r,1,Unsorted);
		register const char *p=m_pWindow->caption();
		while(*p){
			strWidth+=fm.width(*p);
			if(strWidth < maxWidth){
				p++;
				len++;
			} else {
				if(len > 0){ //Go back one char to allow space for points
					strWidth-=fm.width(*p);
					len--;
					p--;
					strWidth-=fm.width(*p);
				}
				maxWidth = 0; //This means that we must draw the points
				break;
			}
		}
		int yPos = KVI_TASKBAR_BORDER+fm.ascent();

#ifdef COMPILE_USE_AA_FONTS
		if(g_pXftFont){
			XftColor color;
//			QColor * clr = &(m_textColor);
			color.color.red = clr->red() | clr->red() << 8;
			color.color.green = clr->green() | clr->green() << 8;
			color.color.blue = clr->blue() | clr->blue() << 8;
			color.color.alpha = 0xffff;
			color.pixel = clr->pixel();
			if(len)XftDrawString8(g_pXftDraw,&color,g_pXftFont,leftPos,yPos,
				(unsigned char *)m_pWindow->caption(),len);
			if(maxWidth==0){
				leftPos +=strWidth;
				XftDrawString8(g_pXftDraw,&color,g_pXftFont,leftPos,yPos,
					(unsigned char *)"...",3);
			}
		} else {
#endif
			if(len)XDrawString(dpy,hMemBuffer,gc_aux,leftPos,yPos,m_pWindow->caption(),len);
			if(maxWidth==0){
				leftPos +=strWidth;
				XDrawString(dpy,hMemBuffer,gc_aux,leftPos,yPos,"...",3);
			}
#ifdef COMPILE_USE_AA_FONTS
		}
#endif
		XSetClipMask(dpy,gc_aux,None);
	}
	XCopyArea(dpy,hMemBuffer,this->handle(),gc_aux,0,0,width(),height(),0,0);
	XFreeGC(dpy,gc_aux);
}

void KviTaskBarButton::mousePressEvent(QMouseEvent *e)
{
	m_bHighlighted = false;
	m_bAltColor = false;
	if(e->button() & LeftButton)m_pTaskBar->buttonLeftClicked(m_pWindow,this);
	else m_pTaskBar->buttonRightClicked(m_pWindow,this);
}

void KviTaskBarButton::setOn(bool bOn)
{
	if(bOn == m_bIsOn)return;
	m_bIsOn = bOn;
	paintEvent(0);
}

void KviTaskBarButton::setAttachState(bool bAttached)
{
	if(m_bAttached == bAttached)return;
	m_bHighlighted = false;
	m_bAltColor = false;
	m_bAttached = bAttached;
	paintEvent(0);
}

void KviTaskBarButton::highlight(bool bAltColor)
{
	if(!m_bIsOn){
		if((!m_bHighlighted) || (bAltColor && !m_bAltColor)){
			m_bHighlighted = true;
			m_bAltColor = bAltColor;
			paintEvent(0);
		}
	}
	g_pApp->globalHighlight(bAltColor);
}

void KviTaskBarButton::updateButton()
{
	//Update the toolTip
	QToolTip::remove(this);
	QToolTip::add(this,m_pWindow->caption());
	paintEvent(0);
}



//####################################################################
//
// KviTaskBar
//
//####################################################################

KviTaskBar::KviTaskBar(KviFrame *parent,QMainWindow::ToolBarDock dock)
:QToolBar("TaskBar",parent,dock,true,"taskbar_toolbar")
{
	m_pFrm = parent;
	m_pBase = new KviTransparentWidget(this,"taskbar_base");

	recalcButtonHeight();

	m_pMemBuffer = new QPixmap(KVI_TASKBAR_MINIMUM_WIDTH,m_iButtonHeight);

	setStretchableWidget(m_pBase);

	m_pButtonList = new QList<KviTaskBarButton>;
	m_pButtonList->setAutoDelete(true);

	setFontPropagation(QWidget::SameFont);
//	setStretchMode(QToolBar::FullWidth);
	setVerticalStretchable(true);
	setHorizontalStretchable(true);
	setBackgroundMode(QWidget::PaletteBackground);
	connect(this,SIGNAL(orientationChanged(Orientation)),this,SLOT(setupMinimumSizes(Orientation)));
}

KviTaskBar::~KviTaskBar()
{
	delete m_pButtonList;
	delete m_pMemBuffer;
}


void KviTaskBar::buttonLeftClicked(KviWindow *win_ptr,KviTaskBarButton *btn)
{
	m_pFrm->taskbarButtonLeftClicked(win_ptr,btn);
}

void KviTaskBar::buttonRightClicked(KviWindow *win_ptr,KviTaskBarButton *btn)
{
	m_pFrm->taskbarButtonRightClicked(win_ptr,btn);
}

KviTaskBarButton * KviTaskBar::addWinButton(KviWindow *win_ptr)
{
	__range_valid(win_ptr);
	KviTaskBarButton *b=new KviTaskBarButton(m_pBase,this,win_ptr,m_pMemBuffer);
	m_pButtonList->append(b);
	b->show(); //????
	layoutButtons();
	return b;
}

void KviTaskBar::removeWinButton(KviWindow *win_ptr)
{
	__range_valid(win_ptr);
	KviTaskBarButton *b=getButton(win_ptr);
	__range_valid(b);
	if(b){
		m_pButtonList->removeRef(b);
		layoutButtons();
	}
}

void KviTaskBar::windowAttached(KviWindow *win_ptr,bool bAttached)
{
	KviTaskBarButton *b=getButton(win_ptr);
	b->setAttachState(bAttached);
}

KviTaskBarButton * KviTaskBar::getButton(KviWindow *win_ptr)
{
	for(KviTaskBarButton *b=m_pButtonList->first();b;b=m_pButtonList->next()){
		if(b->m_pWindow == win_ptr)return b;
	}
	__debug("WARNING: Button not found!");
	return 0;
}

KviTaskBarButton * KviTaskBar::getNextWindowButton(bool bRight,KviWindow *win_ptr)
{
	if(bRight){
		for(KviTaskBarButton *b=m_pButtonList->first();b;b=m_pButtonList->next()){
			if(b->m_pWindow == win_ptr){
				b = m_pButtonList->next();
				if(!b)b = m_pButtonList->first();
				if(win_ptr != b->m_pWindow)return b;
				else return 0;
			}
		}
	} else {
		for(KviTaskBarButton *b=m_pButtonList->first();b;b=m_pButtonList->next()){
			if(b->m_pWindow == win_ptr){
				b = m_pButtonList->prev();
				if(!b)b = m_pButtonList->last();
				if(win_ptr != b->m_pWindow)return b;
				else return 0;
			}
		}
	}
	return 0;
}

void KviTaskBar::setActiveButton(KviWindow *win_ptr)
{
	for(KviTaskBarButton *b=m_pButtonList->first();b;b=m_pButtonList->next()){
		b->setOn(b->m_pWindow == win_ptr);
	}
}

#define KVI_TASKBAR_BUTTON_MINIMUM_WIDTH 80

void KviTaskBar::layoutButtons()
{
	int btnCount = m_pButtonList->count();
	int curPos = 0;
	if(btnCount == 0)return;
	if(orientation() == Qt::Horizontal){
		// Horizontal layout
		setupMinimumSizes(Qt::Horizontal);

		int btnWidth = (m_pBase->width() / btnCount) - 1;
		if(btnWidth < KVI_TASKBAR_BUTTON_MINIMUM_WIDTH)btnWidth = KVI_TASKBAR_BUTTON_MINIMUM_WIDTH;

		m_pMemBuffer->resize(btnWidth,m_pBase->height());
		int theY = 0;
		for(KviTaskBarButton *b=m_pButtonList->first();b;b=m_pButtonList->next()){
			b->setGeometry(curPos,theY,btnWidth,m_iButtonHeight);
			curPos += btnWidth;
			if((curPos + btnWidth) >= m_pBase->width())
			{
				curPos = 0;
				theY += m_iButtonHeight;
			}
		}
	} else {
		// Vertical layout
		setupMinimumSizes(Qt::Vertical);
		int btnWidth = m_pBase->width();
		m_pMemBuffer->resize(btnWidth,m_iButtonHeight);
		//Vertical layout
		for(KviTaskBarButton *b=m_pButtonList->first();b;b=m_pButtonList->next()){
			b->setGeometry(0,curPos,btnWidth,m_iButtonHeight);
			curPos+=m_iButtonHeight;
		}
	}
}

void KviTaskBar::recalcButtonHeight()
{
	QFontMetrics fm(font());
	m_iButtonHeight = fm.height()+KVI_TASKBAR_BUTTON_DOUBLE_BORDER;
	if(m_iButtonHeight < 18)m_iButtonHeight = 18;
	m_pBase->setMinimumSize(QSize(KVI_TASKBAR_MINIMUM_WIDTH,m_iButtonHeight));
}

void KviTaskBar::fontChange(const QFont &oldFont)
{
	recalcButtonHeight();
}

void KviTaskBar::resizeEvent(QResizeEvent *e)
{
	QToolBar::resizeEvent(e);
	layoutButtons();
}

void KviTaskBar::setupMinimumSizes(Orientation o)
{
	if(o == Qt::Horizontal)
	{
		int btnCount = m_pButtonList->count();
		if(btnCount == 0)btnCount = 1;
		int btnWidth = (m_pBase->width() / btnCount) - 1;
		int nRows = 1;
		if(btnWidth < KVI_TASKBAR_BUTTON_MINIMUM_WIDTH){
			int btnPerRow = m_pBase->width() / KVI_TASKBAR_BUTTON_MINIMUM_WIDTH;
			if(btnPerRow == 0)btnPerRow = 1; //Absurd
			nRows = (m_pButtonList->count() / btnPerRow);
			if(((unsigned int)(btnPerRow * nRows)) < m_pButtonList->count())nRows++;
		}
		m_pBase->setMinimumSize(QSize(KVI_TASKBAR_MINIMUM_WIDTH,m_iButtonHeight * nRows));
	} else {
		int hght = m_iButtonHeight * m_pButtonList->count();
		if(hght > m_pFrm->m_pMdi->height() - 18)hght = m_pFrm->m_pMdi->height();
		m_pBase->setMinimumSize(QSize(KVI_TASKBAR_MINIMUM_WIDTH,hght));
	}
}

QSizePolicy KviTaskBar::sizePolicy()
{
	return QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
}

#include "m_kvi_taskbar.moc"
