//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_scriptwidget.h"

#include "kvi_app.h"

#include "kvi_error.h"
#include "kvi_frame.h"

#include "kvi_locale.h"
#include "kvi_defines.h"
#include "kvi_window.h"

#include "kvi_options.h"

#include <qtimer.h>
#include <qcolor.h>
#include <qvariant.h>
#include <qmetaobject.h>

#include "kvi_malloc.h"

#include <qpushbutton.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qmultilineedit.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qstrlist.h>
#include <qobjectlist.h>
#include <qwidgetlist.h>
#include <qcheckbox.h>
#include <qpixmap.h>
#include <qsplitter.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <qspinbox.h>
#include <qcombobox.h>
#include <qtabwidget.h>

extern QPixmap * g_pixViewOut[KVI_OUT_NUM_IMAGES];

/*
#warning "FONTS AND COLORS, MORE EVENTS, ICONS!"
#warning "OnDblCLick ?" 
#warning "Set tooltip ?"
#warning "Font properties"
*/

/*
	@class: widget
	@short:
		Base class for all widgets
	@inherits:
		<a href="class_object.kvihelp">object</a>
	@functions:

		!fn: $show(&lt;bShowChildren&gt;)
		Shows the widget.<br>
		The argument bShowChildren is optional , if it is present and it is non-zero (true)
		the children of the widget are also shown.<br>

		!fn: $hide()
		Hides the widget

		!fn: $x()
		Returns the x coordinate of the widget in pixels , relative to the parent widget
		or to the desktop of the widget is toplevel.<br>

		!fn: $y()
		Returns the y coordinate of the widget in pixels , relative to the parent widget
		or to the desktop of the widget is toplevel.<br>

		!fn: $width()
		Returns the width of the widget in pixels.<br>

		!fn: $height()
		Returns the height of the widget in pixels.<br>

		!fn: $setMinimumWidth()
		Sets the minimum width of the widget.<br>

		!fn: $setMinimumHeight()
		Sets the minimum height of the widget.<br>

		!fn: $isToplevel()
		returns 1 (true) if this widget is a toplevel one (has no parent widget).<br>
		Please note that this does NOT mean that the object itself has no parent!.<br>

		!fn: $isVisible()
		returns 1 (true) if this widget is visible (mapped) , 0 otherwise.<br>

		!fn: $setGeometry(&lt;x&gt;,&lt;y&gt;,&lt;width&gt;,&lt;height&gt;)
		Sets the geometry of the widget.<br>
		All the coordinates are expressed in screen pixels.<br>
		The &lt;x&gt; and &lt;y&gt; coordinates are relative to the parent widget,
		or to the desktop (if this widget is a toplevel one).<br>
		Returns 0 if one of the parameters passed was invalid (not a number ?),
		1 otherwise.<br>

		!fn: $isEnabled()
		Returns 1 if the widget is enabled , 0 otherwise.<br>

		!fn: $setEnabled(&lt;state&gt;)
		Sets the enable state to &lt;state&gt; that is a boolean value:<br>
		If &lt;state&gt; is 0 the widget is disabled , for any other value the widget
		is enabled.<br>

		!fn: $parentWidget()
		If this widget is a toplevel one , this functon is equivalent to parent()
		(inherited from the object class) , otherwise it returns the STRING "0".<br>

		!fn: $setCaption()
		Sets the caption text for this widget.
		This function is meaningful only for the toplevel widgets.<br>

		!fn: $caption()
		Returns the caption text of this widget.<br>

		!fn: $raise()
		Brings this widget to the top of the stack of widgets of its parent widget.<br>

		!fn: $lower()
		Brings this widget to the bottom of the stack of widget of its parent widget.<br>

		!fn: $setFocus()
		Sets the keyboard focus to this widget.<br>

		!fn: $hasFocus()
		Returns 1 if this widget currently has the keyboard focus.<br>

		!fn: $setBackgroundColor(&lt;rgb&gt;)
		Sets the background color of the widget to the color specified
		by the rgb string.<br>
		The rgb string is a hexadecimal string of 6 characters.<br>
		The first two hex-digits rappresent the red component of the color,
		the third and fourth rappresent the green comonent and the remaining part
		rappresents the blue component.<br>
		So for example: 000000 rappresents black , FFFFFF rappresents white,
		FF0000 rappresents red , FFA000 is a mix of red and green that gives an orange colour.<br>
		Returns 0 if the rgb parameter was invalid , 1 otherwise.<br>

		!fn: $backgroundColor()
		Returns the background color as hexadecimal string as accepted by $setBackgroundColor().<br>

		!fn: $qtProperty(&lt;property_name&gt;)
		This is for really advanced scripting.<br>
		All KVIrc widgets are based on the Qt library ones.<br>
		The Qt library widgets allow to set and read special properties.<br>
		You will have to take a look at the Qt documentation for each widget type
		to see the available property names.<br>
		The supported property types are: Rect , Size , Point , Color , String , CString ,
		Int , UInt , Bool and enumeration types.<br>
		For example , the widget's x coordinate can be retrieved by using the [classfnc:widget]$x[/classfnc]()
		function or by calling $property(x).<br>
		There are many properties that are available ony through the "$property()" call:<br>
		For example , you can find out if the widget accepts drops by calling $property(acceptDrops).<br>
		This function will be mainly useful in the [class]qtwrapper[/class] class.<br>

		!fn: $setQtProperty(&lt;property_name&gt;,&lt;property_value&gt;)
		Sets a qt property for this widget.<br>
		This is for advanced scripting , and can control really many features of the Qt widgets.<br>
		For example , the [class]multilineedit[/class] widgets can be set to
		the "password" echo mode only by using this function call:<br>
		<example>
			%X = [fnc]$new[/fnc]([class]multilineedit[/class],[fnc]$root[/fnc],a_name)
			%X->[classfnc:widget]$show[/classfnc]()
			%X->[classfnc:widget]$setQtProperty[/classfnc](echoMode,Password)
		</example>
		The available properties to be set are listed by [classfnc:widget]$listQtProperties[/classfnc]()
		and must appear in the list as writeable.<br>
		This function will be mainly useful in the [class]qtwrapper[/class] class.<br>

		!fn: $listQtProperties()
		Lists the qt properties supported by this object.<br>
		This function will be mainly useful in the [class]qtwrapper[/class] class.<br>

	@events:
		!ev: OnMouseButtonPress($1 = Button; $2 = x $3 = y)
		Triggered when the user press a mouse button while the pointer is on this widget.<br>
		The first parameter is "left" , "right" or "mid" depenging on the mouse button
		that has been pressed.<br>
		The x and y parameters are the coordinates in pixels of the press-point relative to this widget.<br>
		Please note that if the user drags the mouse out of the widget while the button is
		being pressed , the OnMouseButtonRelease event may also be triggered in a different widget.<br>

		!ev: OnMouseButtonRelease($1 = Button; $2 = x $3 = y)
		Triggered when the user releases a mouse button while the pointer is on this widget.<br>
		The first parameter is "left" , "right" or "mid" depenging on the mouse button
		that has been released.<br>
		The x and y parameters are the coordinates in pixels of the release-point relative to this widget.<br>
		Please note that this event may also NOT follow a OnMouseButtonPress event.<br>

		!ev: OnMouseButtonDblClick($1 = Button; $2 = x $3 = y)
		Triggered when the user "double-clicks" a mouse button while the pointer is on this widget.<br>
		The first parameter is "left" , "right" or "mid" depenging on the mouse button
		that has been double-pressed.<br>
		The x and y parameters are the coordinates in pixels of the click-point relative to this widget.<br>

		!ev: OnClose()
		Triggered when the user closes the window with the close caption button.<br>
		This widget object will be destroyed in a while.<br>
		This event is obviously triggered only for toplevel widgets.<br>

		!ev: OnResize($1 = newWidth,$2 = newHeight)
		Triggered when the widget is being resized either by the user or
		by using $setGeometry.<br>
		Please note that the new widget size is available through the parameters of this
		event , and may not yet correspond to the size returned by $width() and $height().<br>
		It depends on the order in that the events are processed.<br>

		!ev: OnMove($1 = newX,$2 = newY)
		Triggered when the widget is being moved either by the user or
		by a $setGeometry() function call.<br>
		Please nothe the the new widget position is available through the parameters
		of this event , and may not correspond (yet) to the position returned by $x() and $y().<br>

		!ev: OnFocusIn()
		Triggered when the widget gains keyboard focus.<br>

		!ev: OnFocusOut()
		Triggered when the widget loses keyboard focus.<br>

	@description:
		This class is the base for all the widget classes.<br>
		It encapsulates the basic functionality of a "physical" widget.<br>
		You create a widget by using the <a href="s_new.kvihelp">$new()</a> function.<br>
		If the object passed as parent inherits the "widget" class , its "physical"
		widget is taken as parent , otherwise this widget becomes a toplevel one (parentless ,
		or if you want , with the desktop as parent).<br>
		The $isToplevel() member function will tell you wheter the widget
		was created as toplevel or not.<br>
		The $parentWidget() function will return the parent object id , if it inherits the
		widget class , the string "0" otherwise (and so this widget is toplevel).<br>
		A toplevel widget usually has a caption bar provided by the window manager.<br>
		It is possible to set the text of the caption by using the $setCaption() function.<br>
		The position of the widget relative to the parent widget (or to the desktop) can be
		set by using $setGeometry() and retrieved by using $x(),$y(),$width() and $height().<br>
		You can react to mouse clicks by implementing the handlers for OnMouseButtonPress,
		OnMouseButtonRelease and OnMouseButtonDblClick.<br>
		When the widget is resized or moved by the user (or by the $setGeometry() function)
		the events OnMove and OnResize are triggered.<br>
		If the widget is being closed by the user by clicking on the caption "close" button
		(if provided by the window manager) , the OnClose event is triggered and after that
		the object is automatically destroyed.<br>

	@examples:
		First simple example: create a toplevel widget and respond to the button press events.<br>
		<example>
		%tmp = <a href="s_new.kvihelp">$new</a>(widget,<a href="s_root.kvihelp">$root</a>,aName)
		<a href="obj_setevent.kvihelp">obj_setevent</a>(%tmp, OnMouseButtonPress){ <a href="s_this.kvihelp">$this</a>-&gt;$setCaption(Pressed); }
		<a href="obj_setevent.kvihelp">obj_setevent</a>(%tmp, OnMouseButtonRelease){ <a href="s_this.kvihelp">$this</a>-&gt;$setCaption(Released); }
		<a href="obj_setevent.kvihelp">obj_setevent</a>(%tmp, OnClose){ <a href="echo.kvihelp">echo</a> Widget closed!; }
		%tmp-&gt;$show();
		</example>
		Now just click on the widget to see the caption changing.<br>
		When you close the widget by clicking the close button in the caption
		the object will be automatically destroyed.<br>
		An alternative implementation using class definitions:<br>
		<example>
		<a href="class.kvihelp">class</a>(mywidget,widget)
		{
		&nbsp;&nbsp;event(OnMouseButtonPress)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;$setCaption(Pressed)
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnMouseButtonRelease)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;$setCaption(Released)
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnClose)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> Widget destroyed!
		&nbsp;&nbsp;}
		}
		%tmp = <a href="s_new.kvihelp">$new</a>(mywidget,<a href="s_root.kvihelp">$root</a>,aName)
		%tmp-&gt;$show();
		</example>
		If you test this script twice , and you get errors about "Classes already defined"
		try to use the <a href="clearobjects.kvihelp">clearobjects</a> command.<br>
		Another example:<br>
		Create a toplevel widget and a child one with a special background color.<br>
		React to clicks on the child widget , and to the parent resize event to adjust
		the child size.<br>
		<example>
		<a href="class.kvihelp">class</a>(myw1,widget)
		{
		&nbsp;&nbsp;event(OnDestroy)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> Object myw1 destroyed
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnMouseButtonPress)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="if.kvihelp">if</a>("<a href="s_this.kvihelp">$this</a>-&gt;$parentWidget()" != "0")<a href="s_this.kvihelp">$this</a>-&gt;$parent()-&gt;$setCaption(Parent click)
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;$setBackgroundColor(000000)
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnMouseButtonRelease)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="if.kvihelp">if</a>("<a href="s_this.kvihelp">$this</a>-&gt;$parentWidget()" != "0")<a href="s_this.kvihelp">$this</a>-&gt;$parent()-&gt;$setCaption(Parent release)
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;$setBackgroundColor(80,80,80)
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnResize)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> Child resized : <a href="s_this.kvihelp">$this</a>-&gt;$x() , <a href="s_this.kvihelp">$this</a>-&gt;$y()  , $1 , $2
		&nbsp;&nbsp;}
		}

		<a href="class.kvihelp">class</a>(myw2,widget)
		{
		&nbsp;&nbsp;function(constructor)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;%child1 = <a href="s_new.kvihelp">$new</a>(myw1,<a href="s_this.kvihelp">$this</a>,myw)
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;$setBackgroundColor(255,255,255)
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="setreturn.kvihelp">setreturn</a> 1
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnResize)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>-&gt;%child1-&gt;$setGeometry(4,4,<a href="s_calc.kvihelp">$calc</a>($1 - 8),<a href="s_calc.kvihelp">$calc</a>($2 - 8));
		&nbsp;&nbsp;}
		&nbsp;&nbsp;event(OnDestroy)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> Object myw2 destroyed
		&nbsp;&nbsp;}
		}

		%O = <a href="s_new.kvihelp">$new</a>(myw2,<a href="s_root.kvihelp">$root</a>,0)
		%O-&gt;$show()
		</example>
	@seealso:
		<a href="class_object.kvihelp">Object class</a>,
		<a href="class_layout.kvihelp">Layout class</a>,
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/


void KviScriptWidget::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("show",(scriptObjectFunction)&KviScriptWidget::builtinFunction_SHOW);
	d->addBuiltinFunction("hide",(scriptObjectFunction)&KviScriptWidget::builtinFunction_HIDE);
	d->addBuiltinFunction("x",(scriptObjectFunction)&KviScriptWidget::builtinFunction_X);
	d->addBuiltinFunction("y",(scriptObjectFunction)&KviScriptWidget::builtinFunction_Y);
	d->addBuiltinFunction("width",(scriptObjectFunction)&KviScriptWidget::builtinFunction_WIDTH);
	d->addBuiltinFunction("height",(scriptObjectFunction)&KviScriptWidget::builtinFunction_HEIGHT);
	d->addBuiltinFunction("setMinimumWidth",(scriptObjectFunction)&KviScriptWidget::builtinFunction_SETMINIMUMWIDTH);
	d->addBuiltinFunction("setMinimumHeight",(scriptObjectFunction)&KviScriptWidget::builtinFunction_SETMINIMUMHEIGHT);
	d->addBuiltinFunction("isToplevel",(scriptObjectFunction)&KviScriptWidget::builtinFunction_ISTOPLEVEL);
	d->addBuiltinFunction("isVisible",(scriptObjectFunction)&KviScriptWidget::builtinFunction_ISVISIBLE);
	d->addBuiltinFunction("setGeometry",(scriptObjectFunction)&KviScriptWidget::builtinFunction_SETGEOMETRY);
	d->addBuiltinFunction("isEnabled",(scriptObjectFunction)&KviScriptWidget::builtinFunction_ISENABLED);
	d->addBuiltinFunction("setEnabled",(scriptObjectFunction)&KviScriptWidget::builtinFunction_SETENABLED);
	d->addBuiltinFunction("parentWidget",(scriptObjectFunction)&KviScriptWidget::builtinFunction_PARENTWIDGET);
	d->addBuiltinFunction("setCaption",(scriptObjectFunction)&KviScriptWidget::builtinFunction_SETCAPTION);
	d->addBuiltinFunction("caption",(scriptObjectFunction)&KviScriptWidget::builtinFunction_CAPTION);
	d->addBuiltinFunction("raise",(scriptObjectFunction)&KviScriptWidget::builtinFunction_RAISE);
	d->addBuiltinFunction("lower",(scriptObjectFunction)&KviScriptWidget::builtinFunction_LOWER);
	d->addBuiltinFunction("setFocus",(scriptObjectFunction)&KviScriptWidget::builtinFunction_SETFOCUS);
	d->addBuiltinFunction("hasFocus",(scriptObjectFunction)&KviScriptWidget::builtinFunction_HASFOCUS);
	d->addBuiltinFunction("setBackgroundColor",(scriptObjectFunction)&KviScriptWidget::builtinFunction_SETBACKGROUNDCOLOR);
	d->addBuiltinFunction("backgroundColor",(scriptObjectFunction)&KviScriptWidget::builtinFunction_BACKGROUNDCOLOR);
	d->addBuiltinFunction("setQtProperty",(scriptObjectFunction)&KviScriptWidget::builtinFunction_SETQTPROPERTY);
	d->addBuiltinFunction("qtProperty",(scriptObjectFunction)&KviScriptWidget::builtinFunction_QTPROPERTY);
	d->addBuiltinFunction("listQtProperties",(scriptObjectFunction)&KviScriptWidget::builtinFunction_LISTQTPROPERTIES);
}

KviScriptWidget::KviScriptWidget(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptObject(cntrl,p,name,pDef)
{
	m_pWidget = 0;
	m_bAutoDestroyControlledWidget = false;
}

KviScriptWidget::~KviScriptWidget()
{
	if(m_pWidget)
	{
		m_pWidget->removeEventFilter(this);
		disconnect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
		if(m_bAutoDestroyControlledWidget)delete m_pWidget;
		m_pWidget = 0;
	}
}

bool KviScriptWidget::init(QList<KviStr> *)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pWidget = new QWidget(((KviScriptWidget *)parent())->m_pWidget,name());
		}
	}
	if(!m_pWidget)m_pWidget = new QWidget(0,name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	return true;
}

void KviScriptWidget::widgetDestroyed()
{
	m_pWidget = 0;
	dieOutOfThisEventStep();
}

bool KviScriptWidget::eventFilter(QObject *o,QEvent *e)
{
	if(o == m_pWidget)
	{

		KviStr parms;
		KviStr tmp;
		switch(e->type())
		{
			case QEvent::MouseButtonPress:
				if(((QMouseEvent *)e)->button() & LeftButton)tmp = "left";
				else if(((QMouseEvent *)e)->button() & RightButton)tmp = "right";
				else if(((QMouseEvent *)e)->button() & MidButton)tmp = "mid";
				parms.sprintf("%s %d %d",tmp.ptr(),((QMouseEvent *)e)->pos().x(),((QMouseEvent *)e)->pos().x());
				triggerEvent("OnMouseButtonPress",parms);
			break;
			case QEvent::MouseButtonRelease:
				if(((QMouseEvent *)e)->button() & LeftButton)tmp = "left";
				else if(((QMouseEvent *)e)->button() & RightButton)tmp = "right";
				else if(((QMouseEvent *)e)->button() & MidButton)tmp = "mid";
				parms.sprintf("%s %d %d",tmp.ptr(),((QMouseEvent *)e)->pos().x(),((QMouseEvent *)e)->pos().x());
				triggerEvent("OnMouseButtonRelease",parms);
			break;
			case QEvent::MouseButtonDblClick:
				if(((QMouseEvent *)e)->button() & LeftButton)tmp = "left";
				else if(((QMouseEvent *)e)->button() & RightButton)tmp = "right";
				else if(((QMouseEvent *)e)->button() & MidButton)tmp = "mid";
				parms.sprintf("%s %d %d",tmp.ptr(),((QMouseEvent *)e)->pos().x(),((QMouseEvent *)e)->pos().x());
				triggerEvent("OnMouseButtonDblClick",parms);
			break;
			case QEvent::Resize:
				parms.sprintf("%d %d",((QResizeEvent *)e)->size().width(),((QResizeEvent *)e)->size().height());
				triggerEvent("OnResize",parms);
			break;
			case QEvent::Move:
				parms.sprintf("%d %d",((QMoveEvent *)e)->pos().x(),((QMoveEvent *)e)->pos().y());
				triggerEvent("OnMove",parms);
			break;
			case QEvent::Close:
				triggerEvent("OnClose",parms);
				dieOutOfThisEventStep();
			break;
			case QEvent::FocusIn:
				triggerEvent("OnFocusIn",parms);
			break;
			case QEvent::FocusOut:
				triggerEvent("OnFocusOut",parms);
			break;
			default:
			break;
		}
	}
	return KviScriptObject::eventFilter(o,e);
}

/*
#warning "OnGenericEvent"
#warning "Settable properties of the KVIrc windows!"
*/

int KviScriptWidget::builtinFunction_SHOW(QList<KviStr> * params,KviStr &buffer)
{
	bool bShowChildren = false;
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)bShowChildren = pS->toInt();
	}
	if(bShowChildren)recursiveShow();
	else m_pWidget->show();
	
	return KVI_ERROR_Success;
}

void KviScriptWidget::recursiveShow()
{
	m_pWidget->show();
	QList<KviScriptObject> * l = childrenList();
	if(l)
	{
		for(KviScriptObject * o = l->first();o;o = l->next())
		{
			if(o->inherits("KviScriptWidget"))((KviScriptWidget *)o)->recursiveShow();
		}
	}
}

int KviScriptWidget::builtinFunction_LISTQTPROPERTIES(QList<KviStr> * params,KviStr &buffer)
{
	KviWindow * w = controller()->mainFrame()->activeWindow();
	w->output(KVI_OUT_INTERNAL,__tr("%cListing Qt Properties for widget object %s (%s)"),KVI_TEXT_BOLD,getName(),id());

	QMetaObject * o = m_pWidget->metaObject();
	do {
		w->output(KVI_OUT_INTERNAL,__tr("Properties class : %c%s%c"),KVI_TEXT_BOLD,o->className(),KVI_TEXT_BOLD);
		QStrList l = o->propertyNames(false);
		for(char * c = l.first();c;c = l.next())
		{
			const QMetaProperty * p = o->property(c);
			if(p)
			{
				KviStr tmp(KviStr::Format,__tr("Property: %c%s%c, type: %s"),KVI_TEXT_BOLD,p->name(),KVI_TEXT_BOLD,p->type());
				if(p->isEnumType())
				{
					tmp.append(__tr(", enum ("));
					QStrList le = p->enumKeys();
					int i = 0;
					for(char * c2 = le.first();c2;c2 = le.next())
					{
						if(i == 0)
						{
							i++;
						} else tmp.append(", ");
						tmp.append(c2);
					}
					tmp.append(')');
				}
				if(p->isSetType())tmp.append(__tr(", set"));
				if(p->writeable())tmp.append(__tr(", writeable"));
				w->output(KVI_OUT_INTERNAL,tmp.ptr());
			} else debug("Oops...no such property %s",c);
		}
		o = o->superClass();
	} while(o);
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_QTPROPERTY(QList<KviStr> *params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			const QMetaProperty * p = m_pWidget->metaObject()->property(pS->ptr(),true);
			if(!p)return KVI_ERROR_NoSuchQtProperty;
			QVariant v = m_pWidget->property(pS->ptr());
			if(!v.isValid())return KVI_ERROR_NoSuchQtProperty;
			if(p->isEnumType())
			{
				const char * c = p->valueToKey(v.toInt());
				buffer.append(c ? c : "");
			} else {
				switch(v.type())
				{
					case QVariant::Int:
						{
							KviStr tmp(KviStr::Format,"%d",v.toInt());
							buffer.append(tmp.ptr());
						}
						break;
					case QVariant::Point:
						{
							QPoint p = v.toPoint();
							KviStr tmp(KviStr::Format,"%d,%d",p.x(),p.y());
							buffer.append(tmp.ptr());
						}
						break;
					case QVariant::Size:
						{
							QSize s = v.toSize();
							KviStr tmp(KviStr::Format,"%d,%d",s.width(),s.height());
							buffer.append(tmp.ptr());
						}
						break;
					case QVariant::Rect:
						{
							QRect r = v.toRect();
							KviStr tmp(KviStr::Format,"%d,%d,%d,%d",r.x(),r.y(),r.width(),r.height());
							buffer.append(tmp.ptr());
						}
						break;
					case QVariant::Color:
						{
							QColor clr = v.toColor();
							char buf[3];
							buf[0] = clr.red();
							buf[1] = clr.green();
							buf[2] = clr.blue();
							KviStr tmp;
							tmp.bufferToHex(buf,3);
							buffer.append(tmp);
						}
						break;
					case QVariant::UInt:
						{
							KviStr tmp(KviStr::Format,"%u",v.toUInt());
							buffer.append(tmp.ptr());
						}
						break;
					case QVariant::Bool:
						{
							KviStr tmp(KviStr::Format,"%d",v.toInt());
							buffer.append(tmp.ptr());
						}
						break;
					case QVariant::String:
						buffer.append(v.toString());
						break;
					case QVariant::CString:
						{
							const char * c = v.toCString().data();
							if(c)buffer.append(c);
						}
						break;
					default:
						return KVI_ERROR_UnsupportedQtProperty;
						break;
				}
			}
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

/*
#warning "QFont & QPixmap ?"
*/

int KviScriptWidget::builtinFunction_SETQTPROPERTY(QList<KviStr> *params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		KviStr * pV = params->next();
		if(pS)
		{
			const QMetaProperty * p = m_pWidget->metaObject()->property(pS->ptr(),true);
			if(!p)return KVI_ERROR_NoSuchQtProperty;
			KviStr type = p->type();
//			debug("Type : %s",type.ptr());
			if(p->isEnumType())
			{
				if(!pV)return KVI_ERROR_MissingParameter;
				int val = p->keyToValue(pV->ptr());
				QVariant v(val);
				m_pWidget->setProperty(pS->ptr(),v);
			} else if(kvi_strEqualCI("QString",type.ptr()))
			{
				QVariant v(_CHAR_2_QSTRING(pV ? pV->ptr() : ""));
				m_pWidget->setProperty(pS->ptr(),v);
			} else if(kvi_strEqualCI("QCString",type.ptr()))
			{
				QVariant v(QCString(pV ? pV->ptr() : ""));
				m_pWidget->setProperty(pS->ptr(),v);
			} else if(kvi_strEqualCI("int",type.ptr()))
			{
				if(!pV)return KVI_ERROR_MissingParameter;
				QVariant v(pV ? pV->toInt() : 0);
				m_pWidget->setProperty(pS->ptr(),v);
			} else if(kvi_strEqualCI("uint",type.ptr()))
			{
				if(!pV)return KVI_ERROR_MissingParameter;
				QVariant v(pV ? pV->toUInt() : 0);
				m_pWidget->setProperty(pS->ptr(),v);
			} else if(kvi_strEqualCI("bool",type.ptr()))
			{
				if(!pV)return KVI_ERROR_MissingParameter;
				QVariant v(pV ? pV->toInt() : 0,0);
				m_pWidget->setProperty(pS->ptr(),v);
			} else if(kvi_strEqualCI("QPoint",type.ptr()))
			{
				if(!pV)return KVI_ERROR_MissingParameter;
				KviStr * pV2 = params->next();
				if(!pV2)return KVI_ERROR_MissingParameter;
				QVariant v(QPoint(pV->toInt(),pV2->toInt()));
				m_pWidget->setProperty(pS->ptr(),v);
			} else if(kvi_strEqualCI("QSize",type.ptr()))
			{
				if(!pV)return KVI_ERROR_MissingParameter;
				KviStr * pV2 = params->next();
				if(!pV2)return KVI_ERROR_MissingParameter;
				QVariant v(QSize(pV->toInt(),pV2->toInt()));
				m_pWidget->setProperty(pS->ptr(),v);
			} else if(kvi_strEqualCI("QRect",type.ptr()))
			{
				if(!pV)return KVI_ERROR_MissingParameter;
				KviStr * pV2 = params->next();
				if(!pV2)return KVI_ERROR_MissingParameter;
				KviStr * pV3 = params->next();
				if(!pV3)return KVI_ERROR_MissingParameter;
				KviStr * pV4 = params->next();
				if(!pV4)return KVI_ERROR_MissingParameter;
				QVariant v(QRect(pV->toInt(),pV2->toInt(),pV3->toInt(),pV4->toInt()));
				m_pWidget->setProperty(pS->ptr(),v);
			} else if(kvi_strEqualCI("QColor",type.ptr()))
			{
				if(!pV)return KVI_ERROR_MissingParameter;
				char * buf = 0;
				int len = pS->hexToBuffer(&buf,false);
				if(len == 3)
				{
					QVariant v(QColor(((unsigned char)buf[0]),((unsigned char)buf[1]),((unsigned char)buf[2])));
					m_pWidget->setProperty(pS->ptr(),v);
					kvi_free(buf);
				} else return KVI_ERROR_InvalidParameter;
				if(len > 0)kvi_free(buf);
			} else return KVI_ERROR_UnsupportedQtProperty;

			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}


int KviScriptWidget::builtinFunction_HIDE(QList<KviStr> *,KviStr &buffer)
{
	m_pWidget->hide();
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_X(QList<KviStr> *,KviStr &buffer)
{
	KviStr tmp(KviStr::Format,"%d",m_pWidget->x());
	buffer.append(tmp);
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_Y(QList<KviStr> *,KviStr &buffer)
{
	KviStr tmp(KviStr::Format,"%d",m_pWidget->y());
	buffer.append(tmp);
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_WIDTH(QList<KviStr> *,KviStr &buffer)
{
	KviStr tmp(KviStr::Format,"%d",m_pWidget->width());
	buffer.append(tmp);
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_HEIGHT(QList<KviStr> *,KviStr &buffer)
{
	KviStr tmp(KviStr::Format,"%d",m_pWidget->height());
	buffer.append(tmp);
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_ISTOPLEVEL(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(m_pWidget->isTopLevel() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_ISVISIBLE(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(m_pWidget->isVisible() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_ISENABLED(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(m_pWidget->isEnabled() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETGEOMETRY(QList<KviStr> *params,KviStr &buffer)
{
//#warning "LOOK HERE ! :)"
// There obviously is a bug in passing parameters in as QList...
// If I can't find it, I'm sure Pragma will :)

	int x = m_pWidget->x();
	int y = m_pWidget->y();
	int w = m_pWidget->width();
	//int h = 10;
	int h = m_pWidget->height();
	if(params){
		KviStr * pS = params->first();
		if(pS)
		{
			x = pS->toInt();
			pS = params->next();
			if(pS)
			{
				y = pS->toInt();
				pS = params->next();
				if(pS)
				{
					w = pS->toInt();
					pS = params->next();
					if(pS)h = pS->toInt();
					
				}
			}
		}
	
// After doing %w->$setGeometr(100, 100, 40) debug() shows:
// widget::setGeometry(100, 100, 40, 0)
// There is no checking for number of parameters !
//		debug( "widget::setGeometry(%i, %i, %i, %i)", x, y, w, h);
		m_pWidget->setGeometry(x,y,w,h);
		buffer.append('1');
	} else buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETENABLED(QList<KviStr> *params,KviStr &buffer)
{
	bool bEn = true;
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			int tmp = pS->toInt();
			if(tmp == 0)bEn = false;
		}	
		m_pWidget->setEnabled(bEn);
	}
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETMINIMUMHEIGHT(QList<KviStr> *params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			int tmp = pS->toInt();
			m_pWidget->setMinimumHeight(tmp);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptWidget::builtinFunction_SETMINIMUMWIDTH(QList<KviStr> *params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			int tmp = pS->toInt();
			m_pWidget->setMinimumWidth(tmp);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}
int KviScriptWidget::builtinFunction_PARENTWIDGET(QList<KviStr> *,KviStr &buffer)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			buffer.append(((KviScriptObject *)parent())->id());
			return KVI_ERROR_Success;
		}
	}
	buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETCAPTION(QList<KviStr> *params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)m_pWidget->setCaption(pS->ptr());
	}
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_CAPTION(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(m_pWidget->caption());
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_RAISE(QList<KviStr> *,KviStr &buffer)
{
	m_pWidget->raise();
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_LOWER(QList<KviStr> *,KviStr &buffer)
{
	m_pWidget->lower();
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETFOCUS(QList<KviStr> *,KviStr &buffer)
{
	m_pWidget->setFocus();
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_HASFOCUS(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(m_pWidget->hasFocus() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_SETBACKGROUNDCOLOR(QList<KviStr> *params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			char * buf = 0;
			int len = pS->hexToBuffer(&buf,false);
			if(len == 3)
			{
				m_pWidget->setBackgroundColor(QColor(((unsigned char)buf[0]),((unsigned char)buf[1]),((unsigned char)buf[2])));
				buffer.append('1');
				kvi_free(buf);
				return KVI_ERROR_Success;
			}
			if(len > 0)kvi_free(buf);
		}
	}
	buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptWidget::builtinFunction_BACKGROUNDCOLOR(QList<KviStr> *params,KviStr &buffer)
{
	QColor clr = m_pWidget->backgroundColor();
	char buf[3];
	buf[0] = clr.red();
	buf[1] = clr.green();
	buf[2] = clr.blue();
	KviStr tmp;
	tmp.bufferToHex(buf,3);
	buffer.append(tmp);
	return KVI_ERROR_Success;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
///////  BUTTON CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////


/*
	@class: button
	@short:
		A push button widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $text()
		Returns the text displayed on the button.<br>

		!fn: $setText(&lt;text&gt;)
		Sets the text to be displayed on the button.<br>

	@events:
		!ev: OnClick()
		Triggered when the user clicks the button.<br>
		<b>This signal has a default event handler that emits the [classsignal:button]clicked[/classsignal] signal</b>.
		If you override the event handler , and still want the signal to be emitted, you have to emit
		it by yourself!.<br>
		The default event handler is the following one:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:button]clicked[/classsignal])
		</example>

	@signals:
		!sg: clicked()
		Emitted when the button has been clicked by the user.<br>
		<b>This signal is emitted by the default [classevent:button]OnClick[/classevent] event handler!<br>
		If you define a new event handler , and still want the signal to be emitted,
		you have to emit it by yourself!

	@description:
		This is a simple GUI button class.<br>
		It shows a simple text label and reacts to the user clicks.<br>
		The constructor for this widget accepts an optional additional parameter:
		the text to be displayed.<br>
		When the user clicks the button , the [classevent:button]OnClick[/classevent]
		event is triggered. The [classevent:button]OnClick[/classevent] event has a default
		event handler that emits the [classsignal:button]clicked[/classsignal] signal.<br>
		If you define a new event handler , and still want the signal to be emitted,
		you have to emit it by yourself!		

	@examples:
		Simple usage:<br><br>
		Construct a simple (toplevel) button displaying the text "Hello".<br>
		<example>
			%btn = [fnc]$new[/fnc]([class]button[/class],[fnc]$root[/fnc],the_button,Hello)
			%btn->[classfnc:widget]$show[/classfnc]()
		</example>
		React to its click by setting a handler to the OnClick event.<br>
		<example>
			%btn = [fnc]$new[/fnc]([class]button[/class],[fnc]$root[/fnc],the_button,Hello)
			[cmd]obj_setevent[/cmd](%btn,[classevent:button]OnClick[/classevent])
			{
			&nbsp;	[cmd]echo[/cmd] Button clicked!
			}
			%btn->[classfnc:widget]$show[/classfnc]()
		</example>
		Please note that in the example above the signal clicked is no longer emitted!.<br>
		In a simple usage case it is not useful.<br>
		If you still want the signal to be emitted when the user clicks the button:<br>
		<example>
			%btn = [fnc]$new[/fnc]([class]button[/class],[fnc]$root[/fnc],the_button,Hello)
			[cmd]obj_setevent[/cmd](%btn,[classevent:button]OnClick[/classevent])
			{
			&nbsp;	[cmd]echo[/cmd] Button clicked!
			&nbsp;  [fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:button]clicked[/classsignal])
			}
			%btn->[classfnc:widget]$show[/classfnc]()
		</example>

		Complex usage:<br><br>

		TODO : put a dialog example here.

	@seealso:
		[class]object[/class] class</a>,
		[class]widget[/class] class</a>,
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/

/*
#warning "TODO : put a dialog example in the button class doc"
*/

void KviScriptButton::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",(scriptObjectFunction)&KviScriptButton::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",(scriptObjectFunction)&KviScriptButton::builtinFunction_TEXT);
	KviScriptEventStruct * s = new KviScriptEventStruct;
	s->szName = "OnClick";
	s->szBuffer = "$this->$emit(clicked)";
	d->addDefaultEvent(s);
}

KviScriptButton::KviScriptButton(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptWidget(cntrl,p,name,pDef)
{
}

KviScriptButton::~KviScriptButton()
{
}

bool KviScriptButton::init(QList<KviStr> * params)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pWidget = new QPushButton(((KviScriptWidget *)parent())->m_pWidget,name());
		}
	}
	if(!m_pWidget)m_pWidget = new QPushButton(0,name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(clicked()),this,SLOT(buttonClicked()));
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)((QPushButton *)m_pWidget)->setText(pS->ptr());
	}
	return true;
}

int KviScriptButton::builtinFunction_TEXT(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(((QPushButton *)m_pWidget)->text());
	return KVI_ERROR_Success;
}

int KviScriptButton::builtinFunction_SETTEXT(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			((QPushButton *)m_pWidget)->setText(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	((QPushButton *)m_pWidget)->setText("");
	return KVI_ERROR_Success;
}

void KviScriptButton::buttonClicked()
{
	// trigger clicked()
	KviStr params;
	triggerEvent("OnClick",params);
}

/////////////////////////////////////////////////////////////////////////////////////////////////
///////  BUTTON CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////


/*
	@class: toolbutton
	@short:
		A tool button widget (mainly for toolbars)
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]

	@functions:
		!fn: $setText(&lt;text&gt;)
		Sets the text to be used as the "text label" and as the tooltip for the button.<br>

		!fn: $text()
		Returns the text label of the button.<br>

		!fn: $setImage(&lt;image_path&gt;)
		Sets the image that should be displayed on the button.<br>
		The &lt;image_path&gt; can be absolute or a simple file name
		(in this case KVIrc will look for the file in a set of predefined
		directories including the local KVIrc pics directory , the global one , the
		user home directory)
		You should provide a 32x32 image : KVIrc will automatically
		resize it when oter formats will be requested.<br>
		Returns 1 if the image has been found and set sucesfully , 0 otherwise.<br>

	@events:
		!ev: OnClick()
		Triggered when the user clicks the button.<br>
		<b>This signal has a default event handler that emits the [classsignal:toolbutton]clicked[/classsignal] signal</b>.
		If you override the event handler , and still want the signal to be emitted, you have to emit
		it by yourself!.<br>
		The default event handler is the following one:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:toolbutton]clicked[/classsignal])
		</example>

	@signals:
		!sg: clicked()
		Emitted when the button has been clicked by the user.<br>
		<b>This signal is emitted by the default [classevent:toolbutton]OnClick[/classevent] event handler!<br>
		If you define a new event handler , and still want the signal to be emitted,
		you have to emit it by yourself!

	@description:
		This is a tool button class, to be used mainly with [class]toolbar[/class] objects as parent.<br>
		It usually shows an image that can be 32x32 or 16x16 pixels.<br>
		You should always provide a 32x32 image : KVIrc will resize it when needed.<br>
		The button can optionally display a text label.<br>
		The text label will be shown if KVIrc has this option enabled.<br>
		You can force the text label to be shown by setting the usesTextLabel qt property to
		true (see [classfnc:widget]$setQtProperty[/classfnc]()).<br>

		The constructor for this object accepts two additional (optional) parameters:<br>
		The path to the image to be displayed and the text label to be set.<br>

		<example>
		%btn = [fnc]$new[/fnc]([class]toolbutton[/class],[fnc]$root[/fnc],a_name,/usr/share/icons/quit.png,Quit the session)
		</example>

		When the user clicks the button , the [classevent:toolbutton]OnClick[/classevent]
		event is triggered. The [classevent:toolbutton]OnClick[/classevent] event has a default
		event handler that emits the [classsignal:toolbutton]clicked[/classsignal] signal.<br>
		If you define a new event handler , and still want the signal to be emitted,
		you have to emit it by yourself!		

	@examples:
		A colorful (but useless) example.<br>
		A toolbar subclass with some buttons using the KVIrc file icons.<br>
		It will be docked initially on the top of the main KVirc window
		and will NOT be dockable to the right edge.<br>

		<example>
		&nbsp;[cmd]class[/cmd] -o (mytoolbar,[class]toolbar[/class])
		&nbsp;{
		&nbsp;	[cmd:class]function[/cmd](constructor)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->$newButton(html,kvi_icon_html.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(h,kvi_icon_h.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(c,kvi_icon_c.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(exe,kvi_icon_exe.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(image,kvi_icon_image.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(lib,kvi_icon_lib.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(rpm,kvi_icon_rpm.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(shell,kvi_icon_shell.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(sound,kvi_icon_sound.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(text,kvi_icon_text.png)
		&nbsp;		[fnc]$this[/fnc]->$newButton(zip,kvi_icon_zip.png)
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](newButton)
		&nbsp;	{
		&nbsp;		%btn = [fnc]$new[/fnc]([class]toolbutton[/class],[fnc]$this[/fnc],button_$1,$2,$1)
		&nbsp;		[cmd]connect[/cmd] %btn [classsignal:toolbutton]clicked[/classsignal] [fnc]$this[/fnc] buttonSlot
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](buttonSlot)
		&nbsp;	{
		&nbsp;		[cmd]echo[/cmd] "Clicked button \"[fnc]$signalSender[/fnc]->[classfnc:toolbutton]$text[/classfnc]()\""
		&nbsp;	}
		&nbsp;}
		&nbsp;
		&nbsp;%bar = [fnc]$new[/fnc](mytoolbar,[fnc]$root[/fnc],TestToolbar)
		&nbsp;
		&nbsp;%bar->[classfnc:toolbar]$setDockEnabled[/classfnc](right,0)
		&nbsp;%bar->[classfnc:toolbar]$dock[/classfnc](top)
		&nbsp;%bar->[classfnc:widget]$show[/classfnc]()
		</example>
	@seealso:
		[class]object[/class] class</a>,<br>
		[class]widget[/class] class</a>,<br>
		[class]toolbar[/class] class</a>,<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/

void KviScriptToolButton::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",(scriptObjectFunction)&KviScriptToolButton::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",(scriptObjectFunction)&KviScriptToolButton::builtinFunction_TEXT);
	d->addBuiltinFunction("setImage",(scriptObjectFunction)&KviScriptToolButton::builtinFunction_SETIMAGE);


	KviScriptEventStruct * s = new KviScriptEventStruct;
	s->szName = "OnClick";
	s->szBuffer = "$this->$emit(clicked)";
	d->addDefaultEvent(s);
}

KviScriptToolButton::KviScriptToolButton(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptWidget(cntrl,p,name,pDef)
{
}

KviScriptToolButton::~KviScriptToolButton()
{
}

bool KviScriptToolButton::init(QList<KviStr> * params)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pWidget = new QToolButton(((KviScriptWidget *)parent())->m_pWidget,name());
		}
	}
	if(!m_pWidget)m_pWidget = new QToolButton(0,name());

	((QToolButton *)m_pWidget)->setUsesBigPixmap(g_pOptions->m_bUseBigToolbarPixmaps);
	((QToolButton *)m_pWidget)->setUsesTextLabel(g_pOptions->m_bUseTextToolbarComment);

	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(clicked()),this,SLOT(buttonClicked()));
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	if(params)
	{
		KviStr * pS1 = params->first();
		KviStr * pS2 = params->next();
		if(pS1)
		{
			KviStr path;
			if(g_pApp->findImage(path,pS1->ptr()))((QToolButton *)m_pWidget)->setPixmap(QPixmap(path.ptr()));
			if(pS2)
			{
				((QToolButton *)m_pWidget)->setTextLabel(pS2->ptr(),true);
			}
		}
	}
	return true;
}

int KviScriptToolButton::builtinFunction_TEXT(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(((QToolButton *)m_pWidget)->textLabel());
	return KVI_ERROR_Success;
}

int KviScriptToolButton::builtinFunction_SETTEXT(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			((QToolButton *)m_pWidget)->setTextLabel(pS->ptr(),true);
			return KVI_ERROR_Success;
		}
	}
	((QToolButton *)m_pWidget)->setTextLabel("",true);
	return KVI_ERROR_Success;
}

int KviScriptToolButton::builtinFunction_SETIMAGE(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			KviStr path;
			if(g_pApp->findImage(path,pS->ptr()))
			{
				((QToolButton *)m_pWidget)->setPixmap(QPixmap(path.ptr()));
				buffer.append('1');
			} else {
				buffer.append('0');
			}
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}


void KviScriptToolButton::buttonClicked()
{
	// trigger clicked()
	KviStr params;
	triggerEvent("OnClick",params);
}


/////////////////////////////////////////////////////////////////////////////////////////////////
///////  LABEL CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////
/*
#warning "Alignment in the label ?"
#warning "An example with tags inside ?"
*/

/*
	@class: label
	@short:
		A simple label widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:

		!fn: $text()
		Returns the text displayed on the label.<br>

		!fn: $setText(&lt;text&gt;)
		Sets the text to be displayed on the label.<br>

		!fn: $setFrameShape(&lt;style&gt;)
		none , box , panel , winpanel , hline , vline

		!fn: $setFrameShadow(&lt;type&gt;)
		plain , sunken , raised

		!fn: $setImage(\[image path\])
		Sets the picture to be displayed on the label.<br>
		The \[image path\] must be an absolute path to the image file.<br>
		If \[image path\] is an empty string , the current picture is unset.<br>
		

	@description:
		This class implements a simple label widget.<br>
		The label can display static text that can be set and retrieved by using
		the [classfnc:label]$setText[/classfnc]() and the [classfnc:label]$text[/classfnc]() function.<br>
		The label has different kind of frames available:<br>
		The frame appearance can be controlled by using the [classfnc:label]$setFrameShape[/classfnc]()
		and [classfnc:label]$setFrameShadow[/classfnc]() functions.<br>
		The constructor for this widget accepts an optional additional parameter:
		the text to be displayed.<br>
		The label text can handle a limited set of html tags to change the text
		appearance.<br>
		The supported tags include : &lt;font&gt; (face and color attributes),<br>
		&lt;h1&gt; (and the other headings), &lt;center&gt; , &lt;br&gt; , &lt;hr&gt; ,
		&lt;li&gt; , &lt;pre&gt; , &lt;tt&gt; , &lt;table&gt; ...<br>
		More tags may be supported , depending on the underlying Qt library.<br>
	@examples:
		Ok...a complex example this time.<br>
		A simple minesweeper game done with labels.<br>
		You can improve this game in many ways...have fun :)<br>
		<example>
		&nbsp;[cmd]if[/cmd]([fnc]$classDefined[/fnc](minebutton))[cmd]clearobjects[/cmd] minebutton
		&nbsp;[cmd]if[/cmd]([fnc]$classDefined[/fnc](minesweeper))[cmd]clearobjects[/cmd] minesweeper
		&nbsp;[cmd]class[/cmd](minebutton,[class]label[/class])
		&nbsp;{
		&nbsp;	[cmd:class]function[/cmd](constructor)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->[classfnc:label]$setFrameShape[/classfnc](winpanel)
		&nbsp;		[fnc]$this[/fnc]->[classfnc:label]$setFrameShadow[/classfnc](raised)
		&nbsp;		[fnc]$this[/fnc]->[classfnc:widget]$setMinimumWidth[/classfnc](20)
		&nbsp;		[fnc]$this[/fnc]->[classfnc:widget]$setMinimumHeight[/classfnc](20)
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](press)
		&nbsp;	{
		&nbsp;		[cmd]if[/cmd](("[fnc]$this[/fnc]->%pressed" == "") <a href="s_calc.kvihelp">&&</a> [fnc]$this[/fnc]->[classfnc:widget]$isEnabled[/classfnc]())
		&nbsp;		{
		&nbsp;			[fnc]$this[/fnc]->[classfnc:label]$setFrameShape[/classfnc](panel)
		&nbsp;			[fnc]$this[/fnc]->[classfnc:label]$setFrameShadow[/classfnc](sunken)
		&nbsp;			[fnc]$this[/fnc]->%pressed = 1
		&nbsp;			[cmd]if[/cmd]([fnc]$this[/fnc]->%nbombs == -1){
		&nbsp;				[fnc]$this[/fnc]->[classfnc:label]$setText[/classfnc]("@")
		&nbsp;				[fnc]$this[/fnc]->[classfnc:object]$parent[/classfnc]()->$bombPressed()
		&nbsp;			} [cmd:if]else[/cmd] {
		&nbsp;				[cmd]if[/cmd]([fnc]$this[/fnc]->%nbombs &gt; 0)[fnc]$this[/fnc]->[classfnc:label]$setText[/classfnc]([fnc]$this[/fnc]->%nbombs)
		&nbsp;				[cmd:if]else[/cmd] [fnc]$this[/fnc]->[classfnc:object]$parent[/classfnc]()->$emptyButtonPressed([fnc]$this[/fnc]->%x,[fnc]$this[/fnc]->%y)
		&nbsp;				[fnc]$this[/fnc]->[classfnc:object]$parent[/classfnc]()->$nonBombPressed()
		&nbsp;			}
		&nbsp;		}
		&nbsp;	}
		&nbsp;	[cmd:class]event[/cmd]([event]OnMouseButtonPress[/event])
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->$press()
		&nbsp;	}
		&nbsp;}
		&nbsp;[cmd]class[/cmd](minesweeper,[class]widget[/class])
		&nbsp;{
		&nbsp;	[cmd:class]function[/cmd](constructor)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->%layout = [fnc]$new[/fnc]([class]layout[/class],[fnc]$this[/fnc],my_layout)
		&nbsp;		[cmd]foreach[/cmd](%x,[fnc]$range[/fnc](0,19))
		&nbsp;		{
		&nbsp;			[fnc]$this[/fnc]->%bombs[[fnc]$rand[/fnc](15)_[fnc]$rand[/fnc](15)] = bomb
		&nbsp;		}
		&nbsp;		[fnc]$this[/fnc]->%totalbombs = 0;
		&nbsp;		[cmd]foreach[/cmd](%x,[fnc]$range[/fnc](0,14))
		&nbsp;		{
		&nbsp;			[cmd]foreach[/cmd](%y,[fnc]$range[/fnc](0,14))
		&nbsp;			{
		&nbsp;				%l = [fnc]$new[/fnc](minebutton,[fnc]$this[/fnc],label%x\_%y)
		&nbsp;				[fnc]$this[/fnc]->%layout->[classfnc:layout]$add[/classfnc](%l,%x,%y)
		&nbsp;				[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[%x\_%y]" == "bomb"){
		&nbsp;					%l->%nbombs = -1
		&nbsp;					[fnc]$this[/fnc]->%totalbombs++
		&nbsp;				} [cmd:if]else[/cmd] {
		&nbsp;					%nbombs = 0;
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x - 1)_[fnc]$calc[/fnc](%y - 1)]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x - 1)_%y]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x - 1)_[fnc]$calc[/fnc](%y + 1)]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x + 1)_[fnc]$calc[/fnc](%y - 1)]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x + 1)_%y]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[[fnc]$calc[/fnc](%x + 1)_[fnc]$calc[/fnc](%y + 1)]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[%x\_[fnc]$calc[/fnc](%y - 1)]" == "bomb")%nbombs++
		&nbsp;					[cmd]if[/cmd]("[fnc]$this[/fnc]->%bombs[%x\_[fnc]$calc[/fnc](%y + 1)]" == "bomb")%nbombs++
		&nbsp;					%l->%nbombs = %nbombs
		&nbsp;				}
		&nbsp;				[fnc]$this[/fnc]->%labels[%x\_%y] = %l
		&nbsp;				%l->%x = %x
		&nbsp;				%l->%y = %y
		&nbsp;			}
		&nbsp;		}
		&nbsp;		[fnc]$this[/fnc]->%status = [fnc]$new[/fnc]([class]label[/class],[fnc]$this[/fnc],status_label)
		&nbsp;		[fnc]$this[/fnc]->%status->[classfnc:label]$setFrameShape[/classfnc](panel)
		&nbsp;		[fnc]$this[/fnc]->%status->[classfnc:label]$setFrameShadow[/classfnc](sunken)
		&nbsp;		[fnc]$this[/fnc]->%layout->[classfnc:layout]$addMultiCell[/classfnc]([fnc]$this[/fnc]->%status,15,15,0,14)
		&nbsp;		[fnc]$this[/fnc]->%status->[classfnc:label]$setText[/classfnc]("New game : total bombs : [fnc]$this[/fnc]->%totalbombs")
		&nbsp;		[fnc]$this[/fnc]->%buttonsLeft = [fnc]$calc[/fnc]((15 * 15) - [fnc]$this[/fnc]->%totalbombs)
		&nbsp;		[fnc]$this[/fnc]->[classfnc:widget]$show[/classfnc](1)
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](pressButton)
		&nbsp;	{
		&nbsp;		[cmd]if[/cmd]("$1" != "")$1->$press();
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](disableChildren)
		&nbsp;	{
		&nbsp;		[cmd]foreach[/cmd](%x,[fnc]$range[/fnc](0,14))
		&nbsp;		{
		&nbsp;			[cmd]foreach[/cmd](%y,[fnc]$range[/fnc](0,14))[fnc]$this[/fnc]->%labels[%x\_%y]->[classfnc:widget]$setEnabled[/classfnc](0)
		&nbsp;		}
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](bombPressed)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->%status->[classfnc:label]$setText[/classfnc]("Boooooom");
		&nbsp;		[fnc]$this[/fnc]->$disableChildren()
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](nonBombPressed)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->%buttonsLeft--
		&nbsp;		[fnc]$this[/fnc]->%status->[classfnc:label]$setText[/classfnc]("Total bombs : [fnc]$this[/fnc]->%totalbombs")
		&nbsp;		[cmd]if[/cmd]([fnc]$this[/fnc]->%buttonsLeft == 0){
		&nbsp;			[fnc]$this[/fnc]->%status->[classfnc:label]$setText[/classfnc]("Great job! All bombs discovered.");
		&nbsp;			[fnc]$this[/fnc]->$disableChildren()
		&nbsp;		}
		&nbsp;	}
		&nbsp;
		&nbsp;	[cmd:class]function[/cmd](emptyButtonPressed)
		&nbsp;	{
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 - 1)_[fnc]$calc[/fnc]($2 - 1)])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 - 1)_$2])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 - 1)_[fnc]$calc[/fnc]($2 + 1)])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 + 1)_[fnc]$calc[/fnc]($2 - 1)])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 + 1)_$2])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[[fnc]$calc[/fnc]($1 + 1)_[fnc]$calc[/fnc]($2 + 1)])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[$1_[fnc]$calc[/fnc]($2 - 1)])
		&nbsp;		[fnc]$this[/fnc]->$pressButton([fnc]$this[/fnc]->%labels[$1_[fnc]$calc[/fnc]($2 + 1)])
		&nbsp;	}
		&nbsp;}
		&nbsp;%x = [fnc]$new[/fnc](minesweeper,[fnc]$root[/fnc],minesweeper)
		&nbsp;%x->[classfnc:widget]$show[/classfnc](1)
		</example>
	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptLabel::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",(scriptObjectFunction)&KviScriptLabel::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",(scriptObjectFunction)&KviScriptLabel::builtinFunction_TEXT);
	d->addBuiltinFunction("setFrameShape",(scriptObjectFunction)&KviScriptLabel::builtinFunction_SETFRAMESHAPE);
	d->addBuiltinFunction("setFrameShadow",(scriptObjectFunction)&KviScriptLabel::builtinFunction_SETFRAMESHADOW);
	d->addBuiltinFunction("setImage",(scriptObjectFunction)&KviScriptLabel::builtinFunction_SETIMAGE);
}

KviScriptLabel::KviScriptLabel(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptWidget(cntrl,p,name,pDef)
{
}

KviScriptLabel::~KviScriptLabel()
{
}

bool KviScriptLabel::init(QList<KviStr> * params)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pWidget = new QLabel(((KviScriptWidget *)parent())->m_pWidget,name());
		}
	}
	if(!m_pWidget)m_pWidget = new QLabel(0,name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)((QLabel *)m_pWidget)->setText(pS->ptr());
	}
	return true;
}

int KviScriptLabel::builtinFunction_TEXT(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(((QLabel *)m_pWidget)->text());
	return KVI_ERROR_Success;
}

int KviScriptLabel::builtinFunction_SETTEXT(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			((QLabel *)m_pWidget)->setText(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	((QLabel *)m_pWidget)->setText("");
	return KVI_ERROR_Success;
}

int KviScriptLabel::builtinFunction_SETIMAGE(QList<KviStr> * params,KviStr &)
{
	if(params)
	{
		KviStr * pS1 = params->first();
		if(pS1)
		{
			KviStr path;
			if(g_pApp->findImage(path,pS1->ptr()))
			{
				((QLabel *)m_pWidget)->setAutoResize(true);
				((QLabel *)m_pWidget)->setPixmap(QPixmap(path.ptr()));
			}
			return KVI_ERROR_Success;
		}
	}

	((QLabel *)m_pWidget)->setAutoResize(false);
	((QLabel *)m_pWidget)->setPixmap(QPixmap()); // null pixmap
	return KVI_ERROR_Success;
}

int KviScriptLabel::builtinFunction_SETFRAMESHAPE(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			if(kvi_strEqualCI(pS->ptr(),"none"))((QLabel *)m_pWidget)->setFrameShape(QFrame::NoFrame);
			else if(kvi_strEqualCI(pS->ptr(),"box"))((QLabel *)m_pWidget)->setFrameShape(QFrame::Box);
			else if(kvi_strEqualCI(pS->ptr(),"panel"))((QLabel *)m_pWidget)->setFrameShape(QFrame::Panel);
			else if(kvi_strEqualCI(pS->ptr(),"winpanel"))((QLabel *)m_pWidget)->setFrameShape(QFrame::WinPanel);
			else if(kvi_strEqualCI(pS->ptr(),"hline"))((QLabel *)m_pWidget)->setFrameShape(QFrame::HLine);
			else if(kvi_strEqualCI(pS->ptr(),"vline"))((QLabel *)m_pWidget)->setFrameShape(QFrame::VLine);
			else return KVI_ERROR_InvalidParameter;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptLabel::builtinFunction_SETFRAMESHADOW(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			if(kvi_strEqualCI(pS->ptr(),"plain"))((QLabel *)m_pWidget)->setFrameShadow(QFrame::Plain);
			else if(kvi_strEqualCI(pS->ptr(),"sunken"))((QLabel *)m_pWidget)->setFrameShadow(QFrame::Sunken);
			else if(kvi_strEqualCI(pS->ptr(),"raised"))((QLabel *)m_pWidget)->setFrameShadow(QFrame::Raised);
			else return KVI_ERROR_InvalidParameter;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
///////  LINEEDIT CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@class: lineedit
	@short:
		A simple single line editor widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:

		!fn: $text()
		Returns the text currently displayed.<br>

		!fn: $setText(&lt;text&gt;)
		Sets the text to be edited.<br>

		!fn: $setReadOnly(&lt;bReadOnly&gt;)
		Sets this line edit to read only mode or read/write mode.<br>
		if &lt;bReadOnly&gt; is '1' then the read-only mode is set ,
		if it is '0' then the read/write mode is set.<br>

		!fn: $readOnly()
		Returns '1' if this widget is in read only mode, '0' otherwise.<br>

	@description:
		This class rappresents a simple text line editor.<br>
		It can be easily used in configuration dialogs.<br>
		The constructor for this class accepts an additional (optional) parameter:
		the initial text that has to be edited.<br>
		In this way you can avoid to call the [classfnc:lineedit]$setText[/classfnc] function.<br>

	@examples:

	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		class [class]label[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptLineEdit::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",(scriptObjectFunction)&KviScriptLineEdit::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",(scriptObjectFunction)&KviScriptLineEdit::builtinFunction_TEXT);
	d->addBuiltinFunction("readOnly",(scriptObjectFunction)&KviScriptLineEdit::builtinFunction_READONLY);
	d->addBuiltinFunction("setReadOnly",(scriptObjectFunction)&KviScriptLineEdit::builtinFunction_SETREADONLY);
	d->addBuiltinFunction("home",(scriptObjectFunction)&KviScriptLineEdit::builtinFunction_HOME);
	d->addBuiltinFunction("end",(scriptObjectFunction)&KviScriptLineEdit::builtinFunction_END);
	d->addBuiltinFunction("clear",(scriptObjectFunction)&KviScriptLineEdit::builtinFunction_CLEAR);
}

KviScriptLineEdit::KviScriptLineEdit(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptWidget(cntrl,p,name,pDef)
{
}

KviScriptLineEdit::~KviScriptLineEdit()
{
}

bool KviScriptLineEdit::init(QList<KviStr> * params)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pWidget = new QLineEdit(((KviScriptWidget *)parent())->m_pWidget,name());
		}
	}
	if(!m_pWidget)m_pWidget = new QLineEdit(0,name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)((QLineEdit *)m_pWidget)->setText(pS->ptr());
	}
	return true;
}

int KviScriptLineEdit::builtinFunction_TEXT(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(((QLineEdit *)m_pWidget)->text());
	return KVI_ERROR_Success;
}

int KviScriptLineEdit::builtinFunction_SETTEXT(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			((QLineEdit *)m_pWidget)->setText(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	((QLineEdit *)m_pWidget)->setText("");
	return KVI_ERROR_Success;
}

int KviScriptLineEdit::builtinFunction_READONLY(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(((QLineEdit *)m_pWidget)->isReadOnly() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptLineEdit::builtinFunction_SETREADONLY(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			((QLineEdit *)m_pWidget)->setReadOnly(kvi_strEqualCI(pS->ptr(),"1"));
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptLineEdit::builtinFunction_HOME(QList<KviStr> * p, KviStr & b)
{
	if(p)
		if(KviStr * pS = p->first())
			((QLineEdit *)m_pWidget)->home(kvi_strEqualCI(pS->ptr(),
				"1"));
	
	return KVI_ERROR_Success;
}

int KviScriptLineEdit::builtinFunction_END(QList<KviStr> * p, KviStr & b)
{
	if(p)
		if(KviStr * pS = p->first())
			((QLineEdit *)m_pWidget)->end(kvi_strEqualCI(pS->ptr(),
				"1"));
	
	return KVI_ERROR_Success;
}

int KviScriptLineEdit::builtinFunction_CLEAR(QList<KviStr> *, KviStr & b)
{
	((QLineEdit *)m_pWidget)->clear();
	return KVI_ERROR_Success;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
///////  MULTILINEEDIT CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@class: multilineedit
	@short:
		A multiline editor widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:

		!fn: $text()
		Returns the text currently displayed.<br>

		!fn: $setText(&lt;text&gt;)
		Sets the text to be edited.<br>

		!fn: $setReadOnly(&lt;bReadOnly&gt;)
		Sets this multilineedit to read only mode or read/write mode.<br>
		if &lt;bReadOnly&gt; is '1' then the read-only mode is set ,
		if it is '0' then the read/write mode is set.<br>

		!fn: $readOnly()
		Returns '1' if this widget is in read only mode, '0' otherwise.<br>

		!fn: $setCursorPosition(&lt;line&gt;,&lt;col&gt;[,&ltmark&gt])
		Moves the cursor to the specified position, extending the current selection
		if &lt;mark&gt; is 1

		!fn: $cursorPosition()
		Returns the current cursor position in the form "line,column"

	@description:

	@examples:

	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		class [class]lineedit[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptMultiLineEdit::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",(scriptObjectFunction)&KviScriptMultiLineEdit::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",(scriptObjectFunction)&KviScriptMultiLineEdit::builtinFunction_TEXT);
	d->addBuiltinFunction("readOnly",(scriptObjectFunction)&KviScriptMultiLineEdit::builtinFunction_READONLY);
	d->addBuiltinFunction("setReadOnly",(scriptObjectFunction)&KviScriptMultiLineEdit::builtinFunction_SETREADONLY);
	d->addBuiltinFunction("setCursorPosition",(scriptObjectFunction)&KviScriptMultiLineEdit::builtinFunction_SETCURSORPOSITION);
	d->addBuiltinFunction("cursorPosition",(scriptObjectFunction)&KviScriptMultiLineEdit::builtinFunction_CURSORPOSITION);
}

KviScriptMultiLineEdit::KviScriptMultiLineEdit(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptWidget(cntrl,p,name,pDef)
{
}

KviScriptMultiLineEdit::~KviScriptMultiLineEdit()
{
}

bool KviScriptMultiLineEdit::init(QList<KviStr> * params)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pWidget = new QMultiLineEdit(((KviScriptWidget *)parent())->m_pWidget,name());
		}
	}
	if(!m_pWidget)m_pWidget = new QMultiLineEdit(0,name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)((QMultiLineEdit *)m_pWidget)->setText(pS->ptr());
	}
	return true;
}

int KviScriptMultiLineEdit::builtinFunction_TEXT(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(((QMultiLineEdit *)m_pWidget)->text());
	return KVI_ERROR_Success;
}

int KviScriptMultiLineEdit::builtinFunction_CURSORPOSITION(QList<KviStr> *,KviStr &buffer)
{
	int line,col;
	((QMultiLineEdit *)m_pWidget)->getCursorPosition(&line,&col);
	buffer.append(KviStr::Format,"%d,%d",line,col);
	return KVI_ERROR_Success;
}

int KviScriptMultiLineEdit::builtinFunction_SETCURSORPOSITION(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			bool bOk;
			int line = pS->toInt(&bOk);
			if(!bOk)return KVI_ERROR_InvalidParameter;
			pS = params->next();
			if(pS)
			{
				int col = pS->toInt(&bOk);
				if(!bOk)return KVI_ERROR_InvalidParameter;
				bool bMark = false;
				pS = params->next();
				if(pS)
				{
					if(kvi_strEqualCI(pS->ptr(),"1"))bMark = true;
				}
				((QMultiLineEdit *)m_pWidget)->setCursorPosition(line,col,bMark);
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptMultiLineEdit::builtinFunction_SETTEXT(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			((QMultiLineEdit *)m_pWidget)->setText(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	((QMultiLineEdit *)m_pWidget)->setText("");
	return KVI_ERROR_Success;
}


int KviScriptMultiLineEdit::builtinFunction_READONLY(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(((QMultiLineEdit *)m_pWidget)->isReadOnly() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptMultiLineEdit::builtinFunction_SETREADONLY(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			((QMultiLineEdit *)m_pWidget)->setReadOnly(kvi_strEqualCI(pS->ptr(),"1"));
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
///////  CHECKBOX CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@class: checkbox
	@short:
		A checkbox widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $text()
		Returns the text displayed on the check box.<br>

		!fn: $setText(&lt;text&gt;)
		Sets the text to be displayed on the check box.<br>

		!fn: $isChecked()
		Returns '1' if the check box is in the checked state , '0' otherwise.<br>

		!fn: $setChecked(&lt;bChecked&gt;)
		Sets the check state of the check box.<br>
		If &lt;bChecked&gt; is 0 (false) the check box is unchecked , otherwise
		it is put into checked state.<br>

	@events:
		!ev: OnToggle($1 = &lt;bChecked&gt;)
		Triggered when the user clicks the check box and changes its state.<br>
		&lt;bChecked&gt; is '1' if the new state is "checked" , '0' otherwise.<br>
		The default event handler emits the [classsignal:checkbox]toggled[/classsignal] signal.<br>
		If you provide a new handler for this event , and still want the signal
		to be emitted , you must emit it by yourself:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:checkbox]toggled[/classsignal],$1)
		</example>

	@signals:
		!sg: toggled($1 = &lt;bChecked&gt;)
		Emitted when the box has been clicked by the user and its state has changed.<br>
		The first parameter is the new state of the checkbox : '1' if "checked" , '0' if not.<br>
		<b>This signal is emitted by the default [classevent:checkbox]OnToggle[/classevent] event handler!<br>
		If you define a new event handler , and still want the signal to be emitted,
		you have to emit it by yourself!

	@description:
		A checkable box widget : useful in implementing boolean option choices for the user.<br>
		It contains a little button that has a check mark if the widget is in checked state.<br>
		A small text label is displayed on the right of the button.<br>
		The constructor for this object class accepts an additional parameter:
		the text to be displayed.<br>

	@examples:

	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptCheckBox::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",(scriptObjectFunction)&KviScriptCheckBox::builtinFunction_SETTEXT);
	d->addBuiltinFunction("text",(scriptObjectFunction)&KviScriptCheckBox::builtinFunction_TEXT);
	d->addBuiltinFunction("isChecked",(scriptObjectFunction)&KviScriptCheckBox::builtinFunction_ISCHECKED);
	d->addBuiltinFunction("setChecked",(scriptObjectFunction)&KviScriptCheckBox::builtinFunction_SETCHECKED);
	KviScriptEventStruct * s = new KviScriptEventStruct;
	s->szName = "OnToggle";
	s->szBuffer = "$this->$emit(toggled,$1)";
	d->addDefaultEvent(s);
}

KviScriptCheckBox::KviScriptCheckBox(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptWidget(cntrl,p,name,pDef)
{
}

KviScriptCheckBox::~KviScriptCheckBox()
{
}

bool KviScriptCheckBox::init(QList<KviStr> * params)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pWidget = new QCheckBox(((KviScriptWidget *)parent())->m_pWidget,name());
		}
	}
	if(!m_pWidget)m_pWidget = new QCheckBox(0,name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	connect(m_pWidget,SIGNAL(toggled(bool)),this,SLOT(boxToggled(bool)));
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)((QCheckBox *)m_pWidget)->setText(pS->ptr());
	}
	return true;
}

void KviScriptCheckBox::boxToggled(bool bChecked)
{
	KviStr params = bChecked ? '1' : '0';
	triggerEvent("OnToggle",params);
}

int KviScriptCheckBox::builtinFunction_TEXT(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(((QCheckBox *)m_pWidget)->text());
	return KVI_ERROR_Success;
}

int KviScriptCheckBox::builtinFunction_SETTEXT(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			((QCheckBox *)m_pWidget)->setText(pS->ptr());
			return KVI_ERROR_Success;
		}
	}
	((QCheckBox *)m_pWidget)->setText("");
	return KVI_ERROR_Success;
}

int KviScriptCheckBox::builtinFunction_ISCHECKED(QList<KviStr> *,KviStr &buffer)
{
	buffer.append( ((QCheckBox *)m_pWidget)->isChecked() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptCheckBox::builtinFunction_SETCHECKED(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			((QCheckBox *)m_pWidget)->setChecked(!kvi_strEqualCI(pS->ptr(),"0"));
			return KVI_ERROR_Success;
		}
	}
	((QCheckBox *)m_pWidget)->setChecked(true);
	return KVI_ERROR_Success;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
///////  TOOLBAR CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@class: toolbar
	@short:
		A toolbar class
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $dock(&lt;docksite&gt;)
		Docks the toolbar to one of the sides of the main window.<br>
		Returns 1 if the operation was sucessfull or 0 if the toolbar is not
		a children or a KVIrc main window and thus can not be docked succesfully.<br>
		The &lt;docksite&gt; can be one of :<br>
		top,left,right,bottom,minimized.<br>

		!fn: $setDockEnabled(&lt;docksite&gt;,&lt;bEnabled&gt;)
		Enables or disables a specific docksite for this toolbar.<br>
		The &lt;docksite&gt; can be one of :<br>
		top,left,right,bottom,minimized.<br>
		&lt;bEnabled&gt; can be '1' to enable the docksite and '0' to disable it.<br>
		Returns 1 if the operation was sucessfull or 0 if the toolbar is not
		a children or a KVIrc main window and thus can not be docked succesfully.<br>

	@description:
		A toolbar widget.<br>
		It is used mainly with the [class]toolbutton[/class] objects.<br>
		If the parent of the toolbar is a non-widget object ([fnc]$root[/fnc] for example),
		the toolbar will be owned by the main KVirc window.<br>
		In that case , it will be possible to dock it to one of the four sides
		of the window and drag it around.<br>

	@examples:

	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		class [class]toolbutton[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptToolBar::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("dock",(scriptObjectFunction)&KviScriptToolBar::builtinFunction_DOCK);
	d->addBuiltinFunction("setDockEnabled",(scriptObjectFunction)&KviScriptToolBar::builtinFunction_SETDOCKENABLED);

}

KviScriptToolBar::KviScriptToolBar(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptWidget(cntrl,p,name,pDef)
{
}

KviScriptToolBar::~KviScriptToolBar()
{
}

bool KviScriptToolBar::init(QList<KviStr> * params)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pWidget = new QToolBar(name(),controller()->mainFrame(),((KviScriptWidget *)parent())->m_pWidget,false,name());
		}
	}
	if(!m_pWidget)m_pWidget = new QToolBar(name(),controller()->mainFrame(),QMainWindow::Top,false,name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	return true;
}

QMainWindow::ToolBarDock KviScriptToolBar::getDockId(const char * str)
{
	if(!str)return QMainWindow::Minimized;
	if(kvi_strEqualCI(str,"top"))return QMainWindow::Top;
	if(kvi_strEqualCI(str,"bottom"))return QMainWindow::Bottom;
	if(kvi_strEqualCI(str,"left"))return QMainWindow::Left;
	if(kvi_strEqualCI(str,"right"))return QMainWindow::Right;
	return QMainWindow::Minimized;
}

int KviScriptToolBar::builtinFunction_DOCK(QList<KviStr> * params,KviStr &buffer)
{
	if(parent()->inherits("KviScriptWidget"))
	{
		// Can't dock
		buffer.append('0');
		return KVI_ERROR_Success;
	}
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			QMainWindow * w = ((QToolBar *)m_pWidget)->mainWindow();
			if(w)
			{
				w->moveToolBar((QToolBar *)m_pWidget,getDockId(pS->ptr()));
				buffer.append('1');
			} else {
				// Can't dock
				buffer.append('0');
			}
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptToolBar::builtinFunction_SETDOCKENABLED(QList<KviStr> * params,KviStr &buffer)
{
	if(parent()->inherits("KviScriptWidget"))
	{
		// Can't dock
		buffer.append('0');
		return KVI_ERROR_Success;
	}
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			KviStr * pS2 = params->next();
			bool bEnable = true;
			if(pS2)
			{
				if(kvi_strEqualCI(pS2->ptr(),"0"))bEnable = false;
			}
			QMainWindow * w = ((QToolBar *)m_pWidget)->mainWindow();
			if(w)
			{
				w->setDockEnabled((QToolBar *)m_pWidget,getDockId(pS->ptr()),bEnable);
				buffer.append('1');
			} else {
				// Can't dock
				buffer.append('0');
			}
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

/////////////////////////////////////////////////////////////////////////////////////////////////
///////  QTWRAPPER CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@class: qtwrapper
	@short:
		A Qt widget wrapper
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@description:
		This is a class for advanced KVIrc scripting.<br>
		It can wrap any existing KVIrc widget.<br>

		The KVIrc Qt widgets are arranged in trees (just as the objects).<br>
		The difference is that there are more toplevel widgets (and so more than one tree).<br>
		You can use [cmd]debug[/cmd] -w dump_objects to take a look at the KVIrc Qt objects tree.<br>
		Here is a part of the tree:<br><br>
		<p>
		[...]<br>
		Ptr 136544360: Top level object : kvirc_main_frame , class KviFrame , visible , rect = 4,0,788,568<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Ptr 136554184: Object : recent_servers_popup , class QPopupMenu<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Ptr 136555360: Object : recent_channels_popup , class QPopupMenu<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Ptr 137333152: Object : recent_nicknames_popup , class QPopupMenu<br>
		&nbsp;&nbsp;&nbsp;&nbsp;Ptr 137336648: Object : main_splitter , class QSplitter<br>
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Ptr 137337464: Object : mdi_manager , class KviMdiManager<br>
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Ptr 137340656: Object : window_list_popup , class QPopupMenu<br>
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Ptr 137514256: Object : mdi_child , class KviMdiChild<br>
		[...]<br>
		</p>
		<br><br>
		Every listed widget has a "name" , a "class" and a set of properties.<br>
		The first entry in this example is a KVIrc server window , class "KviFrame" and name "kvirc_main_frame":
		it is a toplevel widget.<br>
		The "recent_server_popup" , "recent_channels_popup" , "recent_nicknames_popup" and the "main_splitter"
		are direct children of that widget.<br>
		The "mdi_manager" (that is the largest visible widget in each KVIrc main frame) is
		a direct child of the main_splitter.<br>
		To indicate a specific widget we will use the "class::name" form.<br>
		So to indicate the main KVIrc frame you will use "KviFrame::kvirc_main_frame",
		to indicate the MDI manager widget you will user "KviMdiManager::mdi_manager".<br>
		The special cases are:<br>
		"::mdi_manager" (::name) that designates the first widget that is named "mdi_manager" and has ANY class.<br>
		or "QPopupMenu::" (class[::]) that designates the first widget with class "QPopupMenu" and has ANY name.<br>
		The widget to wrap is defined by the constructor arguments of this class:<br>
		the fourth argument indicates the toplevel widget.<br>
		If there are arguments after the fourth , these indicate successive child widgets
		to be looked in the children sub-tree of the previous one.<br>
		To wrap the main KVIrc frame you will use:<br>
		<example>
		%tmp = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],a_name,KviFrame::kvirc_main_frame)
		</example>		
		To wrap the MDI manager you will use:<br>
		<example>
		%tmp = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],a_name,KviFrame::kvirc_main_frame,KviMdiManager::mdi_manager)
		</example>		
		The example above will first find the toplevel KviFrame then look in its sub-tree for the KviMdiManager::mdi_manager
		widget.<br>
		It could be done also in the following way:<br>
		<example>
		%tmp = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],a_name,KviFrame::kvirc_main_frame,QSplitter::main_splitter,KviMdiManager::mdi_manager)
		</example>		
		Since KVIrc can have more than one main frame, there is a special identifier for the
		main frame where the command is executed : '!'.<br>
		So again , the example above can be rewritten as:<br>
		<example>
		%tmp = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],a_name,!,QSplitter::main_splitter,KviMdiManager::mdi_manager)
		</example>
		If there is exactly one KviFrame::kvirc_main_frame, the last two examples do exactly the
		same job , otherwise the first one wraps the "first frame found" , the second one
		wraps the "current frame".<br>
		If there is exactly one mdi_manager in the specified main frame , the example above can
		be rewritten as follows:<br>
		<example>
		%tmp = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],a_name,!,::mdi_manager)
		</example>
		The constructor returns the <b>string</b> "0" if the widget to wrap can not be found,
		and thus the class instance is not created.<br>
		You will find more examples of wrapping later in this documentation.<br>
		Once you have wrapped a widget , you will receive its events (look at the events
		for the [class]widget[/class] class) , you will be able to
		use the [class]widget[/class] class functions , set its qt properties and add children
		widgets to it.<br>
		When the wrapped widget is destroyed , this class instance is destroyed too.<br>
		When this class instance is destroyed , the wrapped widget is NOT (you can't destroy the wrapped widget).<br>
		Widget names and class names ARE case sensitive.<br>
		<b>NOTE:</b><br>
		This class is a *hack* :)<br>
		It allows to do a lot of strange things with the KVIrc executable.<br>
		It <b>IS</b> possible to make a royal mess in a way that KVIrc will be not able to recover.<br>
		The only way that you have to learn to use it well is <b>trial and error</b>.<br>
		In the really worst case the "error" will lead to a segfault :)<br>
		But this should not make you give up: while writing KVIrc I fight with SIGSEGV every day ...<br>
		so just restart again and do not make the error a second time.<br>
		Just be sure that the scripts that use this class are widely tested if you are going
		to distribute it.<br>

	@examples:
		Wrap the Qt desktop widget , get its dimensions and set its color to dark gray.<br>
		<example>
		%desktop = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],just_a_name,QWidget::desktop)
		[cmd]echo[/cmd] Desktop width : %desktop->[classfnc:widget]$width[/classfnc]()
		[cmd]echo[/cmd] Desktop height : %desktop->[classfnc:widget]$height[/classfnc]()
		%desktop->[classfnc:widget]$setBackgroundColor[/classfnc](303030)
		</example>
		Change the color of the status bar of the current KVirc main frame.<br>
		<example>
		%bar =  [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],just_a_name,!,KviStatusBar::status_bar)
		%bar->[classfnc:widget]$setBackgroundColor[/classfnc](000000)
		</example>
		React to right clicks on the empty part of the toolbar in the current KVIrc main frame.<br>
		You could install a right click popup there :).<br>
		<example>
		%bar =  [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],just_a_name,!,KviToolBar::main_tool_bar)
		[cmd]obj_setevent[/cmd](%bar,[classevent:widget]OnMouseButtonPress[/classevent])[cmd]echo[/cmd] Toolbar clicked!
		# Mess with it just to show it better :)
		%bar->[classfnc:widget]$setBackgroundColor[/classfnc](303030)
		%bar->[classfnc:widget]$setQtProperty[/classfnc](frameShadow,Sunken)
		%bar->[classfnc:widget]$setQtProperty[/classfnc](frameShape,Panel)
		</example>
		Wrap the console window splitter and add a child widget to it.<br>
		<example>
		%spl = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],just_a_name,!,KviConsole::@CONSOLE,QSplitter::console_splitter)
		%child = [fnc]$new[/fnc]([class]multilineedit[/class],%spl,child_editor)
		%child->[classfnc:widget]$show[/classfnc]()
		%child->[classfnc:multilineedit]$setText[/classfnc](Hello world!)
		</example>
		Do the same on the main frame splitter<br>
		<example>
		%spl = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],just_a_name,!,QSplitter::main_splitter)
		%child = [fnc]$new[/fnc]([class]multilineedit[/class],%spl,child_editor)
		%child->[classfnc:widget]$show[/classfnc]()
		%child->[classfnc:multilineedit]$setText[/classfnc](Hi :)...[fnc]$lf[/fnc]\I could be replaced with any other widget)
		</example>
		You can add any widget instead of a multilineedit.<br>
		This could be useful for script configuration dialogs.<br>
		<br><br>
		Wrap the main window and list its qt properties<br>
		<example>
		%Main = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],just_a_name,!)
		%Main->[classfnc:widget]$listQtProperties[/classfnc]()
		</example>
		Continuing the example above.<br>
		Set the toolbars drag-movement to opaque mode.<br>
		<example>
		%Main->[classfnc:widget]$setQtProperty[/classfnc](opaqueMoving,1)
		</example>
		Continuing the example above.<br>
		Set the main window geometry and caption.<br>
		<example>
		%Main->[classfnc:widget]$setQtProperty[/classfnc](geometry,30,10,600,460)
		%Main->[classfnc:widget]$setQtProperty[/classfnc](caption,Messing with KVIrc !)
		</example>
		The beginning of a royal mess...<br>
		Wrap the main window splitter and change its orientation.<br>
		<example>
		%spl = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],just_a_name,!,::main_splitter)
		%spl->[classfnc:widget]$setQtProperty[/classfnc](orientation,Vertical)
		</example>
		Wrap a hidden popup menu and show it :)<br>
		<example>
		%pop = [fnc]$new[/fnc](qtwrapper,[fnc]$root[/fnc],just_a_name,!,::window_list_popup)
		%pop->[classfnc:widget]$show[/classfnc]()
		</example>
	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptQtWrapper::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
}

KviScriptQtWrapper::KviScriptQtWrapper(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptWidget(cntrl,p,name,pDef)
{
	m_pWidget = 0;
}

KviScriptQtWrapper::~KviScriptQtWrapper()
{
}

bool KviScriptQtWrapper::init(QList<KviStr> * params)
{
	if(!params)return false;
	m_pWidget = 0;

	for(KviStr * s = params->first();s;s = params->next())
	{
		KviStr szClass;
		KviStr szName;
		int idx = s->findFirstIdx("::");
		if(idx != -1){
			szClass = s->left(idx);
			szName = (s->ptr() + idx + 2);
		} else {
			szClass = s->ptr();
			szName = "";
		}
		if(m_pWidget)m_pWidget = findWidgetToWrap(szClass.hasData() ? szClass.ptr() : 0,
				szName.hasData() ? szName.ptr() : 0,m_pWidget);
		else m_pWidget = kvi_strEqualCS(szClass.ptr(),"!") ?
				controller()->mainFrame() :
				findTopLevelWidgetToWrap(szClass.hasData() ? szClass.ptr() : 0,
					szName.hasData() ? szName.ptr() : 0);

		if(!m_pWidget)return false;
	}
	
	if(!m_pWidget)return false;
	m_bAutoDestroyControlledWidget = false;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	return true;
}

QWidget * KviScriptQtWrapper::findTopLevelWidgetToWrap(const char * szClass,const char * szName)
{
	QWidgetList * list = g_pApp->topLevelWidgets();
	if(!list)return 0;
	QWidgetListIt it(*list);
	while(it.current())
	{
		bool bNameMatch = false;
		bool bClassMatch = false;
		if(szName)bNameMatch = kvi_strEqualCS(it.current()->name(),szName);
		else bNameMatch = true;
		if(szClass)bClassMatch = kvi_strEqualCS(it.current()->className(),szClass);
		else bClassMatch = true;
		if(bNameMatch && bClassMatch)
		{
			QWidget * w = it.current();
			delete list;
			return w;
		}

		++it;
	}
	delete list;
	return 0;
}

QWidget * KviScriptQtWrapper::findWidgetToWrap(const char * szClass,const char *szName,QWidget * childOf)
{
	const QObjectList * list = childOf->queryList(szClass ? szClass : 0,szName ? szName : 0,false,true);

	if(!list)return 0;

	QObjectListIt it(*list);
	while(it.current())
	{
		if(it.current()->isWidgetType())
		{
			QWidget * w = (QWidget *)it.current();
			delete list;
			return w;
		}
		++it;
	}

	delete list;
	return 0;
}



/////////////////////////////////////////////////////////////////////////////////////////////////
/////// SPLITTER CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@class: splitter
	@short:
		A splitter widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $setOrientation(&lt;orientation&gt;)
		Sets the orientation for this splitter widget.<br>
		&lt;orientation&gt; can be "horizontal" or "vertical".<br>

		!fn: $orientation()
		Returns the orientation of this splitter widget.<br>

	@description:
		A splitter widget object class.<br>
		It can manage the layout of one or more widgets , separating them by
		draggable bars.<br>
		It can layout its children widgets horizontally or vertically.<br>
		Basically you create a splitter as a child of any other widget
		and then create children widgets.<br>
		The children will be automatically added to the splitter and it will
		manage their geometries.<br>
	@examples:

	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptSplitter::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("orientation",(scriptObjectFunction)&KviScriptSplitter::builtinFunction_ORIENTATION);
	d->addBuiltinFunction("setOrientation",(scriptObjectFunction)&KviScriptSplitter::builtinFunction_SETORIENTATION);
}

KviScriptSplitter::KviScriptSplitter(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptWidget(cntrl,p,name,pDef)
{
}

KviScriptSplitter::~KviScriptSplitter()
{
}

bool KviScriptSplitter::init(QList<KviStr> * params)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pWidget = new QSplitter(((KviScriptWidget *)parent())->m_pWidget,name());
		}
	}
	if(!m_pWidget)return false;
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	return true;
}


int KviScriptSplitter::builtinFunction_SETORIENTATION(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			if(kvi_strEqualCI(pS->ptr(),"vertical"))
			{
				((QSplitter *)m_pWidget)->setOrientation(QSplitter::Vertical);
				return KVI_ERROR_Success;
			} else if(kvi_strEqualCI(pS->ptr(),"horizontal"))
			{
				((QSplitter *)m_pWidget)->setOrientation(QSplitter::Horizontal);
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSplitter::builtinFunction_ORIENTATION(QList<KviStr> *,KviStr &buffer)
{
	switch(((QSplitter *)m_pWidget)->orientation())
	{
		case QSplitter::Vertical:
			buffer.append("vertical");
			break;
		case QSplitter::Horizontal:
			buffer.append("horizontal");
			break;
	}
	return KVI_ERROR_Success;
}


/////////////////////////////////////////////////////////////////////////////////////////////////
/////// POPUP MENU CLASS
/////////////////////////////////////////////////////////////////////////////////////////////////

/*
	@class: popupmenu
	@short:
		A popup menu widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $insertItem(&lt;text&gt;[,image_id])
		Inserts an item with &lt;text&gt; and an optional image (see [fnc]$icon[/fnc]()).<br>
		Returns the item id.<br>

		!fn: $insertSeparator()
		Inserts a separator

		!fn: $insertMenu(&lt;text&gt;,&lt;menu_object_id&gt;[,image_id])
		Inserts a submenu ite with text &lt;text&gt; and an optional image.<br>
		The &lt;menu_object_id&gt; is the id of the inserted popupmenu class object.<br>

		!fn: $popup();
		Shows the popup and allows the user to select one of the items.<br>
		When the user has clicked on an item the popup will be hidden and the OnItemClicked
		event will be fired.<br>

		!fn: $text(&lt;id&gt;)
		Returns the text displayed on the item with the specified &lt;id&gt;

		!fn: $clear()
		Removes all the items from this popup.<br>

		!fn: $setItemEnabled(&lt;item_id&gt;,&lt;bEnabled&gt)
		Enables or disables the specified item.<br>

		!fn: $setItemChecked(&lt;item_id&gt;,&lt;bChecked&gt)
		Checks or unchecks the specified item.<br>

		!fn: $removeItem(&lt;item_id&gt;)
		Removes the specified item.<br>

		!fn: $isItemEnabled(&lt;item_id&gt;)
		Returns '1' if the specified item is enabled, '0' otherwise.<br>

		!fn: $isItemChecked(&lt;item_id&gt;)
		Returns '1' if the specified item is checked, '0' otherwise.<br>

	@events:
		!ev: OnItemClicked($1 = &lt;itemId&gt;)
		Triggered when the user selects one of the items of the menu.<br>
		&lt;itemId&gt; contains the id of the item that has been selected.<br>
		The default event handler emits the [classsignal:popupmenu]itemClicked[/classsignal] signal.<br>
		If you provide a new handler for this event , and still want the signal
		to be emitted , you must emit it by yourself:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:popupmenu]itemClicked[/classsignal],$1)
		</example>

	@signals:
		!sg: itemClicked($1 = &lt;itemId&gt;)
		Emitted when the user selects one of the menu items.<br>
		&lt;itemId&gt; contains the id of the item that has been selected.<br>
		<b>This signal is emitted by the default [classevent:popupmenu]OnItemClicked[/classevent] event handler!<br>
		If you define a new event handler , and still want the signal to be emitted,
		you have to emit it by yourself!</b>

	@description:
		A popup menu widget.<br>
		The managing of this class isn't obvious:<br>
		You need to call $insertItem() for each item that you want to add.<br>
		$insertItem() will return an unique id that you can use in the OnItemClicked
		event to check which item has been selected by the user.<br>
		Better take a look at the examples :)<br>

	@examples:
		The simplest (and most useless) way to use the menu:<br>
		<example>
		%menu = [fnc]$new[/fnc]([class]popupmenu[/class],[fnc]$root[/fnc],mypopup)
		%menu->$insertItem(Item 1,$icon(echo))
		%menu->$insertItem(Item 2,$icon(privmsg))
		%menu->$insertItem(Item 3,$icon(kill))
		obj_setevent(%menu,OnItemClicked)echo You have clicked the "$this->$text($1)"
		%menu->$popup()
		</example>
		This is kewl! :)<br>
		Try to guess what it does.<br>
		<example>
		[cmd]option[/cmd] szPrivmsgFormatNickLinkCommand "[dbl][cmd]whois[/cmd] \$1[rbt]builtin_nick_popup \$1"
		[cmd]alias[/cmd] builtin_nick_popup {
		&nbsp;	%p = [fnc]$root[/fnc]->[classfnc:object]$findChild[/classfnc](builtin_nick_popup)
		&nbsp;	[cmd]if[/cmd]("%p" != "0")%p->[classfnc:popupmenu]$clear[/classfnc]()
		&nbsp;	[cmd:if]else[/cmd] %p = [fnc]$new[/fnc]([class]popupmenu[/class],[fnc]$root[/fnc],builtin_nick_popup)
		&nbsp;	%p->%nick = $1
		&nbsp;	%p->%idWhois = %p->[classfnc:popupmenu]$insertItem[/classfnc](whois $1,[fnc]$icon[/fnc](whois))
		&nbsp;	%p->%idQuery = %p->[classfnc:popupmenu]$insertItem[/classfnc](query $1,[fnc]$icon[/fnc](query))
		&nbsp;	%p->%idKill  = %p->[classfnc:popupmenu]$insertItem[/classfnc](kill $1,[fnc]$icon[/fnc](kill))
		&nbsp;	[cmd]obj_setevent[/cmd](%p,[classevent:popupmenu]OnItemClicked[/classevent]){
		&nbsp;		[cmd]switch[/cmd]($1){
		&nbsp;			case([fnc]$this[/fnc]->%idWhois)[cmd]whois[/cmd] [fnc]$this[/fnc]->%nick
		&nbsp;			case([fnc]$this[/fnc]->%idQuery)[cmd]query[/cmd] [fnc]$this[/fnc]->%nick
		&nbsp;			case([fnc]$this[/fnc]->%idKill)[cmd]kill[/cmd] [fnc]$this[/fnc]->%nick
		&nbsp;		}
		&nbsp;	}
		&nbsp;%p->[classfnc:popupmenu]$popup[/classfnc]()
		}		
		</example>
	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


void KviScriptPopupMenu::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("insertItem",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_INSERTITEM);
	d->addBuiltinFunction("removeItem",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_REMOVEITEM);
	d->addBuiltinFunction("setItemEnabled",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_SETITEMENABLED);
	d->addBuiltinFunction("setItemChecked",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_SETITEMCHECKED);
	d->addBuiltinFunction("isItemEnabled",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_ISITEMENABLED);
	d->addBuiltinFunction("isItemChecked",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_ISITEMCHECKED);
	d->addBuiltinFunction("insertMenu",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_INSERTMENU);
	d->addBuiltinFunction("insertSeparator",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_INSERTSEPARATOR);
	d->addBuiltinFunction("popup",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_POPUP);
	d->addBuiltinFunction("clear",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_CLEAR);
	d->addBuiltinFunction("text",(scriptObjectFunction)&KviScriptPopupMenu::builtinFunction_TEXT);
	KviScriptEventStruct * s = new KviScriptEventStruct;
	s->szName = "OnItemClicked";
	s->szBuffer = "$this->$emit(itemClicked,$1)";
	d->addDefaultEvent(s);
}

KviScriptPopupMenu::KviScriptPopupMenu(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptWidget(cntrl,p,name,pDef)
{
}

KviScriptPopupMenu::~KviScriptPopupMenu()
{
}


bool KviScriptPopupMenu::init(QList<KviStr> * params)
{
	if(parent())
	{
		if(parent()->inherits("KviScriptWidget"))
		{
			m_pWidget = new QPopupMenu(((KviScriptWidget *)parent())->m_pWidget,name());
		}
	}
	if(!m_pWidget)m_pWidget = new QPopupMenu(0,name());
	connect(m_pWidget,SIGNAL(activated(int)),this,SLOT(itemActivated(int)));
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget,SIGNAL(destroyed()),this,SLOT(widgetDestroyed()));
	return true;
}

int KviScriptPopupMenu::builtinFunction_INSERTITEM(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			KviStr * pS2 = params->next();
			KviStr tmp;
			if(pS2)
			{
				bool bOk = false;
				int id = pS2->toInt(&bOk) % KVI_OUT_NUM_IMAGES;
				if(bOk)tmp.setNum(((QPopupMenu *)m_pWidget)->insertItem(*(g_pixViewOut[id]),pS->ptr()));
				else {
					QPixmap thePix(pS2->ptr());
					if(thePix.isNull())tmp.setNum(((QPopupMenu *)m_pWidget)->insertItem(*(g_pixViewOut[id]),pS->ptr()));
					else tmp.setNum(((QPopupMenu *)m_pWidget)->insertItem(thePix,pS->ptr()));
				}
			} else tmp.setNum(((QPopupMenu *)m_pWidget)->insertItem(pS->ptr()));
			buffer.append(tmp);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_REMOVEITEM(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			bool bOk = false;
			int id = pS->toInt(&bOk);
			if(bOk)((QPopupMenu *)m_pWidget)->removeItem(id);
			else return KVI_ERROR_InvalidParameter;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_SETITEMENABLED(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			KviStr * pS2 = params->next();
			if(pS2)
			{
				bool bOk = false;
				int id = pS->toInt(&bOk);
				if(!bOk)return KVI_ERROR_InvalidParameter;
				int bEnabled = pS2->toInt(&bOk);
				if(!bOk)return KVI_ERROR_InvalidParameter;
				((QPopupMenu *)m_pWidget)->setItemEnabled(id,bEnabled);
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_SETITEMCHECKED(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			KviStr * pS2 = params->next();
			if(pS2)
			{
				bool bOk = false;
				int id = pS->toInt(&bOk);
				if(!bOk)return KVI_ERROR_InvalidParameter;
				int bChecked = pS2->toInt(&bOk);
				if(!bOk)return KVI_ERROR_InvalidParameter;
				((QPopupMenu *)m_pWidget)->setItemChecked(id,bChecked);
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_INSERTMENU(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			KviStr * pS2 = params->next();
			if(pS2)
			{
				KviScriptObject * o = controller()->findObjectById(pS2->ptr());
				if(!o)return KVI_ERROR_ObjectNotFound;
				if(!o->inherits("KviScriptPopupMenu"))return KVI_ERROR_ObjectIsNotAPopup;
				KviStr * pS3 = params->next();
				KviStr tmp;
				if(pS3)
				{
					bool bOk = false;
					int id = pS3->toInt(&bOk) % KVI_OUT_NUM_IMAGES;
					if(bOk)tmp.setNum(((QPopupMenu *)m_pWidget)->insertItem(*(g_pixViewOut[id]),pS->ptr(),
							((QPopupMenu *)(((KviScriptPopupMenu *)o)->m_pWidget))));
					else {
						QPixmap thePix(pS3->ptr());

						if(thePix.isNull())tmp.setNum(((QPopupMenu *)m_pWidget)->insertItem(*(g_pixViewOut[id]),pS->ptr(),
							((QPopupMenu *)(((KviScriptPopupMenu *)o)->m_pWidget))));
						else tmp.setNum(((QPopupMenu *)m_pWidget)->insertItem(thePix,pS->ptr(),
							((QPopupMenu *)(((KviScriptPopupMenu *)o)->m_pWidget))));
					}
				} else tmp.setNum(((QPopupMenu *)m_pWidget)->insertItem(pS->ptr(),((QPopupMenu *)(((KviScriptPopupMenu *)o)->m_pWidget))));
				buffer.append(tmp);
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_TEXT(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			bool bOk = false;
			int id = pS->toInt(&bOk);
			if(!bOk)return KVI_ERROR_InvalidParameter;
			KviStr tmp = ((QPopupMenu *)m_pWidget)->text(id);
			buffer.append(tmp);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_ISITEMENABLED(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			bool bOk = false;
			int id = pS->toInt(&bOk);
			if(!bOk)return KVI_ERROR_InvalidParameter;
			buffer.append(((QPopupMenu *)m_pWidget)->isItemEnabled(id) ? '1' : '0');
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptPopupMenu::builtinFunction_ISITEMCHECKED(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr * pS = params->first();
		if(pS)
		{
			bool bOk = false;
			int id = pS->toInt(&bOk);
			if(!bOk)return KVI_ERROR_InvalidParameter;
			buffer.append(((QPopupMenu *)m_pWidget)->isItemChecked(id) ? '1' : '0');
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}
int KviScriptPopupMenu::builtinFunction_POPUP(QList<KviStr> * params,KviStr &buffer)
{
	((QPopupMenu *)m_pWidget)->popup(QCursor::pos());
	return KVI_ERROR_Success;
}

int KviScriptPopupMenu::builtinFunction_INSERTSEPARATOR(QList<KviStr> * params,KviStr &buffer)
{
	((QPopupMenu *)m_pWidget)->insertSeparator();
	return KVI_ERROR_Success;
}

int KviScriptPopupMenu::builtinFunction_CLEAR(QList<KviStr> * params,KviStr &buffer)
{
	((QPopupMenu *)m_pWidget)->clear();
	return KVI_ERROR_Success;
}


void KviScriptPopupMenu::itemActivated(int id)
{
	KviStr parms(KviStr::Format,"%d",id);
	triggerEvent("OnItemClicked",parms);
}

////////////////////////////////////////////////////////////////////////////////
///// SPINBOX CLASS
////////////////////////////////////////////////////////////////////////////////

/*
	@class: spinbox
	@short:
		A spinbox widget, sometimes called 'little arrows widget' :)
	@inherits:
		[class]widget[/class]<br>
		[class]object[/object]
	@functions:
		!fn: $setValue(&lt;new_val&gt;)
		Sets the current value of the spinbox to &lt;new_val&gt;.<br>
		Returns nothing.

		!fn: $value()
		Returns the current raw value of the spinbox (ie: including
		suffix).<br>

		!fn: $setMinValue()
		Sets the minimal value of the spinbox.<br>
		Returns nothing.

		!fn: $setMaxValue()
		Sets the maximal value of the spinbox.<br>
		Returns nothing.
		
		!fn: $setStep(&lt;new_val&gt;)
		Sets the spinbox value increment/decrement every time an arrow
		button is pressed.<br>
		Returns nothing.

		!fn: $setWrapping(&lt;state&gt;)
		Sets wrapping to &lt;state&gt; which is a boolean value:<br>
		If &lt;state&gt; is 0 wrapping is disabled, if it has any other
		value, wrapping is enabled.<br>
		Enabling wrapping will allow to step from the highest value and
		vice-versa. By default, wrapping is turned off.

		!fn: $wrapping()
		Returns 0 if wrapping is disabled, 1 otherwise.

		!fn: $setSuffix(&lt;new_val&gt;)
		Sets the text to be displayed after the spinbox value.<br>
		Returns nothing.

		!fn: $suffix()
		Returns current spinbox suffix.

		!fn: $cleanValue()
		Returns spinbox value stripped from suffix.
	
	@description:
		The spinbox is a widget sometimes called little arrows widget or
		up-down widget.<br>
		It allows to pick a numeric value either by typing it directly
		to the spinbox, or clicking up/down buttons.<br>

	@examples:
		The simplest way to use a spinbox is:
		<example>
		%s = $new(spinbox, $root, my_spinbox, 1, 24, 1)
		# the spinbox now can show values ranged between 10-100.
		# Every time user clicks an arrow button value will be<br> 
		# increased/decreased by 1.<br>
		# Now you ask the user to set the number of hours he usually<br>
		# sleeps :)<br>
		%s->$setSuffix( " hours" );
		%var = %s->$value();
		echo "Usually you sleep for %var.";
		# That's it :)
		</example>
		Or, you can do it the "hard way":<br>
		<example>
		%s = $new(spinbox, $root, my_spinbox, 1, 24, 1)
		%s->$setSuffix(" hours");
		%var1 = %s->$cleanValue();
		%var2 = %s->$suffix();
		echo "Usually you sleep for %var1%var2.";
		# This will produce same result as previous example.	
		</example>
	
	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>

*/	

void KviScriptSpinBox::
	initializeClassDefinition(KviScriptObjectClassDefinition * d)
{
	d->addBuiltinFunction("setValue",(scriptObjectFunction)&
		KviScriptSpinBox::builtinFunction_SETVALUE);
	d->addBuiltinFunction("value",(scriptObjectFunction)&
		KviScriptSpinBox::builtinFunction_VALUE);
	d->addBuiltinFunction("setMinValue",(scriptObjectFunction)&
		KviScriptSpinBox::builtinFunction_SETMINVALUE);
	d->addBuiltinFunction("setMaxValue",(scriptObjectFunction)&
		KviScriptSpinBox::builtinFunction_SETMAXVALUE);
	d->addBuiltinFunction("wrapping",(scriptObjectFunction)&
		KviScriptSpinBox::builtinFunction_WRAPPING);
	d->addBuiltinFunction("setWrapping",(scriptObjectFunction)&
		KviScriptSpinBox::builtinFunction_SETWRAPPING);
	d->addBuiltinFunction("setStep",(scriptObjectFunction)&
		KviScriptSpinBox::builtinFunction_SETSTEP);
	d->addBuiltinFunction("setSuffix",(scriptObjectFunction)&
		KviScriptSpinBox::builtinFunction_SETSUFFIX);
	d->addBuiltinFunction("suffix",(scriptObjectFunction)&
		KviScriptSpinBox::builtinFunction_SUFFIX);
	d->addBuiltinFunction("cleanValue",(scriptObjectFunction)&
		KviScriptSpinBox::builtinFunction_CLEANVALUE);
}

KviScriptSpinBox::KviScriptSpinBox(KviScriptObjectController * cntrl, 
	KviScriptObject * p, const char * name, KviScriptObjectClassDefinition *
	pDef ):KviScriptWidget(cntrl, p, name, pDef)
{
}

KviScriptSpinBox::~KviScriptSpinBox()
{
}

bool KviScriptSpinBox::init(QList<KviStr> * params)
{
	if(parent())
		if(parent()->inherits("KviScriptWidget"))
			m_pWidget = new QSpinBox(((KviScriptWidget*)parent())->
				m_pWidget, name());
	
	if(!m_pWidget)
		m_pWidget = new QSpinBox(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	if(params)
	{
		KviStr * pS = params->first();
		KviStr * pS2 = params->next();
		KviStr * pS3 = params->next();
		int val;
		bool bOk;
		if(pS)
		{
			val = pS->toInt(&bOk);
			if(!bOk)
				return false;
			((QSpinBox *)m_pWidget)->setMinValue(val);
		}

		if(pS2)
		{
			val = pS2->toInt(&bOk);
			if(!bOk)
				return false;
			((QSpinBox *)m_pWidget)->setMaxValue(val);
		}

		if(pS3)
		{
			val = pS3->toInt(&bOk);
			if(!bOk)
				return false;
			((QSpinBox *)m_pWidget)->setLineStep(val);
		}
	}
	return true;
}

int KviScriptSpinBox::builtinFunction_SETVALUE(QList<KviStr> * p, KviStr & buff)
{
	if(p)
	{
		KviStr * pS = 0;
		if((pS = p->first()))
		{
			bool bOk;
			int val;
			val = pS->toInt(&bOk);
			if(!bOk)
				return KVI_ERROR_InvalidParameter;
			((QSpinBox *)m_pWidget)->setValue(val);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_VALUE(QList<KviStr> * p, KviStr & buffer)
{
	if(p)
	{
		KviStr * pS = 0;
		if((pS = p->first()))
		{
			buffer.append(((QSpinBox *)m_pWidget)->text());
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_SETMINVALUE(QList<KviStr> * p, KviStr & b)
{
	if(p)
	{
		if(KviStr *pS = p->first())
		{
			bool bOk;
			int val = pS->toInt(&bOk);
			if(bOk)
			{
				((QSpinBox *)m_pWidget)->setMinValue(val);
				return KVI_ERROR_Success;
			}
			else
				return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_SETMAXVALUE(QList<KviStr> * p, KviStr & b)
{
	if(p)
	{
		if(KviStr * pS = p->first())
		{
			bool bOk;
			int val = pS->toInt(&bOk);
			if(bOk)
			{
				((QSpinBox *)m_pWidget)->setMaxValue(val);
				return KVI_ERROR_Success;
			}
			else
				return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_WRAPPING(QList<KviStr> *p, KviStr & b)
{
	b.append(((QSpinBox *)m_pWidget)->wrapping() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptSpinBox::builtinFunction_SETWRAPPING(QList<KviStr> *p, KviStr &b)
{
	if(p)
	{
		bool bEn = true;
		if(KviStr * pS = p->first())
		{
			int tmp = pS->toInt();
			if(tmp == 0)
				bEn = false;
		}	
		((QSpinBox *)m_pWidget)->setWrapping(bEn);
		return KVI_ERROR_Success;
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_SETSTEP(QList<KviStr> *p, KviStr &b)
{
	if(p)
	{
		if(KviStr * pS = p->first())
		{
			bool bOk;
			int val = pS->toInt(&bOk);
			if(bOk)
			{
				((QSpinBox *)m_pWidget)->setLineStep(val);
				return KVI_ERROR_Success;
			}
			else
				return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_SETSUFFIX(QList<KviStr> *p, KviStr &b)
{
	if(p)
	{
		KviStr * pS = p->first();
		((QSpinBox *)m_pWidget)->setSuffix(pS->ptr());
		return KVI_ERROR_Success;
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_SUFFIX(QList<KviStr> *p, KviStr &b)
{
	b.append(((QSpinBox *)m_pWidget)->suffix().latin1());
	return KVI_ERROR_Success;
}

int KviScriptSpinBox::builtinFunction_CLEANVALUE(QList<KviStr> *p, KviStr &b)
{
	b.append(((QSpinBox *)m_pWidget)->cleanText().latin1());
	return KVI_ERROR_Success;
}

////////////////////////////////////////////////////////////////////////////////
//////		COMBOBOX
////////////////////////////////////////////////////////////////////////////////

/*
	@class: combobox
	@short:
		A handy selection widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $insertItem(&lt;text&gt;)
		Inserts a text item.
		!fn: $clear()
		Clears the combobox.
		!fn: $currentText()
		Returns the text of currently selected item.
		!fn: $text(&lt;index&gt;)
		Returns the text of an item at given index.
		!fn: $removeItem(&lt;index&gt;)
		Removes item at given index.

	@description:
		This widget is a great widget to display or get user selection.
		It is somewhat similar to a button combined with a popup menu:
		when user clicks a button placed on the right edge of the
		combobox, a popup menu with availible options/selections will
		pop up.
		
	@examples:
		Let's say you have a script that can run in two modes: Normal
		and Advanced. One of the way to show current mode, or to let
		user change it is putting a combobox in your script's config
		dialog:
		<example>
		# we assume that %my_cfg_dlg is a widget (config dialog)<br>
		%cb = $new(combobox, %my_cfg_dlg);
		%cb->$insertItem("Normal");
		%cb->$insertItem("Advanced");
		# now after user makes a choice you can obtain it like this:<br>
		%sMode = %cb->$currentText;
		echo Current script mode is: %sMode;
		</example>
	
		
	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

void KviScriptComboBox::initializeClassDefinition(KviScriptObjectClassDefinition * d)
{
	d->addBuiltinFunction("insertItem",(scriptObjectFunction)&
		KviScriptComboBox::builtinFunction_INSERTITEM);
	d->addBuiltinFunction("clear",(scriptObjectFunction)&
		KviScriptComboBox::builtinFunction_CLEAR);
	d->addBuiltinFunction("currentText",(scriptObjectFunction)&
		KviScriptComboBox::builtinFunction_CURRENTTEXT);
	d->addBuiltinFunction("text",(scriptObjectFunction)&
		KviScriptComboBox::builtinFunction_TEXT);
	d->addBuiltinFunction("removeItem",(scriptObjectFunction)&
		KviScriptComboBox::builtinFunction_REMOVEITEM);
}

KviScriptComboBox::KviScriptComboBox(KviScriptObjectController * cntrl, 
	KviScriptObject * p, const char * name, KviScriptObjectClassDefinition *
	pDef ):KviScriptWidget(cntrl, p, name, pDef)
{
}

KviScriptComboBox::~KviScriptComboBox()
{
}


bool KviScriptComboBox::init(QList<KviStr> * params)
{
	if(parent())
		if(parent()->inherits("KviScriptWidget"))
			m_pWidget = new QComboBox(((KviScriptWidget*)parent())->
				m_pWidget, name());
	
	if(!m_pWidget)
		m_pWidget = new QComboBox(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	connect(m_pWidget, SIGNAL(activated(const QString &)), m_pWidget, \
		SLOT(__slotActivated(const QString &)));
	return true;
}

int KviScriptComboBox::builtinFunction_INSERTITEM(QList<KviStr> *p, KviStr &b)
{
	if(p)
	{
		KviStr * pS = p->first();	// text
		int val = -1;
	// There are problems with the commented stuff, and I can't figure out
	// what it is !!! This is strange: when I do for example:
	//
	// %a=$new(combobox, $root)
	// %a->$insertItem("Blah.. blah...");
	//
	// I get an error message (Invalid parameter). But there is no second
	// parmeter given... It works ok, if I _do_ pass the second parameter.
	// I'll leave this disabled though...
	/*	bool bOk;
	
		if(p->count()>1)
		{
			KviStr * pS2 = p->at(1);	// index
			val = pS2->toInt(&bOk);
			if(!bOk)
				return KVI_ERROR_InvalidParameter;
		}
	*/

		((QComboBox *)m_pWidget)->insertItem(pS->ptr(), val);
		return KVI_ERROR_Success;
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptComboBox::builtinFunction_CLEAR(QList<KviStr> *, KviStr &)
{
	((QComboBox *)m_pWidget)->clear();
	return KVI_ERROR_Success;
}

int KviScriptComboBox::builtinFunction_CURRENTTEXT(QList<KviStr> *, KviStr & b)
{
	b.append(((QComboBox *)m_pWidget)->currentText().latin1());
	return KVI_ERROR_Success;
}

int KviScriptComboBox::builtinFunction_TEXT(QList<KviStr> * p, KviStr & b)
{
	if(!p)
		return KVI_ERROR_MissingParameter;
	
	KviStr * pS = p->first();
	if(!pS)
		return KVI_ERROR_MissingParameter;

	bool bOk;
	int val = pS->toInt(&bOk);
	if(!bOk)
		return KVI_ERROR_InvalidParameter;

	b.append(((QComboBox *)m_pWidget)->text(val).latin1());
	return KVI_ERROR_Success;
}

int KviScriptComboBox::builtinFunction_REMOVEITEM(QList<KviStr> * p, KviStr & b)
{
	if(p)
	{
		if(KviStr * pS = p->first())
		{
			bool bOk;
			int val = pS->toInt(&bOk);
			if(!bOk)
				return KVI_ERROR_InvalidParameter;

			((QComboBox *)m_pWidget)->removeItem(val);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

void KviScriptComboBox::__slotActivated(const QString & str)
{
	KviStr parms(KviStr::Format, "%s", str.latin1());
	triggerEvent("OnActivated", parms);
}

////////////////////////////////////////////////////////////////////////////////
//////		TABWIDGET
////////////////////////////////////////////////////////////////////////////////

/*
	@class: tabwidget
	@short:
		A stack of tabbed widgets
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $addTab(<widget_object>,<label>)
		!fn: $removeTab(<widget_object>)
		!fn: $showTab>(<widget_object>)
	@description:
		
	@examples:
		
	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

void KviScriptTabWidget::
	initializeClassDefinition(KviScriptObjectClassDefinition * d)
{
	d->addBuiltinFunction("addTab",(scriptObjectFunction)&
		KviScriptTabWidget::builtinFunction_ADDTAB);
	d->addBuiltinFunction("removeTab",(scriptObjectFunction)&
		KviScriptTabWidget::builtinFunction_REMOVETAB);
	d->addBuiltinFunction("showTab",(scriptObjectFunction)&
		KviScriptTabWidget::builtinFunction_SHOWTAB);
}

KviScriptTabWidget::KviScriptTabWidget(KviScriptObjectController * cntrl, 
	KviScriptObject * p, const char * name, KviScriptObjectClassDefinition *
	pDef ):KviScriptWidget(cntrl, p, name, pDef)
{
}

KviScriptTabWidget::~KviScriptTabWidget()
{
}

bool KviScriptTabWidget::init(QList<KviStr> * params)
{
	if(parent())
		if(parent()->inherits("KviScriptWidget"))
			m_pWidget = new QTabWidget(((KviScriptWidget*)parent())->m_pWidget, name());
	
	if(!m_pWidget)
		m_pWidget = new QTabWidget(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	return true;
}

int KviScriptTabWidget::builtinFunction_ADDTAB(QList<KviStr> * p, KviStr & b)
{
	if(p)
	{
		KviStr * pS = p->first();
		if(pS)
		{
			KviScriptObject * o = controller()->findObjectById(pS->ptr());
			if(o)
			{
				if(o->inherits("KviScriptWidget"))
				{
					KviStr * pS2 = p->next();
					((QTabWidget *)m_pWidget)->addTab(((KviScriptWidget *)o)->m_pWidget,
						pS2 ? pS2->ptr() : __tr("unnamed"));
				} else return KVI_ERROR_ObjectIsNotAWidget;
			} else return KVI_ERROR_ObjectNotFound;
		} else return KVI_ERROR_MissingObjectId;
	} else return KVI_ERROR_MissingParameter;

	return KVI_ERROR_Success;
}

int KviScriptTabWidget::builtinFunction_REMOVETAB(QList<KviStr> * p, KviStr & b)
{
	if(p)
	{
		KviStr * pS = p->first();
		if(pS)
		{
			KviScriptObject * o = controller()->findObjectById(pS->ptr());
			if(o)
			{
				if(o->inherits("KviScriptWidget"))
				{
					((QTabWidget *)m_pWidget)->removePage(((KviScriptWidget *)o)->m_pWidget);
				} else return KVI_ERROR_ObjectIsNotAWidget;
			} else return KVI_ERROR_ObjectNotFound;
		} else return KVI_ERROR_MissingObjectId;
	} else return KVI_ERROR_MissingParameter;

	return KVI_ERROR_Success;
}


int KviScriptTabWidget::builtinFunction_SHOWTAB(QList<KviStr> * p, KviStr & b)
{
	if(p)
	{
		KviStr * pS = p->first();
		if(pS)
		{
			KviScriptObject * o = controller()->findObjectById(pS->ptr());
			if(o)
			{
				if(o->inherits("KviScriptWidget"))
				{
					((QTabWidget *)m_pWidget)->showPage(((KviScriptWidget *)o)->m_pWidget);
				} else return KVI_ERROR_ObjectIsNotAWidget;
			} else return KVI_ERROR_ObjectNotFound;
		} else return KVI_ERROR_MissingObjectId;
	} else return KVI_ERROR_MissingParameter;

	return KVI_ERROR_Success;
}

////////////////////////////////////////////////////////////////////////////////
//////////		KviScriptTreeViewItem
////////////////////////////////////////////////////////////////////////////////
/*
KviScriptTreeViewItem::KviScriptTreeViewItem(KviScriptObjectController * cntrl, 
	KviScriptObject * p, const char * name, KviScriptObjectClassDefinition *
	pDef ):KviScriptWidget(cntrl, p, name, pDef),
	QListViewItem((QListViewItem *)p);
{
}

KviScriptTreeViewItem::~KviScriptTreeViewItem()
{
}

bool KviScriptTreeViewItem::init(QList<KviStr> * params)
{
	if(parent())
		if(parent()->inherits("KviScriptWidget"))
			m_pWidget = new QTreeListItem(((KviScriptWidget*)parent())->m_pWidget, name());
	
	if(!m_pWidget)
		m_pWidget = new QTabWidget(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	return true;
}
*/


////////////////////////////////////////////////////////////////////////////////
//////		TREEVIEW
////////////////////////////////////////////////////////////////////////////////

/*
	@class: treeview
	@short:
		A tree view widget
	@inherits:
		[class]object[/class]<br>
		[class]widget[/class]
	@functions:
		!fn: $addColumn(&lt;text&gt;)
		Adds a column with &lt;text&gt; to the treeview
		You need at least one column for this widget to work

		!fn: $removeColumn(&lt;index&gt;)
		Removes the column with zero-based index &lt;index&gt;

		!fn: $setColumnText(&lt;index&gt;,&lt;text&gt;)
		Sets a column's text to &lt;text&gt;.

		!fn: $setRootIsDecorated(&lt;boolean&gt;)
		Enables or disables the plus signs beside the items in this treeview.
		This is enabled by default.

		!fn: $setShowSortIndicator(&lt;boolean&gt;)
		Shows or hides the sort indicator

		!fn: $setAllColumnsShowFocus(&lt;boolean&gt;)
		Enables the focus visualisation for the whole items (instead of the first column only)

		!fn: $firstChild()
		Returns the id of the first [class]treeviewitem[/class] in this treeview.
		If there is no first child, this function returns an empty string.
		
	@events:
		!ev: OnSelectionChanged($1 = &lt;itemId&gt;)
		Triggered when the user selects one of the items of the treeview.<br>
		&lt;itemId&gt; contains the object id of the treeviewitem that has been selected or an empty string if the change
		caused to unset any current item.<br>
		The default event handler emits the [classsignal:treeview]selectionChanged[/classsignal] signal.<br>
		If you provide a new handler for this event , and still want the signal
		to be emitted , you must emit it by yourself:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:treeviewitem]selectionChanged[/classsignal],$1)
		</example>

		!ev: OnItemClicked($1 = &lt;itemId&gt;)
		Triggered when the user clicks on the treeview.<br>
		If the user clicks on an item, it's id is passed as the first parameter.<br>
		The default event handler emits the [classsignal:treeview]itemClicked[/classsignal] signal.<br>
		If you provide a new handler for this event , and still want the signal
		to be emitted , you must emit it by yourself:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:treeviewitem]itemClicked[/classsignal],$1)
		</example>

		!ev: OnItemDoubleClicked($1 = &lt;itemId&gt;)
		Triggered when the user double-clicks on the treeview.<br>
		If the user double-clicks on an item, it's id is passed as the first parameter.<br>
		The default event handler emits the [classsignal:treeview]itemDoubleClicked[/classsignal] signal.<br>
		If you provide a new handler for this event , and still want the signal
		to be emitted , you must emit it by yourself:<br>
		<example>
			[fnc]$this[/fnc]->[classfnc:object]$emit[/classfnc]([classsignal:treeviewitem]itemDoubleClicked[/classsignal],$1)
		</example>

	@signals:
		!sg: selectionChanged($1 = &lt;itemId&gt;)
		Emitted when the user selects one of the treeview items.<br>
		&lt;itemId&gt; contains the object id of the treeviewitem that has been selected or an empty string if the change
		caused the treeview to have no current item.<br>
		<b>This signal is emitted by the default [classevent:treeview]OnSelectionChanged[/classevent] event handler!<br>
		If you define a new event handler , and still want the signal to be emitted,
		you have to emit it by yourself!</b>

		!sg: itemClicked($1 = &lt;itemId&gt;)
		Emitted when the user clicks on one of the treeview items.<br>
		&lt;itemId&gt; contains the object id of the treeviewitem that has been clicked or an empty string if the user
		clicked on the empty part of the treeview.<br>
		<b>This signal is emitted by the default [classevent:treeview]OnItemClicked[/classevent] event handler!<br>
		If you define a new event handler , and still want the signal to be emitted,
		you have to emit it by yourself!</b>

		!sg: itemDoubleClicked($1 = &lt;itemId&gt;)
		Emitted when the user double-clicks on one of the treeview items.<br>
		&lt;itemId&gt; contains the object id of the treeviewitem that has been double-clicked or an empty string if the user
		clicked on the empty part of the treeview.<br>
		<b>This signal is emitted by the default [classevent:treeview]OnItemDoubleClicked[/classevent] event handler!<br>
		If you define a new event handler , and still want the signal to be emitted,
		you have to emit it by yourself!</b>

	@description:
		A tree view is a widget that is able to display items in a tree like manner.
		You will have to use also the [class]treeviewitem[/class] that reppresents
		a child item of this widget.<br>
		Basically you create a treeview widget and add a few child treeviewitem
		objects: the treeviewitem objects are arranged in a tree like manner:
		if you create them as children of the treeview , they will be
		appearing as "toplevel" in the treeview; if you create them as children
		of other treeviewitem objects, they will appear as children in the treeview too.
		To remove an item you have just to destroy it. All the items will be destroyed
		when this treeview object is destroyed.<br>
		The object class emits the OnSelectionChanged event (and signal in the default handler),
		when the user changes the currently selected item or unselects any item.<br>
		You can access the tree of items by using the function $firstChild() that returns the
		root item.

	@examples:
		<example>
			# Really simple one

			%tree = [fnc]$new[/fnc](treeview,[fnc]$root[/fnc],mytreeview)
			%tree->$addColumn("Friends")
			%tree->$addColumn("Age")
			%tree->$setRootIsDecorated(1)
			

			# Create the toplevel items

			%men = [fnc]$new[/fnc](treeviewitem,%tree,item,Men)
			%women = [fnc]$new[/fnc](treeviewitem,%tree,item,Women)

			# Create some child items
			%it = $new(treeviewitem,%men,it,Pippo,25)
			%it = $new(treeviewitem,%men,it,Pluto,11)
			%it = $new(treeviewitem,%men,it,Paperino,45)
			%it = $new(treeviewitem,%women,it,Minnie,12)
			%it = $new(treeviewitem,%women,it,Lucy,999)
			# We want the men subtree open
			%men->$setOpen(1)

			[cmd]obj_setevent[/cmd](%tree,OnClose)
			{
				[cmd]destroy[/cmd] [fnc]$this[/fnc]
			}

			[cmd]obj_setevent[/cmd](%tree,OnSelectionChanged)
			{
				[cmd]if[/cmd]("$1" != "")[cmd]echo[/cmd] "Selection changed: item $1->$text(0)"
				else echo "No selection"
			}

			%tree->$show();

			# ...
			# ... navigate thru the items
			# ...
			# this could be implemented gracefully with a recursive function...
			# (left as programming exercise to the scripter :)

			%it = %tree->$firstChild();
			while("%it" != "")
			{
				echo "Navigation: Item: %it->$text(0)"
				%child = %it->$firstChild()
				if("%child" != "")%it = %child
				else %it = %it->$nextSibling()
			}

		</example>
	@seealso:
		class [class]object[/class],<br>
		class [class]widget[/class],<br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

//#warning "Set column pixmap..."

void KviScriptTreeView::
	initializeClassDefinition(KviScriptObjectClassDefinition * d)
{
	d->addBuiltinFunction("addColumn",(scriptObjectFunction)&
		KviScriptTreeView::builtinFunction_ADDCOLUMN);
	d->addBuiltinFunction("firstChild",(scriptObjectFunction)&
		KviScriptTreeView::builtinFunction_FIRSTCHILD);
	d->addBuiltinFunction("removeColumn",(scriptObjectFunction)&
		KviScriptTreeView::builtinFunction_REMOVECOLUMN);
	d->addBuiltinFunction("setColumnText",(scriptObjectFunction)&
		KviScriptTreeView::builtinFunction_SETCOLUMNTEXT);
	d->addBuiltinFunction("setRootIsDecorated",(scriptObjectFunction)&
		KviScriptTreeView::builtinFunction_SETROOTISDECORATED);
	d->addBuiltinFunction("setShowSortIndicator",(scriptObjectFunction)&
		KviScriptTreeView::builtinFunction_SETSHOWSORTINDICATOR);
	d->addBuiltinFunction("setAllColumnsShowFocus",(scriptObjectFunction)&
		KviScriptTreeView::builtinFunction_SETALLCOLUMNSSHOWFOCUS);
	KviScriptEventStruct * s = new KviScriptEventStruct;
	s->szName = "OnSelectionChanged";
	s->szBuffer = "$this->$emit(selectionChanged,$1)";
	d->addDefaultEvent(s);
	s = new KviScriptEventStruct;
	s->szName = "OnItemClicked";
	s->szBuffer = "$this->$emit(itemClicked,$1)";
	d->addDefaultEvent(s);
	s = new KviScriptEventStruct;
	s->szName = "OnItemDoubleClicked";
	s->szBuffer = "$this->$emit(itemDoubleClicked,$1)";
	d->addDefaultEvent(s);
}

KviScriptTreeView::KviScriptTreeView(KviScriptObjectController * cntrl, 
	KviScriptObject * p, const char * name, KviScriptObjectClassDefinition *
	pDef ):KviScriptWidget(cntrl, p, name, pDef)
{
}

KviScriptTreeView::~KviScriptTreeView()
{
}

bool KviScriptTreeView::init(QList<KviStr> * params)
{
	if(parent())
		if(parent()->inherits("KviScriptWidget"))
			m_pWidget = new QListView(((KviScriptWidget*)parent())->m_pWidget, name());
	
	if(!m_pWidget)
		m_pWidget = new QListView(0, name());
	((QListView *)m_pWidget)->setRootIsDecorated(true);
	((QListView *)m_pWidget)->setSelectionMode(QListView::Single);
	connect(m_pWidget,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(selectionChanged(QListViewItem *)));
	connect(m_pWidget,SIGNAL(clicked(QListViewItem *)),this,SLOT(itemClicked(QListViewItem *)));
	connect(m_pWidget,SIGNAL(doubleClicked(QListViewItem *)),this,SLOT(itemDoubleClicked(QListViewItem *)));
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	return true;
}


void KviScriptTreeView::selectionChanged(QListViewItem *it)
{
	KviStr parms;
	if(it)parms.sprintf("%s",((KviScriptListViewItem *)it)->m_pScriptObjectClass->id());
	triggerEvent("OnSelectionChanged",parms);
}

void KviScriptTreeView::itemClicked(QListViewItem *it)
{
	KviStr parms;
	if(it)parms.sprintf("%s",((KviScriptListViewItem *)it)->m_pScriptObjectClass->id());
	triggerEvent("OnItemClicked",parms);
}

void KviScriptTreeView::itemDoubleClicked(QListViewItem *it)
{
	KviStr parms;
	if(it)parms.sprintf("%s",((KviScriptListViewItem *)it)->m_pScriptObjectClass->id());
	triggerEvent("OnItemDoubleClicked",parms);
}

int KviScriptTreeView::builtinFunction_FIRSTCHILD(QList<KviStr> *p,KviStr &b)
{
	KviScriptListViewItem * it = (KviScriptListViewItem *)((QListView *)m_pWidget)->firstChild();

	if(it)b.append(it->m_pScriptObjectClass->id());

	return KVI_ERROR_Success;
}

int KviScriptTreeView::builtinFunction_ADDCOLUMN(QList<KviStr> * p, KviStr & b)
{
	if(p)
	{
		if(KviStr * pS = p->first())((QListView *)m_pWidget)->addColumn(pS->ptr());
		else ((QListView *)m_pWidget)->addColumn(__tr("unnamed"));
	} else return KVI_ERROR_MissingParameter;

	return KVI_ERROR_Success;
}

int KviScriptTreeView::builtinFunction_REMOVECOLUMN(QList<KviStr> * p, KviStr & b)
{
	if(p)
	{
		if(KviStr * pS = p->first())
		{
			bool bOk;
			int idx = pS->toInt(&bOk);
			if(bOk)
			{
				((QListView *)m_pWidget)->removeColumn(idx);
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}


int KviScriptTreeView::builtinFunction_SETCOLUMNTEXT(QList<KviStr> * p, KviStr & b)
{
	if(p)
	{
		if(KviStr * pS = p->first())
		{
			bool bOk;
			int idx = pS->toInt(&bOk);
			if(bOk)
			{
				pS = p->next();
				((QListView *)m_pWidget)->setColumnText(idx,pS ? pS->ptr() : "");
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTreeView::builtinFunction_SETROOTISDECORATED(QList<KviStr> * p, KviStr & b)
{
	bool rd = true;
	if(p)
	{
		if(KviStr * pS = p->first())rd = !kvi_strEqualCI(pS->ptr(),"0");
	}

	((QListView *)m_pWidget)->setRootIsDecorated(rd);

	return KVI_ERROR_Success;
}

int KviScriptTreeView::builtinFunction_SETALLCOLUMNSSHOWFOCUS(QList<KviStr> * p, KviStr &)
{
	bool rd = true;
	if(p)
	{
		if(KviStr * pS = p->first())rd = !kvi_strEqualCI(pS->ptr(),"0");
	}

	((QListView *)m_pWidget)->setAllColumnsShowFocus(rd);

	return KVI_ERROR_Success;
}

int KviScriptTreeView::builtinFunction_SETSHOWSORTINDICATOR(QList<KviStr> * p, KviStr & b)
{
	bool rd = true;
	if(p)
	{
		if(KviStr * pS = p->first())rd = !kvi_strEqualCI(pS->ptr(),"0");
	}

	((QListView *)m_pWidget)->setShowSortIndicator(rd);

	return KVI_ERROR_Success;
}

////////////////////////////////////////////////////////////////////////////////
//////		TREEVIEWITEM
////////////////////////////////////////////////////////////////////////////////

/*
	@class: treeviewitem
	@short:
		A tree view item
	@inherits:
		[class]object[/class]
	@functions:
		!fn: $setText(&lt;idx&gt;,&lt;text&gt;)
		Sets this item's column &lt;idx&gt; text to &lt;text&gt;.
		&lt;idx&gt; is a zero-based index.
		The text will be obviously visible only if
		the column &lt;idx&gt; has been added to the [class]treeview[/class]
		that this item belongs to.

		!fn: $setPixmap(&lt;idx&gt;,&lt;image path&gt;)
		Sets this item's column &lt;idx&gt; pixmap to the specified image.
		If the image can't be loaded or the &lt;image path&gt; is an empty
		string, any previous pixmap for this column is unset.

		!fn: $text(&lt;idx&gt;)
		Returns the text in the column &lt;idx&gt; (zero based index!)

		!fn: $setOpen(&lt;boolean&gt;)
		If &lt;boolean&gt; is non 0 , opens this item showing its children tree,
		otherwise closes this item.

		!fn: $isSelected()
		Returns 1 if this item is currently selected, 0 otherwise

		!fn: $firstChild()
		Returns the id of the first child treeviewitem, or an empty string if this item
		has no children items.

		!fn: $nextSibling()
		Returns the id of the next item that is a child of the parent of this item.

	@description:
		The treeviewitem object class is to be used with the
		[class]treeview[/class] class.<br> Basically, you create
		a treeview widget and then add a tree of treeviewitems to it.<br>
		The treeviewitem is able to display text and icons in the
		columns of the treeview.<br>
		To add a treeviewitem to a [class]treeview[/class] widget
		simply create the treeviewitem object with [fnc]$new[/fnc].<br>
		If you create the treeviewitem object as child of the treeview,
		it will appear as toplevel item in the tree, otherwise
		you must create it as child of another treviewitem (so it
		will be a children in the item tree too).<br>
		When a [class]treeview[/class] widget is destroyed,
		all the contained items are destroyed too.<br>
		To remove a treeviewitem from a [class]treeview[/class]
		just [cmd]destroy[/cmd] the treeviewitem object.
		You can naviage the tree of items by using the $firstChild() and $nextSibling()
		functions.<br>
	@examples:
		See the example for class [class]treeview[/class]
	@seealso:
		class [class]object[/class]
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/


KviScriptListViewItem::~KviScriptListViewItem()
{
	if(m_pScriptObjectClass->m_pItem)
	{
		// someone else deleted us!
		m_pScriptObjectClass->m_pItem = 0;
		m_pScriptObjectClass->die(false);
	}
}

void KviScriptTreeViewItem::
	initializeClassDefinition(KviScriptObjectClassDefinition * d)
{
	d->addBuiltinFunction("setText",(scriptObjectFunction)&KviScriptTreeViewItem::builtinFunction_SETTEXT);
	d->addBuiltinFunction("setPixmap",(scriptObjectFunction)&KviScriptTreeViewItem::builtinFunction_SETPIXMAP);
	d->addBuiltinFunction("text",(scriptObjectFunction)&KviScriptTreeViewItem::builtinFunction_TEXT);
	d->addBuiltinFunction("setOpen",(scriptObjectFunction)&KviScriptTreeViewItem::builtinFunction_SETOPEN);
	d->addBuiltinFunction("isSelected",(scriptObjectFunction)&KviScriptTreeViewItem::builtinFunction_ISSELECTED);
	d->addBuiltinFunction("firstChild",(scriptObjectFunction)&KviScriptTreeViewItem::builtinFunction_FIRSTCHILD);
	d->addBuiltinFunction("nextSibling",(scriptObjectFunction)&KviScriptTreeViewItem::builtinFunction_NEXTSIBLING);
}

KviScriptTreeViewItem::KviScriptTreeViewItem(KviScriptObjectController * cntrl, 
	KviScriptObject * p, const char * name, KviScriptObjectClassDefinition *
	pDef ):KviScriptObject(cntrl, p, name, pDef)
{
	m_pItem = 0;
}

KviScriptTreeViewItem::~KviScriptTreeViewItem()
{
	if(m_pItem)
	{
		KviScriptListViewItem * it = m_pItem;
		m_pItem = 0;
		delete it;
	}
}

bool KviScriptTreeViewItem::init(QList<KviStr> * params)
{
	QString s1,s2,s3,s4;

//	static char * empty = "";

	KviStr * pS = params ? params->first() : 0;
	if(pS)s1 = pS->ptr();
	else s1 = "?";

	pS = params && pS ? params->next() : 0;
	if(pS)s2 = pS->ptr();
	else s2 = QString::null;

	pS = params && pS ? params->next() : 0;
	if(pS)s3 = pS->ptr();
	else s3 = QString::null;

	pS = params && pS ? params->next() : 0;
	if(pS)s4 = pS->ptr();
	else s4 = QString::null;

	if(parent())
	{
		if(parent()->inherits("KviScriptTreeView"))
			m_pItem = new KviScriptListViewItem(this,((QListView *)(((KviScriptWidget*)parent())->m_pWidget)),s1,s2,s3,s4);
		else if(parent()->inherits("KviScriptTreeViewItem"))
			m_pItem = new KviScriptListViewItem(this,((KviScriptListViewItem *)(((KviScriptTreeViewItem*)parent())->m_pItem)),s1,s2,s3,s4);
	}
	
	if(!m_pItem)return false;

	return true;
}

int KviScriptTreeViewItem::builtinFunction_FIRSTCHILD(QList<KviStr> *p,KviStr &b)
{
	KviScriptListViewItem * it = (KviScriptListViewItem *)(m_pItem->firstChild());

	if(it)b.append(it->m_pScriptObjectClass->id());

	return KVI_ERROR_Success;
}


int KviScriptTreeViewItem::builtinFunction_NEXTSIBLING(QList<KviStr> *p,KviStr &b)
{
	KviScriptListViewItem * it = (KviScriptListViewItem *)(m_pItem->nextSibling());

	if(it)b.append(it->m_pScriptObjectClass->id());

	return KVI_ERROR_Success;
}

int KviScriptTreeViewItem::builtinFunction_SETTEXT(QList<KviStr> * p, KviStr &)
{
	if(p)
	{
		if(KviStr * pS = p->first())
		{
			bool bOk;
			int idx = pS->toInt(&bOk);
			if(bOk)
			{
				pS = p->next();
				m_pItem->setText(idx,pS ? pS->ptr() : "");
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}


int KviScriptTreeViewItem::builtinFunction_SETPIXMAP(QList<KviStr> * p, KviStr &)
{
	if(p)
	{
		if(KviStr * pS = p->first())
		{
			bool bOk;
			int idx = pS->toInt(&bOk);
			if(bOk)
			{
				pS = p->next();
				if(pS)
				{
					KviStr path;
					if(g_pApp->findImage(path,pS->ptr()))
					{
						m_pItem->setPixmap(idx,QPixmap(path.ptr()));
						return KVI_ERROR_Success;
					}
				}
				m_pItem->setPixmap(idx,QPixmap());
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTreeViewItem::builtinFunction_TEXT(QList<KviStr> *p,KviStr &buffer)
{
	if(p)
	{
		if(KviStr * pS = p->first())
		{
			bool bOk;
			int idx = pS->toInt(&bOk);
			if(bOk && (idx >= 0))
			{
				buffer.append(m_pItem->text(idx));
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTreeViewItem::builtinFunction_SETOPEN(QList<KviStr> *p,KviStr &)
{
	if(p)
	{
		if(KviStr * pS = p->first())
		{
			bool bOk;
			int idx = pS->toInt(&bOk);
			if(bOk)
			{
				m_pItem->setOpen(idx);
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTreeViewItem::builtinFunction_ISSELECTED(QList<KviStr> *,KviStr &buffer)
{
	buffer.append(KviStr::Format,"%d",m_pItem->isSelected() ? 0 : 1);
	return KVI_ERROR_Success;
}

#include "m_kvi_scriptwidget.moc"
