//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CHECK_RANGE_

#include "kvi_scriptsocket.h"
#include "kvi_error.h"
#include "kvi_netutils.h"
#include "kvi_malloc.h"
#include "kvi_debug.h"
#include "kvi_memmove.h"

#include <sys/socket.h>
#include <netdb.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>

/*
	@class: socket
	@short:
		A socket device.
	@inherits:
		<a href="class_object.kvihelp">object</a>
	@functions:
		!fn: $setDomain(&lt;domain&gt;)
		Set the domain of the socket: valid values for &lt;domain&gt; are:<br>
		inet and unix. (inet is the default for new socket objects)<br>

		!fn: $domain()
		Returns the name of the domain of the socket.

		!fn: $setType(&lt;type&gt;)
		Set the type of the socket: valid values for &lt;type&gt; are:<br>
		stream and raw. (stream is the default for new socket objects)<br>

		!fn: $type()
		Returns the type of the socket

		!fn: $setProtocol(&lt;protocol&gt;)
		Sets the protocol of the socket .<br>
		The &lt;protocol&gt; argument is the protocol name to be matched
		in the /etc/protocols file (see man getprotobyname :).<br>
		(The default protocol is "ip" : that should work in most cases).<br>

		!fn: $protocol()
		Returns the protocol of the socket

		!fn: $lastError()
		Returns the last error string

		!fn: $setMode(&lt;data_mode&gt;)
		Sets the data mode of the socket.<br>
		The valid modes are:<br>
		- Ascii : for text based protocols (pop3 , irc...) , messages separated by combinations of CR and LF characters<br>
		- Hex : hexadecimal endoded , for binary protocols. (see <a href="s_hextostr.kvihelp">$HexToStr()</a> and <a href="s_strtohex.kvihelp">$StrToHex()</a>)

		!fn: $mode()
		Returns the data mode of the socket.

		!fn: $write(&lt;string&gt;)
		Writes data to the socket and returns the length of the written buffer (that may also be zero!).<br>
		If a serious error occures , -1 is returned and the OnDisconnect event is triggered.<br>
		Please note that this function does NOT implicitly write a trailing CR or LF:
		you must add it by yourself (All text based protocols require a messages terminator).<br>
		Please note that this function can be called either in Ascii or Hex mode.<br>

		!fn: $writeHex(&lt;hexstring&gt;)
		Writes data (transformed to binary form) to the socket and returns the length of the written buffer (that may also be zero!).<br>
		If an error occures , the OnDisconnect event is fired and -1 is returned.<br>
		The hexstring is a string of pairs of hexadecimal digits that encode the bytes to be sent.<br>
		This is useful when you want send null or unprintable characters.<br>
		For example $writeHex(68656C6C6F00) writes a null terminated "hello" string to the socket.<br>
		There is no other way to write the null character.<br>
		(see also <a href="s_hextostr.kvihelp">$HexToStr()</a> and <a href="s_strtohex.kvihelp">$StrToHex()</a>)<br>
		Please note that this function can be called either in Ascii or Hex mode.<br>

		!fn: $close()
		Closes the connection. The OnDisconnect event is NOT fired.<br>

		!fn: $state()
		Returns the state of the socket : ready , connecting , listening or connected.<br>

		!fn: $connect(&lt;ipaddress&gt;,&lt;port&gt;)
		Attempts a connection to the specified host on a specified port.<br>
		This function returns 1 if the creation of the socket and the "connect" system call
		have been succesfull. Please note that this does NOT mean that you're connected:
		the connection attempt has been only started.<br>
		From that moment you can receive two events:<br>
		OnConnectFailed , if the connection fails for some reason (for example the specified
		host is unreachable) or OnConnect when the connection has been estabilished.<br>
		If this function returns 0 you can obtain the error string by calling $lastError().<br>

		!fn: $listen(&lt;local_interface_address&gt;,&lt;port&gt;)
		Starts listening for incoming connectons on the specified port.<br>
		&lt;local_interface_address&gt; can be a valid ip address of a local
		network interface or the string "any" (or "default") that will cause to listen
		on all available interfaces.<br>
		This function returns 1 if the socket creation and the system "listen" call
		have been succesfull. Please nothe that this does NOT mean that you're connected
		to some host: the listening has just been started.<br>
		When a connection arrives the "OnIncomingConnection" event will be triggered.<br>
		If this function returns 0 you can obtain the error string by calling $lastError().<br>
		Please note that not all ports are available to be "listened":
		especially , if you're not the root user , you may have no access to ports below 1024.<br>
		This function may fail also if some other "entity" is listening on the specified port.<br>
		Passing 0 as the &lt;port&gt; argument causes the kernel to select the first available port for you,
		(....at least on Linux).<br>

		!fn: $host()
		Returns the ip address of the currently connected remote host

		!fn: $port()
		Returns the remote end port of the current connection

		!fn: $accept(&lt;listening_socket_id&gt;)
		This function accepts an incoming connection that is pending on the specified listening socket.<br>
		Returns 1 on success , 0 otherwise.<br>
		Please note that usually the &lt;listening_socket_id&gt; is NOT *this* socket.<br>
		Commonly , you will create a listening socket and override its OnIncomingConnection event.<br>
		There you will create a new socket object to serve the incoming connection and call its
		$accept member function passing the listening socket id.<br>
		See the OnIncomingConnection example.<br>
		The most common failure causes are:<br>
		- The &lt;listening_socket_id&gt; is not an object id at all (the object can't be found)<br>
		- The &lt;listening_socket_id&gt; object id does not refer to a class that inherits socket<br>
		- The &lt;listening_socket_id&gt; socket has no pending incoming connections.<br>
		You can retrieve the failure reason by calling $lastError().<br>
		After a succesfull call of this function , the caller socket is ready to communicate:<br>
		You can retrieve the remote host and port data with the $host and $port functions.<br>
		The OnConnect event is not triggered (not useful).<br>

		!fn: $localhost()
		Returns the IP address of the local host.<br>
		This identifier returns valid values only when the socket object is in connected state.<br>

	@events:
		!ev: OnConnect($1 = HostIp,$2 = Port)
		Triggered just after a succesfull connection has been estabilished
		The default implementation of this event echoes "Connected to $1 on port $2" to the active window

		!ev: OnIncomingConnection()
		Triggered by a listening socket when a connection request arrives.<br>
		To accept the incoming connection you should create a new socket (or derived class)
		object and call accept(<a href="s_this.kvihelp">$this</a>)<br>
		<example>
			event(OnIncomingConnection)
			{
			&nbsp;&nbsp;%sock = <a href="s_new.kvihelp">$new</a>(socket,<a href="s_this.kvihelp">$this</a>,incoming)
			&nbsp;&nbsp;if(!%sock->$accept(<a href="s_this.kvihelp">$this</a>))
			&nbsp;&nbsp;{
			&nbsp;&nbsp;&nbsp;&nbsp;# Ops...something went really wrong
			&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> Accept failed : %sock->$lastError()
			&nbsp;&nbsp;&nbsp;&nbsp;<a href="destroy.kvihelp">destroy</a> %sock;
			&nbsp;&nbsp;}
			}
		</example>
		If you do not accept in this way the incoming connection in this event,
		it will be automatically discarded (the socket will be closed and the remote end will
		see some error such as "Connection reset by peer").<br>
		You can also accept the incoming connection with <a href="s_this.kvihelp">$this</a> socket
		(the listening one) , but then you will obviously stop listening.<br>
		<example>
			event(OnIncomingConnection)
			{
			&nbsp;&nbsp;if(!<a href="s_this.kvihelp">$this</a>->$accept(<a href="s_this.kvihelp">$this</a>))
			&nbsp;&nbsp;{
			&nbsp;&nbsp;&nbsp;&nbsp;# Ops...something went really wrong
			&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> Accept failed : <a href="s_this.kvihelp">$this</a>->$lastError()
			&nbsp;&nbsp;&nbsp;&nbsp;# Still listening
			&nbsp;&nbsp;}
			}
		</example>
		If $accept succeeds in this case , <a href="s_this.kvihelp">$this</a> socket stops listening
		and starts the communication on the accepted connection.<br>
		If $accept fails , <a href="s_this.kvihelp">$this</a> socket will be still listening.
		The default implementation of this event does nothing

		!ev: OnConnectFailed($1- = Error String)
		Triggered when a connection attempt fails
		The default implementation of this event echoes "Connect failed : $this->$lastError()" to the active window

		!ev: OnDisconnect($1- = Error String)
		Triggered when the socket gets disconnected for some user-independent reason.<br>
		It may be caused by a transmission error or the remote host may simply close the connection
		after a timeout...<br>
		The default implementation of this event echoes "Disconnected from $this->$host() : $this->$lastError()" to the active window

		!ev: OnDataReceived($1 = Data length,$2- = Data String)
		The data string is a "normal" text line if the socket is in the Ascii mode ,
		otherwise it is a sequence of hexadecimal-encoded bytes.<br>
		In Ascii mode you will be loosing the leading and trailing whitespace,
		and the unprintable characters. Also , in Ascii mode this event will NOT fire until
		a CR or LF has been readed from the socket (to signal the end of a line).<br>
		The $1 argument is the length of the Data string argument (in characters).<br>
		In Hex mode the $1 argument is the length of the Data string argument (in characters);
		since each character is encoded by two hexadecimal digits , so <a href="s_calc.kvihelp">$calc</a>($1 / 2) 
		will be the effective length in bytes of the received buffer.<br>
		In hex mode this event will be triggered whenever any data arrives.<br>
		If you need to receive such data , use the socket in Hex mode.<br>
		(see <a href="s_hextostr.kvihelp">$HexToStr()</a> and <a href="s_strtohex.kvihelp">$StrToHex()</a>)<br>
		This event has a default event handler (in case that you don't want to bother yourself with
		the <a href="class.kvihelp">CLASS</a> command) that "echoes" the received data to the active window.<br>
		The body of the event is: "echo -w=$activewindow SOCKET $this : Received data :$2-".<br>
		Obviously you can override it, as usual, with <a href="obj_setevent.kvihelp">obj_setevent</a>
		or in a subclass definition.<br>

	@description:
		A socket device object.<br>
		Theoretically it can work in the Unix and Inet domain
		but actually only the Inet (default) domain has been tested.<br>
		Theoretically it can work as Stream or Raw socket but
		actually only the Stream (default) socket implementation has been tested.<br>
		Theoretically it can work over any valid protocol but by now
		only "ip" (0 - default) has been tested.

	@examples:
		Simple code to check for new messages in a FIXED mailbox.<br>
		The connection is made in Ascii mode since unless messages are transmitted
		the pop3 protocol is text based: we will loose no data.<br>
		You can easily optimize this code and convert it to an alias
		that may accept the username, password and the mailserver as parameters.<br>
		This version also wants the real IP address of the mailserver , not the hostname.<br>
		Adding the DNS extension is left to you as exercise :).<br>
		<example>
		# Create a socket object
		%o=<a href="s_new.kvihelp">$new</a>(socket,<a href="s_root.kvihelp">$root</a>,mysock);
		# OnConnect event : Once connected , send USER , PASS and LIST
		# You will probably want to set a different username and password :)
		<a href="obj_setevent.kvihelp">obj_setevent</a>(%o,OnConnect)
		{
		&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> CHECKMAIL : Connected to the mailserver...
		&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>->$write("USER place_here_your_username<a href="s_cr.kvihelp">$cr</a><a href="s_lf.kvihelp">$lf</a>")
		&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>->$write("PASS place_here_your_password<a href="s_cr.kvihelp">$cr</a><a href="s_lf.kvihelp">$lf</a>")
		&nbsp;&nbsp;<a href="s_this.kvihelp">$this</a>->$write("LIST<a href="s_cr.kvihelp">$cr</a><a href="s_lf.kvihelp">$lf</a>")
		}
		# When we get disconnected for some reason kill the object
		<a href="obj_setevent.kvihelp">obj_setevent</a>(%o,OnDisconnect)
		{
		&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> CHECKMAIL : Disconnected ($1-)
		&nbsp;&nbsp;<a href="destroy.kvihelp">destroy</a> <a href="s_this.kvihelp">$this</a>
		}
		# Data received ...here's the clue
		<a href="obj_setevent.kvihelp">obj_setevent</a>(%o,OnDataReceived)
		{
		&nbsp;&nbsp;<a href="switch.kvihelp">switch</a>($2-)
		&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="switch.kvihelp">match</a>("+OK * messages*")
		&nbsp;&nbsp;&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> CHECKMAIL : The mailbox contains <a href="s_strrightfromfirst.kvihelp">$strRightFromFirst</a>("+OK ",$1-)
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<a href="destroy.kvihelp">destroy</a> <a href="s_this.kvihelp">$this</a>
		&nbsp;&nbsp;&nbsp;&nbsp;}
		&nbsp;&nbsp;&nbsp;&nbsp;<a href="switch.kvihelp">match</a>("-ERR *")
		&nbsp;&nbsp;&nbsp;&nbsp;{
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> CHECKMAIL : Received error : $2-
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> CHECKMAIL : Aborting
		&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<a href="destroye.kvihelp">destroy</a> <a href="s_this.kvihelp">$this</a>
		&nbsp;&nbsp;&nbsp;&nbsp;}
		&nbsp;&nbsp;}
		}
		# If the connection fails for some reason , kill the object
		<a href="obj_setevent.kvihelp">obj_setevent</a>(%o,OnConnectFailed)
		{
		&nbsp;&nbsp;<a href="echo.kvihelp">echo</a> CHECKMAIL : Connect failed : ($1-)
		&nbsp;&nbsp;<a href="destroy.kvihelp">destroy</a> <a href="s_this.kvihelp">$this</a>
		}
		# Ok...ready to go... connect to our mailserver
		# You will probably want to change the ip address :)
		<a href="if.kvihelp">if</a>(!%o->$connect(127.0.0.1,110)){ <a href="echo.kvihelp">echo</a> Mail : Connection failed : %o->$lastError(); <a href="destroy.kvihelp">destroy</a> %o; }
		<a href="if.kvihelp">else</a> <a href="echo.kvihelp">echo</a> Checking mail on 127.0.0.1....
		# And now just wait and watch...
 		</example>
	@seealso:
		<a href="syntax_objects.kvihelp">Objects documentation</a>
*/

KviScriptSocket::KviScriptSocket(KviScriptObjectController * cntrl,KviScriptObject * p,const char *name,KviScriptObjectClassDefinition * pDef)
: KviScriptObject(cntrl,p,name,pDef)
{
	m_sock = -1;
	m_domain = PF_INET;
	m_type = SOCK_STREAM;
	m_protocol = 0;
	m_error = KVI_ERROR_Success;
	m_mode = Ascii;
	m_iReadBufLen = 0;
	m_pReadBuffer = 0;
	m_pRsn = 0;
	m_pTmpSn = 0;
	m_state = Ready;
	m_incomingConnectionSock = -1;
	m_szLocalHostIp = "127.0.0.1";
}

KviScriptSocket::~KviScriptSocket()
{
	reset();
}

void KviScriptSocket::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("mode",(scriptObjectFunction)&KviScriptSocket::builtinFunction_MODE);
	d->addBuiltinFunction("type",(scriptObjectFunction)&KviScriptSocket::builtinFunction_TYPE);
	d->addBuiltinFunction("setDomain",(scriptObjectFunction)&KviScriptSocket::builtinFunction_SETDOMAIN);
	d->addBuiltinFunction("setType",(scriptObjectFunction)&KviScriptSocket::builtinFunction_SETTYPE);
	d->addBuiltinFunction("setMode",(scriptObjectFunction)&KviScriptSocket::builtinFunction_SETMODE);
	d->addBuiltinFunction("setProtocol",(scriptObjectFunction)&KviScriptSocket::builtinFunction_SETPROTOCOL);
	d->addBuiltinFunction("write",(scriptObjectFunction)&KviScriptSocket::builtinFunction_WRITE);
	d->addBuiltinFunction("writeHex",(scriptObjectFunction)&KviScriptSocket::builtinFunction_WRITEHEX);
	d->addBuiltinFunction("close",(scriptObjectFunction)&KviScriptSocket::builtinFunction_CLOSE);
	d->addBuiltinFunction("protocol",(scriptObjectFunction)&KviScriptSocket::builtinFunction_PROTOCOL);
	d->addBuiltinFunction("host",(scriptObjectFunction)&KviScriptSocket::builtinFunction_HOST);
	d->addBuiltinFunction("port",(scriptObjectFunction)&KviScriptSocket::builtinFunction_PORT);
	d->addBuiltinFunction("state",(scriptObjectFunction)&KviScriptSocket::builtinFunction_STATE);
	d->addBuiltinFunction("domain",(scriptObjectFunction)&KviScriptSocket::builtinFunction_DOMAIN);
	d->addBuiltinFunction("connect",(scriptObjectFunction)&KviScriptSocket::builtinFunction_CONNECT);
	d->addBuiltinFunction("listen",(scriptObjectFunction)&KviScriptSocket::builtinFunction_LISTEN);
	d->addBuiltinFunction("lastError",(scriptObjectFunction)&KviScriptSocket::builtinFunction_LASTERROR);
	d->addBuiltinFunction("accept",(scriptObjectFunction)&KviScriptSocket::builtinFunction_ACCEPT);
	d->addBuiltinFunction("localhost",(scriptObjectFunction)&KviScriptSocket::builtinFunction_LOCALHOST);

	KviScriptEventStruct * s = new KviScriptEventStruct;
	s->szName = "OnDataReceived";
	s->szBuffer = "echo -w=$activewindow SOCKET $this : Received data :$2-";
	d->addDefaultEvent(s);
	s = new KviScriptEventStruct;
	s->szName = "OnDisconnect";
	s->szBuffer = "echo -w=$activewindow SOCKET $this : Disconnected from $this->$host() : $this->$lastError()";
	d->addDefaultEvent(s);
	s = new KviScriptEventStruct;
	s->szName = "OnConnect";
	s->szBuffer = "echo -w=$activewindow SOCKET $this : Connected to $1 on port $2";
	d->addDefaultEvent(s);
	s = new KviScriptEventStruct;
	s->szName = "OnConnectFailed";
	s->szBuffer = "echo -w=$activewindow SOCKET $this : Connect failed : $this->$lastError()";
	d->addDefaultEvent(s);
}

int KviScriptSocket::builtinFunction_MODE(QList<KviStr> * params,KviStr &buffer)
{
	if(m_mode == Ascii)buffer.append("ascii");
	else buffer.append("hex");
	return KVI_ERROR_Success;
}

int KviScriptSocket::builtinFunction_LOCALHOST(QList<KviStr> * params,KviStr &buffer)
{
	buffer.append(m_szLocalHostIp);
	return KVI_ERROR_Success;
}

int KviScriptSocket::builtinFunction_SETDOMAIN(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr *pDomain = params->first();
		if(pDomain)
		{
			if(kvi_strEqualCI(pDomain->ptr(),"inet"))m_domain = PF_INET;
			else if(kvi_strEqualCI(pDomain->ptr(),"unix"))m_domain = PF_UNIX;
			else return KVI_ERROR_InvalidParameter;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSocket::builtinFunction_SETTYPE(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr *pType = params->first();
		if(pType)
		{
			if(kvi_strEqualCI(pType->ptr(),"stream"))m_type = SOCK_STREAM;
			else if(kvi_strEqualCI(pType->ptr(),"raw"))m_type = SOCK_RAW;
			else return KVI_ERROR_InvalidParameter;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSocket::builtinFunction_SETMODE(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr *pType = params->first();
		if(pType)
		{
			if(kvi_strEqualCI(pType->ptr(),"ascii"))
			{
				if(m_mode != Ascii)
				{
					m_mode = Ascii;
					m_iReadBufLen = 0;
					if(m_pReadBuffer)kvi_free(m_pReadBuffer);
					m_pReadBuffer = 0;
				}
			} else if(kvi_strEqualCI(pType->ptr(),"hex"))
			{
				if(m_mode != Hex)
				{
					m_mode = Hex;
					m_iReadBufLen = 0;
					if(m_pReadBuffer)kvi_free(m_pReadBuffer);
					m_pReadBuffer = 0;
				}
			}
			else return KVI_ERROR_InvalidParameter;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSocket::builtinFunction_SETPROTOCOL(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr *pProto = params->first();
		if(pProto)
		{
			struct protoent * p = getprotobyname(pProto->ptr());
			if(p)m_protocol = p->p_proto;
			else return KVI_ERROR_InvalidParameter;
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSocket::builtinFunction_WRITE(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr *pData = params->first();
		if(pData)
		{
			KviStr tmp(KviStr::Format,"%d",writeData(pData->ptr(),pData->len()));
			buffer.append(tmp.ptr());
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}


int KviScriptSocket::builtinFunction_WRITEHEX(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr *pData = params->first();
		if(pData)
		{
			char * buf = 0;
			int len = pData->hexToBuffer(&buf);
			KviStr tmp(KviStr::Format,"%d",writeData(buf,len));
			if(buf)kvi_free(buf);
			buffer.append(tmp.ptr());
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSocket::builtinFunction_CLOSE(QList<KviStr> * params,KviStr &buffer)
{
	if(m_sock == -1)buffer.append('0');
	else {
		reset();
		buffer.append('1');
	}
	return KVI_ERROR_Success;
}

int KviScriptSocket::builtinFunction_PROTOCOL(QList<KviStr> * params,KviStr &buffer)
{
	struct protoent * p = getprotobynumber(m_protocol);
	if(p)buffer.append(p->p_name);
	else {
		KviStr num;
		num.setNum(m_protocol);
		buffer.append(num.ptr());
	}
	return KVI_ERROR_Success;
}

int KviScriptSocket::builtinFunction_TYPE(QList<KviStr> * params,KviStr &buffer)
{
	if(m_type == SOCK_STREAM)buffer.append("stream");
	else buffer.append("raw");
	return KVI_ERROR_Success;
}

int KviScriptSocket::builtinFunction_HOST(QList<KviStr> * params,KviStr &buffer)
{
	if(m_state == Connected)buffer.append(m_szHostIp);
	return KVI_ERROR_Success;
}

int KviScriptSocket::builtinFunction_PORT(QList<KviStr> * params,KviStr &buffer)
{
	if(m_state == Connected)buffer.append(m_szPort);
	return KVI_ERROR_Success;
}

int KviScriptSocket::builtinFunction_STATE(QList<KviStr> * params,KviStr &buffer)
{
	if(m_state == Connected)buffer.append("connected");
	else if(m_state == Connecting)buffer.append("connecting");
	else if(m_state == Listening)buffer.append("listening");
	else buffer.append("ready");
	return KVI_ERROR_Success;
}

int KviScriptSocket::builtinFunction_DOMAIN(QList<KviStr> * params,KviStr &buffer)
{
	if(m_domain == PF_INET)buffer.append("inet");
	else buffer.append("unix");
	return KVI_ERROR_Success;
}


int KviScriptSocket::builtinFunction_CONNECT(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr *pHost = params->first();
		if(pHost)
		{
			KviStr *pPort = params->next();
			if(pPort)
			{
				buffer.append(doConnect(pHost,pPort) ? '1' : '0');
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSocket::builtinFunction_LISTEN(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr *pHost = params->first();
		if(pHost)
		{
			KviStr *pPort = params->next();
			if(pPort)
			{
				buffer.append(doListen(pHost,pPort) ? '1' : '0');
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSocket::builtinFunction_LASTERROR(QList<KviStr> * params,KviStr &buffer)
{
	buffer.append(kvi_getErrorString(m_error));
	return KVI_ERROR_Success;
}

int KviScriptSocket::builtinFunction_ACCEPT(QList<KviStr> * params,KviStr &buffer)
{
	if(params)
	{
		KviStr *pObId = params->first();
		if(pObId)
		{
			buffer.append(doAccept(pObId->ptr()) ? '1' : '0');
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}


bool KviScriptSocket::doListen(KviStr *szLocalAddr,KviStr *szPort)
{
	struct in_addr     inAddress;
	struct sockaddr_in hostSockAddr;

	if(m_sock != -1){ m_error = KVI_ERROR_AnotherConnectionInProgress; return false; }

	if(!kvi_stringIpToBinaryIp(szLocalAddr->ptr(),&inAddress))
	{
		if(kvi_strEqualCI("default",szLocalAddr->ptr()) || kvi_strEqualCI("any",szLocalAddr->ptr()))hostSockAddr.sin_addr.s_addr = INADDR_ANY;
		else { m_error = KVI_ERROR_InvalidIpAddress; return false; }
	} else hostSockAddr.sin_addr = inAddress;

	bool bOk = false;
	unsigned short int port = szPort->toUInt(&bOk);
	if(!bOk){ m_error = KVI_ERROR_InvalidPort; return false; }

	hostSockAddr.sin_family = m_domain; //AF_INET == PF_INET && AF_UNIX == PF_UNIX
	hostSockAddr.sin_port   = htons(port);

	m_sock = ::socket(m_domain,m_type,m_protocol);

	if(m_sock < 0){ m_error = KVI_ERROR_SocketCreationFailed; return false; }

	if(fcntl(m_sock, F_SETFL, O_NONBLOCK) < 0){ m_error = KVI_ERROR_AsyncSocketFailed; return false; }

	int theError;
	if((theError = bind(m_sock,(struct sockaddr *)&hostSockAddr,sizeof(hostSockAddr))) < 0)
	{ setErrorFromSystemError(theError); reset(); return false; }
	if((theError = listen(m_sock,1)) < 0)
	{ setErrorFromSystemError(theError); reset(); return false; }

	// and setup the READ notifier...
	m_pTmpSn = new QSocketNotifier(m_sock,QSocketNotifier::Read);

	QObject::connect(m_pTmpSn,SIGNAL(activated(int)),this,SLOT(listenNotifierFired(int)));

	m_pTmpSn->setEnabled(true);

	m_szHostIp = "";
	m_szPort = "";
	m_error = KVI_ERROR_Success;
	m_state = Listening;

	return true;
}

void KviScriptSocket::listenNotifierFired(int)
{
	struct sockaddr_in connectedAddr;

	_this_should_be_socklen_t iSize = sizeof(connectedAddr);
	m_incomingConnectionSock = accept(m_sock,(struct sockaddr*)&connectedAddr,&iSize);
	if(m_incomingConnectionSock == -1)debug("warning : accept returns -1...waiting for the next accept call");
	else {
		// Connected
		m_szIncomingConnectionPort.setNum(ntohs(connectedAddr.sin_port));
		kvi_binaryIpToString(connectedAddr.sin_addr,m_szIncomingConnectionHostIp);

		KviStr parms;
		triggerEvent("OnIncomingConnection",parms);

		if(m_incomingConnectionSock != -1)
		{
			// Not accepted
			close(m_incomingConnectionSock);
			m_incomingConnectionSock = -1;
			m_szIncomingConnectionPort = "";
			m_szIncomingConnectionHostIp = "";
		}
	}
}

bool KviScriptSocket::doAccept(const char *sockObjectId)
{
	KviScriptObject * o = (KviScriptObject *)controller()->findObjectById(sockObjectId);

	if(!o){ m_error = KVI_ERROR_ObjectNotFound; return false; }
	if(!o->inherits("KviScriptSocket")){ m_error = KVI_ERROR_ObjectIsNotASocket; return false; }

	KviScriptSocket * s = (KviScriptSocket *)o;

	if(!s>hasPendingConnection()){ m_error = KVI_ERROR_NoConnectionToAccept; return false; }

	if(m_state != Ready)
	{
		reset(); // if o == this it will work!...reset() will not clear the incomingConnectionSock
	}

	m_sock = s->m_incomingConnectionSock;
	s->m_incomingConnectionSock = -1;
	m_szHostIp = s->m_szIncomingConnectionHostIp;
	s->m_szIncomingConnectionHostIp = "";
	m_szPort = s->m_szIncomingConnectionPort;
	s->m_szIncomingConnectionPort = "";
	m_state = Connected;
	m_error = KVI_ERROR_Success;

	m_pRsn = new QSocketNotifier(m_sock,QSocketNotifier::Read);
	QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(receivedData(int)));
	m_pRsn->setEnabled(true);

	setLocalHostIp();

	return true;
}

void KviScriptSocket::setLocalHostIp()
{
	if(m_state == Connected)
	{
		struct sockaddr_in name;
		_this_should_be_socklen_t len = sizeof(name);
		if(getsockname(m_sock, (struct sockaddr *)&name,&len) >= 0)
		{
			if(kvi_binaryIpToString(name.sin_addr,m_szLocalHostIp))return;
		}
	}
	m_szLocalHostIp = "127.0.0.1";
}

bool KviScriptSocket::doConnect(KviStr * szHost,KviStr *szPort)
{
	struct in_addr     inAddress;
	struct sockaddr_in hostSockAddr;

	if(m_sock != -1){ m_error = KVI_ERROR_AnotherConnectionInProgress; return false; }

	if(!kvi_stringIpToBinaryIp(szHost->ptr(),&inAddress)){ m_error = KVI_ERROR_InvalidIpAddress; return false; }

	bool bOk = false;
	unsigned short int port = szPort->toUInt(&bOk);
	if(!bOk){ m_error = KVI_ERROR_InvalidPort; return false; }

	hostSockAddr.sin_family = m_domain; //AF_INET == PF_INET && AF_UNIX == PF_UNIX
	hostSockAddr.sin_port   = htons(port);
	hostSockAddr.sin_addr   = inAddress;

	// Let's go...
	m_sock = ::socket(m_domain,m_type,m_protocol);

	if(m_sock < 0){ m_error = KVI_ERROR_SocketCreationFailed; return false; }

	if(fcntl(m_sock, F_SETFL, O_NONBLOCK) < 0){ m_error = KVI_ERROR_AsyncSocketFailed; return false; }

	if(::connect(m_sock,(struct sockaddr*)(&hostSockAddr),sizeof(hostSockAddr))<0){
		if(errno != EINPROGRESS){
			int sockError=errno;
			if(sockError==0){
				_this_should_be_socklen_t iSize=sizeof(int);
				if(getsockopt(m_sock,SOL_SOCKET,SO_ERROR,(void *)&sockError,&iSize)<0)sockError=0;
			}
			if(sockError > 0)setErrorFromSystemError(sockError);
			else m_error = KVI_ERROR_UnknownError; //Error 0 ?
			return false;
		}
	}

	// and setup the WRITE notifier...
	m_pTmpSn = new QSocketNotifier(m_sock,QSocketNotifier::Write);

	QObject::connect(m_pTmpSn,SIGNAL(activated(int)),this,SLOT(writeNotifierFired(int)));

	m_pTmpSn->setEnabled(true);
	// set the timer
	m_szHostIp = szHost->ptr();
	m_szPort = szPort->ptr();
	m_error = KVI_ERROR_Success;
	m_state = Connecting;
	return true;
}

void KviScriptSocket::writeNotifierFired(int)
{
	int sockError;
	_this_should_be_socklen_t iSize=sizeof(int);
	if(getsockopt(m_sock,SOL_SOCKET,SO_ERROR,(void *)&sockError,&iSize)<0)sockError = -1;
	if(sockError != 0){
		//failed
		if(sockError > 0)setErrorFromSystemError(sockError);
		else m_error = KVI_ERROR_UnknownError; //Error 0 ?
		KviStr parms = kvi_getErrorString(m_error);
		triggerEvent("OnConnectFailed",parms);
		reset();
	} else {
		//Succesfully connected...
		m_pRsn = new QSocketNotifier(m_sock,QSocketNotifier::Read);
		// normal irc connection
		KviStr parms(KviStr::Format,"%s %s",m_szHostIp.ptr(),m_szPort.ptr());
		triggerEvent("OnConnect",parms);
		if(m_state != Connecting)return; //Killed ?

		QObject::connect(m_pRsn,SIGNAL(activated(int)),this,SLOT(receivedData(int)));
		m_pRsn->setEnabled(true);
		delete m_pTmpSn;
		m_pTmpSn = 0;
		m_state = Connected;
		setLocalHostIp();
	}

}

void KviScriptSocket::reset()
{
	if(m_sock != -1)
	{
		::close(m_sock);
		m_sock = -1;
	}
	m_szHostIp = "";
	m_szPort = "";
	if(m_pReadBuffer)
	{
		kvi_free(m_pReadBuffer);
		m_pReadBuffer = 0;
	}
	m_iReadBufLen = 0;
	if(m_pRsn)
	{
		delete m_pRsn;
		m_pRsn = 0;
	}
	if(m_pTmpSn)
	{
		delete m_pTmpSn;
		m_pTmpSn = 0;
	}
	m_state = Ready;
}

void KviScriptSocket::setErrorFromSystemError(int errorNum)
{
	switch(errorNum){
		case EBADF:        m_error = KVI_ERROR_BadFileDescriptor;     break;
		case EFAULT:       m_error = KVI_ERROR_OutOfAddressSpace;     break;
		case ECONNREFUSED: m_error = KVI_ERROR_ConnectionRefused;     break;
		case ENOTSOCK:     m_error = KVI_ERROR_KernelNetworkingPanic; break;
		case ETIMEDOUT:    m_error = KVI_ERROR_ConnectionTimedOut;    break;
		case ENETUNREACH:  m_error = KVI_ERROR_NetworkUnreachable;    break;
		case EPIPE:        m_error = KVI_ERROR_BrokenPipe;            break;
		// Unhandled error...pass errno to the strerror function
		default:           m_error = -errorNum;                       break;
	}
}

void KviScriptSocket::handleInvalidSocketRead(int readedLength)
{
	if(readedLength==0){
		m_error = KVI_ERROR_RemoteEndClosedConnection;
		KviStr parms = kvi_getErrorString(m_error);
		triggerEvent("OnDisconnect",parms);
		reset();
	} else {
		//check for transmission errors
		if((errno != EAGAIN) && (errno != EINTR)){
			if(errno > 0)setErrorFromSystemError(errno);
			else m_error = KVI_ERROR_RemoteEndClosedConnection;
			KviStr parms = kvi_getErrorString(m_error);
			triggerEvent("OnDisconnect",parms);
			reset();
		} //else transient error...wait again...
	}
}

void KviScriptSocket::receivedData(int fd)
{
	//read data

	char buffer[1025];
	int readLength = read(m_sock,buffer,1024);

	if(readLength <= 0){
		handleInvalidSocketRead(readLength);
		return;
	}

	if(m_mode == Hex)
	{
		KviStr parms;
		KviStr tmp(KviStr::Format,"%d ",readLength * 2);
		parms.bufferToHex(buffer,readLength);
		parms.prepend(tmp.ptr());
		m_pRsn->setEnabled(false);
		triggerEvent("OnDataReceived",parms);
		// check if we were closed() by the meantins
		if(m_state != Connected)return; //closed
		m_pRsn->setEnabled(true);
	} else {
		//terminate our buffer
		(*(buffer+readLength))='\0';

		register char *p=buffer;
		char *beginOfCurData = buffer;
		int   bufLen = 0;
		char *messageBuffer = (char *)kvi_malloc(1);
		//Shut up the socket notifier
		//in case that we enter in a local loop somewhere
		//while processing data...

		m_pRsn->setEnabled(false);

		while(*p){
			if((*p == '\r' )||(*p == '\n')){
				//found a CR or LF...
				//prepare a message buffer
				bufLen = p - beginOfCurData;
				//check for previous unterminated data
				KviStr parms;

				if(m_iReadBufLen > 0){

					__range_valid(m_pReadBuffer);
					messageBuffer = (char *)kvi_realloc(messageBuffer,bufLen + m_iReadBufLen + 1);
					kvi_memmove(messageBuffer,m_pReadBuffer,m_iReadBufLen);
					kvi_memmove((void *)(messageBuffer + m_iReadBufLen),beginOfCurData,bufLen);
					*(messageBuffer + bufLen + m_iReadBufLen) = '\0';
					kvi_free(m_pReadBuffer);
					m_pReadBuffer = 0;

					parms.sprintf("%d %s",bufLen + m_iReadBufLen,messageBuffer);
					m_iReadBufLen = 0;
				} else {

					__range_invalid(m_pReadBuffer);
					messageBuffer = (char *)kvi_realloc(messageBuffer,bufLen + 1);
					kvi_memmove(messageBuffer,beginOfCurData,bufLen);
					*(messageBuffer + bufLen) = '\0';
					parms.sprintf("%d %s",bufLen,messageBuffer);
				}

				triggerEvent("OnDataReceived",parms);
				// check if we were closed() by the meantins
				if(m_state != Connected)
				{
					kvi_free(messageBuffer);
					return;
				}

				while(*p && ((*p=='\r')||(*p=='\n')) )p++;
				beginOfCurData = p;
			} else p++;

			if(inDelayedDestroy())
			{
				// already a zombie! (destroyed while processing data!)
				kvi_free(messageBuffer);
				return;
			}
		}
		//now *p == '\0'
		//beginOfCurData points to '\0' if we have
		//no more stuff to parse , or points to something
		//different than '\r' or '\n'...
		if(*beginOfCurData){
			//Have remaining data...in the local buffer
			bufLen = p - beginOfCurData;
			if(m_iReadBufLen > 0){
				//and there was more stuff saved... (really slow connection)
				__range_valid(m_pReadBuffer);
				m_pReadBuffer =(char *)kvi_realloc(m_pReadBuffer,m_iReadBufLen + bufLen);
				kvi_memmove((void *)(m_pReadBuffer+m_iReadBufLen),beginOfCurData,bufLen);
				m_iReadBufLen += bufLen;
			} else {
				//
				__range_invalid(m_pReadBuffer);
				m_iReadBufLen = bufLen;
				m_pReadBuffer =(char *)kvi_malloc(m_iReadBufLen);
				kvi_memmove(m_pReadBuffer,beginOfCurData,m_iReadBufLen);
			}
		}

		kvi_free(messageBuffer);
		//ready to receive
		m_pRsn->setEnabled(true);
	}
}


int KviScriptSocket::writeData(char * buffer,int len)
{
	if(m_sock == -1)
	{
		m_error = KVI_ERROR_NotConnected;
		return -1;
	}

	int result = write(m_sock,buffer,len);
	if(result >= len)return len;
	else {
		// Oops...error ?
		if((errno == EAGAIN)||(errno == EINTR))return 0;
		{
			// Disconnected... :(
			setErrorFromSystemError(errno);
			KviStr parms = kvi_getErrorString(m_error);
			triggerEvent("OnDisconnect",parms);
			reset();
			return -1;
		}
	}
}
#include "m_kvi_scriptsocket.moc"
