//
//   File : kvi_scripteditor.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_scripteditor.cpp)
//   Last major modification : Sun Mar 28 1999 16:12:41 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_SCRIPTEDITOR_CPP_

#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_scripteditor.h"


#include "kvi_options.h"

#include "kvi_uparser.h"

#include "kvi_app.h"

#include <qpainter.h>
#include <qnamespace.h>
#include <qlayout.h>
#include <qtoolbutton.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include "kvi_fileutils.h"
#include "kvi_locale.h"

#include <X11/Xlib.h>
#ifdef COMPILE_USE_AA_FONTS
#include <X11/Xft/Xft.h>
#endif

#include <ctype.h>

#define KVI_SCRIPTEDITOR_TAB_SIZE 4
#define KVI_SCRIPTEDITOR_BORDER 3

#define KVI_SCRIPTEDITOR_STATE_NEWCOMMAND 0
#define KVI_SCRIPTEDITOR_STATE_INCOMMAND 1
#define KVI_SCRIPTEDITOR_STATE_INSTRING 2
#define KVI_SCRIPTEDITOR_STATE_INCOMMENT 3

QRgb defaultScriptEditorRgb[KVI_SCRIPTEDITOR_NUM_COLORS] = {
	qRgb(0,255,0),     // TEXT
	qRgb(0,235,0),     // COMMAND
	qRgb(0,170,0),     // COMMENT
	qRgb(135,135,255), // STRING
	qRgb(255,0,255),   // ESCAPE
	qRgb(255,0,0),     // BRACE
	qRgb(194,194,194), // LOCAL VARIABLE
	qRgb(255,255,255), // GLOBAL VARIABLE
	qRgb(180,180,20),  // OPERATOR
	qRgb(255,190,0),   // THIS
	qRgb(255,255,0),   // IDENTIFIER
	qRgb(180,180,180), // DISABLED BACK
	qRgb(0,0,0)        // ENABLED BACK
};

//#warning "BACKGROUND IS WHITE INSTEAD OF BLACK"

#ifdef COMPILE_USE_AA_FONTS
	extern XftFont        * g_pXftFont;
	extern XftDraw        * g_pXftDraw;
	extern int qt_use_xft (void); // qpainter_x11.cpp
	extern void *qt_ft_font (const QFont *f); // qfont_x11.cpp
	extern XftDraw * qt_lookup_ft_draw (Drawable draw, bool paintEventClipOn, QRegion *crgn);
#endif

KviScriptEditorWidget::KviScriptEditorWidget(QWidget *parent)
:QMultiLineEdit(parent)
{
	m_pFm = 0;
	m_pMemBuffer = new QPixmap(cellWidth(),cellHeight());

	setFont(font()); // force update the font variables

//	setBackgroundColor(*g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_ENABLEDBACK]);

	QPalette pal = palette();
	pal.setColor(QColorGroup::Base,*(g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_ENABLEDBACK]));
	pal.setColor(QPalette::Disabled,QColorGroup::Base,*(g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_DISABLEDBACK]));
	setPalette(pal);

	getCursorPosition(&m_iLastCursorRow,&m_iLastCursorCol);

	setDefaultTabStop(KVI_SCRIPTEDITOR_TAB_SIZE);
}

KviScriptEditorWidget::~KviScriptEditorWidget()
{
	delete m_pFm;
	delete m_pMemBuffer;
}

int KviScriptEditorWidget::extractNextWord(const char *s)
{
	int len=0;

	// spaces | tabs
	if((*s==' ')||(*s=='\t')){
		len++; s++;
		while((*s==' ') || (*s=='\t')){ len++; s++; }
		return len; 
	}

	if((*s == '#') && m_textState == KVI_SCRIPTEDITOR_STATE_NEWCOMMAND){
		len++; s++;
		while((*s) && (*s != ' ') && (*s != '\t') && (*s != ';') && (*s != '\n')){ len++; s++; }
		return len;
	}

	if(m_textState == KVI_SCRIPTEDITOR_STATE_INCOMMENT){
		while((*s) && (*s != ' ') && (*s != '\t') && (*s != ';') && (*s != '\n')){ len++; s++; }
		return len;		
	}

	if(*s == '\\'){
		// escape!
		len++; s++;
		if(*s && (*s != '\n') && (*s != ' ') && (*s != '\t'))len++;
		return len;
	}

	if(*s == '"')return 1;

	if(isalnum(*s) || (*s== '_') || (*s=='.')){
		// command or normal text
		s++; len++;
		while(*s && (isalnum(*s) || (*s == '_') || (*s == '.'))){ s++; len++; }
		return len;
	}

	if((*s == '%')||(*s == '$')){
		// Variable or identifier
		s++; len++;
		while(*s && (isalnum(*s) || (*s == '_') || (*s == '.') || (*s == ':'))){ s++; len++; }
		return len;
	}

	if((*s=='{')||(*s=='}'))return 1;

	if((*s=='(')||(*s==')')||(*s=='[')||
		(*s==']')||(*s==',')||(*s==':')||
		(*s==';')||(*s=='=')||(*s=='-')||
		(*s=='<')||(*s=='>')||(*s=='&')||
		(*s=='|')||(*s=='+')||(*s=='*')||
		(*s=='/')||(*s=='!')||(*s=='^')){
		len++; s++;
		while((*s=='(')||(*s==')')||(*s=='[')||
				(*s==']')||(*s==',')||(*s==':')||
				(*s==';')||(*s=='=')||(*s=='-')||
				(*s=='<')||(*s=='>')||(*s=='&')||
				(*s=='|')||(*s=='+')||(*s=='*')||
				(*s=='/')||(*s=='!')||(*s=='^')){ len++; s++; }
		return len;
	}

	//senseless string
	s++; len++;
	while((*s) && (isalnum(*s))){ len++; s++; }
	return len;
}

QColor * KviScriptEditorWidget::getColor(const char *s,int len)
{
	if((*s==' ')||(*s=='\t'))return 0;

	if(m_textState == KVI_SCRIPTEDITOR_STATE_NEWCOMMAND){
		if(*s == '#'){
			m_textState = KVI_SCRIPTEDITOR_STATE_INCOMMENT;
			return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_COMMENT];
		} else {
			m_textState = KVI_SCRIPTEDITOR_STATE_INCOMMAND;
			if(isalnum(*s) || (*s== '_') || (*s=='.'))return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_COMMAND];
		}
	} else if(m_textState == KVI_SCRIPTEDITOR_STATE_INCOMMENT){
		if((*s != ';')&&(*s != '\n'))return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_COMMENT];
	}


	if(*s == '"'){
		if(m_textState != KVI_SCRIPTEDITOR_STATE_INSTRING)m_textState = KVI_SCRIPTEDITOR_STATE_INSTRING;
		else m_textState = KVI_SCRIPTEDITOR_STATE_INCOMMAND;
		return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_STRING];
	}

	if(m_textState == KVI_SCRIPTEDITOR_STATE_INSTRING)return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_STRING];


	if(*s == ';')m_textState = KVI_SCRIPTEDITOR_STATE_NEWCOMMAND;

	if(*s == '\\')return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_ESCAPE];

	if((*s=='{')||(*s=='}'))return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_BRACE];
	if(*s == '%'){
		s++;
		if((!*s) && (!(isalnum(*s))) && (*s != '_') && (*s != '.'))return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_OPERATOR];
		if(islower(*s))return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_LOCVAR]; //local variable
		return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_GLOBVAR];
	}
	if(*s == '$'){
		s++;
		len--;
		if(len && (*s == 't') || (*s == 'T')){
			s++;
			len--;
			if(len && (*s == 'h') || (*s == 'H')){
				s++;
				len--;
				if(kvi_strEqualCIN(s,"is",len))return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_THIS];
			}
		}
		return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_IDENT];
	}

	if((*s=='(')||(*s==')')||(*s=='[')||
		(*s==']')||(*s==',')||(*s==':')||
		(*s==';')||(*s=='=')||(*s=='-')||
		(*s=='<')||(*s=='>')||(*s=='&')||
		(*s=='|')||(*s=='+')||(*s=='*')||
		(*s=='/')||(*s=='!')||(*s=='^'))return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_OPERATOR];

	return g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_TEXT];
}

int KviScriptEditorWidget::getSpacesAndTabsLen(const char *s,int nChars,int curNChars,int *width)
{
	int lastChars=curNChars;
	(*width)=0;
	while(nChars && *s){
		if(*s==' '){
			curNChars++;
			(*width)+=m_spaceWidth;
		} else { //A TAB!!!!
			int upToNextTab=KVI_SCRIPTEDITOR_TAB_SIZE-(curNChars % KVI_SCRIPTEDITOR_TAB_SIZE);
			if(upToNextTab==0){
				curNChars+=KVI_SCRIPTEDITOR_TAB_SIZE;
				(*width)+=m_tabWidth;
			} else {
				curNChars+=upToNextTab;
				(*width)+=(m_xWidth * upToNextTab);
			}
		}
		s++;
		nChars--;
	}
#ifdef _KVI_DEBUG_CHECK_RANGE_
	if(nChars)debug("Oops... nChars was incorrect!");
#endif
	return (curNChars-lastChars);
}

int KviScriptEditorWidget::textWidthWithTabs(const char *s,int nChars)
{
    if(!s)return 0;
    if(nChars == -1)nChars = strlen(s);
    int dist = - m_pFm->minLeftBearing();
    const char *t = s;
	int curNChars = 0;
    while ( *t && ((t - s) < nChars)){
		if ( *t == '\t'){
			int upToNextTab=KVI_SCRIPTEDITOR_TAB_SIZE-(curNChars % KVI_SCRIPTEDITOR_TAB_SIZE);
			if(upToNextTab==0){
				curNChars+=KVI_SCRIPTEDITOR_TAB_SIZE;
				dist+=m_tabWidth;
			} else {
				curNChars+=upToNextTab;
				dist+=(m_xWidth*upToNextTab);
			}
//			dist = (((t - s) / KVI_SCRIPTEDITOR_TAB_SIZE) +1) * KVI_SCRIPTEDITOR_TAB_WIDTH;
		} else {
			dist+=m_pFm->width(t,1);
			curNChars++;
		}
		t++;
    }
	return dist;
}

void KviScriptEditorWidget::paintCell(QPainter *painter,int row,int col)
{
	QString s=textLine(row);
    if(s.isNull())return;

    QRect updateR   = cellUpdateRect();

	int rowY;
	if(!rowYPos(row,&rowY))return; //invisible row

	if(m_pMemBuffer->width() != cellWidth())m_pMemBuffer->resize(cellWidth(),cellHeight());

	Display *dspy	=m_pMemBuffer->x11Display();
	HANDLE hnd	=m_pMemBuffer->handle();
	GC gc_aux       =XCreateGC(dspy,hnd,0,0);

#ifdef COMPILE_USE_AA_FONTS
	if(qt_use_xft())
	{
		g_pXftFont = (XftFont *)qt_ft_font(&(font()));
		g_pXftDraw = qt_lookup_ft_draw (hnd,false,0);
		if(!g_pXftDraw)
		{
			XSetFont(dspy,gc_aux,font().handle());
			g_pXftFont = 0;
		}
	} else {
#endif
		XSetFont(dspy,gc_aux,font().handle());
#ifdef COMPILE_USE_AA_FONTS
		g_pXftFont = 0;
		g_pXftDraw = 0;
	}
#endif

	//Draw the background
	XSetLineAttributes(dspy,gc_aux,0,LineSolid,CapButt,JoinMiter);

 	XSetForeground(dspy,gc_aux,isEnabled() 
		? g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_ENABLEDBACK]->pixel()
		: g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_DISABLEDBACK]->pixel());
 	XSetBackground(dspy,gc_aux,isEnabled() 
		? g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_ENABLEDBACK]->pixel()
		: g_pOptions->m_pScriptEditorClr[KVI_SCRIPTEDITOR_CLR_DISABLEDBACK]->pixel());

	XSetFillStyle(dspy,gc_aux,FillSolid);

  	XFillRectangle(dspy,hnd,gc_aux,0,0,cellWidth(),cellHeight());

//	m_spaceWidth = m_pFm->width(' ');

	int markX1 = 0;
	int markX2 = 0;

    if(hasMarkedText()){
		int markBeginX,markBeginY,markEndX,markEndY;
		getMarkedRegion(&markBeginY,&markBeginX,&markEndY,&markEndX);
		if ( row >= markBeginY && row <= markEndY ) {
			if ( row == markBeginY ) {
				markX1 = markBeginX;
				if ( row == markEndY )markX2 = markEndX;
				else markX2 = s.length();
			} else {
				if ( row == markEndY ) {
					markX1 = 0;
					markX2 = markEndX;
				} else {
					markX1 = 0;
					markX2 = s.length();
				}
			}
		}
	}


	const char *pC=s.ascii();
	int len=0;
	int xLeft=KVI_SCRIPTEDITOR_BORDER;

	int totLen=0;
	int myyPos=m_pFm->ascent();


	QColor * clr;

	m_textState = KVI_SCRIPTEDITOR_STATE_NEWCOMMAND;

	while((*pC) && (*pC != '\n')){

		len=extractNextWord(pC);
		clr=getColor(pC,len);

		if(clr){
			// no spaces or tabs
			XSetForeground(dspy,gc_aux,clr->pixel());
#ifdef COMPILE_USE_AA_FONTS
			if(g_pOptions->m_bUseFontAntiAliasing && g_pXftFont){
				XftColor color;
//				QColor * clr = &m_textColor;
				color.color.red = clr->red() | clr->red() << 8;
				color.color.green = clr->green() | clr->green() << 8;
				color.color.blue = clr->blue() | clr->blue() << 8;
				color.color.alpha = 0xffff;
				color.pixel = clr->pixel();
				XftDrawString8(g_pXftDraw,&color,g_pXftFont,xLeft,
					myyPos,(unsigned char *)pC,len);
			} else
#endif
				XDrawString(dspy,hnd,gc_aux,xLeft,myyPos,pC,len);
			xLeft+=m_pFm->width(pC,len);
			totLen+=len;
		} else {
			int addWidth;
			int nSpaces=getSpacesAndTabsLen(pC,len,totLen,&addWidth);
			xLeft+=addWidth;
			totLen+=nSpaces;
		}
		pC+=len;
	}

    if(markX1!=markX2)
	{
		int sLength   = s.length();
		int xpos1     = KVI_SCRIPTEDITOR_BORDER + textWidthWithTabs(s.latin1(),markX1);
		int xpos2     = KVI_SCRIPTEDITOR_BORDER + textWidthWithTabs(s.latin1(),markX2) - 1;
		int fillxpos1 = xpos1;
		int fillxpos2 = xpos2;
		if (markX1 == 0)fillxpos1 -= 2;
		if (markX2 == sLength)fillxpos2 += 3;

		XSetFunction(dspy,gc_aux,GXinvert);
	  	XFillRectangle(dspy,hnd,gc_aux,fillxpos1,0,fillxpos2-fillxpos1,cellHeight());
		XSetFunction(dspy,gc_aux,GXcopy);

	}
	int cursorY,cursorX;
	getCursorPosition(&cursorY,&cursorX);
	if(cursorY != m_iLastCursorRow)
	{
		m_iLastCursorRow = cursorY;
		emit cursorRowChanged(cursorY);
	}
	if(cursorX != m_iLastCursorCol)
	{
		m_iLastCursorCol = cursorX;
		emit cursorColChanged(cursorX);
	}
    if ((row==cursorY) && cursorOn){
		int cursorPos = QMIN((int)s.length(),cursorX);
		int cXPos     = KVI_SCRIPTEDITOR_BORDER + textWidthWithTabs(s.latin1(),cursorPos)-1;
		int cYPos     = 0;
		if(hasFocus()){
			XSetForeground(dspy,gc_aux,Qt::red.pixel());
			XDrawLine(dspy,hnd,gc_aux,cXPos-2,cYPos,cXPos + 2,cYPos );
			XDrawLine(dspy,hnd,gc_aux,cXPos,cYPos,cXPos,cYPos+m_pFm->height()-2);
			XDrawLine(dspy,hnd,gc_aux,cXPos-2,cYPos+m_pFm->height()-2,cXPos+2,cYPos+m_pFm->height()-2);

		}
    }

//	debug("rowY(%d),update.top(%d),update.height(%d),yOffset(%d)",rowY,updateR.top(),updateR.height(),yOffset());

	XCopyArea(dspy,hnd,this->handle(),gc_aux,
			updateR.left(),
			updateR.top(),
			updateR.width(),updateR.height(),
			frameWidth() + updateR.left() - xOffset(),
			rowY + updateR.top());
	XFreeGC(dspy,gc_aux);
}

void KviScriptEditorWidget::resizeEvent(QResizeEvent *e)
{
	m_pMemBuffer->resize(cellWidth(),cellHeight());
	QMultiLineEdit::resizeEvent(e);
}

void KviScriptEditorWidget::setFont(const QFont &font)
{
	if(m_pFm)delete m_pFm;
	m_pFm = new QFontMetrics(font);

	m_xWidth = m_pFm->width('x');
	m_tabWidth = m_xWidth * KVI_SCRIPTEDITOR_TAB_SIZE;
	m_spaceWidth = m_pFm->width(' ');

	QMultiLineEdit::setFont(font);

	m_pMemBuffer->resize(cellWidth(),cellHeight());
}

void KviScriptEditorWidget::setEnabled(bool bEnabled)
{
	QMultiLineEdit::setEnabled(bEnabled);
	cursorOn = false;
	setBackgroundColor(*(g_pOptions->m_pScriptEditorClr[bEnabled ? KVI_SCRIPTEDITOR_CLR_ENABLEDBACK : KVI_SCRIPTEDITOR_CLR_DISABLEDBACK]));
}

void KviScriptEditorWidget::completion()
{
	KviUserParser * p = g_pApp->anyUserParser();
	if(!p)return;

	int row,col;
	getCursorPosition(&row,&col);

	QString * pLine = getString(row);
	if(!pLine)return;
	KviStr line = pLine->ascii();
	if(line.isEmpty() || (col == 0))return;

	// get the word before the cursor
	KviStr word = line.left(col);
	int idx = word.findLastIdx(' ');
	bool bFirstWordInLine = false;
	if(idx > -1)word.cutLeft(idx+1);
	else bFirstWordInLine = true;
	word.stripWhiteSpace();
	if(word.isEmpty())return;

	KviStr match;
	KviStr multiple;
	int iMatches = 0;

	if(*(word.ptr()) == '$')
	{
		// function/identifer completion
		word.cutLeft(1);
		iMatches = p->completeFunctionOrIdentifier(word,match,multiple);
	} else iMatches = p->completeCommand(word,match,multiple);


	if(iMatches != 1)
	{
		if(iMatches == 0)emit message(_CHAR_2_QSTRING(__tr("[Completion]: No matches")));
		else emit message(_CHAR_2_QSTRING(__tr("[Completion]: Multiple matches")));
	}

	if(match.hasData())
	{
		match.toLower();
		setCursorPosition(row, col - word.len());
		killLine();
		insert(match.ptr());
		line.cutLeft(col);
		insert(line.ptr());
		setCursorPosition(row, col - word.len() + match.len());
	}
}



void KviScriptEditorWidget::keyPressEvent(QKeyEvent * e)
{
	if(e->state() & ControlButton)
	{
		switch(e->key())
		{
			case Qt::Key_W:
				completion();   return;
				break;
			case Qt::Key_K:
				insert("$K\\"); return;
				break;
			case Qt::Key_B:
				insert("$B\\");   return;
				break;
			case Qt::Key_O:
				insert("$O\\");   return;
				break;
			case Qt::Key_U:
				insert("$U\\");   return;
				break;
			case Qt::Key_R:
				insert("$R\\");   return;
				break;
		}
	}
	
	QMultiLineEdit::keyPressEvent(e);
}




KviScriptEditor::KviScriptEditor(QWidget * parent)
:QWidget(parent)
{
	QGridLayout * g = new QGridLayout(this,2,6,0,0);
	m_pEditor = new KviScriptEditorWidget(this);
	g->addMultiCellWidget(m_pEditor,0,0,0,5);
	g->setRowStretch(0,1);

	QToolButton * b = new QToolButton(UpArrow,this);
	g->addWidget(b,1,0);

	QPopupMenu * pop = new QPopupMenu(b);
	pop->insertItem(__tr("&Save to File..."),this,SLOT(saveToFile()));
	pop->insertItem(__tr("&Load from File..."),this,SLOT(loadFromFile()));
	b->setPopup(pop);
	b->setPopupDelay(0);

	QLabel * l = new QLabel(__tr("Ctrl+W completes commands and functions"),this);
	l->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	g->setColStretch(1,1);
	g->addWidget(l,1,1);
	m_pAux1 = new QLabel(__tr("Row:"),this);
	m_pAux1->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	g->addWidget(m_pAux1,1,2);
	m_pRowLabel = new QLabel("0",this);
	m_pRowLabel->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	m_pRowLabel->setMinimumWidth(80);
	g->addWidget(m_pRowLabel,1,3);
	m_pAux2 = new QLabel(__tr("Col:"),this);
	m_pAux2->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	g->addWidget(m_pAux2,1,4);
	m_pColLabel = new QLabel("0",this);
	m_pColLabel->setFrameStyle(QFrame::Sunken | QFrame::Panel);
	m_pColLabel->setMinimumWidth(80);
	g->addWidget(m_pColLabel,1,5);
	QObject::connect(m_pEditor,SIGNAL(cursorRowChanged(int)),this,SLOT(cursorRowChanged(int)));
	QObject::connect(m_pEditor,SIGNAL(cursorColChanged(int)),this,SLOT(cursorColChanged(int)));
	QObject::connect(m_pEditor,SIGNAL(message(const QString &)),l,SLOT(setText(const QString &)));
}

KviScriptEditor::~KviScriptEditor()
{
}

void KviScriptEditor::cursorRowChanged(int row)
{
	QString tmp;
	tmp.setNum(row);
	m_pRowLabel->setText(tmp);
}

void KviScriptEditor::cursorColChanged(int col)
{
	QString tmp;
	tmp.setNum(col);
	m_pColLabel->setText(tmp);
}

void KviScriptEditor::setEnabled(bool bEnabled)
{
	QWidget::setEnabled(bEnabled);
	m_pEditor->setEnabled(bEnabled);
	m_pAux1->setEnabled(bEnabled);
	m_pAux2->setEnabled(bEnabled);
	m_pColLabel->setEnabled(bEnabled);
	m_pRowLabel->setEnabled(bEnabled);
}

void KviScriptEditor::saveToFile()
{
	KviStr fName = kvi_askForSaveFileName();
	if(fName.hasData())
	{
		if(kvi_fileExists(fName.ptr()))
		{
			if(QMessageBox::warning(0,__tr("Script Center"),
				__tr("The file already exists. Do you wish to overwrite it?"),__tr("Yes"),__tr("No")) == 1)return;
		}
		KviStr buffer = text();
		if(!kvi_writeFile(fName.ptr(),buffer))g_pApp->warningBox(__tr("Can't open file %s for writing. Save failed."),fName.ptr());
	}
}

void KviScriptEditor::loadFromFile()
{
	KviStr fName = kvi_askForOpenFileName();
	if(fName.hasData())
	{
		KviStr buffer;
		if(kvi_loadFile(fName.ptr(),buffer))setText(buffer.ptr());
		else g_pApp->warningBox(__tr("Unable to to open file %s"),fName.ptr());
	}
}

#include "m_kvi_scripteditor.moc"
