#ifndef _KVI_IRCSOCKET_H_INCLUDED_
#define _KVI_IRCSOCKET_H_INCLUDED_

//
//   File : kvi_ircsocket.h (/usr/build/NEW_kvirc/kvirc/kvirc/kvi_ircsocket.h)
//   Last major modification : Mon Jan 11 1999 19:52:18 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include <qobject.h>
#include <qsocketnotifier.h>
#include <qtimer.h>

#include "kvi_netutils.h"
#include "kvi_string.h"

class KviFrame;
class KviServerParser;
class KviSysTrayIoLed;

typedef struct KviIrcSocketMsgEntry
{
	char                 * data_ptr;
	int                    data_len;
	KviIrcSocketMsgEntry * next_ptr;
};

typedef struct KviProxyStruct
{
	KviStr szIp;
	KviStr szPort;
	KviStr szPassword;
	KviStr szUsername;
	bool   bProtoV5;
	bool   bProtoHttp;
};

typedef struct KviInternalProxyData
{
	unsigned short int ircHostPort;
	struct in_addr ircHostAddress;
	KviStr szPassword;
	KviStr szUsername;
	bool bProtoV5;
	bool bProtoHttp;
	int state;
};

#define KVI_PROXYSTATE_INACTIVE 0
#define KVI_PROXYSTATE_CONNECTING 1
#define KVI_PROXYSTATE_SELECTMETHOD 2
#define KVI_PROXYSTATE_WAITFORUSERANDPASS 3
#define KVI_PROXYSTATE_V4FINAL 4
#define KVI_PROXYSTATE_V5FINAL 5
#define KVI_PROXYSTATE_HTTPHANDSHAKE 6

class KviIrcSocket : public QObject
{
	friend class KviFrame;
	Q_OBJECT
public:
	KviIrcSocket(KviFrame *pFrm,KviServerParser *pPrs,KviSysTrayIoLed *pSysTrayIoLed);
	~KviIrcSocket();
private:
	int              m_sock;
	KviFrame        *m_pFrm;
	KviSysTrayIoLed *m_pSysTrayIoLed;
	QSocketNotifier *m_pWsn;
	QSocketNotifier *m_pRsn;
	int              m_error;
	int              m_iReadBufLen;
	char            *m_pReadBuffer;
	uint             m_uReadBytes;
	uint             m_uSentBytes;
	uint             m_uReadPackets;
	uint             m_uSentPackets;
	KviInternalProxyData m_proxy;
	enum ircSocketState { Ready , Connecting, Connected };
	ircSocketState      m_state;
	KviIrcSocketMsgEntry * m_pSendQueueHead;
	KviIrcSocketMsgEntry * m_pSendQueueTail;
	QTimer               * m_pFlushTimer;
	KviServerParser      * m_pServerParser;
public:
	/**
	* Attempt to connect to a specified host.
	* Returns true if the connection has been succesfully initiated
	* otherwise returns false and sets m_error.
	*/
	bool connectTo(const char *szIp,const char *szPort,KviProxyStruct *prx=0,bool bIpV6 = false);
	/**
	* Aborts the current connection , or (brutally) disconnects from the server
	*/
	void abortConnection();
	/**
	* Sends (NOCRLF) data buffer.
	* Returns if the message has been sucessfully queued
	* false otherwise (sets m_error)
	*/
	bool sendRawData(const char *buffer,int buflen);
	/**
	* Sends a formatted buffer of data
	* the text is formatted up to 512 bytes and then truncated
	* A CRLF is ALWAYS appended to the end of the string
	*/
	bool sendFmtData(const char *fmt,...);
	/**
	* Sends a buffer of non-formatted data.
	* The text is truncated to 510 bytes and CRLF is ALWAYS appended.
	* If buflen is -1 , the length of the buffer is calculated internally.
	*/
	bool sendData(const char *buffer,int buflen = -1);
	/**
	* Returns the last error identifier (kvi_error.h)
	*/
	int getError();
	//Utils
	/**
	* Blocking DNS
	* Returns false on failure and sets m_error
	*/
	bool getHostByName(const char *host,KviStr &szIp,bool bIpV6 = false);
	//This one works only when connected (returns this socket addres in network byte order)
	/**
	* Works only on connected socket.
	* Returns the local address as unsigned long in network byte order
	*/
	unsigned long getSockAddress();
	/**
	* Wrapper for getSockAddress
	* Returns the same address as above , but in a numbers-and-dots notation
	*/
	bool getLocalHostIp(KviStr &szIp,bool bIpV6 = false);
	bool isConnected() { return m_state == Connected; };
protected slots:
	void timeout();
	void slot_readIrcData(int);
	void slot_readProxyData(int);
	int slot_readLine(char *buffer, int bufferSize);
	void slot_writeNotifierFired(int);
	void handleInvalidSocketRead(int readedLength,int eventToFireInCaseOfError);
	void flushSendQueue();
private:
	void proxy_loginV4();
	void proxy_loginV5();
	void proxy_loginHttp();
	void proxy_authUserPassV5();
	void proxy_sendTargetDataV5();
	void proxy_handleV4FinalReply(unsigned char reply);
	void proxy_handleV5MethodReply(unsigned char reply);
	void proxy_handleV5AuthReply(unsigned char reply);
	void proxy_handleV5FinalReply(unsigned char reply);
	void proxy_handleHttpHandshake(char *reply);
	void setErrorFromSystemError(int errorNum);
	void reset_killReadNotifier();
	void reset_killWriteNotifier();
	void reset_killSocket();
	void reset_destroyReadBuffer();
	void reset_clearProxyState();
	/**
	* Calls killReadNotifier , killWriteNotifier ,<br>
	* killSocket and then and sets m_state to Ready
	* calls also while(dequeueMessage());
	*/
	void reset();
	void queue_insertMessage(KviIrcSocketMsgEntry *msg_ptr);
	/**
	* Returns true if the send queue has other entries
	* Gracefully segfaults if the queue has no entries
	*/
	bool queue_removeMessage();
	void queue_removeAllMessages();

	void reset_clearSendQueue();
	void clearInternalCounters();
};

#define KVI_SOCKEVENT_Connected 0
#define KVI_SOCKEVENT_ConnectionFailed 1
#define KVI_SOCKEVENT_Disconnected 2
//#define KVI_SOCKEVENT_OneMessageSent 3 //UNUSED
#define KVI_SOCKEVENT_PartialSocketWrite 4
#define KVI_SOCKEVENT_MessageTruncated 5
#define KVI_SOCKEVENT_UsingDefaultPort6667 6
#define KVI_SOCKEVENT_UsingDefaultPort1080 7
#define KVI_SOCKEVENT_ConnectedToProxy 8
#define KVI_SOCKEVENT_UsingProxyProtoV4 9
#define KVI_SOCKEVENT_UsingProxyProtoV5 10
#define KVI_SOCKEVENT_ProxySentV4Request 11
#define KVI_SOCKEVENT_ProxySentV5MethodRequest 12
#define KVI_SOCKEVENT_ProxyMethodSelectedNoAuth 13
#define KVI_SOCKEVENT_ProxyMethodSelectedUserPass 14
#define KVI_SOCKEVENT_ProxyReply90RequestGranted 15
#define KVI_SOCKEVENT_ProxySentUserAndPass 16
#define KVI_SOCKEVENT_ProxyAuthOK 17
#define KVI_SOCKEVENT_ProxySentTargetData 18
#define KVI_SOCKEVENT_ProxyReply00Success 19
#define KVI_SOCKEVENT_ContactingHost 20
#define KVI_SOCKEVENT_UsingProxyProtoHttp 21
#define KVI_SOCKEVENT_ProxySentHttpConnectRequest 22
#define KVI_SOCKEVENT_ProxyReplyHttp200Success 23


inline int KviIrcSocket::getError()
{ return m_error; }

inline void KviIrcSocket::abortConnection()
{ reset(); }

#endif //!_KVI_IRCSOCKET_H_INCLUDED_
