//
//   File : kvi_frame.cpp (/usr/build/NEW_kvirc/kvirc/src/kvirc/kvi_frame.cpp)
//   Last major modification : Tue Jul 6 1999 14:43:32 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#define _KVI_DEBUG_CHECK_RANGE_
//#define _KVI_DEBUG_CLASS_NAME_ "KviFrame"

#include "kvi_debug.h"

#include <qvariant.h>

#include "kvi_app.h"
#include "kvi_locale.h"
#include "kvi_frame.h"
#include "kvi_menubar.h"
#include "kvi_toolbar.h"
#include "kvi_utoolbar.h"
#include "kvi_statusbar.h"
#include "kvi_taskbar.h"
#include "kvi_mdi.h"
#include "kvi_mdibutton.h"
#include "kvi_ircserver.h"
#include "kvi_netutils.h"
#include "kvi_userlist.h"
#include "kvi_window.h"
#include "kvi_ircview.h"
#include "kvi_console.h"
#include "kvi_channel.h"
#include "kvi_query.h"
#include "kvi_uwindow.h"
#include "kvi_dcc_chat.h"
#include "kvi_dcc_send.h"
#include "kvi_dcc_voice.h"
#include "kvi_links.h"
#include "kvi_list.h"
#include "kvi_ircsocket.h"
#include "kvi_error.h"
#include "kvi_dns.h"
#include "kvi_options.h"
#include "kvi_sparser.h"
#include "kvi_uparser.h"
#include "kvi_ircview.h"
#include "kvi_systray.h"
#include "kvi_winproperties.h"
#include "kvi_dcc_manager.h"
#include "kvi_regusersdb.h"
#include "kvi_listbox.h"
#include "kvi_event.h"
#include "kvi_process.h"
#include "kvi_asyncwhois.h"
#include "kvi_dirbrowser.h"
#include "kvi_helpwindow.h"

#include <qcursor.h>
#include <qclipboard.h>
#include <qobjcoll.h>
#include <qtoolbutton.h>
#include <qsplitter.h>
#include <qtooltip.h>

//###########################################
//
// CREATION PROCESS
//
//###########################################


//This comes from kvi_app.cpp
extern QPopupMenu *g_pTaskBarPopup;
extern QPopupMenu *g_pViewSelectPopup;
extern QPopupMenu *g_pWindowPopup;

extern KviUserToolBarTemplate *g_pUserToolBarTemplate;

extern KviEventManager * g_pEventManager;
extern QPixmap * g_pixViewOut[KVI_OUT_NUM_IMAGES];

//=============================== constructor =================================//

KviFrame::KviFrame()
:QMainWindow(0,"kvirc_main_frame")
{
	setIcon(*g_pixViewOut[KVI_OUT_KVIRC]);
	// Register with the app
	g_pApp->insertNewFrame(this);
	// Create the local list of windows
	m_pWinList = new QList<KviWindow>;
	m_pWinList->setAutoDelete(false);
	// This seems to be needed (re-check it after Qt2.0 comed out)
	setFocusPolicy(ClickFocus);
	// Reset socket state
	m_state      = Ready;
	m_pDns       = 0;
	m_pConsole   = 0; //temp hack, so resetGlobals will not segfault
	m_bInSDIMode = false;

	m_global.pDnsWhoisPending = new QList<KviStr>;
	m_global.pDnsWhoisPending->setAutoDelete(true);
	m_global.pMultimediaFileOfferts = new QList<KviStr>;
	m_global.pMultimediaFileOfferts->setAutoDelete(true);

	m_pChannelsBeforeDisconnect = 0;

	m_pReconnectTimer = new QTimer();
	connect(m_pReconnectTimer,SIGNAL(timeout()),this,SLOT(reconnectToServer()));

	m_pMultimediaFileOffertTimer = new QTimer();
	connect(m_pMultimediaFileOffertTimer,SIGNAL(timeout()),this,SLOT(cleanupMultimediaFileOffertList()));

	m_pRecentServersPopup = new QPopupMenu(this,"recent_servers_popup");
	m_pRecentChannelsPopup = new QPopupMenu(this,"recent_channels_popup");
	m_pRecentNicknamesPopup = new QPopupMenu(this,"recent_nicknames_popup");

	connect(m_pRecentServersPopup,SIGNAL(activated(int)),this,SLOT(connectToRecentServer(int)));
	connect(m_pRecentChannelsPopup,SIGNAL(activated(int)),this,SLOT(joinRecentChannel(int)));
	connect(m_pRecentNicknamesPopup,SIGNAL(activated(int)),this,SLOT(setRecentNickname(int)));

	// Reset global struct
	m_pUserParser = 0;

	resetGlobals();

	// And start creating self
	createSplitter();
	createMdiManager();
	createMenuBar();
	createToolBar();
	createMdiToolBar();

	// Create the SDI mode buttons
	createSDIModeButtons();

//	setRightJustification(true);
	setDockMenuEnabled(false); // coredumps with dynamically added toolbars

	createStatusBar();
	createTaskBar();
	createSysTrayBar();
	createUserToolBar();

	// IRC parsers and socket
	m_pServerParser       = new KviServerParser(this);
	m_pUserParser         = new KviUserParser(this);
	m_pSysTrayIoLed       = (KviSysTrayIoLed * )(m_pSysTrayBar->m_pSysTray->findSysTrayWidget("KviSysTrayIoLed"));
	m_pSysTrayOnLineTimer = (KviSysTrayOnLineTimer *)(m_pSysTrayBar->m_pSysTray->findSysTrayWidget("KviSysTrayOnLineTimer"));
	m_pSocket             = new KviIrcSocket(this,m_pServerParser,m_pSysTrayIoLed);
	m_pDccManager         = new KviDccManager(this);
	// Global user list
	m_pUserList           = new KviIrcUserList();
	m_pDirBrowser         = 0;
	m_pHelpWindow         = 0;

	// Apply options for the MDI manager
	applyOptions();
	createConsole();
	// Now everything is created...set up the remaining things...
	m_pServerParser->setup(m_pSocket,m_pConsole,m_pUserParser);
	m_pUserParser->setup(m_pSocket,m_pConsole);

	// No active window yet (when shown, the console should be the active one)
	m_global.pCurrentActiveWindow = 0;
	m_iUserListUpdateTimer        = 0;
	m_iNotifyListUpdateTimer      = 0;

	m_pAccel = installAccelerators(this);

	updateCaption();

	fillRecentServersPopup();
	fillRecentChannelsPopup();
	fillRecentNicknamesPopup();

	// Set the geometry
	setGeometry(g_pOptions->m_rectFrameGeometry); //does not work with KWM... ???

	restoreToolBarPositions();

	bool bHalt = false;

	if(g_pEventManager->eventEnabled(KviEvent_OnStartup)){
		KviStr tmp(KviStr::Format,"%d %s",(g_pApp->m_pFrameList->count() == 1) ? "1" : "0", KVI_VERSION);
		bHalt = m_pUserParser->callEvent(KviEvent_OnStartup,m_pConsole,tmp.ptr());
	}

	if(!bHalt)m_pConsole->output(KVI_OUT_KVIRC,KVI_VERSION);

	m_pIdleTimer = new QTimer(this);
	connect(m_pIdleTimer, SIGNAL(timeout()), SLOT(triggerOnIdleEvent()));

	if(g_pEventManager->eventEnabled(KviEvent_OnIdleStart) || g_pEventManager->eventEnabled(KviEvent_OnIdleStop))
		m_pIdleTimer->start(g_pOptions->m_iOnIdleAfter * 1000);

	m_userIdle = false;
}


//=============================== destructor =================================//

KviFrame::~KviFrame()
{
	// Fire the last event of this frame
	if(g_pEventManager->eventEnabled(KviEvent_OnShutdown)){
		m_pUserParser->callEvent(KviEvent_OnShutdown,m_pConsole,KviStr(""));
	}

	// Stop timers
	if(m_pReconnectTimer->isActive())m_pReconnectTimer->stop();
	delete m_pReconnectTimer;
	if(m_pMultimediaFileOffertTimer->isActive())m_pMultimediaFileOffertTimer->stop();
	delete m_pMultimediaFileOffertTimer;

	// Close the connection if necessary...
	if(m_state == Connected){
		// Send a quit....in extremis
		if(g_pOptions->m_szQuitMessage.hasData()){
			m_pSocket->sendFmtData("QUIT :%s",g_pOptions->m_szQuitMessage.ptr());
		} else m_pSocket->sendData("QUIT");
	}

	if(m_pChannelsBeforeDisconnect)
	{
		delete m_pChannelsBeforeDisconnect;
		m_pChannelsBeforeDisconnect = 0;
	}

	// Stop the notify list
	stopNotifyOrWatchList();
	clearConsoleListBox(); // Do it before deleting the global userlist

	// Save the last geometry
	if(g_pOptions->m_bUseHackedFrameGeometry){
		g_pOptions->m_rectFrameGeometry.moveTopLeft(frameGeometry().topLeft());
		g_pOptions->m_rectFrameGeometry.setSize(size());
	} else g_pOptions->m_rectFrameGeometry = geometry();


	saveToolBarPositions();

//	__debug_1arg("KviFrame : Child list with %d items",m_pWinList->count());
	// safely close the windows so properties are saved...
	KviWindow *pWnd = 0;
	while((pWnd = m_pWinList->first()))closeWindow(pWnd);
	delete m_pWinList;
	delete m_pServerParser;
	delete m_pUserParser;
	delete m_pSocket;
	delete m_pDccManager;
	if(m_pDns)delete m_pDns;
	delete m_pUserList;
	delete m_global.pDnsWhoisPending;
	delete m_global.pMultimediaFileOfferts;
	delete m_pUserToolBar;
	g_pApp->removeFrame(this);
}

#define KVI_ACCEL_SWITCH_WINDOWS_LEFT 1001
#define KVI_ACCEL_SWITCH_WINDOWS_RIGHT 1002

// 899 shortcuts should be enough
#define KVI_ACCEL_SHORTCUT 1100
#define KVI_ACCEL_LASTSHORTCUT 1999

#define KVI_ACCEL_DOCK_UNDOCK 2050
#define KVI_ACCEL_MAXIMIZE 2051
#define KVI_ACCEL_RESTORE_MINIMIZE 2052
#define KVI_ACCEL_MINIMIZE 2053
#define KVI_ACCEL_PREVLINE 2054
#define KVI_ACCEL_NEXTLINE 2055
#define KVI_ACCEL_PREVPAGE 2056
#define KVI_ACCEL_NEXTPAGE 2057
#define KVI_ACCEL_FINDTEXT 2058

int g_KVIrcAccelerators[]=
{
	Qt::Key_0 + Qt::CTRL   , Qt::Key_1 + Qt::CTRL   , Qt::Key_2 + Qt::CTRL   ,
	Qt::Key_3 + Qt::CTRL   , Qt::Key_4 + Qt::CTRL   , Qt::Key_5 + Qt::CTRL   ,
	Qt::Key_6 + Qt::CTRL   , Qt::Key_7 + Qt::CTRL   , Qt::Key_8 + Qt::CTRL   ,
	Qt::Key_9 + Qt::CTRL   , Qt::Key_F1             , Qt::Key_F2             ,
	Qt::Key_F3             , Qt::Key_F4             , Qt::Key_F5             ,
	Qt::Key_F6             , Qt::Key_F7             , Qt::Key_F8             ,
	Qt::Key_F9             , Qt::Key_F10            , Qt::Key_F11            ,
	Qt::Key_F12            , Qt::Key_F1 + Qt::SHIFT , Qt::Key_F2 + Qt::SHIFT ,
	Qt::Key_F3 + Qt::SHIFT , Qt::Key_F4 + Qt::SHIFT , Qt::Key_F5 + Qt::SHIFT ,
	Qt::Key_F6 + Qt::SHIFT , Qt::Key_F7 + Qt::SHIFT , Qt::Key_F8 + Qt::SHIFT ,
	Qt::Key_F9 + Qt::SHIFT , Qt::Key_F10+ Qt::SHIFT , Qt::Key_F11+ Qt::SHIFT ,
	Qt::Key_F12+ Qt::SHIFT , 0
};

QAccel * KviFrame::installAccelerators(QWidget *w)
{
	QAccel * a = new QAccel(w,"kvirc_accel");
	connect(a,SIGNAL(activated(int)),this,SLOT(accelActivated(int)));

	a->insertItem(Qt::Key_Left     + CTRL,KVI_ACCEL_SWITCH_WINDOWS_LEFT);
	a->insertItem(Qt::Key_Right    + CTRL,KVI_ACCEL_SWITCH_WINDOWS_RIGHT);

	int i = 0;
	while(g_KVIrcAccelerators[i])
	{
		a->insertItem(g_KVIrcAccelerators[i],KVI_ACCEL_SHORTCUT + i);
		i++;
	}

	a->insertItem(Qt::Key_End      + CTRL,KVI_ACCEL_DOCK_UNDOCK);
	a->insertItem(Qt::Key_Up       + CTRL,KVI_ACCEL_MAXIMIZE);
	a->insertItem(Qt::Key_Down     + CTRL,KVI_ACCEL_RESTORE_MINIMIZE);
	a->insertItem(Qt::Key_Escape         ,KVI_ACCEL_MINIMIZE);
	a->insertItem(Qt::Key_PageUp   + CTRL,KVI_ACCEL_PREVLINE);
	a->insertItem(Qt::Key_PageDown + CTRL,KVI_ACCEL_NEXTLINE);
	a->insertItem(Qt::Key_PageUp         ,KVI_ACCEL_PREVPAGE);
	a->insertItem(Qt::Key_PageDown       ,KVI_ACCEL_NEXTPAGE);
	a->insertItem(Qt::Key_F        + CTRL,KVI_ACCEL_FINDTEXT);


	return a;
}

void KviFrame::accelActivated(int id)
{
	KviStr tmp;
	KviWindow * w = activeWindow();
	if(!w)return;

	if((id >= KVI_ACCEL_SHORTCUT) && (id <= KVI_ACCEL_LASTSHORTCUT))
	{
		if(g_pEventManager->eventEnabled(KviEvent_OnKeyShortcut))
		{
			int k = m_pAccel->key(id);
			if(k)
			{
				tmp = m_pAccel->keyToString(k);
				m_pUserParser->callEvent(KviEvent_OnKeyShortcut,w,tmp);
			}
		}
		return;
	}

	switch(id)
	{
		case KVI_ACCEL_SWITCH_WINDOWS_LEFT:
			switchWindows(false);
			break;
		case KVI_ACCEL_SWITCH_WINDOWS_RIGHT:
			switchWindows(true);
			break;
		case KVI_ACCEL_DOCK_UNDOCK:
			if(w->isAttached())w->detach();
			else w->attach();
			break;
		case KVI_ACCEL_MAXIMIZE:
			if(!w->isMaximized())w->maximize();
			break;
		case KVI_ACCEL_RESTORE_MINIMIZE:
			if(w->isMaximized())w->restore();
			else w->minimize();
			break;
		case KVI_ACCEL_MINIMIZE:
			w->minimize();
			break;
		case KVI_ACCEL_PREVPAGE:
			if(w->m_pView)w->m_pView->prevPage();
			break;
		case KVI_ACCEL_PREVLINE:
			if(w->m_pView)w->m_pView->prevLine();
			break;
		case KVI_ACCEL_NEXTPAGE:
			if(w->m_pView)w->m_pView->nextPage();
			break;
		case KVI_ACCEL_NEXTLINE:
			if(w->m_pView)w->m_pView->nextLine();
			break;
		case KVI_ACCEL_FINDTEXT:
			if(!w->isMinimized() && w->m_pView)w->findText();
			break;
	}

}


//============ applyOptions ============//

void KviFrame::applyMdiOptions()
{
	m_pMdi->setMdiCaptionInactiveBackColor(g_pOptions->m_clrMdiCaptionInactiveBack);
	m_pMdi->setMdiCaptionInactiveForeColor(g_pOptions->m_clrMdiCaptionInactiveFore);
	m_pMdi->setMdiCaptionActiveBackColor(g_pOptions->m_clrMdiCaptionActiveBack);
	m_pMdi->setMdiCaptionActiveForeColor(g_pOptions->m_clrMdiCaptionActiveFore);
	m_pMdi->setMdiCaptionFont(g_pOptions->m_fntMdiCaption);
	m_pMdi->setOpaqueMove(g_pOptions->m_bMdiOpaqueMove);
	m_pMdi->setOpaqueResize(g_pOptions->m_bMdiOpaqueResize);

	if(g_pOptions->m_pixMdiBack)
	{
		if(!g_pOptions->m_pixMdiBack->isNull()){
			m_pMdi->setBackgroundPixmap(*(g_pOptions->m_pixMdiBack));
			return;
		}
	}
	m_pMdi->setBackgroundColor(g_pOptions->m_clrMdiBack);
}

void KviFrame::forceUpdateToAllMdiChildren()
{
	for(KviWindow *w = m_pWinList->first();w;w= m_pWinList->next()){
		w->applyOptions();
		QWidget *wdgt = w;
		if(w->mdiParent())wdgt =w->mdiParent();
		// Really ugly hack to FORCE the resize event
		// a resize(width(),height()) won't work...
		wdgt->resize(wdgt->width()+1,wdgt->height()+1);
		wdgt->resize(wdgt->width()-1,wdgt->height()-1);
	}
}

void KviFrame::applyOptions()
{
	if(m_pHelpWindow)m_pHelpWindow->resetStyleSheet();
	if(m_pDirBrowser)m_pDirBrowser->applyOptions();

	applyMdiOptions();
	forceUpdateToAllMdiChildren();

	m_pSysTrayBar->m_pSysTray->repaint();
}

bool KviFrame::focusNextPrevChild(bool next)
{
	QWidget * w = focusWidget();
	if(w){
		QVariant v = w->property("KviProperty_FocusOwner");
		if(v.isValid())return false; // Do NOT change the focus widget!
		if(w->inherits("KviMdiCaptionButton")){
			// Ensure the proper focus when restoring children
			m_pMdi->focusTopChild();
			return true;
		}
	}
	return QMainWindow::focusNextPrevChild(next);
}

void KviFrame::createConsole()
{
	//Create the console window.
	m_pConsole = new KviConsole(this);
	addWindow(m_pConsole);
	if(g_pOptions->m_bAutoLogConsole)m_pConsole->m_pView->toggleLogging();
}

//============ createMdiManager ============//
void KviFrame::createSplitter()
{
	m_pSplitter = new QSplitter(QSplitter::Horizontal,this,"main_splitter");
	setCentralWidget(m_pSplitter);
	m_pSplitter->setFocusPolicy(QWidget::NoFocus);
}

void KviFrame::createMdiManager()
{
	m_pMdi=new KviMdiManager(m_pSplitter,"mdi_manager");
	connect(m_pMdi,SIGNAL(enterSDIMode(bool)),this,SLOT(mdiManagerSDIModeChange(bool)));
}


void KviFrame::toggleHelpWindow()
{
	if(m_pHelpWindow)
	{
		delete m_pHelpWindow;
		m_pHelpWindow = 0;
		m_pMdi->setFocus(); //ensure that the top child gets the focus
	} else {
		m_pHelpWindow = new KviHelpWindow(this,m_pSplitter);
		m_pHelpWindow->show();
	}
}

void KviFrame::requestHelpOn(const char *topic)
{
	if(!m_pHelpWindow)toggleHelpWindow();
	m_pHelpWindow->setDocument(topic);
}

void KviFrame::requestHelpSearchOn(const char *topic)
{
	if(!m_pHelpWindow)toggleHelpWindow();
	m_pHelpWindow->doExactSearchForString(topic);
}

bool KviFrame::dirBrowserOpen()
{
	if(g_pOptions->m_bDirBrowserOpenAsMdiWindow)return false;
	return (m_pDirBrowser != 0);
}

void KviFrame::toggleDirBrowser()
{
	if(g_pOptions->m_bDirBrowserOpenAsMdiWindow)
	{
		KviDirectoryBrowser * b = createDirectoryBrowser(0);
		if(g_pOptions->m_bDirBrowserOpenAlwaysUndocked && b->isAttached())b->detach();
		return;
	}
	// open as child of the splitter
	if(m_pDirBrowser)
	{
		delete m_pDirBrowser;
		m_pDirBrowser = 0;
		m_pMdi->setFocus(); //ensure that the top child gets the focus
	} else {
		m_pDirBrowser = new KviDirectoryBrowserWidget(m_pSplitter,this,0);
//		m_pSplitter->moveToLast(m_pDirBrowser);
		m_pDirBrowser->show();
	}
}

QMainWindow::ToolBarDock g_dockSites[7]= {
	QMainWindow::Top ,
	QMainWindow::Left ,
	QMainWindow::Right ,
	QMainWindow::Bottom ,
	QMainWindow::TornOff ,
	QMainWindow::Unmanaged ,
	QMainWindow::Minimized
};

void KviFrame::saveToolBarPositions()
{
	KviStr szTemp;
	g_pApp->getLocalKVircDirectory(szTemp,KviApp::Config,"kvi.toolbars.conf");
	KviConfig cfg(szTemp.ptr());

	int idx = 0;

	for(int i=0;i<7;i++)
	{
		QList<QToolBar> l = toolBars(g_dockSites[i]);
	
		for(QToolBar * b = l.first();b;b = l.next())
		{
			if(b->parent() == this)
			{
				cfg.setGroup("Main");
				KviStr tmp(KviStr::Format,"Toolbar%d",idx);
				KviStr lab = b->label();
				cfg.writeEntry(tmp.ptr(),lab.ptr());
				saveToolBarState(b,&cfg);
			}
			idx++;
		}
	}

	cfg.setGroup("Main");
	cfg.writeEntry("NumToolbars",idx);
}


void KviFrame::restoreToolBarPositions()
{
	KviStr szTemp;
	g_pApp->getLocalKVircDirectory(szTemp,KviApp::Config,"kvi.toolbars.conf");
	KviConfig cfg(szTemp.ptr());
	cfg.setGroup("Main");

	int nToolbars = cfg.readIntEntry("NumToolbars",0);

	if(nToolbars > 0)
	{
		for(int i=0;i<nToolbars;i++)
		{
			cfg.setGroup("Main");
			KviStr tmp(KviStr::Format,"Toolbar%d",i);
			KviStr label = cfg.readEntry(tmp.ptr(),"");
			if(label.hasData())
			{
				QObjectList * l = queryList("QToolBar",0,false,false);
				if(l)
				{
					QObjectListIt it(*l);
					while(it.current())
					{
						if(it.current()->parent() == this)
						{
							KviStr tbLab = ((QToolBar *)it.current())->label();
							if(kvi_strEqualCI(tbLab.ptr(),label.ptr()))
							{
								restoreToolBarState((QToolBar *)it.current(),&cfg,QMainWindow::Top,0,false,0);
								break;
							}
						}
						++it;
					}
					delete l;
				}
			}
		}
	}
}

void KviFrame::saveToolBarState(QToolBar * t,KviConfig * cfg)
{
	QMainWindow::ToolBarDock dock;
	int index;
	bool bNl;
	int extra;

	if(getLocation(t,dock,index,bNl,extra))
	{
		KviStr tmp =t->label();
		cfg->setGroup(tmp.ptr());

		cfg->writeEntry("Visible",t->isVisible());
		cfg->writeEntry("Dock",(int)dock);
		cfg->writeEntry("Index",index);
		cfg->writeEntry("Newline",bNl);
		cfg->writeEntry("ExtraSpace",extra);
	}
}

void KviFrame::restoreToolBarState(QToolBar * t,KviConfig * cfg,QMainWindow::ToolBarDock defDock,int defIndex,bool bDefNewline,int defExtra)
{
	QMainWindow::ToolBarDock dock;
	int index , extra;
	bool bVisible , bNl;

	KviStr tmp = t->label().ascii();

	cfg->setGroup(tmp.ptr());

	bVisible   = cfg->readBoolEntry("Visible",true);
	dock       = (QMainWindow::ToolBarDock)cfg->readIntEntry("Dock",defDock);
	index      = cfg->readIntEntry("Index",defIndex);
	bNl        = cfg->readBoolEntry("Newline",bDefNewline);
	extra      = cfg->readIntEntry("ExtraSpace",defExtra);

	if(bVisible)t->show();
	else t->hide();

	moveToolBar(t,dock,bNl,index,extra);
}

void KviFrame::slot_lineUpToolBarsNormal()
{
	lineUpToolBars(false);
}

void KviFrame::slot_lineUpToolBarsCompact()
{
	lineUpToolBars(true);
}



//============ createMenuBar ============//
void KviFrame::createMenuBar()
{
	// Here we create the menu bar
	m_pMenuBar = new KviMenuBar(this);
}

//============ createToolBar ============//
void KviFrame::createToolBar()
{
	m_pToolBar = new KviToolBar(this,QMainWindow::Top);
	applyToolbarOptions();
}

void KviFrame::createMdiToolBar()
{
	m_pMdiToolBar = new KviMdiToolBar(this,QMainWindow::Top);
}

void KviFrame::createSysTrayBar()
{
	m_pSysTrayBar = new KviSysTrayBar(this,QMainWindow::Top);
	setDockEnabled(m_pSysTrayBar,QMainWindow::Left,false);
	setDockEnabled(m_pSysTrayBar,QMainWindow::Right,false);
	applyToolbarOptions();
}

void KviFrame::slot_toggleSysTrayBar()
{
	if(m_pSysTrayBar->isVisible())m_pSysTrayBar->hide();
	else m_pSysTrayBar->show();
}


void KviFrame::createSDIModeButtons()
{
	m_pSDICloseButton = new KviMdiCaptionButton(m_pMenuBar,KviMdiCaptionButton::Close,"sdi_close_button");
	m_pSDICloseButton->hide();
	QToolTip::add(m_pSDICloseButton,__tr("Close"));
	connect(m_pSDICloseButton,SIGNAL(buttonPressed()),m_pMdi,SLOT(closeTopChild()));
	m_pSDIRestoreButton = new KviMdiCaptionButton(m_pMenuBar,KviMdiCaptionButton::Restore,"sdi_restore_button");
	m_pSDIRestoreButton->hide();
	QToolTip::add(m_pSDIRestoreButton,__tr("Restore"));
	connect(m_pSDIRestoreButton,SIGNAL(buttonPressed()),m_pMdi,SLOT(restoreTopChild()));
	m_pSDIMinimizeButton = new KviMdiCaptionButton(m_pMenuBar,KviMdiCaptionButton::Minimize,"sdi_minimize_button");
	m_pSDIMinimizeButton->hide();
	QToolTip::add(m_pSDIMinimizeButton,__tr("Minimize"));
	connect(m_pSDIMinimizeButton,SIGNAL(buttonPressed()),m_pMdi,SLOT(minimizeTopChild()));
}

//============ createToolBar ============//
void KviFrame::createUserToolBar()
{
	//Here we create the tool bar
	m_pUserToolBar=new KviUserToolBar(this,QMainWindow::Top);
	updateUserToolBar();
}

void KviFrame::updateUserToolBar()
{
	if(!g_pUserToolBarTemplate->isEmpty())
	{
		m_pUserToolBar->copyFromTemplate(g_pUserToolBarTemplate);
		if(m_pUserToolBar->isVisible()) m_pUserToolBar->show();
	} else m_pUserToolBar->hide();
}

void KviFrame::applyToolbarOptions()
{
	// Update all the toolbars
	QObjectList * tblist = queryList("QToolBar",0,false);
	if(tblist)
	{
		QObjectListIt tbit(*tblist);

		while(tbit.current())
		{
//			debug(tbit.current()->name());
			QObjectList * l = tbit.current()->queryList("QToolButton",0,false);

			if(l)
			{
				if(!kvi_strEqualCS(tbit.current()->name(),"KviUserToolbar"))
				{
					QObjectListIt it1(*l);
					while(it1.current()){
						((QToolButton *)it1.current())->setUsesTextLabel(g_pOptions->m_bUseTextToolbarComment);
						((QToolButton *)it1.current())->setUsesBigPixmap(g_pOptions->m_bUseBigToolbarPixmaps);
						++it1;
					}
					delete l;
				} else {
					QObjectListIt it1(*l);
					while(it1.current()){
						((QToolButton *)it1.current())->setUsesTextLabel(false);
						((QToolButton *)it1.current())->setUsesBigPixmap(g_pOptions->m_bUseBigUToolbarPixmaps);
						++it1;
					}
					delete l;
				}
			}
	
			++tbit;
		}

		delete tblist;
	}
}

void KviFrame::slot_toggleUserToolBar()
{
	if(m_pUserToolBar->isVisible())m_pUserToolBar->hide();
	else m_pUserToolBar->show();
}

void KviFrame::slot_toggleToolBar()
{
	if(m_pToolBar->isVisible())m_pToolBar->hide();
	else m_pToolBar->show();
}

//============ createStatusBar ============//
void KviFrame::createStatusBar()
{
	m_pStatusBar = new KviStatusBar(this);
	if(g_pOptions->m_bStatusBarVisible)m_pStatusBar->show();
	else m_pStatusBar->hide();
}

void KviFrame::slot_toggleStatusBar()
{
	if(m_pStatusBar->isVisible())
	{
		m_pStatusBar->hide();
		g_pOptions->m_bStatusBarVisible = false;
	} else {
		m_pStatusBar->show();
		setUpLayout(); //trick for qt! (so it really gets added to the layout when shown for the first time)
		g_pOptions->m_bStatusBarVisible = true;
	}
}

//============ createTaskBar ==============//
void KviFrame::createTaskBar()
{
	m_pTaskBar = new KviTaskBar(this,QMainWindow::Bottom);
}

void KviFrame::slot_toggleMdiToolBar()
{
	if(m_pMdiToolBar->isVisible())m_pMdiToolBar->hide();
	else m_pMdiToolBar->show();
}

void KviFrame::slot_toggleTaskBar()
{
	if(m_pTaskBar->isVisible())m_pTaskBar->hide();
	else m_pTaskBar->show();
}

void KviFrame::slot_showScriptCenter()
{
	g_pApp->showScriptCenter(this);
}

void KviFrame::fillRecentServersPopup()
{ fillRecentPopup(m_pRecentServersPopup,g_pOptions->m_szRecentServers); }
void KviFrame::fillRecentChannelsPopup()
{ fillRecentPopup(m_pRecentChannelsPopup,g_pOptions->m_szRecentChannels); }
void KviFrame::fillRecentNicknamesPopup()
{ fillRecentPopup(m_pRecentNicknamesPopup,g_pOptions->m_szRecentNicknames); }

void KviFrame::fillRecentPopup(QPopupMenu * popup,KviStr &data)
{
	popup->clear();
	const char *aux = data.ptr();
	KviStr token;
	while(*aux){
		aux = kvi_extractToken(token,aux,',');
		if(token.hasData()){
			popup->insertItem(token.ptr());
		}
	}
}

void KviFrame::addRecentServer(KviStr &serverName,KviStr &port)
{
	KviStr entry = serverName;
	entry.stripWhiteSpace();
	entry.append(':');
	if(port.isUnsignedNum())entry.append(port);
	else entry.append("6667");

	if(addRecentEntry(g_pOptions->m_szRecentServers,entry)){
		for(KviFrame * f=g_pApp->m_pFrameList->first();f;f=g_pApp->m_pFrameList->next())
			fillRecentServersPopup();
	}
}

void KviFrame::addRecentChannel(const char *chan)
{
	KviStr channel(chan);
	channel.stripWhiteSpace();
	if(addRecentEntry(g_pOptions->m_szRecentChannels,channel)){
		for(KviFrame * f=g_pApp->m_pFrameList->first();f;f=g_pApp->m_pFrameList->next())
			fillRecentChannelsPopup();
	}
}

void KviFrame::addRecentNickname(KviStr &nickname)
{
	nickname.stripWhiteSpace();
	if(addRecentEntry(g_pOptions->m_szRecentNicknames,nickname)){
		for(KviFrame * f=g_pApp->m_pFrameList->first();f;f=g_pApp->m_pFrameList->next())
			fillRecentNicknamesPopup();
	}
}

bool KviFrame::addRecentEntry(KviStr &list,KviStr &entry)
{
	if(list.findFirstIdx(entry.ptr(),false) >= 0)return false;
	int numEntries = list.contains(',') + 1;
	if(numEntries >= KVI_MAX_RECENT_ENTRIES){
		KviStr dummy;
		list.getToken(dummy,',');
	}
	if(!list.isEmpty())list.append(',');
	list.append(entry);
	return true;
}

void KviFrame::connectToRecentServer(int id)
{
	KviStr serverAndPort = m_pRecentServersPopup->text(id);
	serverAndPort.stripWhiteSpace();
	if(serverAndPort.isEmpty())return;
	KviStr server;
	serverAndPort.getToken(server,':');
	if(server.isEmpty())return;
	g_pOptions->m_pServerManager->iWantThisServerToBeCurrent(server.ptr(),serverAndPort.ptr());
	// Jump out of this handler, call it from the main event loop...
	doAsyncServerCommandCall();
}

void KviFrame::joinRecentChannel(int id)
{
	KviStr channel = m_pRecentChannelsPopup->text(id);
	channel.stripWhiteSpace();
	if(channel.isEmpty())return;
	if(m_state != Connected)return;
	if(!m_pSocket->sendFmtData("JOIN %s",channel.ptr()))m_pConsole->output(KVI_OUT_ERROR,"Not connected to server");
}

void KviFrame::setRecentNickname(int id)
{
	KviStr nick = m_pRecentNicknamesPopup->text(id);
	nick.stripWhiteSpace();
	if(nick.isEmpty())return;
	if(m_state != Connected)return;
	if(!m_pSocket->sendFmtData("NICK %s",nick.ptr()))m_pConsole->output(KVI_OUT_ERROR,"Not connected to server");
}



void KviFrame::clearConsoleListBox()
{
	if(!m_pConsole)return;
	if(m_pConsole->m_pListBox->count() > 0)m_pConsole->m_pListBox->partAll();
}

//============ closeEvent ============//
void KviFrame::closeEvent(QCloseEvent *e)
{
	e->accept();
	delete this;
}

//============= createChannel ===============//
KviChannel * KviFrame::createChannel(const char *name)
{
	KviChannel * c = new KviChannel(this,name);
	addWindow(c);
	if(g_pOptions->m_bAutoLogChannels)c->m_pView->toggleLogging();
	return c;
}
//============= createLinksWindow ===============//
KviLinksWindow * KviFrame::createLinksWindow()
{
	__range_invalid(findWindow(KVI_LINKS_WINDOW_NAME));
	KviLinksWindow * c= new KviLinksWindow(this);
	addWindow(c);
	return c;
}
KviLinksWindow * KviFrame::getLinksWindow()
{
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(w->type() == KVI_WND_TYPE_LINKS)return ((KviLinksWindow *)w);
	}
	return createLinksWindow();
}

//============= createListWindow ===============//
KviListWindow * KviFrame::createListWindow()
{
	__range_invalid(findWindow(KVI_LIST_WINDOW_NAME));
	KviListWindow * c= new KviListWindow(this);
	addWindow(c);
	return c;
}

KviListWindow * KviFrame::getListWindow()
{
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(w->type() == KVI_WND_TYPE_LIST)return ((KviListWindow *)w);
	}
	return createListWindow();
}

void KviFrame::activateListWindow()
{
	KviListWindow * w = getListWindow();
	if(activeWindow() != w){
		w->raise();
		w->setFocus();
	}
}

//============= createQuery ===============//
KviQuery * KviFrame::createQuery(const char *name, bool isRemote)
{
	KviQuery * q=new KviQuery(this,name);
	addWindow(q,!g_pOptions->m_bCreateIconifiedQuery);
	if(g_pOptions->m_bCreateIconifiedQuery)q->minimize(false); //do not animate
	if(g_pOptions->m_bAutoLogQueries)q->m_pView->toggleLogging();

	if (g_pOptions->m_bBeepOnNewQuery && g_pOptions->m_bEnableBeeping && isRemote)QApplication::beep();

	return q;
}

KviQuery * KviFrame::raiseOrCreateQuery(const char *name)
{
	KviQuery * query = findQuery(name);
	if(!query)query = createQuery(name, false);
	query->delayedAutoRaise();
	return query;
}

//============= createUWindow ===============//
KviUWindow * KviFrame::createUWindow(const char *name)
{
	KviUWindow * u=new KviUWindow(this,name);
	addWindow(u,true);
	return u;
}

KviDirectoryBrowser * KviFrame::createDirectoryBrowser(const char *dirname)
{
	KviDirectoryBrowser * d = new KviDirectoryBrowser(this,dirname);
	addWindow(d,true);
	return d;
}

//void KviFrame::createNewDirBrowser()
//{
//	createDirectoryBrowser(0);
//}

//============= createDccChat ===============//
KviDccChat * KviFrame::createDccChat(const char *name)
{
	KviDccChat * q=new KviDccChat(this,name);
	addWindow(q,!g_pOptions->m_bCreateIconifiedDccChat);
	if(g_pOptions->m_bCreateIconifiedDccChat)q->minimize(false); //do not animate
	if(g_pOptions->m_bAutoLogChats)q->m_pView->toggleLogging();
	return q;
}

//============= createDccSend ===============//
KviDccSend * KviFrame::createDccSend(const char *name)
{
	KviDccSend * q=new KviDccSend(this,name);
	addWindow(q,!g_pOptions->m_bCreateIconifiedDccSend);
	if(g_pOptions->m_bCreateIconifiedDccSend)q->minimize(false); //do not animate
	if(g_pOptions->m_bAutoLogSends)q->m_pView->toggleLogging();
	return q;
}
//============= createDccVoice ===============//
KviDccVoice * KviFrame::createDccVoice(const char *name)
{
	KviDccVoice * q=new KviDccVoice(this,name);
	addWindow(q,!g_pOptions->m_bCreateIconifiedDccVoice);
	if(g_pOptions->m_bCreateIconifiedDccVoice)q->minimize(false); //do not animate
	return q;
}
//================ addWindow ================//
void KviFrame::addWindow(KviWindow *pWnd,bool bShow)
{
	// The window can be added only once :)
	__range_valid(m_pWinList->findRef(pWnd) == -1);
	// Append the window to the list
	m_pWinList->append(pWnd);
	// Create a taskbar button
	pWnd->setTaskBarButton(m_pTaskBar->addWinButton(pWnd));

	KviWindowPackedProperty * prop = g_pOptions->m_pWinPropertiesList->findProperty(pWnd->caption());
	if(prop){
		KviWindowProperty p;
		g_pOptions->m_pWinPropertiesList->unpackProperty(&p,prop);
		pWnd->setProperties(&p);
		if(p.isDocked){
			attachWindow(pWnd,bShow,false,&(p.rect)); //attach, do not cascade
//			debug("p.rect() = %d,%d,%d,%d",p.rect.x(),p.rect.y(),p.rect.width(),p.rect.height());
			if(p.isMaximized)pWnd->maximize();
		} else {
			pWnd->setGeometry(p.rect);
			pWnd->installAccelerators();
			if(bShow)pWnd->show();
			pWnd->youAreDetached();
			if(bShow)pWnd->setFocus();
		}
	} else attachWindow(pWnd,bShow);

//	connect((QObject *)pWnd->m_pInput, SIGNAL(keyPressed()), this,
//		SLOT(inputKeyPressed()));
//	connect((QObject *)pWnd->m_pInput, SIGNAL(textInput(KviStr *)), this,
//		SLOT(textInput(KviStr *)));
}

//============ attachWindow ============//
void KviFrame::attachWindow(KviWindow *pWnd,bool bShow,bool overrideGeometry,QRect *r)
{
	//Creates a new KviMdiChild in m_pMdi manager
	//Attacches the KviWindow to that child and starts the manager
	//	__enter("attachWindow");
	__range_valid(pWnd->parent() == 0);
	KviMdiChild *lpC=new KviMdiChild(m_pMdi,"mdi_child");
	lpC->setClient(pWnd);
	pWnd->youAreAttached(lpC);
	pWnd->destroyAccelerators();
	m_pTaskBar->windowAttached(pWnd,true);
	if(!overrideGeometry && r)lpC->setGeometry(*r);
	m_pMdi->manageChild(lpC,bShow,overrideGeometry);
	m_pMdi->setFocus();
	__leave("attachWindow");
}

//============= detachWindow ==============//
void KviFrame::detachWindow(KviWindow *pWnd)
{
	//Reparents pWnd to 0
	//Destroys the KviMdiChild that contained it
	__range_valid(pWnd->parent());
	if(!pWnd->parent())return;
	__range_valid(pWnd->parent()->inherits("KviMdiChild"));
	KviMdiChild *lpC=pWnd->mdiParent();
	lpC->unsetClient();
	pWnd->youAreDetached();
	pWnd->installAccelerators();
	m_pTaskBar->windowAttached(pWnd,false);
	m_pMdi->destroyChild(lpC,false); //Do not focus the new top child, we loose focus...
	pWnd->setFocus();
}

//============== closeWindow ==============//
void KviFrame::closeWindow(KviWindow *pWnd)
{
	//Closes a child window. sends no close event : simply deletes it
//	__debug_1arg("KviFrame : deleting window %s",pWnd->caption());
	pWnd->saveProperties();
#ifdef _KVI_DEBUG_CHECK_RANGE_
	__range_valid(m_pWinList->removeRef(pWnd));
#else
	m_pWinList->removeRef(pWnd);
#endif
	if(pWnd == m_pConsole)m_pConsole = 0;
	if(pWnd == m_global.pCurrentActiveWindow)
		m_global.pCurrentActiveWindow = ((pWnd == m_pConsole) ? 0 : m_pConsole);
	pWnd->setTaskBarButton(0);
	m_pTaskBar->removeWinButton(pWnd);
	if(pWnd->isAttached())m_pMdi->destroyChild(pWnd->mdiParent());
	else delete pWnd;
}

//================== findWindow =================//
KviWindow * KviFrame::findWindow(const char *caption)
{
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(kvi_strEqualCI(w->caption(),caption))return w;
	}
	return 0;
}
//================== findChannel =================//
KviChannel * KviFrame::findChannel(const char *chanName)
{
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(w->type() == KVI_WND_TYPE_CHANNEL){
			if(kvi_strEqualCI(w->caption(),chanName))return ((KviChannel *)w);
		}
	}
	return 0;
}
//================== findQuery =================//
KviQuery * KviFrame::findQuery(const char *queryName)
{
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(w->type() == KVI_WND_TYPE_QUERY){
			if(kvi_strEqualCI(w->caption(),queryName))return ((KviQuery *)w);
		}
	}
	return 0;
}
//================== findUWindow =================//
KviUWindow * KviFrame::findUWindow(const char *uwindowName)
{
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(w->type() == KVI_WND_TYPE_UWINDOW){
			if(kvi_strEqualCI(w->caption(),uwindowName))return ((KviUWindow *)w);
		}
	}
	return 0;
}
//================== findWaitingDccSend =================//
KviDccSend * KviFrame::findWaitingDccSend(unsigned short uPort,const char *nick)
{
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(w->type() == KVI_WND_TYPE_SEND){
			if(!((KviDccSend *)w)->m_bThreadRunning){
				if(((KviDccSend *)w)->m_dccSendStruct != 0){
					if((uPort == ((KviDccSend *)w)->m_dccSendStruct->uPort) || g_pOptions->m_bIgnorePortNumberInResumeRequests){
						if(kvi_strEqualCI(((KviDccSend *)w)->m_dccSendStruct->nick.ptr(),nick))return ((KviDccSend *)w);
					}
				}
			}
		}
	}
	return 0;
}
//================== findListeningDccSend =================//
KviDccSend * KviFrame::findListeningDccSend(unsigned short uPort,const char *nick)
{
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(w->type() == KVI_WND_TYPE_SEND){
			if(((KviDccSend *)w)->youAreListening(uPort,nick))return ((KviDccSend *)w);
		}
	}
	return 0;
}
//================== activeWindow ===================//
KviWindow * KviFrame::activeWindow()
{
	__range_valid(windowExists(m_global.pCurrentActiveWindow));
	return m_global.pCurrentActiveWindow ? m_global.pCurrentActiveWindow : m_pConsole;
}
//================== windowExists ? =================//
bool KviFrame::windowExists(KviWindow *pWnd)
{
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(w == pWnd)return true;
	}
	return false;
}

QPopupMenu * KviFrame::windowPopup(KviIrcView * pView,KviWindow * pWnd,bool bIncludeTaskbarPopup)
{
	__range_valid(pWnd);
	__range_valid(pView);
	g_pWindowPopup->clear();
	if(bIncludeTaskbarPopup){
		g_pWindowPopup->insertItem(__tr("&Window"),taskBarPopup(pWnd,false));
		g_pWindowPopup->insertSeparator();
	}
	if(pView->isLogging())g_pWindowPopup->insertItem(__tr("Stop &Logging"),pView,SLOT(toggleLogging()));
	else g_pWindowPopup->insertItem(__tr("Start &Logging (Default File)"),pView,SLOT(toggleLogging()));
	g_pWindowPopup->insertItem(__tr("L&og to File..."),pView,SLOT(logToFile()));
	g_pWindowPopup->insertSeparator();
	if(pView->isBeeping())g_pWindowPopup->insertItem(__tr("Don't &Beep on New Text"),pView,SLOT(toggleBeeping()));
	else g_pWindowPopup->insertItem(__tr("&Beep on New Text"),pView,SLOT(toggleBeeping()));
	if(pView->timestamp())g_pWindowPopup->insertItem(__tr("&Timestamp Off"),pView,SLOT(toggleTimestamp()));
	else g_pWindowPopup->insertItem(__tr("&Timestamp On"),pView,SLOT(toggleTimestamp()));
	if(pView->imagesVisible())g_pWindowPopup->insertItem(__tr("Hide Output &Images"),pView,SLOT(toggleImages()));
	else g_pWindowPopup->insertItem(__tr("Show Output &Images"),pView,SLOT(toggleImages()));
	g_pWindowPopup->insertSeparator();
	g_pWindowPopup->insertItem(__tr("&Find Text..."),pWnd,SLOT(findText()));
	//if(pWnd->isMinimized())g_pWindowPopup->setItemEnabled(id,false);
	g_pWindowPopup->insertItem(__tr("&Clear Text Buffer"),pView,SLOT(clearBuffer()));
	g_pWindowPopup->insertItem(__tr("&Save Buffer to File"),pView,SLOT(saveBufferToFile()));
//#warning "missing setBackgroundImage & setFont, then save these options too in winproperties"
	return g_pWindowPopup;
}

//================ taskBarPopup =================//
QPopupMenu * KviFrame::taskBarPopup(KviWindow *pWnd,bool bIncludeWindowPopup)
{
	//returns the g_pTaskBarPopup filled according to the KviWindow state
	g_pTaskBarPopup->clear();
	if(pWnd->isAttached()){
		g_pTaskBarPopup->insertItem(_i18n_("&Undock"),pWnd,SLOT(detach()));
		g_pTaskBarPopup->insertSeparator();
		if(pWnd->isMinimized() || pWnd->isMaximized())
			g_pTaskBarPopup->insertItem(_i18n_("&Restore"),pWnd,SLOT(restore()));
		if(!pWnd->isMinimized())g_pTaskBarPopup->insertItem(_i18n_("Mi&nimize"),pWnd,SLOT(minimize()));
		if(!pWnd->isMaximized())g_pTaskBarPopup->insertItem(_i18n_("Ma&ximize"),pWnd,SLOT(maximize()));
	} else g_pTaskBarPopup->insertItem(_i18n_("&Dock"),pWnd,SLOT(attach()));
	if(pWnd->type() != KVI_WND_TYPE_CONSOLE){
		g_pTaskBarPopup->insertSeparator();
		g_pTaskBarPopup->insertItem(_i18n_("&Close"),pWnd,SLOT(close()));
	}
	if(pWnd->m_pView && bIncludeWindowPopup){
		// the window has a view...get the window popup
		g_pTaskBarPopup->insertSeparator();
		g_pTaskBarPopup->insertItem(__tr("&Operations"),windowPopup(pWnd->m_pView,pWnd,false));  //alvoid recursion
	}
	return g_pTaskBarPopup;
}

//################################################
// The following functions are called DIRECTLY
// from the KviInput class.
//################################################
void KviFrame::switchWindows(bool bRight)
{
	KviMdiChild * pActMdi = m_pMdi->topChild();
	if(!pActMdi)return;
	KviWindow * pAct = (KviWindow *)pActMdi->m_pClient;
	if(!pAct)return;
	KviTaskBarButton * pNext = m_pTaskBar->getNextWindowButton(bRight,pAct);
	if(!pNext)return;
	if(pNext->m_pWindow->isAttached())taskbarButtonLeftClicked(pNext->m_pWindow,pNext);
	else pNext->m_pWindow->attach();
	
}
//################################################
// The following functions are called DIRECTLY
// from the KviTaskBar class.
//################################################
void KviFrame::taskbarButtonLeftClicked(KviWindow *pWnd,KviTaskBarButton *btn)
{
	if(pWnd->isAttached()){
		if(pWnd->hasFocus() && (pWnd->mdiParent() == m_pMdi->topChild())){
			btn->setOn(false);
			pWnd->minimize();
		} else {
			btn->setOn(true);
			pWnd->restore();
			pWnd->setFocus();
		}
	} else { //not so cool...bu cannot do more...
		btn->setOn(true);
		g_pApp->raiseTopLevelWidget(pWnd);
	}
}
void KviFrame::taskbarButtonRightClicked(KviWindow *pWnd,KviTaskBarButton *btn)
{
	taskBarPopup(pWnd,true)->popup(QCursor::pos());
}

//################################################
// The following functions are called DIRECTLY
// from the KviWindow class.
//################################################

void KviFrame::childWindowGainFocus(KviWindow *pWnd)
{
	m_global.pCurrentActiveWindow = pWnd;
//	if(m_bInSDIMode)m_pSDICloseButton->setEnabled(pWnd->mdiParent()->closeEnabled());
	m_pTaskBar->setActiveButton(pWnd);

}

void KviFrame::childWindowCloseRequest(KviWindow *pWnd)
{
	//Ignore console close requests
	if(pWnd != m_pConsole)
	{
		// Other window types
		if(pWnd->type() == KVI_WND_TYPE_CHANNEL)
			partChannel(pWnd);
		else closeWindow(pWnd);
	}
}

void KviFrame::partChannel(KviWindow *pWnd)
{
	__range_valid(pWnd);
	__range_valid(pWnd->type() == KVI_WND_TYPE_CHANNEL);
	__range_valid(m_pWinList->findRef(pWnd));
	if(((KviChannel *)pWnd)->m_bOnChannel){
		pWnd->outputNoFmt(KVI_OUT_INTERNAL,__tr("Part message sent to server, waiting for reply..."));
		if(g_pOptions->m_szPartMessage.isEmpty())m_pSocket->sendFmtData("PART %s",pWnd->caption());
		else m_pSocket->sendFmtData("PART %s :%s",pWnd->caption(),g_pOptions->m_szPartMessage.ptr());
	} else closeWindow(pWnd);
}

void KviFrame::slot_requestLinks()
{
	if(!m_pSocket->sendData("LINKS"))activeWindow()->outputNoFmt(KVI_OUT_ERROR,__tr("Oops..cannot request LINKS: socket write failed"));
}

bool KviFrame::checkHighlight(const char *msg)
{
	for(KviStr * s=g_pOptions->m_pHighlightWords->first();s;s=g_pOptions->m_pHighlightWords->next()){
		const char *aux = msg;
		char c = tolower(*(s->ptr())); //newer \0 !
		// Run thru the string
		while(*aux){
			while(*aux  && (tolower(*aux) != c))aux++; //run until we find a probable match
			if(*aux){
				// probable match...check
				if(kvi_strEqualCIN(s->ptr(),aux,s->len()))return true;
				else aux++;
			} // else finished the string
		}
	}
	return false;
}

void KviFrame::outputPrivmsg(KviWindow *wnd,int type,const char *talker,
		const char *talker_mask,const char *msg)
{
	__range_valid(wnd);

	KviStr prefix;
	if ((*talker == '@')||(*talker == '%')||(*talker == '+')||(*talker == '-')||(*talker == '*')){
		prefix = *talker;
		talker++;
	}
	if(g_pOptions->m_bWordHighlighting){
		if(checkHighlight(msg))type = KVI_OUT_HIGHLIGHT;
	}
	if(g_pOptions->m_bUseExtendedPrivmsgView){
		// <PREFIX>nick[!user@host]<POSTFIX>This is a test message
		if(g_pOptions->m_bShowUserAndHostInPrivmsgs){
			if(g_pOptions->m_bPrivmsgFormatNickIsALink)
			{
				while(*talker_mask && (*talker_mask != '!'))talker_mask++;
				wnd->output(type,__tr("%s%s\r!%s%s\r%s\r%s%s"),g_pOptions->m_szExtPrivmsgPrefix.ptr(),
					prefix.ptr(),g_pOptions->m_szPrivmsgFormatNickLinkCommand.ptr(),talker,talker_mask,
					g_pOptions->m_szExtPrivmsgPostfix.ptr(),msg);
			} else
				wnd->output(type,__tr("%s%s%s%s%s"),g_pOptions->m_szExtPrivmsgPrefix.ptr(),prefix.ptr(),
					talker_mask,g_pOptions->m_szExtPrivmsgPostfix.ptr(),msg);
		} else {
			if(g_pOptions->m_bPrivmsgFormatNickIsALink)
				wnd->output(type,__tr("%s%s\r!%s\r%s\r%s%s"),g_pOptions->m_szExtPrivmsgPrefix.ptr(),
					prefix.ptr(),g_pOptions->m_szPrivmsgFormatNickLinkCommand.ptr(),talker,
					g_pOptions->m_szExtPrivmsgPostfix.ptr(),msg);
			else
				wnd->output(type,__tr("%s%s%s%s%s"),g_pOptions->m_szExtPrivmsgPrefix.ptr(),prefix.ptr(),
					talker,g_pOptions->m_szExtPrivmsgPostfix.ptr(),msg);
		}
	} else {
		if(g_pOptions->m_bPrivmsgFormatNickIsALink)
			wnd->output(type,__tr("<%s\r!%s\r%s\r> %s"),prefix.ptr(),g_pOptions->m_szPrivmsgFormatNickLinkCommand.ptr(),talker,msg);
		else
			wnd->output(type,__tr("<%s%s%s> %s"),prefix.ptr(),talker,msg);
	}
}

//################################################
// Userlist updates...
//################################################

void KviFrame::startUpdatingUserList()
{
	if(m_iUserListUpdateTimer != 0)return; //Already running
	if(m_state != Connected)return; //Cannot do it, man...
	if(g_pOptions->m_bDisableUpdateUserList)return; //User do not want this...Lag ?
	KviStr buffer;
	m_pUserList->getFirstFiveNicksToUpdate(buffer);
	if(buffer.isEmpty())return; //None to update
	// Do the first update now...
	if(!m_pSocket->sendFmtData("USERHOST %s",buffer.ptr()))return; //Oppps...something wrong ?
	m_global.bUserHostInProgress = true;
	if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Starting user list updates (%s)"),buffer.ptr());
	__debug_1arg("Updating user list: %s",buffer.ptr());
	if(g_pOptions->m_iUserListUpdateTimeoutInSecs < 5)g_pOptions->m_iUserListUpdateTimeoutInSecs = 5; //Minimum...avoid self-lag
	m_iUserListUpdateTimer = startTimer(g_pOptions->m_iUserListUpdateTimeoutInSecs * 1000);

}

void KviFrame::stopUpdatingUserList()
{
	if(m_iUserListUpdateTimer == 0)return; //Already stopped
	if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("User list updates stopped"));
	killTimer(m_iUserListUpdateTimer);
	m_global.bUserHostInProgress = false;
	m_iUserListUpdateTimer = 0;
}

void KviFrame::timerEvent(QTimerEvent *e)
{
	if(e->timerId() == m_iUserListUpdateTimer){
		if(g_pOptions->m_bDisableUpdateUserList){
			stopUpdatingUserList();
			return; //User do not want this...Lag ?
		}
		if(m_state == Connected){
			if(m_global.bUserHostInProgress)return; //Need to wait , the last call has not returned yet...
			KviStr buffer;
			m_pUserList->getFirstFiveNicksToUpdate(buffer);
			if(buffer.hasData()){
				if(m_pSocket->sendFmtData("USERHOST %s",buffer.ptr())){
					if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Updating user list (%s)"),buffer.ptr());
					m_global.bUserHostInProgress = true;
					return; //all ok
				}
			}
		}
		stopUpdatingUserList();
	} else if(e->timerId() == m_iNotifyListUpdateTimer){
		if(m_global.szNotifyListString.hasData()){
			m_pSocket->sendFmtData("ISON %s",m_global.szNotifyListString.ptr());
		} else stopNotifyList();
	} else QMainWindow::timerEvent(e);
}

//=====================================================================================================
// Start NOTIFY or WATCH LIST
// If g_pOptions->m_bEnableDalNetStyleNotifyMethod is true
// use the WATCH list , otherwise use the old ISON ping.
//

void KviFrame::startNotifyOrWatchList()
{
	__range_valid(m_global.szNotifyListString.isEmpty());
	__range_valid(m_iNotifyListUpdateTimer == 0);

	if(g_pOptions->m_bEnableNotifyList && (m_state == Connected)){
		// Ok....should run...
		if(g_pOptions->m_bEnableDalNetStyleNotifyMethod && m_global.bServerSupportsWatchList){
			if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Notify : watch list is enabled"));
			startWatchList();
		} else {
			if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Notify : notify list is enabled"));
			startNotifyList();
		}
	}
}

void KviFrame::startWatchList()
{
	__range_valid(m_global.szNotifyListString.isEmpty());
	__range_valid(m_iNotifyListUpdateTimer == 0);
	__range_valid(g_pOptions->m_bEnableDalNetStyleNotifyMethod);

	if(g_pOptions->m_bEnableNotifyList && g_pOptions->m_bEnableDalNetStyleNotifyMethod &&
		(m_state == Connected) && m_global.szNotifyListString.isEmpty() && m_global.bServerSupportsWatchList)
	{
		// was not running
		g_pOptions->m_pRegUsersDb->getWatchString(m_global.szNotifyListString);
		if(m_global.szNotifyListString.hasData()){
			if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Starting watch list..."));
			m_pSocket->sendFmtData("WATCH %s",m_global.szNotifyListString.ptr());
		}
	}
}

void KviFrame::startNotifyList()
{
	__range_valid(m_global.szNotifyListString.isEmpty());
	__range_valid(m_iNotifyListUpdateTimer == 0);
//	__range_invalid(g_pOptions->m_bEnableDalNetStyleNotifyMethod);

	if(g_pOptions->m_bEnableNotifyList && !g_pOptions->m_bEnableDalNetStyleNotifyMethod && (m_iNotifyListUpdateTimer == 0) && (m_state == Connected)){ //Not started yet...
		// Try to start
		g_pOptions->m_pRegUsersDb->getNotifyString(m_global.szNotifyListString);
		if(m_global.szNotifyListString.hasData()){
			// Ok...users to notify
			m_pSocket->sendFmtData("ISON %s",m_global.szNotifyListString.ptr());
			if(g_pOptions->m_iNotifyListCheckTimeoutInSecs < 3)g_pOptions->m_iNotifyListCheckTimeoutInSecs = 3;
			m_iNotifyListUpdateTimer = startTimer(g_pOptions->m_iNotifyListCheckTimeoutInSecs * 1000);
			if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Starting notify list..."));
			return;
		}
	}
	// If we're still here...we could not start it for some reason...clear the vars...
	m_global.szNotifyListString = "";
	m_iNotifyListUpdateTimer = 0;
}

void KviFrame::stopNotifyOrWatchList()
{
	if(!m_pConsole)return; // this might be called from the constructor
	if(m_iNotifyListUpdateTimer)stopNotifyList();
	else if(m_global.szNotifyListString.hasData())stopWatchList();
	// Just to be sure...clear everything...
	clearConsoleListBox();
	m_global.szNotifyListString = "";
	m_iNotifyListUpdateTimer = 0;
}

void KviFrame::stopNotifyList()
{
//	__range_valid(m_iNotifyListUpdateTimer != 0);
//	__range_valid(m_global.szNotifyListString.hasData());

	if(m_iNotifyListUpdateTimer != 0){
		// was running
		killTimer(m_iNotifyListUpdateTimer);
		if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Notify list stopped"));
	}
	clearConsoleListBox();
	m_global.szNotifyListString = "";
	m_iNotifyListUpdateTimer = 0;
}

void KviFrame::stopWatchList()
{
	__range_valid(m_iNotifyListUpdateTimer == 0);
	__range_valid(m_global.szNotifyListString.hasData());
	if(m_global.szNotifyListString.hasData() && (m_state == Connected)){
		if(m_global.bServerSupportsWatchList)m_pSocket->sendFmtData("WATCH clear"); //avoid clearing is unsupported
		if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Watch list stopped"));
	}
	clearConsoleListBox();
	m_global.szNotifyListString = "";
}

void KviFrame::updateNotifyOrWatchList()
{
	if((!g_pOptions->m_bEnableNotifyList) || (m_state != Connected)){
		// kill everything...
		if(m_iNotifyListUpdateTimer)stopNotifyList();
		if(m_global.szNotifyListString.hasData())stopWatchList();
		clearConsoleListBox();
		m_global.szNotifyListString = "";
		m_iNotifyListUpdateTimer = 0;
		return;
	}
	// Something is enabled now...
	if(g_pOptions->m_bEnableDalNetStyleNotifyMethod && m_global.bServerSupportsWatchList){
		// Watch list is enabled now!
		if(m_iNotifyListUpdateTimer){
			// Notify list was running
			stopNotifyList(); // Kill the notify list
			startWatchList(); // and start the Watch list
		} else {
			// Notify list was not running
			if(m_global.szNotifyListString.isEmpty()){
				// Watch list was not running yet...start it
				startWatchList();
			} else {
				// Watch list was already running...update it
				KviStr tmp;
				g_pOptions->m_pRegUsersDb->getWatchString(tmp);
				// Calculate the Diff :)
				if(kvi_strEqualCI(tmp.ptr(),m_global.szNotifyListString.ptr()))return; //no changes
				// Watch list changed!
				// For now just resend the new list to the server
				if(tmp.isEmpty()){
					// ooops..no more nicks to watch...
					if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Watch list changed ... no users to watch"));
					stopWatchList();
					return;
				}
				m_pSocket->sendFmtData("WATCH clear");
				clearConsoleListBox();
				m_pSocket->sendFmtData("WATCH %s",tmp.ptr());
				m_global.szNotifyListString = tmp;
				if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Watch list changed ... sending modifications to server"));
			}
		}
	} else {
		// Notify list is enabled now!
		if((m_iNotifyListUpdateTimer == 0) && m_global.szNotifyListString.hasData()){
			// The watch list was running before
			stopWatchList();
			startNotifyList();
		} else {
			if(m_iNotifyListUpdateTimer == 0){
				// Nothing was running before
				startNotifyList();
			} else {
				// Notify list was already running...
				KviStr tmp;
				g_pOptions->m_pRegUsersDb->getNotifyString(tmp);
				if(kvi_strEqualCI(tmp.ptr(),m_global.szNotifyListString.ptr()))return; // No changes...
				if(tmp.isEmpty()){
					if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Notify list changed ... no users to look for"));
					stopNotifyList();
					return;
				}
				// Ok...users to notify
				m_global.szNotifyListString = tmp;
				m_pSocket->sendFmtData("ISON %s",m_global.szNotifyListString.ptr());
				if(g_pOptions->m_iNotifyListCheckTimeoutInSecs < 3)g_pOptions->m_iNotifyListCheckTimeoutInSecs = 3;
				killTimer(m_iNotifyListUpdateTimer);
				clearConsoleListBox();
				m_iNotifyListUpdateTimer = startTimer(g_pOptions->m_iNotifyListCheckTimeoutInSecs * 1000);
				if(g_pOptions->m_bBeVerbose)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Notify list changed ... sending modifications to server"));
			}
		}
	}
}


//################################################
// The following functions are called DIRECTLY
// from the KviIrcView class.
//################################################


//================================================
// TEXT SELECTION
//================================================

void KviFrame::ircViewTextSelected(KviIrcView *pView,KviWindow *pWnd,const char *text)
{
	// The user selected some text in a view
	__range_valid(text);
	__range_valid(pWnd);
	__range_valid(pView);
	if(!(*text)){
		m_global.szCurrentSelectionText = ""; //selected nothing ?
		return;
	}
	m_global.szCurrentSelectionText = text; //remember it now...

	if(g_pOptions->m_bDoNotShowThePopupOnTextSelected){
		copyCurrentSelectionTextToClipboard();
		return;
	}

	// Extract the first token
	KviStr firstLine;
	kvi_extractToken(firstLine,text,'\n');
	firstLine.stripWhiteSpace();
	KviStr firstToken;
	kvi_extractToken(firstToken,firstLine.ptr(),' ');

	g_pViewSelectPopup->clear();
	g_pViewSelectPopup->insertItem(__tr("&Copy to Clipboard"),this,SLOT(copyCurrentSelectionTextToClipboard()));

	firstToken.stripWhiteSpace();

	if(firstToken.hasData()){
		g_pViewSelectPopup->insertSeparator();
		KviStr tmp(KviStr::Format,"&Help on %s",firstToken.ptr());
		g_pViewSelectPopup->insertItem(tmp.ptr(),this,SLOT(doHelpOnCurrentlySelectedTextToken()));
		tmp.sprintf(__tr("&Search Help for %s..."),firstToken.ptr());
		g_pViewSelectPopup->insertItem(tmp.ptr(),this,SLOT(doHelpSearchOnCurrentlySelectedTextToken()));
		g_pViewSelectPopup->insertSeparator();
		if(tmp.contains('.')){
			tmp.sprintf("R&esolve Host %s",firstToken.ptr());
			g_pViewSelectPopup->insertItem(tmp.ptr(),this,SLOT(hostCurrentlySelectedToken()));
		}
		tmp.sprintf("&Run Program %s",firstToken.ptr());
		g_pViewSelectPopup->insertItem(tmp.ptr(),this,SLOT(execCurrentlySelectedText()));

		if(isConnected()){
			g_pViewSelectPopup->insertSeparator();
			firstToken.stripWhiteSpace();
			tmp.sprintf("&Whois User %s",firstToken.ptr());
			g_pViewSelectPopup->insertItem(tmp.ptr(),this,SLOT(whoisCurrentlySelectedToken()));
			tmp.sprintf("&Query User %s",firstToken.ptr());
			g_pViewSelectPopup->insertItem(tmp.ptr(),this,SLOT(queryCurrentlySelectedToken()));
			tmp.sprintf("&DCC Chat with User %s",firstToken.ptr());
			g_pViewSelectPopup->insertItem(tmp.ptr(),this,SLOT(dccChatCurrentlySelectedToken()));
			tmp.sprintf("D&NS User %s",firstToken.ptr());
			g_pViewSelectPopup->insertItem(tmp.ptr(),this,SLOT(dnsCurrentlySelectedToken()));
			if((*(firstToken.ptr()) == '#') || (*(firstToken.ptr()) == '&')
				|| (*(firstToken.ptr()) == '!')){
				// A channel name ?
				tmp.sprintf("&Join Channel %s",firstToken.ptr());
				g_pViewSelectPopup->insertItem(tmp.ptr(),this,SLOT(joinCurrentlySelectedToken()));				
			}
		}
		if(pWnd->type() == KVI_WND_TYPE_CHANNEL){
			tmp.sprintf("Set %s &Topic to %s",pWnd->caption(),firstToken.ptr());
			g_pViewSelectPopup->insertItem(tmp.ptr(),this,SLOT(topicCurrentlySelectedToken()));			
		}
	}

	m_global.szCurrentSelectionToken = firstToken;
	m_global.pCurrentSelectionWindow = pWnd;
	g_pViewSelectPopup->popup(QCursor::pos());
}

void KviFrame::actionForCurrentlySelectedTextToken(const char *action)
{
	if(!m_global.pCurrentSelectionWindow)return;
	KviStr token = m_global.szCurrentSelectionToken;
	if(token.hasData()){
		// Just redirect to the user parser
		token.prepend(' ');
		token.prepend(action);
		token.prepend('/');
		m_pUserParser->parseUserCommand(token,m_global.pCurrentSelectionWindow);
	}
}


void KviFrame::whoisCurrentlySelectedToken()
{ actionForCurrentlySelectedTextToken("WHOIS"); }

void KviFrame::queryCurrentlySelectedToken()
{ actionForCurrentlySelectedTextToken("QUERY"); }

void KviFrame::dccChatCurrentlySelectedToken()
{ actionForCurrentlySelectedTextToken("DCC CHAT"); }

void KviFrame::hostCurrentlySelectedToken()
{ actionForCurrentlySelectedTextToken("HOST"); }

void KviFrame::dnsCurrentlySelectedToken()
{ actionForCurrentlySelectedTextToken("DNS"); }

void KviFrame::joinCurrentlySelectedToken()
{ actionForCurrentlySelectedTextToken("JOIN"); }

void KviFrame::doHelpOnCurrentlySelectedTextToken()
{ actionForCurrentlySelectedTextToken("HELP"); }

void KviFrame::topicCurrentlySelectedToken()
{
	if(!m_global.pCurrentSelectionWindow)return;
	KviStr token = m_global.szCurrentSelectionToken;
	if(token.hasData()){
		// Just redirect to the user parser
		token.prepend(" ");
		token.prepend(m_global.pCurrentSelectionWindow->caption());
		token.prepend("/TOPIC ");
		m_pUserParser->parseUserCommand(token,m_global.pCurrentSelectionWindow);
	}
}

void KviFrame::doHelpSearchOnCurrentlySelectedTextToken()
{
	if(m_global.szCurrentSelectionText.hasData()){
		// Just redirect to the user parser
		KviStr tmp = m_global.szCurrentSelectionText;
		tmp.prepend("/HELP -s ");
		m_pUserParser->parseUserCommand(tmp,m_global.pCurrentSelectionWindow);
	}	
}

void KviFrame::execCurrentlySelectedText()
{
	if(m_global.szCurrentSelectionText.hasData()){
		// Just redirect to the user parser
		KviStr tmp = m_global.szCurrentSelectionText;
		tmp.prepend("/EXEC -s ");
		m_pUserParser->parseUserCommand(tmp,m_global.pCurrentSelectionWindow);
	}	
}

void KviFrame::copyCurrentSelectionTextToClipboard()
{
	QApplication::clipboard()->setText(m_global.szCurrentSelectionText.ptr());
}

void KviFrame::windowPopupRequested(KviIrcView * pView,KviWindow *pWnd)
{
	// true includes taskBarPopup !
	windowPopup(pView,pWnd,true)->popup(QCursor::pos());
}

//
// CONNECTION STUFF
//

//##################################################
// Socket handlers
// Called directly from KviIrcSocket class
//##################################################

void KviFrame::socketEvent(int sockEventId)
{
	switch(sockEventId){
		case KVI_SOCKEVENT_Connected:
			handleConnectedToIrcServer();
			m_pSysTrayOnLineTimer->run();
			break;
		case KVI_SOCKEVENT_ConnectionFailed:
			connectionAttemptFailed(m_pSocket->getError());
			break;
		case KVI_SOCKEVENT_Disconnected:
			m_pConsole->output(KVI_OUT_SOCKET,__tr("Disconnected: %s"),kvi_getErrorString(m_pSocket->m_error));
			handleDisconnectedFromServer();
			break;
//		case KVI_SOCKEVENT_OneMessageSent: //Newer sent
//			m_pConsole->output(KVI_OUT_INTERNAL,"OneMessageSent");
//			break;
		case KVI_SOCKEVENT_PartialSocketWrite:
			if(g_pOptions->m_bBeVerbose)m_pConsole->outputNoFmt(KVI_OUT_INTERNAL,__tr("Partial socket write"));
			break;
		case KVI_SOCKEVENT_MessageTruncated:
			if(g_pOptions->m_bBeVerbose)m_pConsole->outputNoFmt(KVI_OUT_INTERNAL,__tr("Outgoing data buffer truncated to 510 bytes"));
			break;
		case KVI_SOCKEVENT_UsingDefaultPort6667:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("Invalid server port %s , using default: 6667"),m_global.szCurrentServerPort.ptr());
			m_global.szCurrentServerPort = "6667";
			break;
		case KVI_SOCKEVENT_UsingDefaultPort1080:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("Invalid proxy port %s , using default: 1080"),m_global.szCurrentProxyPort.ptr());
			m_global.szCurrentProxyPort = "1080";
			break;

		case KVI_SOCKEVENT_ConnectedToProxy:
			m_pConsole->output(KVI_OUT_SOCKET,
				_i18n_("Connected to proxy: %s (%s:%s)"),
				m_global.szCurrentProxyHost.ptr(),
				m_global.szCurrentProxyIp.ptr(),
				m_global.szCurrentProxyPort.ptr());
			break;
		case KVI_SOCKEVENT_UsingProxyProtoV4:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("Using proxy protocol SOCKSv4"));
			break;
		case KVI_SOCKEVENT_UsingProxyProtoV5:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("Using proxy protocol SOCKSv5"));
			break;
		case KVI_SOCKEVENT_UsingProxyProtoHttp:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("Using proxy protocol HTTP/1.0"));
			break;
		case KVI_SOCKEVENT_ProxySentV4Request:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("SOCKSv4: Sent proxy connection request : Waiting for acknowledgement..."));
			break;
		case KVI_SOCKEVENT_ProxyReply90RequestGranted:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("SOCKSv4: Received proxy acknowledgement: 90 Request granted"));
			break;
		case KVI_SOCKEVENT_ProxySentV5MethodRequest:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("SOCKSv5: Sent proxy authentication method request : Waiting for acknowledgement..."));
			break;
		case KVI_SOCKEVENT_ProxyMethodSelectedNoAuth:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("SOCKSv5: Proxy: No auth selected"));
			break;
		case KVI_SOCKEVENT_ProxyMethodSelectedUserPass:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("SOCKSv5: Proxy: User/Pass auth method selected"));
			break;
		case KVI_SOCKEVENT_ProxySentUserAndPass:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("SOCKSv5: Sent proxy user/pass : Waiting for acknowledgement..."));
			break;
		case KVI_SOCKEVENT_ProxyAuthOK:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("SOCKSv5: Proxy authentication successful"));
			break;
		case KVI_SOCKEVENT_ProxySentTargetData:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("SOCKSv5: Sent target connection data : Waiting for acknowledgement..."));
			break;
		case KVI_SOCKEVENT_ProxyReply00Success:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("SOCKSv5: Received proxy acknowledgement: 00 Request granted"));
			break;
		case KVI_SOCKEVENT_ProxySentHttpConnectRequest:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("HTTP/1.0: Sent connect request : Waiting for reply..."));
			break;
		case KVI_SOCKEVENT_ProxyReplyHttp200Success:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("HTTP/1.0: Received proxy reply: 200 Connection established"));
			break;
		case KVI_SOCKEVENT_ContactingHost:
			m_pConsole->output(KVI_OUT_SOCKET,_i18n_("Connection in progress..."));
			break;
		default: //Huh?
			break;
	}
}

// #################################################
// Connected to server
// #################################################

void KviFrame::handleConnectedToIrcServer()
{
	// Fire the event...
	bool bHalt = false;
	if(g_pEventManager->eventEnabled(KviEvent_OnConnect)){
		KviStr tmp(KviStr::Format,"%s %s %s %s",
			m_global.szCurrentServerHost.ptr(),
			m_global.szCurrentServerIp.ptr(),
			m_global.szCurrentServerPort.ptr(),
			m_global.szCurrentNetwork.ptr());
		bHalt = m_pUserParser->callEvent(KviEvent_OnConnect,m_pConsole,tmp);
	}
	// Output
	if(!bHalt){
		m_pConsole->output(KVI_OUT_INTERNAL,
			_i18n_("Connected to server: %s (%s:%s) [%s]"),
			m_global.szCurrentServerHost.ptr(),
			m_global.szCurrentServerIp.ptr(),
			m_global.szCurrentServerPort.ptr(),
			m_global.szCurrentNetwork.ptr());
	}
	// And start logging in
	resolveLocalHost();
	setState(LoggingIn);
	loginToIrcServer();
	addRecentServer(m_global.szCurrentServerHost,m_global.szCurrentServerPort);
}

void KviFrame::handleDisconnectedFromServer()
{
	// Save the "cause" of the disconnect
	// If the user caused the connection termination m_global.bSentQuit is true
	// otherwise if g_pOptions->m_bAutoReconnectOnDisconnect is true
	// try reconnecting to the same server
	bool bEventuallyReconnect = (!m_global.bSentQuit);

	stopUpdatingUserList();

	if(bEventuallyReconnect && g_pOptions->m_bRejoinChannelsOnReconnect)
	{
		if(m_pChannelsBeforeDisconnect)delete m_pChannelsBeforeDisconnect;
		m_pChannelsBeforeDisconnect = new QList<KviStr>;
		m_pChannelsBeforeDisconnect->setAutoDelete(true);
	}

	QList<KviWindow> *winlist=new QList<KviWindow>;
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		switch(w->type())
		{
			case KVI_WND_TYPE_QUERY:
				if(g_pOptions->m_bCloseWindowsOnDisconnect)closeWindow(w);
				else w->output(KVI_OUT_SOCKET,__tr("Disconnected"));
			break;
			case KVI_WND_TYPE_CHANNEL:
				if(bEventuallyReconnect && g_pOptions->m_bRejoinChannelsOnReconnect)
					m_pChannelsBeforeDisconnect->append(new KviStr(w->caption()));
				if(g_pOptions->m_bCloseWindowsOnDisconnect)winlist->append(w);
				else {
					((KviChannel *)w)->m_bOnChannel=false;
					w->output(KVI_OUT_SOCKET,__tr("Disconnected"));
				}
			break;
			case KVI_WND_TYPE_LIST:
				((KviListWindow *)w)->notifyDisconnected();
			break;
		}
	}
	// Clean up those window that need to be cleaned.
	// Doing it in a separate loop prevents the iteration of m_pWinList
	// getting messed up when we start dereferencing its elements.
	for(KviWindow *w=winlist->first();w;w=winlist->next())
	{
		closeWindow(w);
	}

	// Fire the event...
	bool bHalt = false;
	if(g_pEventManager->eventEnabled(KviEvent_OnDisconnect)){
		KviStr tmp(KviStr::Format,"%s %s %s %s",
			m_global.szCurrentServerHost.ptr(),
			m_global.szCurrentServerIp.ptr(),
			m_global.szCurrentServerPort.ptr(),
			m_global.szCurrentNetwork.ptr());
		bHalt = m_pUserParser->callEvent(KviEvent_OnDisconnect,m_pConsole,tmp);
	}
	// Output
	if(!bHalt){
		m_pConsole->output(KVI_OUT_INTERNAL,__tr("Connection terminated: %s (%s:%s) [%s]"),
			m_global.szCurrentServerHost.ptr(),
			m_global.szCurrentServerIp.ptr(),
			m_global.szCurrentServerPort.ptr(),
			m_global.szCurrentNetwork.ptr());
	}

	resetGlobals();

	if(g_pOptions->m_bCloseWindowsOnDisconnect)m_pUserList->clearList(); //remove all nicks...
	m_pSysTrayOnLineTimer->stop();
	m_pSysTrayIoLed->clear();
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(w->type() == KVI_WND_TYPE_LINKS)((KviLinksWindow *)w)->reset();
	}

	setState(Ready);

	if(bEventuallyReconnect){
		if(g_pOptions->m_bAutoReconnectOnDisconnect){
			setState(WaitingForReconnect);
			m_pConsole->output(KVI_OUT_INTERNAL,__tr("Reconnect attempt in %d seconds"),KVI_RECONNECT_TIMEOUT_IN_SECS);
			__range_invalid(m_pReconnectTimer->isActive());
			// And do a single shot
			m_pReconnectTimer->start(KVI_RECONNECT_TIMEOUT_IN_SECS * 1000,true);
		}
	}
}

void KviFrame::reconnectToServer()
{
	__range_valid(m_state == WaitingForReconnect);
//	m_pReconnectTimer->stop();
	connectToServer();
}

void KviFrame::getArtificialNickname(KviStr &nickToChange)
{
	static const char subst[] =
	{
			'o', '0',
			'i', '|',
			'e', '3',
			'l', '1',
			'a', '4',
			's', '5',
	};

	__range_valid(nickToChange.hasData());

// iterate through the nick characters and try to find the ones we can change to
// others, looking similar
	for( int j = 0; j < nickToChange.len(); j++ )
	{
		for( int i = 0; i < (int)(sizeof(subst)/sizeof(subst[0])); i++ )
		{
			if( tolower(nickToChange.at(j)) == subst[i++])
			{
				nickToChange.cut(j, 1);
				nickToChange.insert( j, subst[i] );
				return;
			}
		}
	}
// other methods:
// turn my_nick into [my_nick]
	if( nickToChange.len() > 7 )
	{
		if( nickToChange.at(0) != '[' )
		{
			nickToChange.insert(0,'[');
			nickToChange.insert(nickToChange.len(), ']' );
			return;
		}
		else	// turn [my_nick] into {my_nick}
		{
			nickToChange.cut(0, 1);
			nickToChange.cut(nickToChange.len(), 1);
			nickToChange.insert(0, '{');
			nickToChange.insert(nickToChange.len(), '}' );
			return;
		}
	}
	else
// if everything fails append '_'
		nickToChange.append('_');
}

void KviFrame::loginToIrcServer()
{
	// Find the nickname
	g_pOptions->m_szNickName1.stripWhiteSpace();
	g_pOptions->m_szNickName2.stripWhiteSpace();
	if(g_pOptions->m_szNickName1.isEmpty())g_pOptions->m_szNickName1 = KVI_DEFAULT_NICKNAME1;
	if(g_pOptions->m_szNickName2.isEmpty())g_pOptions->m_szNickName2 = KVI_DEFAULT_NICKNAME2;
	if(kvi_strEqualCS(g_pOptions->m_szNickName1.ptr(),g_pOptions->m_szNickName2.ptr())){
		// Refuse equal nicknames
		getArtificialNickname(g_pOptions->m_szNickName2);
	}
	m_global.szCurrentNick = g_pOptions->m_szNickName1;
	// Find the username
	if(g_pOptions->m_szUserName.hasData())m_global.szCurrentUserName = g_pOptions->m_szUserName;
	else m_global.szCurrentUserName = KVI_DEFAULT_USERNAME;
	// Find the realname
	if(g_pOptions->m_szRealName.hasData())m_global.szCurrentRealName = g_pOptions->m_szRealName;
	else m_global.szCurrentRealName = KVI_DEFAULT_REALNAME;
	// Say that we're logging in
	if(!g_pOptions->m_bUseTelnetProxy){
		m_pConsole->output(KVI_OUT_INTERNAL,__tr("Logging in as %s (%s : %s)"),
			m_global.szCurrentNick.ptr(),
			m_global.szCurrentUserName.ptr(),
			m_global.szCurrentRealName.ptr());
		if(m_global.szCurrentPassword.hasData()){
			m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Sending %s as password"),
				m_global.szCurrentPassword.ptr());
			m_pSocket->sendFmtData("PASS %s",m_global.szCurrentPassword.ptr());
		}
		// Login
		m_pSocket->sendFmtData("USER %s %s %s :%s",
			m_global.szCurrentUserName.ptr(),
			m_global.szLocalHostIp.ptr(),
			m_global.szCurrentServerHost.ptr(),
			m_global.szCurrentRealName.ptr());
		m_pSocket->sendFmtData("NICK %s",m_global.szCurrentNick.ptr());
	} else m_pConsole->output(KVI_OUT_INTERNAL,
	__tr("Not logging in automatically. Send USER, NICK, and (if needed) PASS manually"));
	// And wait for reply...
}

void KviFrame::loginNickRefused()
{
	// This one is called from the KviServerParser class
	// when we're still logging in and we receive an ERR_NICKNAMEINUSE
	// or a similar error message that says that the server
	// cannot accept our login nickname
	if(kvi_strEqualCS(m_global.szCurrentNick.ptr(),g_pOptions->m_szNickName1.ptr())){
		// First nickname refused , use the second one
		m_pConsole->output(KVI_OUT_INTERNAL,
			__tr("Login nickname %s refused, using alternate nick %s"),
			m_global.szCurrentNick.ptr(),g_pOptions->m_szNickName2.ptr());
			m_global.szCurrentNick = g_pOptions->m_szNickName2;
	} else {
		// Second or artificial nickname refused. Use another artificial one
		KviStr tmp = m_global.szCurrentNick;
		getArtificialNickname(tmp);
		m_pConsole->output(KVI_OUT_INTERNAL,
			__tr("Login nickname %s refused, using artificial alternate nick %s"),
			m_global.szCurrentNick.ptr(),tmp.ptr());
		m_global.szCurrentNick = tmp;
	}
	m_pSocket->sendFmtData("NICK %s",m_global.szCurrentNick.ptr());
	// And wait for reply...
}

void KviFrame::userModeChange(const char *umode)
{
	while((*umode)&&(*umode == ' '))umode++;

	bool bAdd = true;
	while(*umode && (*umode != ' ')){
		if((*umode == '+')||(*umode == '-')){
			bAdd = (*umode == '+');
			umode++;
			continue;
		}
		// else an user mode
		int idx = m_global.szCurrentUserMode.findFirstIdx(*umode);
		if(idx != -1){
			if(!bAdd)m_global.szCurrentUserMode.cut(idx,1);
		} else {
			if(bAdd)m_global.szCurrentUserMode += (*umode);
		}
		umode++;
	}
	updateCaption();
	if(m_global.szCurrentUserMode.hasData()){
		KviStr tmp(KviStr::Format,__tr("Changed umode to +%s"),m_global.szCurrentUserMode.ptr());
		m_pStatusBar->tempText(tmp.ptr(),KVI_TEMP_TEXT_TIMEOUT);
	} else m_pStatusBar->tempText(__tr("Umode reset"),KVI_TEMP_TEXT_TIMEOUT);
}

//##################################################
// Toolbar slots
//##################################################

void KviFrame::slot_connectToServer()
{
	//The button 'connect' was pressed...
	switch(m_state){
		case Ready:               connectToServer();      break;
		case LoggingIn:           disconnectFromServer(); break;
		case Connected:           disconnectFromServer(); break;
		case WaitingForConnect:   abortConnection();      break;
		case WaitingForDns:       abortDns();             break;
		case WaitingForReconnect: abortReconnect();       break;
	}
}

//##################################################
// Connection management
//##################################################

//
// Connect stuff
//

// Flow chart of function calls up to the connect() call

// connectToServer()
//    |
// have a serve name ? ---No--> connectionAttemptFailed()
//    |Yes
// use a proxy ? --------Yes--> have a proxy to contact ? -----Yes--> lookupProxyHostname()
//    |                          |                                     |     |
//    | <-do not use proxy <---No- <---------------Failed to run kvidns-   kvidns
//    |                                                                      |
//    | <------------------- haveProxyIp() <----Resolved proxy host--------proxyNameResolved()
//    |                                                                      |
// lookupServerHostname() <-do not use proxy-------------Proxy not found------
//    |         |
//    |         ----Failed to run kvidns and to resolve host "by hand"->conncetionAttemptFailed()
// kvidns
//    |
//    |
// serverNameResolved() ---Host not found----> connectionAttemptFailed()
//    |
// haveServerIp()
//    |
// still use a proxy? ---Yes---> connect(proxyIp)-----
//    |                                              |
//    -------------------No----> connect(serverIp)----
//                                                   |
//                                        handleConnectedToIrcServer();

void KviFrame::connectToServer()
{
	// We always start from here.
	__range_valid((m_state == Ready) || (m_state == WaitingForReconnect));
	// Reset everything...
	resetGlobals();
	// Find the server that we want to contact
	if(!findCurrentServerToContact()){
		connectionAttemptFailed(KVI_ERROR_NoServerEntries);
		return;
	}
	// Here we go!
	m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Attempting connection to %s (%s) on port %s"),
		m_global.szCurrentServerHost.ptr(),m_global.szCurrentNetwork.ptr(),m_global.szCurrentServerPort.ptr());

#ifdef COMPILE_NEED_IPV6
	if(m_global.bIpV6Mode) {
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Entering IPv6 mode... good luck :)"));
		if(g_pOptions->m_bUseProxy||g_pOptions->m_bProxyProtoHttp) {
			(g_pOptions->m_bUseProxy) ? g_pOptions->m_bUseProxy : g_pOptions->m_bProxyProtoHttp = false;
			m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Proxy support for IPv6 not implemented : Resuming standard connection mode"));
		}
	}
#endif

	// Check if we have to use a proxy
	if(g_pOptions->m_bUseProxy||g_pOptions->m_bProxyProtoHttp){
		// Yes...
		if(findCurrentProxyHost()){
			// OK...have a proxy to try...
			m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Proxy host: %s on port %s"),
				m_global.szCurrentProxyHost.ptr(),m_global.szCurrentProxyPort.ptr());
			// Lookup the proxy hostname (the secondary way will rejoin the normal
			// process in lookupServerHostname())
			lookupProxyHostname();
			return;
		} else {
			// No proxy entry...resume direct connection.
			m_pConsole->output(KVI_OUT_ERROR,
				_i18n_("Cannot find the proxy host to contact : Resuming direct connection"));
			(g_pOptions->m_bUseProxy) ? g_pOptions->m_bUseProxy : g_pOptions->m_bProxyProtoHttp = false;
		}
	}
	// OK...time to lookup the server hostname
	lookupServerHostname();
}

bool KviFrame::isValidStringIp(const char *szIp)
{
#ifdef COMPILE_NEED_IPV6
	if(m_global.bIpV6Mode)return kvi_isValidStringIp_V6(szIp);
#endif
	return kvi_isValidStringIp(szIp);
}

void KviFrame::lookupProxyHostname()
{
	// Alternative way to lookupServerHostname()
	// Check if the ip was cached
	if(isValidStringIp(m_global.szCurrentProxyIp.ptr())){
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Using cached proxy IP address..."));
		haveProxyIp();
		return;
	}
	// Check if the hostname is an IP
	if(isValidStringIp(m_global.szCurrentProxyHost.ptr())){
		m_global.szCurrentProxyIp = m_global.szCurrentProxyHost;
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Using absolute proxy IP address..."));
		haveProxyIp();
		return;
	}
	// Nope...must resolve the host
	m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Looking up proxy hostname..."));
	__range_invalid(m_pDns); //must be zero!
	m_pDns = new KviAsyncDns();
	connect(m_pDns,SIGNAL(dnsFinished(KviDnsStruct *)),this,SLOT(proxyNameResolved(KviDnsStruct *)));

	if(!m_pDns->resolve(m_global.szCurrentProxyHost.ptr())){
		// Failed to start?
		delete m_pDns;
		m_pDns = 0;
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Cannot start the DNS thread : Blocking DNS query"));
		if(!m_pSocket->getHostByName(m_global.szCurrentProxyHost.ptr(),m_global.szCurrentProxyIp)){
			// Invalid host...or something
			m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Cannot resolve proxy hostname : Resuming direct connection"));
			(g_pOptions->m_bUseProxy) ? g_pOptions->m_bUseProxy : g_pOptions->m_bProxyProtoHttp = false;
			// Forget proxy names...
			resetProxyGlobals();
			// Jump back to normal connection process
			lookupServerHostname();
			return;
		}
		haveProxyIp();
	} else {
		if(g_pOptions->m_bBeVerbose)m_pConsole->outputNoFmt(KVI_OUT_INTERNAL,__tr("DNS thread started"));
		// Now wait for kvidns to return
		setState(WaitingForDns);
	}
}

void KviFrame::lookupServerHostname()
{
	// Check if the ip was cached
	if(isValidStringIp(m_global.szCurrentServerIp.ptr())){
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Using cached server IP address..."));
		haveServerIp();
		return;
	}
	// Check if the hostname is an ip address
	if(isValidStringIp(m_global.szCurrentServerHost.ptr())){
		m_global.szCurrentServerIp = m_global.szCurrentServerHost;
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Using absolute server IP address..."));
		haveServerIp();
		return;
	}
	// Nope  , must resolve the host
	m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Looking up server hostname..."));
	__range_invalid(m_pDns); //must be zero!
	m_pDns = new KviAsyncDns();
	connect(m_pDns,SIGNAL(dnsFinished(KviDnsStruct *)),this,SLOT(serverNameResolved(KviDnsStruct *)));
#ifdef COMPILE_NEED_IPV6
	if(!m_pDns->resolve(m_global.szCurrentServerHost.ptr(),0,m_global.bIpV6Mode)){
#else
	if(!m_pDns->resolve(m_global.szCurrentServerHost.ptr())){
#endif
		delete m_pDns;
		m_pDns = 0;
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Cannot start the DNS thread : Trying blocking DNS query"));
#ifdef COMPILE_NEED_IPV6
		if(!m_pSocket->getHostByName(m_global.szCurrentServerHost.ptr(),m_global.szCurrentServerIp,m_global.bIpV6Mode)){
#else
		if(!m_pSocket->getHostByName(m_global.szCurrentServerHost.ptr(),m_global.szCurrentServerIp)){
#endif
			connectionAttemptFailed(m_pSocket->getError());
			return;
		}
		haveServerIp();
	} else {
		if(g_pOptions->m_bBeVerbose)m_pConsole->outputNoFmt(KVI_OUT_INTERNAL,__tr("DNS thread started"));
		setState(WaitingForDns);
	}
}

bool KviFrame::findCurrentServerToContact()
{
	if ((m_state == WaitingForReconnect) && m_global.szCurrentNetwork.hasData())return true;
//	if (!m_global.szConnectNetwork.hasData()){
		KviIrcNetwork * net = g_pOptions->m_pServerManager->getCurrentNetwork();
		if(!net)return false;
		KviIrcServer * srv = net->currentServer();
		if(!srv)return false;
		m_global.szCurrentNetwork    = net->name();
//		m_global.szConnectNetwork    = m_global.szCurrentNetwork;
		m_global.szCurrentServerHost = srv->szHost;
//		m_global.szConnectServerHost = m_global.szCurrentServerHost;
		m_global.szCurrentServerIp   = srv->szIp;
//		m_global.szConnectServerIp   = m_global.szCurrentServerIp;
		m_global.szCurrentServerPort = srv->szPort;
//		m_global.szConnectServerPort = m_global.szCurrentServerPort;
		m_global.szCurrentPassword   = srv->szPassword;
//		m_global.szConnectPassword   = m_global.szCurrentPassword;
#ifdef COMPILE_NEED_IPV6
		m_global.bIpV6Mode           = srv->bIpV6;
//		m_global.bCIpV6Mode          = m_global.bIpV6Mode;
#endif
//	} else if(m_global.szConnectNetwork.hasData()){
//		m_global.szCurrentNetwork    = m_global.szConnectNetwork;
//		m_global.szCurrentServerHost = m_global.szConnectServerHost;
//		m_global.szCurrentServerIp   = m_global.szConnectServerIp;
//		m_global.szCurrentServerPort = m_global.szConnectServerPort;
//		m_global.szCurrentPassword   = m_global.szConnectPassword;
//#ifdef COMPILE_NEED_IPV6
//		m_global.bIpV6Mode           = m_global.bCIpV6Mode;
//#endif
//	}
	return true;
}

bool KviFrame::findCurrentProxyHost()
{
	if ((m_state == WaitingForReconnect) && m_global.szCurrentProxyHost.hasData())return true;
//	if (!m_global.szConnectProxyHost.hasData()){
		KviIrcProxy * prx = g_pOptions->m_pProxyManager->currentProxy();
		if(!prx)return false;
		m_global.szCurrentProxyHost     = prx->szHost;
//		m_global.szConnectProxyHost     = m_global.szCurrentProxyHost;
		m_global.szCurrentProxyIp       = prx->szIp;
//		m_global.szConnectProxyIp       = m_global.szCurrentProxyIp;
		m_global.szCurrentProxyPort     = prx->szPort;
//		m_global.szConnectProxyPort     = m_global.szCurrentProxyPort;
		m_global.szCurrentProxyPassword = prx->szPassword;
//		m_global.szConnectProxyPassword = m_global.szCurrentProxyPassword;
		m_global.szCurrentProxyUsername = prx->szUsername;
//		m_global.szConnectProxyUsername = m_global.szCurrentProxyUsername;
//	} else if (m_global.szConnectProxyHost.hasData()){
//		m_global.szCurrentProxyHost     = m_global.szConnectProxyHost;
//		m_global.szCurrentProxyIp       = m_global.szConnectProxyIp;
//		m_global.szCurrentProxyPort     = m_global.szConnectProxyPort;
//		m_global.szCurrentProxyPassword = m_global.szConnectProxyPassword;
//		m_global.szCurrentProxyUsername = m_global.szConnectProxyUsername;
//	}
	return true;
}

bool KviFrame::findNextServerToContact()
{
	KviIrcNetwork * net = g_pOptions->m_pServerManager->getCurrentNetwork();
	if(!net)return false;
	KviIrcServer * srv = net->nextServer();
	if(!srv)return false;
	m_global.szCurrentNetwork    = net->name();
	m_global.szCurrentServerHost = srv->szHost;
	m_global.szCurrentServerIp   = srv->szIp;
	m_global.szCurrentServerPort = srv->szPort;
	m_global.szCurrentPassword   = srv->szPassword;
	return true;
}

void KviFrame::connectionAttemptFailed(int errNum)
{
	//Dead....reset the entire state
	m_pConsole->output(KVI_OUT_ERROR,_i18n_("[Socket error]: %s"),kvi_getErrorString(errNum));
	m_pConsole->output(KVI_OUT_ERROR,_i18n_("Connection attempt failed (%s)"),m_global.szCurrentServerHost.ptr());
	resetGlobals();
	setState(Ready);

	if(g_pOptions->m_bTryNextServerOnConnectFailed){
		if(findNextServerToContact()){
			setState(WaitingForReconnect);
			m_pConsole->output(KVI_OUT_INTERNAL,__tr("Trying next server in the current network: %s"),m_global.szCurrentServerHost.ptr());
			m_pConsole->output(KVI_OUT_INTERNAL,__tr("New connection attempt in %d seconds. Click the 'Connect' button to abort."),KVI_RECONNECT_TIMEOUT_IN_SECS);
			__range_invalid(m_pReconnectTimer->isActive());
			// And do a single shot
			m_pReconnectTimer->start(KVI_RECONNECT_TIMEOUT_IN_SECS * 1000,true);
		}
	}

}

void KviFrame::serverNameResolved(KviDnsStruct * dns)
{
	//Slot called by the KviAsyncDns class.
	//We resolved the hostname.
	//If we get a DNS error here , it has no sense to try a blocking DNS call
	//we will get the same error.
	__range_valid(dns->dnsParent = m_pDns);
	if(dns->iError != KVI_ERROR_Success){
//		int error;
//		ptr->getError(query,&error);
		connectionAttemptFailed(dns->iError);
		delete m_pDns;
		m_pDns = 0;
		return;
	}
	m_global.szCurrentServerIp = dns->szIp;
//	ptr->getResult(query,hostname,m_global.szCurrentServerIp,alias1,alias2);
	if(dns->szHostname != m_global.szCurrentServerHost){
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Server real hostname is %s"),dns->szHostname.ptr());
		m_global.szCurrentServerHost = dns->szHostname;
	}
	//Save the server IP for next time...
	g_pOptions->m_pServerManager->updateServerIp(m_global.szCurrentNetwork.ptr(),
			dns->szHostToResolve.ptr(),m_global.szCurrentServerIp.ptr());
	//Curious part...just some fun
	if((!dns->szAlias1.isEmpty()) && (dns->szAlias1 != "*")){
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Server primary alias is %s"),dns->szAlias1.ptr());
		if((!dns->szAlias2.isEmpty()) && (dns->szAlias2 != "*")){
			m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Server secondary alias is %s"),dns->szAlias2.ptr());
		}
	}
	//ok...go ahead
	delete m_pDns;
	m_pDns = 0;
	haveServerIp();
}

void KviFrame::proxyNameResolved(KviDnsStruct *dns)
{
	//Slot called by the KviDns class.
	//We resolved the hostname.
	//If we get a DNS error here , it has no sense to try a blocking DNS call
	//we will get the same error.
	__range_valid(dns->dnsParent == m_pDns);
//	KviStr query,hostname,alias1,alias2;
	if(dns->iError != KVI_ERROR_Success){
//		int error;
//		ptr->getError(query,&error);
		m_pConsole->output(KVI_OUT_ERROR,_i18n_("[Socket error] : %s"),kvi_getErrorString(dns->iError));
		m_pConsole->output(KVI_OUT_INTERNAL,
			_i18n_("Cannot resolve proxy hostname : Resuming direct connection"));
		(g_pOptions->m_bUseProxy) ? g_pOptions->m_bUseProxy : g_pOptions->m_bProxyProtoHttp = false;
		delete m_pDns;
		m_pDns = 0;
		resetProxyGlobals();
		lookupServerHostname();
		return;
	}
	m_global.szCurrentProxyIp = dns->szIp;
//	ptr->getResult(query,hostname,m_global.szCurrentProxyIp,alias1,alias2);
	if(dns->szHostname != m_global.szCurrentProxyHost){
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Proxy real hostname is %s"),dns->szHostname.ptr());
		m_global.szCurrentProxyHost = dns->szHostname;
	}
	//Save the server IP for next time...
	g_pOptions->m_pProxyManager->updateProxyIp(dns->szHostToResolve.ptr(),m_global.szCurrentProxyIp.ptr());
	//Curious part...just some fun
	if((!dns->szAlias1.isEmpty()) && (dns->szAlias1 != "*")){
		m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Proxy primary alias is %s"),dns->szAlias1.ptr());
		if((!dns->szAlias2.isEmpty()) && (dns->szAlias2 != "*")){
			m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Proxy secondary alias is %s"),dns->szAlias2.ptr());
		}
	}
	//ok...go ahead
	delete m_pDns;
	m_pDns = 0;
	haveProxyIp();
}

void KviFrame::resolveLocalHost()
{
	//Finds the local hostname
	//In the worst case sets m_global.szLocalHostIp to 127.0.0.1
	//On IRC it is not so important , but on DCC it will be.
#ifdef COMPILE_NEED_IPV6
	if(!m_pSocket->getLocalHostIp(m_global.szLocalHostIp,m_global.bIpV6Mode)){
		m_global.szLocalHostIp = g_pOptions->m_szLocalHostIp.isEmpty() ? (m_global.bIpV6Mode ? "::1" : "127.0.0.1") : g_pOptions->m_szLocalHostIp.ptr();
#else
	if(!m_pSocket->getLocalHostIp(m_global.szLocalHostIp)){
		m_global.szLocalHostIp = g_pOptions->m_szLocalHostIp.isEmpty() ? "127.0.0.1" : g_pOptions->m_szLocalHostIp.ptr();
#endif
		m_pConsole->output(KVI_OUT_ERROR,_i18n_("Unable to resolve local host %s, using %s"),kvi_getErrorString(m_pSocket->getError()),
			m_global.szLocalHostIp.ptr());
		return;
	}
	m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Local host is %s"),m_global.szLocalHostIp.ptr());
}

void KviFrame::haveProxyIp()
{
	m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Proxy hostname resolved to %s"),m_global.szCurrentProxyIp.ptr());
	// Ok...go back standard way now...
	lookupServerHostname();
}

void KviFrame::haveServerIp()
{
	// The last in the connection sequence...
	// We have a good server Ip here...
	// and we can try to contact a server.
	// Next function called in this class will be socketEvent() (from KviIrcSocket)
	// or abortConnection();
	m_pConsole->output(KVI_OUT_INTERNAL,_i18n_("Server hostname resolved to %s"),m_global.szCurrentServerIp.ptr());

	// If m_global.szCurrentProxyIp is a valid Ip we are going to connect
	// to the server using a proxy.
	KviProxyStruct prx;
	KviProxyStruct *ptr=0;
	if(m_global.szCurrentProxyIp.hasData() && isValidStringIp(m_global.szCurrentProxyIp.ptr())){
		prx.szIp       = m_global.szCurrentProxyIp;
		prx.szPort     = m_global.szCurrentProxyPort;
		prx.szPassword = m_global.szCurrentProxyPassword;
		prx.szUsername = m_global.szCurrentProxyUsername;
		prx.bProtoV5   = g_pOptions->m_bProxyProtoV5;
		prx.bProtoHttp = g_pOptions->m_bProxyProtoHttp;
		ptr            = &prx;
	}
#ifdef COMPILE_NEED_IPV6
	if(m_pSocket->connectTo(m_global.szCurrentServerIp.ptr(),m_global.szCurrentServerPort.ptr(),ptr,m_global.bIpV6Mode)){
#else
	if(m_pSocket->connectTo(m_global.szCurrentServerIp.ptr(),m_global.szCurrentServerPort.ptr(),ptr)){
#endif
		setState(WaitingForConnect);
	} else {
		connectionAttemptFailed(m_pSocket->getError());
	}
}

void KviFrame::ircLoginSuccesful(KviStr &szAcceptedNickname)
{
	// This one is called from the KviServerParser class
	// when the first numeric message is received.
	if(szAcceptedNickname.hasData()){
		if(!kvi_strEqualCI(szAcceptedNickname.ptr(),m_global.szCurrentNick.ptr())){
			// The server changed our nickname in some way...update it
			m_pConsole->output(KVI_OUT_INTERNAL,__tr("The server changed your nickname to %s"),
				szAcceptedNickname.ptr());
			m_global.szCurrentNick = szAcceptedNickname;
		}
	}
	// Temp hack...updated when we join the first channel
	m_global.szCurrentMaskFromServer.sprintf("%s!%s@%s",m_global.szCurrentNick.ptr(),
		m_global.szCurrentUserName.ptr(),m_global.szLocalHostIp.ptr());

	setState(Connected);
	// Set the requested user modes now.
	KviStr szModes;
	if(g_pOptions->m_bModeWOnConnect)szModes+="w";
	if(g_pOptions->m_bModeIOnConnect)szModes+="i";
	if(g_pOptions->m_bModeSOnConnect)szModes+="s";
	if(szModes.hasData()){
		m_pConsole->output(KVI_OUT_INTERNAL,__tr("Setting umode +%s"),szModes.ptr());
		m_pSocket->sendFmtData("MODE %s +%s",szAcceptedNickname.ptr(),szModes.ptr());
	} else m_pSocket->sendFmtData("USERHOST %s",szAcceptedNickname.ptr()); //Need to know exactly our mask
	startNotifyOrWatchList();

	// Fire the event...
	bool bHalt = false;
	if(g_pEventManager->eventEnabled(KviEvent_OnIrc)){
		bHalt = m_pUserParser->callEvent(KviEvent_OnIrc,m_pConsole,m_global.szCurrentNick);
	}
	if(!bHalt)m_pConsole->output(KVI_OUT_INTERNAL,__tr("Login operations complete: happy ircing :)"));
	
	KviStr allChans;
//		QList<KviWindow> l(*m_pWinList);
//		l.setAutoDelete(false);
	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
		if(w->type() == KVI_WND_TYPE_CHANNEL){
			if(m_pChannelsBeforeDisconnect == 0){
				if(allChans.hasData())allChans.append(',');
				allChans.append(w->caption());
			}
			w->output(KVI_OUT_SOCKET,__tr("Attempting to rejoin..."));
		}
	}

	if(m_pChannelsBeforeDisconnect){
		if(g_pOptions->m_bRejoinChannelsOnReconnect)
		{
			for(KviStr * s =  m_pChannelsBeforeDisconnect->first();s;s = m_pChannelsBeforeDisconnect->next())
			{
				if(allChans.hasData())allChans.append(',');
				allChans.append(s->ptr());
			}
		}
		delete m_pChannelsBeforeDisconnect;
		m_pChannelsBeforeDisconnect = 0;
	}
	
	if(allChans.hasData()){
		m_pConsole->output(KVI_OUT_INTERNAL,__tr("Rejoining channels (%s)..."),allChans.ptr());
		m_pSocket->sendFmtData("JOIN %s",allChans.ptr());
	}

//	// Allow requesting the channel list now
//	for(KviWindow *w=m_pWinList->first();w;w=m_pWinList->next()){
//		if(w->type() == KVI_WND_TYPE_LIST){
//			((KviListWindow *)w)->notifyConnected();
//			return;
//		}
//	}
}

//
// Disconnect stuff
//

void KviFrame::disconnectFromServer()
{
	__range_valid(m_state == Connected || m_state == LoggingIn );

	if(g_pOptions->m_bBrutalQuit || m_global.bSentQuit){
		m_pSocket->abortConnection();
		m_global.bSentQuit = true;

		handleDisconnectedFromServer();
	} else {

		if(g_pOptions->m_szQuitMessage.hasData()){
			m_pSocket->sendFmtData("QUIT :%s",g_pOptions->m_szQuitMessage.ptr());
			m_pConsole->output(KVI_OUT_INTERNAL,__tr("Sent QUIT (%s)"),g_pOptions->m_szQuitMessage.ptr());
		} else {
			m_pSocket->sendData("QUIT");
			m_pConsole->output(KVI_OUT_INTERNAL,__tr("Sent QUIT"));
		}

		m_pConsole->output(KVI_OUT_INTERNAL,__tr("Waiting for server to close the connection..."));
//		m_pConsole->output(KVI_OUT_INTERNAL,__tr("Press the 'disconnect' button again to force a 'brutal quit'"));

		m_global.bSentQuit = true;
	}
}

void KviFrame::abortDns()
{
	__range_valid(m_state == WaitingForDns);
	__range_valid(m_pDns);
//	__range_valid(m_pDns->isActive());
	m_pConsole->output(KVI_OUT_INTERNAL,__tr("Connection aborted"));
	if(m_pDns)delete m_pDns;
	m_pDns = 0;
	resetGlobals();
	setState(Ready);
}

void KviFrame::abortConnection()
{
	__range_valid(m_state == WaitingForConnect);
	m_pSocket->abortConnection();
	m_pConsole->output(KVI_OUT_INTERNAL,__tr("Connection aborted"));
	resetGlobals();
	setState(Ready);
}

void KviFrame::abortReconnect()
{
	__range_valid(m_state == WaitingForReconnect);
	m_pConsole->output(KVI_OUT_INTERNAL,__tr("Reconnect aborted"));
	__range_valid(m_pReconnectTimer->isActive());
	if(m_pReconnectTimer->isActive())m_pReconnectTimer->stop();
	resetGlobals();
	setState(Ready);
}

//##################################################
// Utils
//##################################################

void KviFrame::addMultimediaFileOffert(const KviStr &offert)
{
	KviStr *s = new KviStr(offert);
	s->stripWhiteSpace();
	m_global.pMultimediaFileOfferts->append(s);
	if(m_pMultimediaFileOffertTimer->isActive())m_pMultimediaFileOffertTimer->stop();
	m_pMultimediaFileOffertTimer->start(120000); // 2 mins offert
}

void KviFrame::cleanupMultimediaFileOffertList()
{
	while(!m_global.pMultimediaFileOfferts->isEmpty())m_global.pMultimediaFileOfferts->removeFirst();
	__range_valid(m_pMultimediaFileOffertTimer->isActive());
	m_pMultimediaFileOffertTimer->stop();
}

void KviFrame::findMultimediaFileOffert(KviStr &retPath,KviStr &filename)
{
	for(KviStr *s=m_global.pMultimediaFileOfferts->first();s;s=m_global.pMultimediaFileOfferts->next())
	{
		if(s->len() >= filename.len())
		{
			// xoom.mp3 len = 8
			// /usr/build/xoom.mp3 len = 19
			if(kvi_strEqualCI(filename.ptr(),(char *)(s->ptr() + (s->len() - filename.len()))))
			{
				retPath = s->ptr();
				return;
			}
		}
	}
}


void KviFrame::resetGlobals()
{
	m_global.szCurrentServerHost    = "";
	m_global.szCurrentNetwork       = "";
	m_global.szCurrentServerIp      = "";
	m_global.szCurrentServerPort    = "";
	m_global.szCurrentPassword      = "";
	m_global.szLocalHostIp          = "127.0.0.1";
	m_global.szCurrentNick          = "";
	m_global.szCurrentUserName      = "";
	m_global.szCurrentRealName      = "";
	m_global.szCurrentSelectionText = "";
	m_global.szCurrentMaskFromServer= "";
	m_global.szLocalHostName        = "";
	m_global.szCurrentUserMode      = "";
	m_global.bServerSupportsEMode   = true; //Assume that it supports it until we receive an error
	m_global.bServerSupportsWatchList = true; //Same as above
	m_global.bUserHostInProgress    = false;
	m_global.bSentQuit              = false;
	m_global.szNotifyListString     = "";
#ifdef COMPILE_NEED_IPV6
	m_global.bIpV6Mode              = false;
#endif
	while(!m_global.pDnsWhoisPending->isEmpty())m_global.pDnsWhoisPending->removeFirst();
	while(!m_global.pMultimediaFileOfferts->isEmpty())m_global.pMultimediaFileOfferts->removeFirst();
	if(m_pMultimediaFileOffertTimer->isActive())
	{
		m_pMultimediaFileOffertTimer->stop();
	}

	if(m_pUserParser)m_pUserParser->m_pAsyncWhoisController->clearList();

	stopNotifyOrWatchList();

	resetProxyGlobals();
}

void KviFrame::resetProxyGlobals()
{
	m_global.szCurrentProxyHost     = "";
	m_global.szCurrentProxyIp       = "";
	m_global.szCurrentProxyPort     = "";
	m_global.szCurrentProxyPassword = "";
	m_global.szCurrentProxyUsername = "";
}

void KviFrame::setState(KviIrcConnectionState state)
{
//	__range_valid(m_state != state); //sanity check, to be removed later (at least on connect errors it is false)
	m_state = state;
	switch(state){
		case Ready:
			m_pToolBar->setConnectButtonState(KVI_CONNECT_BUTTON_STATE_READY);
			m_pMenuBar->setConnectItemState(false);
			m_pMenuBar->enableOnLinePopup(false);
			break;
		case WaitingForDns:
			m_pToolBar->setConnectButtonState(KVI_CONNECT_BUTTON_STATE_BUSY);
			m_pMenuBar->setConnectItemState(false);
			m_pMenuBar->enableOnLinePopup(false);
			break;
		case WaitingForConnect:
			m_pToolBar->setConnectButtonState(KVI_CONNECT_BUTTON_STATE_BUSY);
			m_pMenuBar->setConnectItemState(false);
			m_pMenuBar->enableOnLinePopup(false);
			break;
		case LoggingIn:
			m_pToolBar->setConnectButtonState(KVI_CONNECT_BUTTON_STATE_BUSY);
			m_pMenuBar->enableOnLinePopup(false);
			break;
		case Connected:
			m_pToolBar->setConnectButtonState(KVI_CONNECT_BUTTON_STATE_CONNECTED);
			m_pMenuBar->setConnectItemState(true);
			m_pMenuBar->enableOnLinePopup(true);
			break;
		case WaitingForReconnect:
			m_pToolBar->setConnectButtonState(KVI_CONNECT_BUTTON_STATE_BUSY);
			m_pMenuBar->setConnectItemState(false);
			m_pMenuBar->enableOnLinePopup(false);
			break;
	}
	updateCaption();
}

void KviFrame::updateCaption()
{
	const char *captionText = g_pOptions->m_szCaptionMessage.hasData() ? g_pOptions->m_szCaptionMessage.ptr() : "KVIrc";
	KviStr tmp = captionText;
	if(m_bInSDIMode){
		KviMdiChild * lpC = m_pMdi->topChild();
		if(lpC){
			__range_valid(lpC->state() == KviMdiChild::Maximized);
			KviStr tmp2(KviStr::Format,"[%s] ",lpC->caption());
			tmp.prepend(tmp2.ptr());
		}
	}
	switch(m_state){
		case Ready:
			break;
		case WaitingForDns:
			tmp.append(__tr(": Waiting for DNS"));
			break;
		case WaitingForConnect:
			tmp.append(__tr(": Connection in progress"));
			break;
		case WaitingForReconnect:
			tmp.append(__tr(": Waiting for reconnect"));
			break;
		case LoggingIn:
			tmp.append(__tr(": Login in progress"));
			break;
		case Connected:
			if(m_global.szCurrentUserMode.hasData()){
				KviStr tmp2(KviStr::Format,__tr("[%s (+%s) on %s] "),m_global.szCurrentNick.ptr(),
					m_global.szCurrentUserMode.ptr(),m_global.szCurrentServerHost.ptr());
				tmp.prepend(tmp2.ptr());
			} else {
				KviStr tmp2(KviStr::Format,__tr("[%s on %s] "),m_global.szCurrentNick.ptr(),
					m_global.szCurrentServerHost.ptr());
				tmp.prepend(tmp2.ptr());
			}
			break;
	}

	setCaption(tmp.ptr());
}

void KviFrame::mdiManagerSDIModeChange(bool bEntering)
{
//	debug("sdiModeChange %d",bEntering);
	if(m_bInSDIMode != bEntering)
	{
		if(bEntering)
		{
			__range_invalid(m_pSDICloseButton->isVisible());
			m_pSDICloseButton->show();
			m_pSDIMinimizeButton->show();
			m_pSDIRestoreButton->show();
			adjustSDIButtons();
		} else {
			__range_valid(m_pSDICloseButton->isVisible());
			m_pSDICloseButton->hide();
			m_pSDIMinimizeButton->hide();
			m_pSDIRestoreButton->hide();
		}
		m_bInSDIMode = bEntering;
	}
	if(bEntering)
	{
		KviMdiChild * lpC = m_pMdi->topChild();
		if(lpC)m_pSDICloseButton->setEnabled(lpC->closeEnabled());
	}
	updateCaption();
}

void KviFrame::adjustSDIButtons()
{
	int menuWidth = width();
	int buttonSize = 16;
	int border = (m_pMenuBar->height() - 16) / 2;
	if(border < 1)
	{
		border = 1;
		buttonSize = m_pMenuBar->height() - 2;
	}
	m_pSDICloseButton->setGeometry(menuWidth - (buttonSize + 1),border,buttonSize,buttonSize);
	m_pSDIRestoreButton->setGeometry(menuWidth - ((buttonSize * 2) + 2),border,buttonSize,buttonSize);
	m_pSDIMinimizeButton->setGeometry(menuWidth - ((buttonSize * 3) + 3),border,buttonSize,buttonSize);
}

void KviFrame::resizeEvent(QResizeEvent *e)
{
	adjustSDIButtons();
	QMainWindow::resizeEvent(e);
}

void KviFrame::doAsyncServerCommandCall()
{
	// This is called from the /SERVER command parsing routine
	// It needs to "jump out" of that code and call the
	// slot_connectedToServer from the main event loop;
	// just to avoid return segfaults :)
	QTimer::singleShot(100,this,SLOT(asyncServerCommand()));
}

void KviFrame::asyncServerCommand()
{
	if(m_state != Ready){
		// Still connected...Force a brutal QUIT...
		m_pSocket->sendFmtData("QUIT :%s",__tr("Changing server..."));
		m_global.bSentQuit = true;
		slot_connectToServer(); //will call disconnect...
	}
	slot_connectToServer();
}

//void KviFrame::inputKeyPressed()
//{
//	if(m_pIdleTimer->isActive())
//		m_pIdleTimer->stop();
//
//	if(!m_userIdle)
//		m_pIdleTimer->start(g_pOptions->m_iOnIdleAfter * 1000);
//}

void KviFrame::textInput(KviStr * buff)
{
//#warning "Add $1 to OnIdleStop - how long user has been idle"
	if(!g_pOptions->m_bCancelIdleOnAnyInput)
	{
		if(buff->stripLeftWhiteSpace().at(0) == '/' || buff->stripLeftWhiteSpace().at(0) == g_pOptions->m_cPersonalCommandPrefix)
			return; // if it's a command don't trigger OnIdleStop
	}
	// ok...first of all..If we were idle, exit this state
	if(m_userIdle)
	{
		m_userIdle = false;
		g_pApp->anyUserParser()->callEvent( \
			KviEvent_OnIdleStop, activeWindow(),
			KviStr(";-)"));
		// and start the timer for the next idle state
		
	} // else we were NOT idle yet...just restart the timer
	restartIdleTimer();
}

void KviFrame::triggerOnIdleEvent()
{
	// we're idle now
	m_pIdleTimer->stop();
	m_userIdle = true;

	if(g_pEventManager->eventEnabled(KviEvent_OnIdleStart))
	{
		g_pApp->anyUserParser()->callEvent(KviEvent_OnIdleStart, activeWindow(),
			KviStr(KviStr::Format, "%d", g_pOptions->m_iOnIdleAfter));
	}

}

void KviFrame::restartIdleTimer()
{
	m_pIdleTimer->stop();
	if(g_pEventManager->eventEnabled(KviEvent_OnIdleStart) || g_pEventManager->eventEnabled(KviEvent_OnIdleStop))
	{
		m_pIdleTimer->start(g_pOptions->m_iOnIdleAfter * 1000);
	}
}

//##################################################
// Internal handlers
//##################################################

void KviFrame::focusInEvent(QFocusEvent *)
{
	m_pMdi->setFocus();
}

#include "m_kvi_frame.moc"
