/*
	miscutils.cpp - Misc functions
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <sys/types.h>

#include "configuration.h"
#include "miscutils.h"
#include "mmainwindow.h"
#include "msystemtray.h"

#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>

#include <kaction.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <knotifyclient.h>
#include <kpassivepopup.h>
#include <kpushbutton.h>
#include <krun.h>
#include <kwindowinfo.h>

// public

void MiscUtils::closeCDTray()
{
	if (kshutdownrc->cdTrayClose)
		runShellCommand(kshutdownrc->cdTrayCloseCommand, KProcess::Block);
}

void MiscUtils::customMessage(const QString &text, const QString &testCommand)
{
	QString c;
	if (testCommand.isNull())
		c = kshutdownrc->customMessageCommand;
	else
		c = testCommand;

	if (kshutdownrc->customMessageEnabled)
	{
		c.replace("%appname", "KShutDown");
		c.replace("%text", text);
		c.replace("%title", ks_main->caption());
		runShellCommand(c);
	}
}

QString MiscUtils::formatDateTime(const int secs, const QString &format)
{
	if ((secs >= 86400) || (secs < 0)) // 24h
		return ">24";

	int h = secs / 3600;
	int m = (secs / 60) % 60;
	int s = secs % 60;
	QTime t(h, m, s);

	if (format.isNull())
		return t.toString();

	return t.toString(format);
}

QString MiscUtils::formatDateTime(const QDateTime &dt)
{
	QDateTime now = QDateTime::currentDateTime();

	return
		dt.toString(
			KGlobal::locale()->use12Clock()
				? "hh:mm ap, dddd"
				: "hh:mm, dddd"
		) +
		" (+" + formatDateTime(now.secsTo(dt), "hh:mm") + ")";
}

bool MiscUtils::isRestricted(const QString &key)
{
	return !kapp->authorize("kshutdown_" + key);
}

void MiscUtils::notifyUser(const int secs)
{
	if ((secs < 60) && (secs % 3 == 0)) {
		KWindowInfo::showMessage(ks_main, i18n("Warning"), SmallIcon("messagebox_warning"), 1000);
	}

	switch (secs)
	{
		case 3600: // 1 hour
			customMessage(i18n("1 hour warning"));
			break;
		case 300: // 5 minutes
			notifyUser("kshutdown-5m", i18n("5 minutes warning"));
			customMessage(i18n("5 minutes warning"));
			break;
		case 60: // 1 minute
			notifyUser("kshutdown-1m", i18n("1 minute warning"));
			customMessage(i18n("1 minute warning"));
			break;
		case 10: // 10 seconds
			customMessage(i18n("10 seconds warning"));
			break;
	}
}

void MiscUtils::passiveMessage(const QString &text, QWidget *parent) {
	KPassivePopup *popup = KPassivePopup::message(
		"KShutDown",
		text,
		SmallIcon("kshutdown"),
		parent
	);
	if (parent == 0)
		popup->move(0, 0);
}

void MiscUtils::plug(const KAction *action, KPushButton *pushButton) {
	pushButton->disconnect(); // disconnect all
	pushButton->connect(pushButton, SIGNAL(clicked()), action, SLOT(activate()));
	pushButton->setIconSet(action->iconSet());
	QString text = action->text();
	if (text.contains("%1"))
		text = text.arg(action->shortcutText());
	pushButton->setText(text);
}

bool MiscUtils::runCommand(const QString &command)
{
	pid_t pid = KRun::run(command, KURL::List());

	if (pid)
		return true; // ok

	showRunErrorMessage(command);

	return false; // error
}

void MiscUtils::runCommandBeforeAction(const QString &configEntry)
{
	KConfig *conf = kshutdownrc->config();

	if (!conf->hasGroup(configEntry))
		return;

	conf->setGroup(configEntry);

	if (!conf->readBoolEntry("RunCommandBeforeAction", false))
		return;

	QString command = conf->readEntry("CommandBeforeAction", "");
	int pause = conf->readNumEntry("CommandBeforeActionPause", 10);
	if ((pause < 0) || (pause > 300))
		pause = 10;
	runShellCommand(command, KProcess::DontCare, pause);
}

bool MiscUtils::runShellCommand(const QString &command, const KProcess::RunMode mode, const int pause)
{
	if (command.isEmpty())
		return false;

	KProcess *p = new KProcess();
	if (!p)
	{
		showRunErrorMessage(command);

		return false; // error
	}

	bool retVal = true; // assume ok

	p->setUseShell(true);
	*p << command;
	if (!p->start(mode))
	{
		showRunErrorMessage(command);

		retVal = false; // error
	}
	else
	{
		// pause
		if (pause > 0)
			p->wait(pause);
	}
	delete p;

	return retVal;
}

void MiscUtils::setAutostart(const bool yes) {
	KConfig *config = kapp->config();
	config->setGroup("KShutDown");
	config->writeEntry("Autostart", yes);
}

void MiscUtils::setHint(QWidget *widget, const QString &value)
{
	QToolTip::add(widget, value);
	QWhatsThis::add(widget, value);
}

void MiscUtils::showRunErrorMessage(const QString &command)
{
	notifyUser("kshutdown-runerror", i18n("Could not run \"%1\"!").arg(command));
}

void MiscUtils::showTestMessage(const QString &message)
{
	KMessageBox::information(0, message, i18n("Test"));
}

// private

void MiscUtils::notifyUser(const QString &name, const QString &text)
{
	WId id;
	if (MSystemTray::isInstance())
		id = ks_tray->winId();
	else
		id = ks_main->winId();
	KNotifyClient::event(id, name, text);
}
