// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "IsoImage.h"
#include "IsoFile.h"
#include "ProgressDialog.h"
#include "appmacros.h"
#include "IsoImage.moc"

#include <kapp.h>
#include <klocale.h>
#include <kconfig.h>
#include <kprocess.h>

#include <qmessagebox.h>
#include <qstring.h>
#include <qstrlist.h>

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>

#define OPTION_DEFAULTS 4

static const bool option_defaults[OPTION_DEFAULTS][10]= {
  /*otdot ndeep long  leado nover  rr   anorr trans joli  all*/
  {false,false,true ,true ,false,false,true ,false,false,true }, /* UNIX RockRidge*/
  {false,true ,true ,true ,false,false,true ,false,true ,true }, /* UNIX + WIN */
  {false,true ,true ,true ,false,false,false,false,true ,true }, /* WIN */
  {false,false,false,false,false,false,false,false,false,true }  /* MS-DOS*/
};

ISOImage::ISOImage(void) {
  KConfig *config;
  QString qs("/");
  rootObject=new ISOFile(ISOFile::ISO_ISODir,&qs);
  connect(rootObject,SIGNAL(treeChanged()),this,SLOT(treeNotify()));
  fileTree=0;

  config=kapp->config();
  config->setGroup("ISOOptions");
  loadConfig(config);
  prepareImageTree();
  validSize=false;
}

ISOImage::ISOImage(const ISOImage &object):ProcessInterface() {
  rootObject=new ISOFile(*object.rootObject);
  fileTree=0;

  calculatedSize=object.calculatedSize;
  strcpy(applicationID,object.applicationID);
  strcpy(preparerID,object.preparerID);
  strcpy(publisherID,object.publisherID);
  strcpy(volumeID,object.volumeID);
  strcpy(systemID,object.systemID);

  omitTrailingDot=object.omitTrailingDot;
  noDeepRelocation=object.noDeepRelocation;
  longISONames=object.longISONames;
  allowLeadingDot=object.allowLeadingDot;
  omitVersions=object.omitVersions;
  withRockRidge=object.withRockRidge;
  withAnonymousRockRidge=object.withAnonymousRockRidge;
  makeTransTab=object.makeTransTab;
  withJoliet=object.withJoliet;
  allFiles=object.allFiles;

  sessionMerge=object.sessionMerge;
  sessionWritable=object.sessionWritable;
  validSize=object.validSize;
  prepareImageTree();
}

ISOImage::ISOImage(KConfig *config) {
  char entryName[40];
  int index;
  ISOFile *cparent=0;
  QStrList xlist;

  xlist.setAutoDelete(true);

  rootObject=0;
  fileTree=0;
  validSize=false;

  loadConfig(config);
  index=0;
  
  while ( (index==0) || (cparent!=0) ) {
    int narg;
    char type;
    QString qs1,qs2;
    char *tp;
    sprintf(entryName,"ISOTree%d",index);
    narg=config->readListEntry(entryName,xlist,'*');
    if (narg==0) {
      if (index==0)  {
        QString qs("/");
        rootObject=new ISOFile(ISOFile::ISO_ISODir,&qs);
      }
      break;
    }
    type=xlist.first()[0];
    tp=xlist.next();
    qs1=QString(tp);
    tp=xlist.next();
    qs2=QString(tp);
    switch (type) {
    case 'I':
      {
        ISOFile *newfile;
        newfile=new ISOFile(ISOFile::ISO_ISODir,&qs1);
        if (cparent==0) {
          rootObject=newfile;
	} else {
          cparent->addObject(newfile);
	}
        cparent=newfile;
        break;
      }
    case 'D':
      {
        ISOFile *newfile;
        newfile=new ISOFile(ISOFile::ISO_RealDir,&qs1,&qs2);
        cparent->addObject(newfile);
        break;
      }
    case 'F':
      {
        ISOFile *newfile;
        newfile=new ISOFile(ISOFile::ISO_RealFile,&qs1,&qs2);
        cparent->addObject(newfile);
        break;
      }
    case 'X':
      {
        cparent=cparent->getParent();
      }
    }
    index++;
  }

  connect(rootObject,SIGNAL(treeChanged()),this,SLOT(treeNotify()));
  prepareImageTree();
}


void ISOImage::loadConfig(KConfig *config) {
  QString ts;
  ts=config->readEntry("ApplicationID","KreateCD");
  strcpy(applicationID,ts.data());
  ts=config->readEntry("PublisherID","No publisher");
  strcpy(publisherID,ts.data());
  ts=config->readEntry("PreparerID","No preparer");
  strcpy(preparerID,ts.data());
  ts=config->readEntry("VolumeID","UNNAMED");
  strcpy(volumeID,ts.data());
  ts=config->readEntry("SystemID","LINUX");
  strcpy(systemID,ts.data());

  omitTrailingDot=config->readBoolEntry("NoTrailingDots",false);
  noDeepRelocation=config->readBoolEntry("NoDeepRelocation",false);
  longISONames=config->readBoolEntry("LongISONames",true);
  allowLeadingDot=config->readBoolEntry("LeadingDots",true);
  omitVersions=config->readBoolEntry("OmitVersions",false);
  withRockRidge=config->readBoolEntry("RockRidge",false);
  withAnonymousRockRidge=config->readBoolEntry("AnonymousRockRidge",true);
  makeTransTab=config->readBoolEntry("TransTab",false);
  withJoliet=config->readBoolEntry("Joliet",false);
  allFiles=config->readBoolEntry("AllFiles",true);
  sessionMerge=config->readLongNumEntry("SessionMerge",-1);
  sessionWritable=config->readLongNumEntry("SessionWritable",0);
  validSize=false;
}

void ISOImage::saveConfig(KConfig *config) {
  config->writeEntry("ApplicationID",applicationID);
  config->writeEntry("PublisherID",publisherID);
  config->writeEntry("PreparerID",preparerID);
  config->writeEntry("VolumeID",volumeID);
  config->writeEntry("SystemID",systemID);

  config->writeEntry("NoTrailingDots",omitTrailingDot);
  config->writeEntry("NoDeepRelocation",noDeepRelocation);
  config->writeEntry("LongISONames",longISONames);
  config->writeEntry("LeadingDots",allowLeadingDot);
  config->writeEntry("OmitVersions",omitVersions);
  config->writeEntry("RockRidge",withRockRidge);
  config->writeEntry("AnonymousRockRidge",withAnonymousRockRidge);
  config->writeEntry("TransTab",makeTransTab);
  config->writeEntry("Joliet",withJoliet);
  config->writeEntry("AllFiles",allFiles);
  config->writeEntry("SessionMerge",sessionMerge);
  config->writeEntry("SessionWritable",sessionWritable);
}

ISOImage::~ISOImage(void) {
  if (fileTree!=0) deleteImageTree();
  delete(rootObject);
}


ISOFile *ISOImage::imageRoot(void) {
  return(rootObject);
}

void ISOImage::setImageRoot(ISOFile *root) {
  if (fileTree!=0) deleteImageTree();
  delete(rootObject);
  rootObject=new ISOFile(*root); 
  connect(rootObject,SIGNAL(treeChanged()),this,SLOT(treeNotify()));
  prepareImageTree();
  validSize=false;
  emit(imageChanged());
}

bool ISOImage::prepareImageTree(void) {
  KConfig *config;
  QString tempdir;
  int i=0,r;
  char numbersuf[16];

  if (fileTree!=0) deleteImageTree();

  config=kapp->config();
  config->setGroup("Path");

  while (i<500) {
    r=rand()%1000;
    sprintf(numbersuf,"/isotree_%d",r);
    tempdir=config->readEntry("PathTemporary","/tmp");
    tempdir+=numbersuf;
    if (mkdir(tempdir.data(),0700)==0) break;
    ++i;
  }
  if (i==500) {
    return(false);
  }
  fileTree=new QString(tempdir.data());
  fileTree_pathinfo=new QString(*fileTree);
  fileTree_excludeinfo=new QString(*fileTree);
  fileTree_dummyDir=new QString(*fileTree);
  *fileTree_pathinfo+="/pathlist";
  *fileTree_excludeinfo+="/excludelist";
  *fileTree_dummyDir+="/dummydir";

  if (mkdir(fileTree_dummyDir->data(),0700)!=0) {
    deleteImageTree();
    return(false);
  }


  return(true);
}

bool ISOImage::deleteImageTree(void) {
  bool status=true;

  if (fileTree==0) return(true);

  remove(fileTree_pathinfo->data());
  remove(fileTree_excludeinfo->data());
  rmdir(fileTree_dummyDir->data());
  if (rmdir(fileTree->data())!=0) {
    status=false;
  }

  delete(fileTree);
  delete(fileTree_pathinfo);
  delete(fileTree_excludeinfo);
  delete(fileTree_dummyDir);
  fileTree=0;
  fileTree_pathinfo=0;
  fileTree_excludeinfo=0;
  fileTree_dummyDir=0;
  return(status);
}

bool ISOImage::buildImageTree(void) {
  FILE *pathlist;

  if (fileTree==0) return (false);
  pathlist=fopen(fileTree_pathinfo->data(),"wt");
  if (!rootObject->createTree("",pathlist,(FILE*) 0 ,*fileTree_dummyDir)) {
    return(false);
  }
  fclose(pathlist);
  return(true);
}


void ISOImage::setOptionsDefault(ISOImage::ISOType type) {
  int itype;
  if (type==CustomType) return;
  itype=type-1;
  omitTrailingDot=option_defaults[itype][0];
  noDeepRelocation=option_defaults[itype][1];
  longISONames=option_defaults[itype][2];
  allowLeadingDot=option_defaults[itype][3];
  omitVersions=option_defaults[itype][4];
  withRockRidge=option_defaults[itype][5];
  withAnonymousRockRidge=option_defaults[itype][6];
  makeTransTab=option_defaults[itype][7];
  withJoliet=option_defaults[itype][8];
  allFiles=option_defaults[itype][9];
  emit(imageChanged());
  validSize=false;
}

ISOImage::ISOType ISOImage::getOptionsDefault(void) {
  int itype;
  for (itype=0;itype<OPTION_DEFAULTS;++itype) {
    if (omitTrailingDot!=option_defaults[itype][0]) continue;
    if (noDeepRelocation!=option_defaults[itype][1]) continue;
    if (longISONames!=option_defaults[itype][2]) continue;
    if (allowLeadingDot!=option_defaults[itype][3]) continue;
    if (omitVersions!=option_defaults[itype][4]) continue;
    if (withRockRidge!=option_defaults[itype][5]) continue;
    if (withAnonymousRockRidge!=option_defaults[itype][6]) continue;
    if (makeTransTab!=option_defaults[itype][7]) continue;
    if (withJoliet!=option_defaults[itype][8]) continue;
    if (allFiles!=option_defaults[itype][9]) continue;
    return( (ISOImage::ISOType) (itype+1) );
  }
  return(CustomType);
}

void ISOImage::setOmitTrailingDot(bool b) {
  omitTrailingDot=b;
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setNoDeepRelocation(bool b) {
  noDeepRelocation=b;
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setLongISONames(bool b) {
  longISONames=b;
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setAllowLeadingDot(bool b) {
  allowLeadingDot=b;
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setOmitVersions(bool b) {
  omitVersions=b;
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setWithRockRidge(bool b) {
  withRockRidge=b;
  if ( withRockRidge && withAnonymousRockRidge) withAnonymousRockRidge=false;
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setWithAnonymousRockRidge(bool b) {
  withAnonymousRockRidge=b;
  if ( withRockRidge && withAnonymousRockRidge) withRockRidge=false;
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setMakeTransTab(bool b) {
  makeTransTab=b;
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setWithJoliet(bool b) {
  withJoliet=b;
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setAllFiles(bool b) {
  allFiles=b;
  emit(imageChanged());
  validSize=false;
}

static void scopy(char *dest,const char *source,int i) {
  while ( (i>0) && (*source!=0) ) {
    *(dest++)=*(source++);
    i--;
  }
  *dest='\0';
}

void ISOImage::setApplicationID(const QString &id) {
  scopy(applicationID,id.data(),128);
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setPreparerID(const QString &id) {
  scopy(preparerID,id.data(),128);
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setPublisherID(const QString &id) {
  scopy(publisherID,id.data(),128);
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setVolumeID(const QString &id) {
  scopy(volumeID,id.data(),32);
  emit(imageChanged());
  validSize=false;
}

void ISOImage::setSystemID(const QString &id) {
  scopy(systemID,id.data(),32);
  emit(imageChanged());
  validSize=false;
}

bool ISOImage::getOmitTrailingDot(void) {
  return(omitTrailingDot);
}

bool ISOImage::getNoDeepRelocation(void) {
  return(noDeepRelocation);
}

bool ISOImage::getLongISONames(void) {
  return(longISONames);
}

bool ISOImage::getAllowLeadingDot(void) {
  return(allowLeadingDot);
}

bool ISOImage::getOmitVersions(void) {
  return(omitVersions);
}

bool ISOImage::getWithRockRidge(void) {
  return(withRockRidge);
}

bool ISOImage::getWithAnonymousRockRidge(void) {
  return(withAnonymousRockRidge);
}

bool ISOImage::getMakeTransTab(void) {
  return(makeTransTab);
}

bool ISOImage::getWithJoliet(void) {
  return(withJoliet);
}

bool ISOImage::getAllFiles(void) {
  return(allFiles);
}

const char *ISOImage::getApplicationID(void) {
  return(applicationID);
}

const char *ISOImage::getPreparerID(void) {
  return(preparerID);
}

const char *ISOImage::getPublisherID(void) {
  return(publisherID);
}

const char *ISOImage::getSystemID(void) {
  return(systemID);
}

const char *ISOImage::getVolumeID(void) {
  return(volumeID);
}


long int ISOImage::calculateImage(void) {
  ProgressDialog *prog;
  prog=new ProgressDialog();
  prog->attachProcess(this);
  if (makeImage((const char*) 0)==false) {
    delete prog;
    return(0);
  }
  if (!prog->exec()) {
    delete prog;
    return(0);    
  }
  delete prog;
  validSize=true;
  return(calculatedSize);
}

bool ISOImage::makeImage(const char *filename) {
  KConfig *config;
  char devline[256];
  int unit,host;
  bool needSuid=false;

  if (sessionMerge!=-1) {
      needSuid=true;
  }

  if (filename!=0) {
    strcpy (imageFilename,filename);
  } else {
    imageFilename[0]=0;
  }

  prepareProcess();
  config=kapp->config();
  config->setGroup("Path");
  if FREE_MKISOFS {
    *this<<config->readEntry("PathMkisofs",PATH_MKISOFS);
  } else {
    if (needSuid) {
      *this<<PATH_WRAPPER<<"mkisofs";
    } else {
      *this<<PATH_MKISOFS;
    }
  }

  // options to support : -abstract      -allow-lowercase     -allow-multidot  -biblio  -b ,-c(bootimg)     -C+-M (multisess) -copyright
  // -f   -hide-list     -exclude-list   -print-size    -relaxed-filenames    -no-iso-translate

  if ( (sessionWritable!=0) || (sessionMerge!=-1) ) {
    if (sessionMerge!=-1) {
      *this<<"-M";
      config->setGroup("SCSI");
      host=config->readNumEntry("SCSIWriterHost",0);
      unit=config->readNumEntry("SCSIWriterUnit",-1);
      if (unit==-1) return(false);
      sprintf(devline,"%d,%d,0",host,unit);
      *this<<devline;
      sprintf(devline,"%ld,%ld",sessionMerge,sessionWritable);
      *this<<"-C";
      *this<<devline;
    } else {
      sprintf(devline,"%d,%ld",0,sessionWritable);
      *this<<"-C";
      *this<<devline;

    }
  }

  if (!allFiles) *this<<"-no-bak";
  if (omitTrailingDot) *this<<"-d";
  if (noDeepRelocation) *this<<"-D";
  if (longISONames) *this<<"-l";
  if (allowLeadingDot) *this<<"-L";
  if (omitVersions) *this<<"-N";
  if (withRockRidge) *this<<"-R";
  if (withAnonymousRockRidge) *this<<"-r";
  if (makeTransTab) *this<<"-T";
  if (withJoliet) *this<<"-J";

  if (imageFilename[0]!=0) {
    *this<<"-o"<<imageFilename;
  } else {
    *this<<"-print-size";
  }

  if (applicationID[0]!=0) *this<<"-A"<<applicationID;
  if (publisherID[0]!=0) *this<<"-P"<<publisherID;
  if (preparerID[0]!=0) *this<<"-p"<<preparerID;
  if (volumeID[0]!=0) *this<<"-V"<<volumeID;
  if (systemID[0]!=0) *this<<"-sysid"<<systemID;

  *this<<"-graft-points";

  *this<<"-path-list";
  *this<<*fileTree_pathinfo;

  reportAction(i18n("Creating directory tree..."));
  /*showDialog();*/

  if (!buildImageTree()) {
    QMessageBox::critical(0,QString::null,i18n("Unable to build directory tree!"));
    closeProcess();
    return(false);
  }
  reportAction(imageFilename[0]!=0?i18n("Creating ISO 9660 image..."):i18n("Calculating ISO 9660 size ..."));
  isoflag=0;
  calculatedSize=0;
  startProcess();   // 1=ripping was OK  -1=abort -2=ripping error  0=window hard closed
  return(1);
}

int ISOImage::processCleanup(int retval) { 
  if ( (imageFilename[0]!=0) && (retval!=1) ) remove(imageFilename);
  if ( ((imageFilename[0]==0) && (calculatedSize==0)) || ( (imageFilename[0]!=0) && 
						   (retval==-2)) ) {
    QMessageBox::critical(0,QString::null,i18n("Unable to create ISO image!"));
  }

  closeProcess();

  if ( ((imageFilename[0]==0) && (calculatedSize==0)) || ( (imageFilename[0]!=0) && 
						   (retval!=1)) ) {
    return(0);
  }
  return(1);
}

int ISOImage::processExited(void) {
  return(isoflag?1:-2);
}

bool ISOImage::processStderrLine(char *linebuffer) {
  if (strncmp(linebuffer,"Total extents scheduled to be written = ",40)==0) {
    unsigned long siz;
    siz=strtoul(linebuffer+40,0,10);
    calculatedSize=siz;
    return(true);
  }
  if (strstr(linebuffer,"done, estimate finish")!=0) {
    unsigned long fin;
    fin=strtoul(linebuffer,0,10);
    reportProgress(fin,100);
    return(true);
  }

  if (strstr(linebuffer,"extents written")!=0) {
    isoflag=1;
    return(true);
  }
  return(true);
}

void ISOImage::saveImage(KConfig *config) {
  saveConfig(config);
  initConfigPutter(config);
  putISOFile(rootObject);  
}

void ISOImage::initConfigPutter(class KConfig *config) {
  cputConfig=config;
  cputIndex=0;
}

void ISOImage::putConfigObject(const char *string1,
			       const QString *string2,
			       const QString *string3) { 
  QStrList slist=QStrList();
  char entryName[40];
  
  slist.append(string1);
  if (string2!=0) slist.append(string2->data());
  if (string3!=0) slist.append(string3->data());
  slist.setAutoDelete(true);
  sprintf(entryName,"ISOTree%d",cputIndex);
  cputConfig->writeEntry(entryName,slist,'*');
  cputIndex++;
}

void ISOImage::putISOFile(ISOFile *ifil) {
  char tstr[2];
  ISOFile::ISOType type;
  type=ifil->type();
  tstr[1]=0;
  switch (type) {
  case ISOFile::ISO_RealFile:
    tstr[0]='F';
    break;
  case ISOFile::ISO_RealDir:
    tstr[0]='D';
    break;
  case ISOFile::ISO_ISODir:
    tstr[0]='I';
    break;
  }
  putConfigObject((const char *) tstr,(const QString*) ifil->name(),
		  (const QString *) ifil->reference());
  if (type==ISOFile::ISO_ISODir) {
    ISOFile *walker=0;
    while ( (walker=ifil->getChildren(walker))!=0) {
      putISOFile(walker);
    }
    putConfigObject("X");
  }
}

void ISOImage::setMultisession(long int writable,long int lba) {
  sessionMerge=lba;
  sessionWritable=writable;
}

long int ISOImage::getMSWrite(void) {
  return (sessionWritable);
}

long int ISOImage::getMSLBA(void) {
  return (sessionMerge);
}

void ISOImage::treeNotify(void) {
  emit(imageChanged());
  validSize=false;
}

bool ISOImage::sizeValid(void) {
  return(validSize);
}
