/* PSK31 -- Viterbi and Varicode encoder / decoder
 * based on PSK31-Code by Andrew Senior, G0TJZ
 * (C) 1998,1999 Hansi Reiser, DL9RDZ
 * Subject to GPL -- see LICENSE for details
 */

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "psk31-coder.h"

/**********************************************************************/

int psk31_coder::decotab[2048];
int psk31_coder::encotab[256];
unsigned char psk31_coder::symbols[32];

unsigned char distance[4][4]={ {0, 2, 3, 2}, 
			       {2, 0, 2, 3}, 
			       {3, 2, 0, 2},
			       {2, 3, 2, 0} };

/* BPSK/QPSK-Encoder */
unsigned int psk31_coder::encode_qpsk(unsigned int sreg)
{
	unsigned int symb;
	
	symb =  psk31_coder::symbols[~sreg & 0x1F];
	if(!lsb) symb = (4 - symb) & 3;  /* Reverse 90/270 exchanges 
					    * if on LSB */
	return symb;
}
unsigned int psk31_coder::encode_bpsk(unsigned int sreg)
{
	 if(sreg & 1) return 0;   /* 1 -> no change */
	 else         return 2;   /* 0 -> reversal */
 }
unsigned int psk31_coder::encode(unsigned int sreg, int use_qpsk)
{
	if(use_qpsk) return encode_qpsk(sreg);
	else         return encode_bpsk(sreg);
}

/* BPSK/QPSK-decoder */

/**********************************************************************/
/* functions used for receiving */

int psk31_coder::IQ2iphase(cmplx IQval)
{
	float phase;
	int iphase;
	
	phase=atan2(IQval.y, IQval.x);
	iphase=(int)(phase*32767/M_PI);
	return iphase;
}


int psk31_coder::encode_varicode(int symb)
{
	return encotab[symb];
}

int psk31_coder::decode_varicode(int bit)
{
	unsigned char c;

	bit = !bit;	/* Invert received bit */
	//fprintf(stderr,"%d",bit);
	
	rxreg >>= 1;
	if((rxreg & 0x0002) > 0)
		vlerror = 1;
	if(bit)
		rxreg |= 0x8000;
	if(((rxreg & 0xC000) == 0) && (rxreg != 0)) {
		while((rxreg & 0x0001) == 0)
			rxreg >>= 1;
		rxreg >>= 1;

		c=vlerror?'':decotab[rxreg]; if(c==13) c='\n';
		vlerror = 0;
		rxreg = 0;

		return c;
	}
	return NO_CHAR; /* no new char generated */
}

float psk31_coder::x_distance(cmplx rxsymb, int lastphase, int delta)
{
	int ph[4]={0, 0x40, 0x80, 0xC0};
	float d;

	if(!lsb) { ph[1]=0xC0; ph[3]=0x40; }  // change 90<>270 deg

	/* Wir sollten jetzt bei lastphase+delta landen, sind aber bei
	   rxsymb... */
	/* we should be at: */
	float x,y;
	int tmp=-(ph[delta]-lastphase);
	x=cos( tmp/128.0*M_PI ); y=sin( tmp/128.0*M_PI );

	d=sqrt( (x-rxsymb.x)*(x-rxsymb.x) + (y-rxsymb.y)*(y-rxsymb.y) );

	return d*1.5; /* scaled to original scale -- unnecesary */
}
	

int psk31_coder::decode_qpsk(cmplx rxsymb)
{
	float dists[32]={0}, min;
	long ests[32];
	int newabs[32];
	unsigned char select, vote;
	int i;
	int rxphase;


	ampl=sqrt(rxsymb.x*rxsymb.x+rxsymb.y*rxsymb.y);
	if(ampl>1e-5) {
		agc=1/ampl;
		rxsymb.x*=agc; rxsymb.y*=agc;
	}
	// rxsymbol now scaled to absolute value of 1 
	//fprintf(stderr," %f'%f ",rxsymb.x,rxsymb.y);

	rxphase=IQ2iphase(rxsymb)>>8;

	int diffphase= (rxphase-lastphase)&0xFF;
	unsigned char rcvd = ((diffphase + 0x20) & 0xC0 ) >> 6;
	if(lsb) rcvd = (4 - rcvd) & 3;

	lastphase=rxphase;

	min = 1e6; /* Preset minimum = huge */
	for(i = 0; i < 32; i++) {
#if 0
		dists[i] = states[i/2].dist + distance[rcvd][symbols[i]];
#else
		/* Contrary to the original version use a "soft" decoder... */
		dists[i] = states[i / 2].dist + 
		    x_distance(rxsymb,states[i/2].last_abs_phase,symbols[i]);
#endif
		newabs[i]=rxphase;

		if(dists[i] < min) min = dists[i];
		ests[i] = ((states[i / 2].estimate) << 1) + (i & 1);
	}
	//fprintf(stderr," min=%f ",min);
	for(i = 0; i < 16; i++) {
		if(dists[i] < dists[16 + i])
			select = 0;
		else
			select = 16;
		states[i].dist = dists[select + i] - min;
		states[i].estimate = ests[select + i];
		states[i].last_abs_phase=newabs[select + i];
	}
	vote = 0;
	for(i = 0; i < 16; i++)
		if(states[i].estimate & (1L << 20))
			vote++;
	if(vote == 8)
		return rand()/(RAND_MAX>>1);
	else
		return (unsigned char)(vote > 8);
}

int psk31_coder::decode(cmplx rxsymb, int symbol)
{
	if(qpsk)
		return decode_varicode(decode_qpsk(rxsymb));
	else
		return decode_varicode(symbol);
}




void psk31_coder::prime_decoder() {
	cmplx prime={1,0};
	/* Prime the Viterbi decoder */
	for(int k = 0; k < 20; k++) {
		decode_qpsk(prime); prime.x=-prime.x;
	}
}


unsigned char psk31_coder::parity(unsigned char u)
{
	unsigned char k, p;
	p=0;
	for(k=0; k<8; k++) {
		p ^= u & 1;
		u >>= 1;
	}
	return p;
}

int psk31_coder::init_tables()
{
  // this is the psk31 coder table

const char* psk31cod[] = {"1010101011",
"1011011011",
"1011101101",
"1101110111",
"1011101011",
"1101011111",
"1011101111",
"1011111101",
"1011111111",
"11101111",
"11101",
"1101101111",
"1011011101",
"11111",
"1101110101",
"1110101011",
"1011110111",
"1011110101",
"1110101101",
"1110101111",
"1101011011",
"1101101011",
"1101101101",
"1101010111",
"1101111011",
"1101111101",
"1110110111",
"1101010101",
"1101011101",
"1110111011",
"1011111011",
"1101111111",
"1",
"111111111",
"101011111",
"111110101",
"111011011",
"1011010101",
"1010111011",
"101111111",
"11111011",
"11110111",
"101101111",
"111011111",
"1110101",
"110101",
"1010111",
"110101111",
"10110111",
"10111101",
"11101101",
"11111111",
"101110111",
"101011011",
"101101011",
"110101101",
"110101011",
"110110111",
"11110101",
"110111101",
"111101101",
"1010101",
"111010111",
"1010101111",
"1010111101",
"1111101",
"11101011",
"10101101",
"10110101",
"1110111",
"11011011",
"11111101",
"101010101",
"1111111",
"111111101",
"101111101",
"11010111",
"10111011",
"11011101",
"10101011",
"11010101",
"111011101",
"10101111",
"1101111",
"1101101",
"101010111",
"110110101",
"101011101",
"101110101",
"101111011",
"1010101101",
"111110111",
"111101111",
"111111011",
"1010111111",
"101101101",
"1011011111",
"1011",
"1011111",
"101111",
"101101",
"11",
"111101",
"1011011",
"101011",
"1101",
"111101011",
"10111111",
"11011",
"111011",
"1111",
"111",
"111111",
"110111111",
"10101",
"10111",
"101",
"110111",
"1111011",
"1101011",
"11011111",
"1011101",
"111010101",
"1010110111",
"110111011",
"1010110101",
"1011010111",
"1110110101",
"1110111101",
"1110111111",
"1111010101",
"1111010111",
"1111011011",
"1111011101",
"1111011111",
"1111101011",
"1111101101",
"1111101111",
"1111110101",
"1111110111",
"1111111011",
"1111111101",
"1111111111",
"10101010101",
"10101010111",
"10101011011",
"10101011101",
"10101011111",
"10101101011",
"10101101101",
"10101101111",
"10101110101",
"10101110111",
"10101111011",
"10101111101",
"10101111111",
"10110101011",
"10110101101",
"10110101111",
"10110110101",
"10110110111",
"10110111011",
"10110111101",
"10110111111",
"10111010101",
"10111010111",
"10111011011",
"10111011101",
"10111011111",
"10111101011",
"10111101101",
"10111101111",
"10111110101",
"10111110111",
"10111111011",
"10111111101",
"10111111111",
"11010101011",
"11010101101",
"11010101111",
"11010110101",
"11010110111",
"11010111011",
"11010111101",
"11010111111",
"11011010101",
"11011010111",
"11011011011",
"11011011101",
"11011011111",
"11011101011",
"11011101101",
"11011101111",
"11011110101",
"11011110111",
"11011111011",
"11011111101",
"11011111111",
"11101010101",
"11101010111",
"11101011011",
"11101011101",
"11101011111",
"11101101011",
"11101101101",
"11101101111",
"11101110101",
"11101110111",
"11101111011",
"11101111101",
"11101111111",
"11110101011",
"11110101101",
"11110101111",
"11110110101",
"11110110111",
"11110111011",
"11110111101",
"11110111111",
"11111010101",
"11111010111",
"11111011011",
"11111011101",
"11111011111",
"11111101011",
"11111101101",
"11111101111",
"11111110101",
"11111110111",
"11111111011",
"11111111101",
"11111111111",
"101010101011",
"101010101101",
"101010101111",
"101010110101",
"101010110111",
"101010111011",
"101010111101",
"101010111111",
"101011010101",
"101011010111",
"101011011011",
"101011011101",
"101011011111",
"101011101011",
"101011101101",
"101011101111",
"101011110101",
"101011110111",
"101011111011",
"101011111101",
"101011111111",
"101101010101",
"101101010111",
"101101011011", NULL};

	/* Generate convolutional coder symbols table */
	int k;
	for(k = 0; k < 32; k++) {
		symbols[k] = 2 * parity(k & poly1) + parity(k & poly2);
	}

	/* Initialize the Varicode tables */
	int n;
	unsigned int codeword, mask;
	char buf[15];

	for(k = 0; k < 2048; k++)
		decotab[k] = 248;	/* '' indicates invalid codeword */
	for(k = 0; k < 256; k++) {
		codeword = 4;	/* Preset 100 pattern */
    strcpy(buf, psk31cod[k]);
		n=strlen(buf);
		while( buf[n]!='1' && buf[n]!='0' && n ) n--;
		for( ; n >= 0; n--) {
			codeword <<= 1;
			if(buf[n] == '1')
				codeword++;
		}
		encotab[k] = codeword;
		mask = 0x8000;	/* Start at MSB */
		while((mask & codeword) == 0)
			mask >>= 1;	/* Find dummy '1' MSB added earlier */
		codeword &= ~mask;	/* Remove it */
		codeword >>= 1;	/* Remove (always '1') LSB */
		decotab[codeword] = k;
	}
	return 0;
}

