/***************************************************************************
                          kpgserver.cpp  -  description
                             -------------------
    begin                : � led 6 2004
    copyright            : (C) 2004 by Lumir Vanek
    email                : lvanek@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kpgserver.h"

// include files for Qt
#include <qstringlist.h>
#include <qregexp.h>

// include files for KDE
#include <kiconloader.h>
#include <kdebug.h>
#include <klocale.h>
#include <kcursor.h>

#include "kpgdatabase.h"
#include "kpgconnection.h"
#include "kpgusersfolder.h"
#include "kpggroupsfolder.h"
#include "kpgtablespacesfolder.h"
#include "../kpgsqldialog.h"


KPGServer::KPGServer(QListView *parent)
  : KPGTreeItem(parent)
{
  	setPixmap(0, *m_pIconServer);

  	m_pDbConnection = 0;
	
	m_pFolderUsers = 0;
	m_pFolderGroups = 0;
	m_pFolderTableSpaces = 0;
  
	m_uiVersionMinor = 0;
	m_uiVersionMiddle = 0;
	m_uiVersionMajor = 0;
}

KPGServer::~KPGServer()
{
    if(m_pDbConnection)
    {
        if(m_pDbConnection->is_open() == true)
		{        
			m_pDbConnection->disconnect();
		}
        delete m_pDbConnection;
    }
}

/**
  * Connect to PostgreSQL server
  *
  * @throw Exception, when failed 
  */
void KPGServer::connectToServer(
  const QString strHost,
  const QString strPort,
  const QString strDatabase,
  const QString strUser,
  const QString strPassword
 )
{
    QString strConnectString = (strDatabase.length() == 0) ?
        makeConnectionString(strHost, strPort, strUser, strPassword)
        :
        makeConnectionString(strHost, strPort, strDatabase, strUser, strPassword)
        ;
    
    try
    {
        m_pDbConnection = new KPGConnection(strConnectString);
        m_pDbConnection->setPassword(strPassword); // store password for future use
    }
    catch(const std::exception &e)
    {
        kdError() << k_funcinfo << " Failed to open connection "  << endl << e.what() << endl;
        throw;
    }
    
    setText(0, strHost);
    
    getVersionInfo();
}

void KPGServer::refresh() throw(const KPGSqlException &)
{
	// delete all child items
    while(QListViewItem * pItem = firstChild())
        delete pItem; 
	
    m_pFolderUsers = 0;
    m_pFolderGroups = 0;
    m_pFolderTableSpaces = 0;
    
    try
    {
	   refreshDatabaseList(false);
	
        m_pFolderUsers = new KPGUsersFolder(this);
        m_pFolderGroups = new KPGGroupsFolder(this);
            
        m_pFolderUsers->refresh();
        m_pFolderGroups->refresh();
        
        if(m_uiVersionMajor >= 8)
        {
            m_pFolderTableSpaces = new KPGTablespacesFolder(this); // Tablespaces suppor introduced in PostgreSQL 8
            m_pFolderTableSpaces->refresh();
        }
    }
    catch (const KPGSqlException &e)
    {
        listView()->setCursor(KCursor::arrowCursor());
        KPGSqlDialog dlg(0, e.sql(), e.message());
        dlg.exec();
    }
		
    setOpen(true);
}

// Intelligent refresh - find new or removed databases
void KPGServer::smartRefresh() throw(const KPGSqlException &)
{
	// obtain list of tables
    refreshDatabaseList(true);
}

// Obtain PostgreSQL version info
void KPGServer::getVersionInfo()
{
    QString strQuery("SELECT version();");
	
	pqxx::result pqxxResult; 
    try
    {
        pqxxResult = connection()->runQuery(strQuery);
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(connection(), e.what(), strQuery);
    } 
    
    if (pqxxResult.size() >= 1)
    {
        m_strVersionInfo = pqxxResult[0][0].c_str();
    }
  
    // Parse version
    
    //--- major
    QRegExp patternMajor("PostgreSQL \\d+", false); // major version
                
    int pos=patternMajor.search(m_strVersionInfo, 0);
    if(pos >= 0)
    {
        int l = patternMajor.matchedLength();
        QString s(m_strVersionInfo.mid(pos + 11, l - 11));
        
        m_uiVersionMajor = s.toInt();
        
        //--- middle
        QRegExp patternNumber("\\d+", false); // middle version
                    
        pos=patternNumber.search(m_strVersionInfo, pos + l);
        if(pos >= 0)
        {
			l = patternNumber.matchedLength();
			s = m_strVersionInfo.mid(pos, l);
			
			m_uiVersionMiddle = s.toInt();
			
			// minor
			pos=patternNumber.search(m_strVersionInfo, pos + l);
			if(pos >= 0)
			{
				l = patternNumber.matchedLength();
				s = m_strVersionInfo.mid(pos, l);
			
				m_uiVersionMinor = s.toInt();
			}
        }
    } 
    else
    {
        kdError() << "Cannot parse Version info" << endl;
        throw PGSTD::runtime_error("Cannot parse Version info");
    } 
}

// Force PostgreSQL server to reload configuration
bool KPGServer::reloadConfig()
{
	QString strQuery("SELECT pg_reload_conf();");
	
	pqxx::result pqxxResult; 
    try
    {
        pqxxResult = connection()->runQuery(strQuery);
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(connection(), e.what(), strQuery);
    } 
    
    if (pqxxResult.size() >= 1)
    {
        bool bResult;
        pqxxResult[0][0].to(bResult);
        return bResult;
    }
    
    return false;
}

// Force PostgreSQL server to rotate log file
bool KPGServer::rotateLogFile()
{
	QString strQuery("SELECT pg_rotate_logfile();");
	
	pqxx::result pqxxResult; 
    try
    {
        pqxxResult = connection()->runQuery(strQuery);
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(connection(), e.what(), strQuery);
    } 
    
    if (pqxxResult.size() >= 1)
    {
        bool bResult;
        pqxxResult[0][0].to(bResult);
        return bResult;
    }
    
    return false;
}

// Cancel a backed's current query on PostgreSQL
bool KPGServer::cancelBackend(int iPID)
{
	QString strQuery(QString("SELECT pg_cancel_backend(%1);").arg(iPID));
	
	pqxx::result pqxxResult; 
    try
    {
        pqxxResult = connection()->runQuery(strQuery);
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(connection(), e.what(), strQuery);
    } 
    
    if (pqxxResult.size() >= 1)
    {
        bool bResult;
        pqxxResult[0][0].to(bResult);
        return bResult;
    }
    
    return false;
}

void KPGServer::refreshDatabaseList(bool bSmart) throw(const KPGSqlException &)
{
    bool bVersion80_OrNewer = false;
	bool bVersion81_OrNewer = false;
	
	// Is it 8.0 or newer ?
	if(versionMajor() > 7)
    {             
       bVersion80_OrNewer = true;
    }     
	    
    // Is it 8.1 or newer ?
	if(((versionMajor() == 8) && (versionMiddle() >= 1)) || ((versionMajor() > 8))) 
	{
		bVersion81_OrNewer = true;
	}
	    
    // obtain list of databases
    QString strQuery("SELECT db.oid, datname, pg_get_userbyid(datdba) AS datowner");
   	    
    if(!bVersion80_OrNewer)
        strQuery.append(", datpath "); // deprecated in PostgreSQL 8
    else
        strQuery.append(", dattablespace, spcname ");
        
    strQuery.append(", pg_encoding_to_char(encoding) AS serverencoding");
    
    if(bVersion81_OrNewer) 
	{
		strQuery.append(", pg_catalog.pg_size_pretty(pg_catalog.pg_database_size(db.oid)) AS size_pretty");
	}
    
    strQuery.append(", has_database_privilege(db.oid, 'CREATE') as cancreate, datconfig, datallowconn, datistemplate, datacl ");
        
    if(bVersion81_OrNewer) 
	{
		strQuery.append(", pg_catalog.pg_database_size(db.oid) AS database_size ");
	}
        
    strQuery.append("FROM pg_catalog.pg_database db ");
    
    if(bVersion80_OrNewer)
        strQuery.append("LEFT JOIN pg_catalog.pg_tablespace ts ON db.dattablespace=ts.oid ");
    
    strQuery.append("ORDER BY datname");  
        
    try
    {
        m_pqxxResultDatabases = connection()->runQuery(strQuery);
    }
    catch (const std::exception &e)
    {
        kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(connection(), e.what(), strQuery);
    } 
    
    // Remove deleted databases from Server childrens list
    QStringList strListOfExistingDatabases;
    for (result::size_type i = 0; i != m_pqxxResultDatabases.size(); ++i)
    strListOfExistingDatabases.append(QString(m_pqxxResultDatabases[i]["datname"].c_str()));
        
    QListViewItem * pItem = firstChild();
    while(pItem)
    {
        if((static_cast <KPGTreeItem *> (pItem))->type() == KPGTreeItem::nodeDatabase)
        {
            if(strListOfExistingDatabases.contains(pItem->text(0)) == false)
            {
                delete pItem; // database not exists - delete it
                pItem = firstChild(); // restart cycle
                continue;
            }
        }			
        pItem = pItem->nextSibling();
    }
        
    KPGDatabase *pDatabase = 0;
    
    // Append databases to Server childrens list
    for (result::size_type i = 0; i != m_pqxxResultDatabases.size(); ++i)
    {
            pqxx::oid oid;
            m_pqxxResultDatabases[i][0].to(oid);
            
            bool bCreateNew = true; // default
            if(bSmart)
            {
                // look, if this database already exists
                pItem = firstChild();
                while(pItem)
                {
                    if((static_cast <KPGTreeItem *> (pItem))->type() == KPGTreeItem::nodeDatabase)
                    {
                        if(pItem->text(0) == QString(m_pqxxResultDatabases[i]["datname"].c_str()))
                        {
                            bCreateNew = false; // database exists, skip creation
                            break;
                        }
                    }			
                    pItem = pItem->nextSibling();
                }
            }
            
            if(bCreateNew)
            {
                if(pDatabase == 0)
                    pDatabase = new KPGDatabase(
                            this,
                            m_pqxxResultDatabases[i]["datname"].c_str(),
                            oid
                            );
                else
                    pDatabase = new KPGDatabase(
                        this,
                        pDatabase,
                        m_pqxxResultDatabases[i]["datname"].c_str(),
                        oid
                        );
            
                pDatabase->setProperties(m_pqxxResultDatabases[i], bVersion80_OrNewer, bVersion81_OrNewer);
        }
    }
}

result KPGServer::runQuery(const QString &strQuery)
{
  	// run query
  	return connection()->runQuery(strQuery);
}

const QString KPGServer::url() const 
{ 
  	if(m_pDbConnection == 0) return "";
  
  	return QString("host=" + QString(m_pDbConnection->hostname()) +
	               " port=" + QString(m_pDbConnection->port()) +
                 " dbname=" + QString(m_pDbConnection->dbname()) +
	               " user=" + QString(m_pDbConnection->username())
                ); 
}

// Query server activity 
pqxx::result KPGServer::queryActivity() throw(const KPGSqlException &)
{
	QString strQuery("SELECT datname, procpid, usename, current_query, query_start FROM pg_catalog.pg_stat_activity");
	
	try
	{
		return m_pDbConnection->runQuery(strQuery);
	}
	catch (const std::exception &e)
	{
		kdError() << k_funcinfo << e.what() << endl;
        throw KPGSqlException(connection(), e.what(), strQuery);
	}
}
