/***************************************************************************
                          kpgconnection.cpp  -  description
                             -------------------
    begin                : � led 13 2004
    copyright            : (C) 2004 by Lumir Vanek
    email                : dsacher@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kpgconnection.h"

// application specific includes
#include "../kpogreview.h"
#include "../kpgconfiguration.h"
#include "../kpggeneralsettings.h"

// include files for libpgxx 
#include <pqxx/nontransaction> 
#include <pqxx/robusttransaction> 
 
// include files for Qt
#include <qstring.h>
#include <qtextcodec.h>
#include <qstringlist.h>

// include files for KDE
#include <kdebug.h>
#include <klocale.h>
#include <qtextcodec.h>

KPGConnection::KPGConnection(const QString strConnectString)
  : connection(strConnectString.latin1())
{
	// Try obtain codec for connection client encoding
	m_pTextCodec = QTextCodec::codecForName(encoderName().latin1());
	if(m_pTextCodec == 0)
	{
        // If not found, force UTF8
		kdDebug() << k_funcinfo << "Unknown codec: " << encoderName() << " - trying UTF8"  << endl;  
		set_client_encoding("utf8"); 
        
        // Get codec for connection client encoding
		m_pTextCodec = QTextCodec::codecForName(encoderName().latin1());
		if(!m_pTextCodec)
		{	
			kdDebug() << k_funcinfo << "Unknown codec: " << encoderName() << endl;  
			m_pTextCodec = QTextCodec::codecForLocale(); // hope that correct codec is returned
		}
	}
	
	m_pNoticer = new KPGNoticer(m_pTextCodec);
	  
    // This is not a memory leak: C stores MyNoticer in an auto_ptr that will delete the object on destruction.
	set_noticer(PGSTD::auto_ptr<noticer> (m_pNoticer));
    
    m_pFileTrace = 0;
}


KPGConnection::~KPGConnection() throw ()
{
	if(is_open())
		disconnect();
    
	if(m_pFileTrace)
	{
		fclose(m_pFileTrace);
	}
}

// Run SQL statement, make conversion regard connection encoding,
// Get XAct type from global setting. Don't make commit, use it
// only for SELECT
pqxx::result KPGConnection::runQuery(const QString& strQuery)
{
    KPGConnection::ETransType transType = KPoGreView::configuration()->general()->transactionType();
    
    QCString cstrDecodedQuery(m_pTextCodec->fromUnicode(strQuery));
 	QString strDecodedQuery(cstrDecodedQuery);
    
    if(transType == KPGConnection::eTransNormal)
    {    
        //kdDebug() << k_funcinfo << "eTransNormal"  << endl;  
        pqxx::transaction<> pqxxXaction(*this, "runQuery");
        return pqxxXaction.exec(strDecodedQuery);
    }
    else if(transType == eTransNone)
	{
		//kdDebug() << k_funcinfo << "eTransNone"  << endl;  
		pqxx::nontransaction pqxxXaction(*this, "runQuery");
		return pqxxXaction.exec(strDecodedQuery);
	}	
    
    //	eTransRobust:
    //kdDebug() << k_funcinfo << "eTransRobust"  << endl;  
    pqxx::robusttransaction<> pqxxXaction(*this, "runQuery"); // name.operator std::string()
    return pqxxXaction.exec(strDecodedQuery);
}

// Run SQL statement, make conversion regard connection encoding
pqxx::result KPGConnection::runQuery(const QString& strQuery, ETransType transType)
{
 	QCString cstrDecodedQuery(m_pTextCodec->fromUnicode(strQuery));
 	QString strDecodedQuery(cstrDecodedQuery);
 	
 	// run query
    if(transType == eTransNormal)
	{
		pqxx::transaction<> pqxxXaction(*this, "runQuery");
		result pqxxResult(pqxxXaction.exec(strDecodedQuery));
		pqxxXaction.commit();
  		return pqxxResult;
	}
	else if(transType == eTransNone)
	{
		pqxx::nontransaction pqxxXaction(*this, "runQuery");
		return pqxxXaction.exec(strDecodedQuery);
	}	
		
	//	eTransRobust:
	pqxx::robusttransaction<> pqxxXaction(*this, "runQuery");
	result pqxxResult(pqxxXaction.exec(strDecodedQuery));
	pqxxXaction.commit();
	return pqxxResult;
}

// Run prepared SQL statement
pqxx::result KPGConnection::runPreparedStatement(const std::string &stdstrPrepStmtName, const ListFunctionArgumentsWithValues & listArgumentsWithValues, ETransType transType)
{
 	// run query
    if(transType == eTransNormal)
	{
		pqxx::transaction<> pqxxXaction(*this, stdstrPrepStmtName);
		prepare::invocation pqxxInvocation = pqxxXaction.prepared(stdstrPrepStmtName);
		
		for(ListFunctionArgumentsWithValues::const_iterator cit = listArgumentsWithValues.begin(); cit != listArgumentsWithValues.end(); ++cit)
		{
            if((*cit).mode() == KPGFunctionArgument::output)
            {
                kdDebug() << "Can;t set value for argument with output mode" << endl;
                continue;
            }
            
			bool bSuccess = setInvocationValue(pqxxInvocation, (*cit).typName(), (*cit).value());
			if(!bSuccess)
			{
				unprepare(stdstrPrepStmtName);
				
				if((*cit).name().length() > 0)
				{			
					throw PGSTD::invalid_argument(i18n("Conversion for argument %1 failed !").arg((*cit).name())); 
				}
				else
				{			
					throw PGSTD::invalid_argument(i18n("Conversion for argument %1 failed !").arg((*cit).sequence())); 
				}
			}
		}
		
		//try
		//{
			result pqxxResult(pqxxInvocation.exec());
			pqxxXaction.commit();
			unprepare(stdstrPrepStmtName);
			return pqxxResult;
		//}
		/*catch(...)
		{
			try
			{
				unprepare(stdstrPrepStmtName);
			}
			catch (const std::exception &e)
			{
				QString strError(e.what());
				kdError() << "Can't unprepare. " << e.what() << endl;
			}
			throw;
		}*/
	}
	else if(transType == eTransNone)
	{
		pqxx::nontransaction pqxxXaction(*this, "runQuery");
		prepare::invocation pqxxInvocation = pqxxXaction.prepared(stdstrPrepStmtName);
		
        for(ListFunctionArgumentsWithValues::const_iterator cit = listArgumentsWithValues.begin(); cit != listArgumentsWithValues.end(); ++cit)
		{
            if((*cit).mode() == KPGFunctionArgument::output)
            {
                kdDebug() << "Can;t set value for argument with output mode" << endl;
                continue;
            }
            
			bool bSuccess = setInvocationValue(pqxxInvocation, (*cit).typName(), (*cit).value());
			if(!bSuccess)
			{
				unprepare(stdstrPrepStmtName);
					
				if((*cit).name().length() > 0)
				{			
					throw PGSTD::invalid_argument(i18n("Conversion for argument %1 failed !").arg((*cit).name())); 
				}
				else
				{			
					throw PGSTD::invalid_argument(i18n("Conversion for argument %1 failed !").arg((*cit).sequence())); 
				}
			}
		}
			
		result pqxxResult(pqxxInvocation.exec());
		unprepare(stdstrPrepStmtName);
		return pqxxResult;
	}
		
	//	eTransRobust:
	pqxx::robusttransaction<> pqxxXaction(*this, "runQuery");
	prepare::invocation pqxxInvocation = pqxxXaction.prepared(stdstrPrepStmtName);
		
	for(ListFunctionArgumentsWithValues::const_iterator cit = listArgumentsWithValues.begin(); cit != listArgumentsWithValues.end(); ++cit)
	{
        if((*cit).mode() == KPGFunctionArgument::output)
        {
            kdDebug() << "Can;t set value for argument with output mode" << endl;
            continue;
        }
        
		bool bSuccess = setInvocationValue(pqxxInvocation, (*cit).typName(), (*cit).value());
		if(!bSuccess)
		{
			unprepare(stdstrPrepStmtName);
				
			if((*cit).name().length() > 0)
			{			
				throw PGSTD::invalid_argument(i18n("Conversion for argument %1 failed !").arg((*cit).name())); 
			}
			else
			{			
				throw PGSTD::invalid_argument(i18n("Conversion for argument %1 failed !").arg((*cit).sequence())); 
			}
		}
	}
		
	result pqxxResult(pqxxInvocation.exec());
	pqxxXaction.commit();
	unprepare(stdstrPrepStmtName);
	return pqxxResult;
}


// Return encoder name for QTextCodec regards to connection client encoding
// Source
// http://doc.trolltech.com/3.3/qtextcodec.html
// http://www.postgresql.org/docs/8.1/static/multibyte.html
const QString KPGConnection::encoderName()
{
    QString strEncoding(clientEncoding());
    
	if(strEncoding == "LATIN1") return "ISO8859-1";
	if(strEncoding == "LATIN2") return "ISO8859-2";
	if(strEncoding == "LATIN3") return "ISO8859-3";
	if(strEncoding == "LATIN4") return "ISO8859-4";
	if(strEncoding == "LATIN5") return "ISO8859-9";	
	if(strEncoding == "LATIN6") return "ISO8859-10";
	if(strEncoding == "LATIN7") return "ISO8859-13";
	if(strEncoding == "LATIN8") return "ISO8859-14";
	if(strEncoding == "LATIN9") return "ISO8859-15";
	if(strEncoding == "LATIN10") return "ISO8859-16";
	
	if(strEncoding == "ISO_8859_5") return "ISO8859-5";
	if(strEncoding == "ISO_8859_6") return "ISO8859-6";
	if(strEncoding == "ISO_8859_7") return "ISO8859-7";
	if(strEncoding == "ISO_8859_8") return "ISO8859-8";
	
	if(strEncoding == "WIN1250") return "CP1250";
	if(strEncoding == "WIN1251") return "CP1251";
	if(strEncoding == "WIN1252") return "CP1252";
	if(strEncoding == "WIN1256") return "CP1256";
	if(strEncoding == "WIN1258") return "CP1258";
	
	if(strEncoding == "UNICODE") return "utf8";
	
	return QString(strEncoding);
}

// Enable tracing to file
void KPGConnection::setTrace()
{
	if(m_pFileTrace)
	{
		kdError() << "KPGConnection::setTrace : trace already set" << endl;
		return;
	}
	
	m_pFileTrace = fopen("/tmp/KPoGre.log","w");
	
	trace((FILE *) m_pFileTrace);
}

// Set prepare declaration for column, regards of its datatype
void KPGConnection::setPrepareDeclaration(pqxx::prepare::declaration &pqxxPrepDecl, const QString &strTypName)
{
	//QString strTypName = m_listArguments[numArgument].typName();
	
	if((strTypName == "varchar") || (strTypName == "bpchar") || (strTypName == "text"))
	{	
		pqxxPrepDecl(strTypName, pqxx::prepare::treat_string);
		return;
	}
	
	if((strTypName == "int2") || (strTypName == "int4") || (strTypName == "int8") || (strTypName == "float4") || (strTypName == "float8"))
	{	
		pqxxPrepDecl(strTypName, pqxx::prepare::treat_direct);
		return;
	}
	
	if((strTypName == "bool"))
	{	
		pqxxPrepDecl(strTypName, pqxx::prepare::treat_bool);
		return;
	}
	
	if((strTypName == "bytea"))
	{	
		pqxxPrepDecl(strTypName, pqxx::prepare::treat_binary);
		return;
	}
	
	pqxxPrepDecl(strTypName, pqxx::prepare::treat_string);
}

// Set value to prepared statement invocation, regards of its datatype. Return false, if conversion fails
bool KPGConnection::setInvocationValue(prepare::invocation &pqxxInvocation, const QString &strTypName, const QString &strValue)
{
	if((strTypName == "varchar") || (strTypName == "bpchar"))
	{	
		pqxxInvocation((std::string) m_pTextCodec->fromUnicode(strValue)); 
		return true;
	}
	
	if(strTypName == "int2")
	{	
		bool bOk;
		short sValue = strValue.toShort(&bOk);
		if(!bOk) return false;
		pqxxInvocation(sValue); 
		return true;
	}
	
	if(strTypName == "int4")
	{	
		bool bOk;
		int iValue = strValue.toInt(&bOk);
		if(!bOk) return false;
		pqxxInvocation(iValue); 
		return true;
	}
	
	if(strTypName == "int8")
	{	
		bool bOk;
		long lValue = strValue.toLong(&bOk);
		if(!bOk) return false;
		pqxxInvocation(lValue); 
		return true;
	}
	
	if(strTypName == "float4")
	{	
		bool bOk;
		float fValue = strValue.toFloat(&bOk);
		if(!bOk) return false;
		pqxxInvocation(fValue); 
		return true;
	}
	
	if(strTypName == "float8")
	{	
		bool bOk;
		double dValue = strValue.toDouble(&bOk);
		if(!bOk) return false;
		pqxxInvocation(dValue); 
		return true;
	}
	
	if((strTypName == "bool"))
	{	
		bool bValue = false;
		
		if((strValue == "t") || (strValue == "T") || (strValue == "y") || (strValue == "Y") || (strValue == "1"))
		{
			bValue = true;
		} 
		else if((strValue == "f") || (strValue == "F") || (strValue == "n") || (strValue == "N") || (strValue == "0"))
		{
			bValue = false;
		}
		else
		{
			 return false;
		}
		pqxxInvocation(bValue);
		return true;
	}
		
	//kdDebug() << k_funcinfo "Uncovered datatype: " <<  strTypName << ", " << m_listArguments[numRow].name() << endl;	

    std::string stdstrValue((std::string) m_pTextCodec->fromUnicode(strValue));
	pqxxInvocation(stdstrValue);
	
	return true;
}

// Return given string converted to UNICODE using current codec
QString KPGConnection::toUnicode(const char * pszString) const
{
	return m_pTextCodec->toUnicode(pszString);
}

