#ifndef SERVICEPROVIDER_H
#define SERVICEPROVIDER_H

#include <kore/kore.h>
#include <kore/module.h>
#include <vector>
#include <string>

//using namespace std;

namespace kore
{

class ServiceManager;

/**
 * class kore::ServiceProvider - a ServiceProvider is a Module which provides
 * or more services to be used by the other modules.
 */
class KORE_API ServiceProvider : public Module
{
public:
    /**
     * class kore::ServiceProvider::Service - helper class which holds service
     * information, such as service's provider, name and description.
     * Service naming scheme: a service name consists of one or more alfanumeric
     * strings separated by '/' (slash). Example: "foo/bar/baz" is a legal
     * service name. A service name has 2 components: path ("foo/bar/") and leaf
     * ("baz"). Under some circumstances the leaf may be missing ("foo/bar/").
     * The path constists of one or more path elements ("foo" and "bar")
     * separated by '/'.
     * Illegal characters: '/' is an illegal character in both path elements and
     * leaf (since '/' is the path separator). '*' (star) and '?' (question mark)
     * are illegal characters in path elements, but they can be used as leaf
     * elements. Please note that '*' and '?' as leaf elements have special
     * meanings for the service naming sheme.
     * The service which have '?' as their leaf element are called "active
     * services". All the others are 'passive services'.
     */
    class KORE_API Service
    {
    public:
        /**
         * Constructor.
         * @param provider - the ServiceProvider this Service is associated to.
         * @param name - the Service name.
         * @param description - Service description string.
         */
        Service(ServiceProvider* provider,const char* name,const char* description);

        /**
         * Gets the ServiceProvider this Service is associated to.
         * @return - who provides this service.
         */
        ServiceProvider* provider() const;
        /**
         * Gets the service name.
         * @return - service name.
         */
        const char* name() const;
        /**
         * This is a convenience alias to name().
         * @return - service name.
         */
        operator const char* () const;
        /**
         * Gets the service description.
         * @return - service description string.
         */
        const char* description() const;
    private:
        // the list of Service's path elements.
        vector<string> _path;
        // Service's leaf element.
        string _leaf;
        // Service's parent (unused)
        Service* _parent;
        // Service's provider
        ServiceProvider* _provider;
        // service name
        const char* _name;
        // service description
        const char* _description;
    };

    /**
     * Gets the list services provided by this ServiceProvider.
     * @return - a null-terminated array of Services.
     */
    virtual const Service** services() const;
    /**
     * Checks whether this SP provides a service who's name matches
     * the given srv->name().
     * The default implementation returns ServiceProvider::activeService(...),
     * or if the activeService()'s result is NULL,
     * it returns ServiceProvider::passiveService(...).
     * @param srv - the service who's name is used for mathcing.
     * @return - the actual service, if a match was found, NULL otherwise.
     */
    virtual const Service* service(const Service* srv) const;
    /**
     * Checks whether this SP provides a service who's name matches
     * the given srv->name().
     * The default implementation creates a temporary Service object, sets
     * its name to srv, and calls ServiceProvider::service(const Service*).
     * @param srv - the service name used for mathcing.
     * @return - the actual service, if a match was found, NULL otherwise.
     */
    virtual const Service* service(const char* srv) const;

    /**
     * Callback method triggered right BEFORE REGISTERING this ServiceProvider
     * to a ServiceManager.
     * @param sm - the ServiceManager this SP will get registered to.
     */
    virtual void registeringProvider(ServiceManager* sm);
    /**
     * Callback method triggered right AFTER REGISTERING this ServiceProvider
     * to a ServiceManager.
     * @param sm - the ServiceManager this SP got registered to.
     */
    virtual void providerRegistered(ServiceManager* sm);
    /**
     * Callback method triggered right BEFORE UNREGISTERING this ServiceProvider
     * from a ServiceManager.
     * @param sm - the ServiceManager this SP will get unregistered from.
     */
    virtual void unregisteringProvider(ServiceManager* sm);
    /**
     * Callback method triggered right AFTER UNREGISTERING this ServiceProvider
     * from a ServiceManager.
     * @param sm - the ServiceManager this SP got unregistered from.
     */
    virtual void providerUnregistered(ServiceManager* sm);

    /**
     * Callback method triggered right BEFORE REGISTERING to a ServiceManager
     * one of the Services provided by this SP.
     * @param sm - ServiceManager the Service will get registered to.
     * @param srv - the Service that will get registered.
     */
    virtual void registeringService(ServiceManager* sm,const Service* srv);
    /**
     * Callback method triggered right AFTER REGISTERING to a ServiceManager
     * one of the Services provided by this SP.
     * @param sm - ServiceManager the Service got registered to.
     * @param srv - the Service that got registered.
     */
    virtual void serviceRegistered(ServiceManager* sm,const Service* srv);
    /**
     * Callback method triggered right BEFORE UNREGISTERING from a ServiceManager
     * one of the Services provided by this SP.
     * @param sm - ServiceManager the Service will get unregistered from.
     * @param srv - the Service that will get unregistered.
     */
    virtual void unregisteringService(ServiceManager* sm,const Service* srv);
    /**
     * Callback method triggered right AFTER UNREGISTERING from a ServiceManager
     * one of the Services provided by this SP.
     * @param sm - ServiceManager the Service got unregistered from.
     * @param srv - the Service that got unregistered.
     */
    virtual void serviceUnregistered(ServiceManager* sm,const Service* srv);

protected:
    /**
     * Default constructor. Creates an empty ServiceProvider.
     */
    ServiceProvider();
    /**
     * Creates a ServiceProvider and sets its Info.
     * @param info - the ServiceProvider info.
     */
    ServiceProvider(const Info* info);
    /**
     * Destructor.
     */
    virtual ~ServiceProvider();

    /**
     * Convenience method for adding a Service to this SP services list.
     * @param service - the service to be added.
     */
    virtual void addService(const Service* service);
    /**
     * Searches for a passive service who's name matches service->name().
     * This function gets called by ServiceProvider::service(const Service*) function.
     * @param srv - the service name used for mathcing.
     * @return - the actual passive service, if a match was found, NULL otherwise.
     */
    virtual const Service* passiveService(const Service* service) const;
    /**
     * Searches for an active service who's name matches service->name().
     * This function gets called by ServiceProvider::service(const Service*) functions,
     * only when ServiceProvider::passiveService() returns NULL.
     * Subclasses may override this function in order to tell whether a given service
     * name mathces or not on of the active services this SP provides.
     * @param srv - the service name used for mathcing.
     * @return - the actual active service, if a match was found, NULL otherwise.
     */
    virtual const Service* activeService(const Service* service) const;
private:
    // the list of services provided by this SP.
    vector<const Service*> _services;
};
};

#endif
