/***************************************************************************
                          infoquery.h  -  description
                             -------------------
    begin                : Mon Nov 22 1999
    copyright            : (C) 1999 by Seth Hartbecke
    email                : gandalf@netins.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef INFOQUERY_H
#define INFOQUERY_H


#include <qobject.h>

#include <xmlnode.h>
#include <jabberid.h>

/*
  @author Seth Hartbecke

*/

class QueryManager;

/**
 * The info query class abstracts the whole iq packet concept of jabber's.  This is really just an XMLNode with some
 * added methods to simplifiy the setting and retrieving of important attributes on an iq packet (to, id, type, xmlns),
 * but can be used as an XMLNode with no problems.
 *
 * You cannot/should not create a InfoQuery packet by hand, here is a simple example of how to create a request:
 *
 * <pre>
 * InfoQuery *myRequest = jabber->queryManager()->createQuery();
 * myRequest->setTo( "jabber.org" );
 * myRequest->setType( "get" );
 * myRequest->setNameSpace( "jabber:iq:agents" );
 * connect( myRequest, SIGNAL( responceRecieved( InfoQuery *, InfoQuery * ) )
 *        , this, SLOT( agentsRecieved( InfoQuery *, InfoQuery * ) ) );
 * myRequest->send();
 * </pre>
 *
 * This will request an agent list from jabber.org and call agentsRecieved for you when the result is recieved.  The
 * query manager handles setting the ID's and routing of responces (you will never accidentally recieve a responce
 * you did not ask for).
 *
 * @short A <iq> packet
 * @author Seth Hartbecke (gandalf@netins.net)
 * @see QueryManager
 * @see XMLNode
 */
class InfoQuery
        : public QObject, public XMLNode
{
Q_OBJECT
public:
        InfoQuery( QueryManager *parent );
        InfoQuery( QueryManager *parent, XMLNode *t );

	/**
	 * Sets the to attribute on the infoquery tag, this must be a valid Jabber identeifier (however, the
	 * class will not check for you, so be carefull).
	 *
	 * @param QString
	 */
	void setTo( QString sendTo ) { setAttribute( "to", sendTo ); };
	/**
	 * Sets the to attribute to the JabberID (assumed not to be null).
	 *
	 * @param JabberID
	 */
        void setTo( JabberID &jid ) { setTo( jid.getJID() ); };
	/**
	 * Get's the to attribute.  Please be aware, that if you are looking at the responce, the to attribute
	 * will be you!
	 *
	 * @return QString
	 */
        QString getTo() { return getAttribute( "to" ); };
	/**
	 * Get's the to attribute.  Please be aware, that if you are looking at the responce, the to attribute
	 * will be you!
	 *
	 * @return JabberID
	 */
        JabberID getToJID() { JabberID jid( getAttribute( "to" ) ); return jid; };


	/**
	 * This will set the ID parameter on the IQ.  You should never have to call this yourself as the
	 * QueryManager will set a unique one for you.  However, if you do set your own ID, the QueryManager
	 * will not change it (even if it is not unique)
	 *
	 * @param QString
         */
	void setID( QString newID ) { setAttribute( "id", newID ); };
	/**
	 * Gets the current ID
	 *
	 * @return QString
	 */
        QString getID() { return getAttribute( "id" ); };


	/**
	 * Sets the current type of infoquery.  The last time I read the IQ spec on jabber.org the only
	 * valid values are get, set, and result.  But, because I'm often behind on my reading, the class
	 * will <b>NOT</b> check to see if the type you are telling it to use is valid.
	 *
	 * @param QString
	 */
        void setType( QString type ) { setAttribute( "type", type ); };
	/**
	 * Tells you the query type, valid ones are get, set, result (there should be no others).
	 *
	 * @return QString
	 */
        QString getType() { return getAttribute( "type" ); };

	/**
	 * Set the namespace you wish to use, a valid list is available at http://protocol.jabber.org/ns.html
	 *
	 * @param QString
	 */
        void setNameSpace( QString newNS ) { getChildNode("query")->setAttribute("xmlns", newNS); };
	/**
	 * Gets the current namespace of the iq tag
	 *
	 * @return QString
	 */
        QString getNameSpace() { return getChildNode("query")->getAttribute("xmlns"); };

	/**
	 * Sends the IQ packet (does not check to see if you are online, you should do that first yourself).
	 * This is where the ID gets set if you have not specified one.
	 *
	 * <b>WARNING</b> DO NOT DELETE THE
	 * InfoQuery OBJECT!!!!  You will really mess things up if you do (like the application WILL CRASH
	 * if a responce for that id is ever recieved).  The QueryManager does all creation and deletion of
	 * InfoQuery objects.
	 */
        void send();

        friend class QueryManager;
signals:
        /**
         * This signal is emitted when a responce to your request is recieved.  The first parameter is a
	 * pointer to the request (just to be nice, in case you have many oustanding requests going to the
	 * same slot) and the secound is a pointer to the responce.
	 *
	 * Do NOT DELETE either of these (the QueryManager does all creation and deletion of InfoQuery objects)
	 * If you really want to have a copy, I recommend copying the data into an XMLNode.  The simpliest way
	 * to do this would be:
	 *
	 * <pre>
	 * XMLNode *copy = new XMLNode( *iqPacket );
	 * </pre>
	 *
	 * @param InfoQuery
	 * @param InfoQuery
	 */
        void responceRecieved(InfoQuery *query, InfoQuery *responce);

private:
        // This is called by the query manager when a responce to the
        // query is recieved.
        // This will emit the responceRecieved() signal.
        void responce(InfoQuery *query);

        QueryManager *_parent;
};


/**
 * This class is used primarally by the QueryManager to keep track of who wants to recieve what callbacks.
 *
 * @short Internal use
 * @author Seth Hartbece (gandalf@netins.net)
 * @see QueryManager
 */
class QueryXMLNS
        : public QObject
{
Q_OBJECT
public:
        QueryXMLNS( QueryManager *parent, QString xmlns, QObject *reciever, const char *slot );

        QString getNS() { return _xmlns; };

        friend class QueryManager;

signals:
        /**
	 * This would be about the only thing that a programmer (one not redesining the IQ sub-system :) would be
	 * inetersted in, this is the signal that is emmited when this xmlns is encountered
	 */
        void queryRecieved( InfoQuery *incoming );

private:
        void recievedIQTag( XMLNode *t );

	QString _xmlns;
	QueryManager *_parent;
};

/**
 * The QueryManager is the interface to the Info/Query subsystem in konverse.  This creates any new InfoQuery objects,
 * routes the responces to the correct place, keeps track of who is interested in what xmlns's (get or set requests) and
 * routes those correctly.  This class also sets the ID's on all InfoQuery packets sent.
 *
 * @author Seth Hartbecke (gandalf@netins.net)
 * @see InfoQuery
 */
class QueryManager
        : public QObject
{
Q_OBJECT
public:
        QueryManager(QWidget *parent=0, const char *name=0);

        /**
	 * This returns a new (empty) InfoQuery object for you to use.  If you call send() on the returned
	 * value DO NOT DELETE IT, but you can delete it any time before then.  After you call send the QueryManager
	 * will take note of it's ID and store a pointer to it (so that responces to that ID can be correctly routed).
	 * If you delete a InfoQuery and a responce for that ID arrives the application WILL CRASH!
	 */
	InfoQuery *createQuery();

	/**
	 * This method is used to register handlers for get or set requests.  For example, to register a handler
	 * for the jabber:iq:version namespace, you need to do something like:
	 *
	 * <pre>
	 * jabber->queryManager()->registerNS( "jabber:iq:version", this, SLOT(versionRequest(InfoQuery *)) );
	 * </pre>
	 *
	 * Once again, the standard warning, DO NOT DELETE THE INFO QUERY OBJECT!  (As there may be other handlers
	 * for that namespace), the QueryManager will delete it once everybody has had a chance to look at the request
	 */
	QueryXMLNS *registerNS( QString xmlns, QObject *reciever, const char *slot );

        friend class InfoQuery;
        friend class QueryXMLNS;
public slots:
        void incomingPacket(XMLNode *t);

private:
        // This should be called only by a InfoQuery object
        void sendQuery( InfoQuery * sendThis );

        int nextAvailID;
        QList<QueryXMLNS> _XMLNSCallbacks;
        QList<InfoQuery> _pendingRequests;
};

#endif
