/*******************************************************************************
**                         configurator.cpp
**                             part of
**                konserve -- A small backup application
**			 -------------------
**  copyright: (C) 2002 - 2004 by Florian Simnacher
**  email    : simnacher AT gmx DOT de
*******************************************************************************/

/*******************************************************************************
**  This program is free software; you can redistribute it and/or             **
**  modify it under the terms of the GNU General Public License               **
**  as published by the Free Software Foundation; either version 2            **
**  of the License, or (at your option) any later version.                    **
**                                                                            **
**  This program is distributed in the hope that it will be useful,           **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of            **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             **
**  GNU General Public License for more details.                              **
**                                                                            **
**  You should have received a copy of the GNU General Public License         **
**  along with this program; if not, write to the Free Software               **
**  Foundation, Inc., 59 Temple Place - Suite 330, Boston,                    **
**  MA  02111-1307, USA.                                                      **
******************************************************************************/

// KDE includes
#include <kapplication.h>
#include <kmessagebox.h>
#include <klocale.h>

// My includes
#include "konservedebug.h"
#include "configurator.h"

static const QString CONFIG_VERSION = QString( "1" );


////////////////////////////////////////////////////////////////////////////////
///////	       class Configurator                                        ///////
////////////////////////////////////////////////////////////////////////////////

Configurator::Configurator()
    : mpTheConfig( kapp->config() )
{
    TRACE();
    mpTheConfig->setGroup( "General" );
    mProfileNames      = mpTheConfig->readListEntry( "Profilenames" );
    mConfigFileVersion = mpTheConfig->readEntry( "Version", CONFIG_VERSION );

    // Check if all files listed in Profilenames (config file) are there
    QValueList<QString>::iterator it;
    for ( it = mProfileNames.begin(); *it; ++it )
        if ( ! mpTheConfig->hasGroup( *it ) )
        {
            KMessageBox::error(
                0L, i18n( "Your configuration file is corrupted" )
                );
            // Why the fuck does the following not terminate the
            // program (if uncommented of course)?
            // kapp->quit();
        }
}

Configurator::~Configurator()
{
    TRACE();
    this->sync();
    mpTheConfig->sync();
}

void Configurator::inserted( BackupProfile *b )
{
    TRACE();
    insertProfile( b );
    this->sync();
}
void Configurator::changed( BackupProfile *b )
{
    TRACE();
    changeProfile( b );
    this->sync();
}
void Configurator::removed( QString s )
{
    TRACE();
    removeProfile( s );
    this->sync();
}

void Configurator::doConfigure()
{
    TRACE();
    QValueList<QString>::iterator it = mProfileNames.begin();

    while ( mProfileNames.end() != it )
    {
        ProfileMngrInterface::insert( nextProfile( *it ) );
        it++;
    }
}

void Configurator::sync()
{
    TRACE();
    mpTheConfig->setGroup( "General" );
    mpTheConfig->writeEntry( "Profilenames", mProfileNames );
    mpTheConfig->writeEntry( "Version", CONFIG_VERSION );
    mpTheConfig->sync();
}


void Configurator::insertProfile( BackupProfile *b )
{
    TRACE();
    if ( mpTheConfig->hasGroup( b->getIdentifier() ) )
        changeProfile( b );
    else
        newProfile( b );
}

void Configurator::newProfile( BackupProfile *b )
{
    TRACE();
    if ( mpTheConfig->hasGroup( b->getIdentifier() ) )
        if ( KMessageBox::No == KMessageBox::questionYesNo(
                 0L, i18n( "The backupprofile: %1 already exists.\n"
                           "Shall I replace it in the config file?" ).arg(
                               b->getIdentifier()
                               ),
                 i18n( "Profile already exists" )
                 )
            )
        {
            return;
        }
        else
        {
            changeProfile( b );
        }
    mpTheConfig->setGroup( b->getIdentifier() );
    mpTheConfig->writeEntry( "SourceUrl",  b->getSourceUrl().url() );
    mpTheConfig->writeEntry( "ArchiveUrl", b->getArchiveUrl().url() );
    mpTheConfig->writeEntry(
        "TimeNumber", b->getTimeBetweenBackups().value()
        );
    mpTheConfig->writeEntry(
        "Measurement", b->getTimeBetweenBackups().measurement()
        );
    mpTheConfig->writeEntry( "LastBackup", b->getLastTime() );
    mpTheConfig->writeEntry( "isActivated", b->isActivated() );

    mProfileNames.append( b->getIdentifier() );
}

void Configurator::changeProfile( BackupProfile *b )
{
    TRACE();
    Q_ASSERT( mpTheConfig->hasGroup( b->getIdentifier() ) );
    Q_ASSERT( 1 == mProfileNames.contains( b->getIdentifier() ) );

    mpTheConfig->deleteGroup( b->getIdentifier() );

    mpTheConfig->setGroup( b->getIdentifier() );
    mpTheConfig->writeEntry( "SourceUrl",  b->getSourceUrl().url() );
    mpTheConfig->writeEntry( "ArchiveUrl", b->getArchiveUrl().url() );
    mpTheConfig->writeEntry(
        "TimeNumber", b->getTimeBetweenBackups().value()
        );
    mpTheConfig->writeEntry(
        "Measurement", b->getTimeBetweenBackups().measurement()
        );
    mpTheConfig->writeEntry( "LastBackup", b->getLastTime() );
    mpTheConfig->writeEntry( "isActivated", b->isActivated() );
}

void Configurator::removeProfile( QString q )
{
    TRACE();
    Q_ASSERT( 1 == mProfileNames.contains( q ) );
    mProfileNames.remove( q );

    Q_ASSERT( mpTheConfig->hasGroup( q ) );
    mpTheConfig->deleteGroup( q );
}

BackupProfile *Configurator::nextProfile( QString aProfileName )
{
    TRACE();
    mpTheConfig->setGroup( aProfileName );

    KURL source( mpTheConfig->readEntry( "SourceUrl" ));
    KURL archive( mpTheConfig->readEntry( "ArchiveUrl" ) );

    QString          id = aProfileName;
    int            time = mpTheConfig->readNumEntry( "TimeNumber" );
    QString measurement = mpTheConfig->readEntry( "Measurement" );
    bool         active = mpTheConfig->readBoolEntry( "isActivated" );
    QDateTime         q = mpTheConfig->readDateTimeEntry( "LastBackup" );

    BackupProfile *tmp =  BackupProfile::newBackupProfile(
        source, archive, aProfileName, time, measurement, active, q
        );

    return tmp;
}
