/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "connection.h"
#include "tool.h"

#include "addressbook_record.h"

using namespace Form;

Entry AddressBookRecord::loadEntry( const QString &id, PagePolicy::State )
{
  QString filter( "objectClass=*" );
  QLdapResponse response = Connection::self()->search( id, QLdap::Base, filter );

  if ( !response.isValid() || response.entries().isEmpty() )
    return Entry();

  return Entry::fromLdapEntry( response.entries().first() );
}

bool AddressBookRecord::saveEntry( const Entry &entry, PagePolicy::State state, QString &errorMsg )
{
  if ( state == PagePolicy::Add || state == PagePolicy::Modify ) {

    QString dnRoot = QString( "cn=external,%1" ).arg( Connection::self()->baseDn() );
    QLdapEntry ldapEntry;

    ldapEntry.addValue( "objectClass", "top" );
    ldapEntry.addValue( "objectClass", "inetOrgPerson" );
    ldapEntry.addValue( "objectClass", "kolabInetOrgPerson" );

    ldapEntry.setValue( "givenName", entry.value( "givenName" ) );
    ldapEntry.setValue( "sn", entry.value( "sn" ) );
    ldapEntry.setValue( "cn", QString( "%1 %2" ).arg( entry.value( "givenName" ), entry.value( "sn" ) ) );

    if ( !entry.value( "title" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "title", entry.value( "title" ) );

    if ( !entry.value( "mail" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "mail", entry.value( "mail" ) );

    if ( !entry.value( "alias" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "alias", entry.value( "alias" ) );

    if ( !entry.value( "o" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "o", entry.value( "o" ) );

    if ( !entry.value( "ou" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "ou", entry.value( "ou" ) );

    if ( !entry.value( "roomNumber" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "roomNumber", entry.value( "roomNumber" ) );

    if ( !entry.value( "street" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "street", entry.value( "street" ) );

    if ( !entry.value( "postOfficeBox" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "postOfficeBox", entry.value( "postOfficeBox" ) );

    if ( !entry.value( "postalCode" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "postalCode", entry.value( "postalCode" ) );

    if ( !entry.value( "l" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "l", entry.value( "l" ) );

    if ( !entry.value( "c" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "c", entry.value( "c" ) );

    if ( !entry.value( "telephoneNumber" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "telephoneNumber", entry.value( "telephoneNumber" ) );

    if ( !entry.value( "facsimileTelephoneNumber" ).isEmpty() || state == PagePolicy::Modify )
      ldapEntry.setValue( "facsimileTelephoneNumber", entry.value( "facsimileTelephoneNumber" ) );

    if ( state == PagePolicy::Modify ) {
      QString newDn = QString( "cn=%1,%2" ).arg( ldapEntry.value( "cn" ), dnRoot );

      /**
       * The 'cn' of the address has changed, so we have to perform some
       * extra checks.
       */
      if ( entry.id() != newDn ) {
        if ( !Connection::self()->add( newDn, ldapEntry ) ) {
          errorMsg = QObject::tr( "Could not modify address '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }

        if ( !Connection::self()->remove( entry.id() ) ) {
          errorMsg = QObject::tr( "Could not modify address '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }

      } else {
        if ( !Connection::self()->modifyAttributes( entry.id(), ldapEntry ) ) {
          errorMsg = QObject::tr( "Could not modify address '%1': %2" )
                                .arg( ldapEntry.value( "cn" ), Connection::self()->errorString() );
          return false;
        }
      }
    } else if ( state == PagePolicy::Add ) {
      QString dn = QString( "cn=%1,%2" ).arg( ldapEntry.value( "cn" ), dnRoot );

      if ( !Connection::self()->add( dn, ldapEntry ) ) {
        errorMsg = QObject::tr( "Could not add address '%1': %2" )
                              .arg( entry.value( "cn" ), Connection::self()->errorString() );
        return false;
      }
    }
  } else if ( state == PagePolicy::Delete ) {
    QString ldapError;
    if ( !Tool::deleteObject( Connection::self(), entry.id(), ldapError, true ) ) {
      errorMsg = QObject::tr( "Could not delete address '%1': %2" )
                            .arg( entry.value( "cn" ), ldapError );
      return false;
    }
  }

  return true;
}
