/***************************************************************************
 *
 * knetworkmanager-storage.h - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Will Stephenson    <wstephenson@suse.de>, <wstephenson@kde.org>
 *         Stefan Bogner      <sbogner@suse.de>, <bochi@kmobiletools.org>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *         Helmut Schaa       <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include "knetworkmanager-devicestore.h"
#include "knetworkmanager-encryption.h"
#include "knetworkmanager-storage.h"
#include "settings.h"

#include <kdebug.h>
#include <kstaticdeleter.h>

static KStaticDeleter<KNetworkManagerStorage> sd;

KNetworkManagerStorage *KNetworkManagerStorage::m_instance = 0L;

KWallet::Wallet* CredentialsRequest::_wallet = 0L;

CredentialsRequest::CredentialsRequest(const QString &id)
{
	_id = id;
	_canceled = false;
}

CredentialsRequest::~CredentialsRequest()
{

}

void 
CredentialsRequest::slotWalletOpened(bool success)
{

	QMap<QString, QString> value;
	if (success)
	{
		(_wallet->setFolder ("knetworkmanager") && _wallet->readMap ( _id, value ));
	}
	State::getInstance()->setWaitingForKey(false);
	emit credentialsLoaded(_id, value, !success);
}

void
CredentialsRequest::slotCancelRequest()
{
	// signal cancel
	_canceled = true;
	QMap<QString, QString> value;
	State::getInstance()->setWaitingForKey(false);
	emit credentialsLoaded(_id, value, true);
}

void CredentialsRequest::loadCredentials()
{
    KNetworkManagerStorage* storage = KNetworkManagerStorage::getInstance();

    if (storage->getStoreKeysUnencrypted())
    {
      // read keys from config file and emit signal
    	QMap< QString, QString> value;
      value = KGlobal::config()->entryMap("Secret_" + _id);
    	emit credentialsLoaded(_id, value, false);
    }
    else
    {
        // check if the requested key is in the wallet -> if not we can return without the need to open the wallet
        if (KWallet::Wallet::keyDoesNotExist(KWallet::Wallet::NetworkWallet(), "knetworkmanager", _id))
				{
						// key is not in the wallet -> emit signal with empty map
            QMap<QString, QString> value;
			    	emit credentialsLoaded(_id, value, false);
				}
				else
				{
					if (_wallet && _wallet->isOpen())
					{
						// if we have already an open wallet reference make use of it
						slotWalletOpened(true);
					}
					else
					{
						// key is in wallet -> open the wallet async and wait for the openeing result
						State::getInstance()->setWaitingForKey(true);
        		_wallet = KWallet::Wallet::openWallet (KWallet::Wallet::NetworkWallet (), 0, KWallet::Wallet::Asynchronous);
						connect(_wallet, SIGNAL(walletOpened(bool)), this, SLOT(slotWalletOpened(bool)));
					}
				}
    }
}

KNetworkManagerStorage 
*KNetworkManagerStorage::getInstance ()
{
    if (!m_instance) 
        sd.setObject( m_instance, new KNetworkManagerStorage () );
    return m_instance;
}

KNetworkManagerStorage::KNetworkManagerStorage () : QObject ()
{
    m_wallet = 0L;
    m_walletRefCount = 0;
}


KNetworkManagerStorage::~KNetworkManagerStorage ()
{
    m_instance = 0L;
    slotWalletClosed ();
}

QStringList
KNetworkManagerStorage::vpnConnectionGroups() const
{
    QStringList groups = KGlobal::config()->groupList();
    QStringList connections;
    const QStringList::Iterator end = groups.end();
    for ( QStringList::Iterator it = groups.begin(); it != end; ++it )
    {
        if ( !(*it).startsWith( "VPNConnection_" ) )
            continue;
        connections.append( *it );
    }
    return connections;
}

QString 
KNetworkManagerStorage::vpnConnectionNewGroup() 
{
    return QString("VPNConnection_").append( KApplication::randomString( 16 ) );
}

void 
KNetworkManagerStorage::slotWalletClosed ()
{
    m_walletRefCount--;

    if (m_walletRefCount == 0) 
    {
        delete m_wallet;
        m_wallet = 0L;
    }
}

void
KNetworkManagerStorage::slotCancelAllCredentialRequests()
{
	while(_requests.begin() != _requests.end())
	{
		(*(_requests.begin()))->slotCancelRequest();
	}
}

void
KNetworkManagerStorage::slotCredentialsRequestDestroyed(QObject* obj)
{
	_requests.remove((CredentialsRequest*)obj);
}

CredentialsRequest*
KNetworkManagerStorage::credentialsAsync(const QString& id)
{
	CredentialsRequest* req = new CredentialsRequest(id);
	_requests.append(req);
	connect(req, SIGNAL(destroyed(QObject*)), this, SLOT(slotCredentialsRequestDestroyed(QObject*)));
	return req;
}

QMap< QString, QString > 
KNetworkManagerStorage::credentials(const QString &id)
{
    QMap< QString, QString> value;
    if (getStoreKeysUnencrypted())
    {
      // read keys from config file
      value = KGlobal::config()->entryMap("Secret_" + id);
    }
    else
    {
        // check if the requested key is in the wallet -> if not we can return without the need to open the wallet
        if (KWallet::Wallet::keyDoesNotExist(KWallet::Wallet::NetworkWallet(), "knetworkmanager", id))
            return QMap<QString, QString>();

        if (!m_wallet)
        {
            m_wallet = KWallet::Wallet::openWallet (KWallet::Wallet::NetworkWallet (), 0, KWallet::Wallet::Synchronous);

            if (m_wallet)
            {
                m_walletRefCount++;
                connect ( m_wallet, SIGNAL( walletClosed () ), this, SLOT( slotWalletClosed () ) );
            }
        }
        (m_wallet && m_wallet->setFolder ("knetworkmanager") && m_wallet->readMap ( id, value ));
    }
   return value;
}

bool
KNetworkManagerStorage::hasCredentialsStored(const QString& id)
{
    if (getStoreKeysUnencrypted())
    {
        QMap< QString, QString> value;
        value = KGlobal::config()->entryMap("Secret_" + id);
        return !value.empty();
    }
    else
    {
        return !(KWallet::Wallet::keyDoesNotExist(KWallet::Wallet::NetworkWallet(), "knetworkmanager", id));
    }
}

bool
KNetworkManagerStorage::storeKey( const QString &id, const QString &key )
{
    bool result = false;

    if (KWallet::Wallet::isEnabled () && !getStoreKeysUnencrypted() )
    {
        if ( !m_wallet )
        {
            m_wallet = KWallet::Wallet::openWallet(KWallet::Wallet::NetworkWallet(), 0, KWallet::Wallet::Synchronous);

            if ( m_wallet ) 
            {
                m_walletRefCount++;
                connect (m_wallet, SIGNAL(walletClosed()), this, SLOT(slotWalletClosed()));
            }
        }

        if ( m_wallet )
        {
            if( !m_wallet->hasFolder ( "knetworkmanager" ) ) m_wallet->createFolder ( "knetworkmanager" );
            m_wallet->setFolder ( "knetworkmanager" );
            QMap< QString, QString> map;
            map.insert( "password", key );
            m_wallet->writeMap (id, map);
            // FIXME: m_wallet methods can also fail, should check for this errors
            result = true;
        }
    }
    else
    {
        // store in config file
        KGlobal::config()->setGroup("Secret_" + id);
        KGlobal::config()->writeEntry("password", key);
        result = true;
    }

    return result;
}

bool
KNetworkManagerStorage::storeCredentials( const QString &id, const QMap< QString, QString> &map )
{
    bool result = false;

    if (KWallet::Wallet::isEnabled () && !getStoreKeysUnencrypted() )
    {
        if ( !m_wallet )
        {
            m_wallet = KWallet::Wallet::openWallet(KWallet::Wallet::NetworkWallet(), 0, KWallet::Wallet::Synchronous);

            if ( m_wallet )
            {
                m_walletRefCount++;
                connect (m_wallet, SIGNAL(walletClosed()), this, SLOT(slotWalletClosed()));
            }
        }

        if ( m_wallet )
        {
            if( !m_wallet->hasFolder ( "knetworkmanager" ) ) m_wallet->createFolder ( "knetworkmanager" );
            m_wallet->setFolder ( "knetworkmanager" );
            m_wallet->writeMap (id, map);
            result = true;
        }
    }
    else
    {
        KGlobal::config()->setGroup("Secret_" + id);
        for (QMap<QString, QString>::ConstIterator it = map.begin(); it != map.end(); ++it)
        {
            KGlobal::config()->writeEntry(it.key(), it.data());
        }
        result = true;
    }

    return result;
}

QStringList
KNetworkManagerStorage::networks() const
{
	QStringList groups = KGlobal::config()->groupList();
	QStringList networks;
	const QStringList::Iterator end = groups.end();
	for ( QStringList::Iterator it = groups.begin(); it != end; ++it )
	{
		if ( !(*it).startsWith( "Network_" ) )
			continue;

		KConfigGroup networkGrp( KGlobal::config(), *it );
		QString ESSID = networkGrp.readEntry( "ESSID" );
		networks.append( ESSID );
	}
	return networks;
}

Network * KNetworkManagerStorage::networkProperties( const QString & name, const QString &hwAddr, bool * hwAddrMatched )
{
	QString groupName = lookupNetworkGroupName( name, hwAddr, hwAddrMatched );
	Network * net = 0;

	if ( !groupName.isEmpty() )
	{
		KConfig * config = KGlobal::config();
		const char* version = config->readEntry ("Version", "0");

		KConfigGroup networkGrp( config, groupName );
		net = new Network();
		net->restore( &networkGrp, version );
	}
	return net;
}


void
KNetworkManagerStorage::storeNetwork( const Network* net, bool updateTimestamp )
{
	QString groupName = lookupNetworkGroupName( net->getEssid(), net->getHardwareAddresses().first() );
	if ( groupName.isEmpty() )
		groupName = QString("Network_").append( KApplication::randomString( 16 ) );

	KConfigGroup networkGrp( KGlobal::config(), groupName );

	//We assume that encrypted network always has a secret key
	net->persist( &networkGrp, updateTimestamp, net->isEncrypted() ); 
}

void KNetworkManagerStorage::removeNetwork( const Network * net )
{
	bool fuzzy = false;
	QString netName = lookupNetworkGroupName( net->getEssid(), net->getHardwareAddresses().first(), &fuzzy );
	if ( !netName.isEmpty() )
		KGlobal::config()->deleteGroup( netName, true );
}

QString KNetworkManagerStorage::lookupVPNConnectionGroupName( const QString & name) const
{
	KConfig * config = KGlobal::config();
	QStringList groups = config->groupList();
	const QStringList::Iterator end = groups.end();

	for ( QStringList::Iterator it = groups.begin(); it != end; ++it )
	{
		if ( (*it).startsWith("VPNConnection_" ) )
		{
			KConfigGroup candidate( config, *it );
			if ( candidate.readEntry( "Name" ) == name )
				return *it;
		}
	}
	return QString();
}

QString KNetworkManagerStorage::lookupNetworkGroupName( const QString & essid, const QString & hwAddr, bool * hwAddrMatched ) const
{
	KConfig * config = KGlobal::config();
	QStringList groups = config->groupList();
	const QStringList::Iterator end = groups.end();
	bool matchOnEssid = hwAddr.isEmpty();
	QStringList fuzzyCandidates;

	for ( QStringList::Iterator it = groups.begin(); it != end; ++it )
	{
		if ( (*it).startsWith("Network_" ) )
		{
			KConfigGroup candidate( config, *it );
			if ( matchOnEssid ) // just return the first network matching on essid
			{
				if ( candidate.readEntry( "ESSID" ) == essid )
				{
					if ( hwAddrMatched )
						*hwAddrMatched = false;
					return *it;
				}
			}
			else // check the hwaddresses for a match
			{
				if ( candidate.readEntry( "ESSID" ) == essid )
				{
					fuzzyCandidates.append( *it );
					QStringList addresses = candidate.readListEntry( "HardwareAddresses", NET_STORAGE_SEPARATOR );
					if ( addresses.find( hwAddr ) != addresses.end() )
					{
						if ( hwAddrMatched )
							*hwAddrMatched = true;
						return *it;
					}
				}
			}
		}
	}
	// if we got here, we didn't match on the caller's desired criteria
	// see if there is a fuzzy candidate match
	if ( hwAddrMatched )
		*hwAddrMatched = false;
	if ( fuzzyCandidates.isEmpty() )
	{
		return QString();
	}
	else
		return fuzzyCandidates.first();
}

void KNetworkManagerStorage::updateNetwork( Network * net, bool automatic )
{
	if ( net->isModified() ) {
		storeNetwork( net, !automatic );
	} else if ( !automatic ) {
		QString groupName = lookupNetworkGroupName( net->getEssid(), net->getHardwareAddresses().first() );
		// New networks (without groupName) are always dirty, so this "if" is "just to be sure"
		if ( !groupName.isEmpty() ) {
			KConfigGroup networkGrp( KGlobal::config(), groupName );
			net->persistTimestamp( &networkGrp );
		}
		else
			kdWarning() << k_funcinfo << "Newly created networks can't have isModified() == false" << endl;
	}
}

void KNetworkManagerStorage::persistWireless (bool state)
{
	KGlobal::config ()->setGroup("General");
	KGlobal::config ()->writeEntry ("WirelessEnabled", state); 
}

bool KNetworkManagerStorage::getWireless (void)
{
	bool ret = false;

	KGlobal::config ()->setGroup("General");
	ret = KGlobal::config ()->readBoolEntry ("WirelessEnabled", true);

	return ret;
}

void KNetworkManagerStorage::persistOfflineMode (bool state)
{
	KGlobal::config ()->setGroup("General");
	KGlobal::config ()->writeEntry ("OfflineModeEnabled", state); 
}

bool KNetworkManagerStorage::getOfflineMode (void)
{
	bool ret = false;

	KGlobal::config ()->setGroup("General");
	ret = KGlobal::config ()->readBoolEntry ("OfflineModeEnabled", false);

	return ret;
}

bool KNetworkManagerStorage::getStoreKeysUnencrypted(void)
{
	bool ret = false;

	KGlobal::config ()->setGroup("General");
	ret = KGlobal::config ()->readBoolEntry ("StoreKeysUnencrypted", false);

	// KWallet may not be uses if it is disabled
	if (!ret && !KWallet::Wallet::isEnabled())
		ret = true;
	

	return ret;
}

bool KNetworkManagerStorage::setStoreKeysUnencrypted(bool val)
{
	bool retval = true;
	
	// if the wallet is not enabled we may not use it
	if (!val && !KWallet::Wallet::isEnabled())
		val = true;

	if (val != getStoreKeysUnencrypted())
	{
		// ok, settings changes
		if (!m_wallet && KWallet::Wallet::isEnabled())
		{
			m_wallet = KWallet::Wallet::openWallet (KWallet::Wallet::NetworkWallet (), 0, KWallet::Wallet::Synchronous);
			if (m_wallet)
			{
				m_walletRefCount++;
				connect( m_wallet, SIGNAL( walletClosed() ), this, SLOT( slotWalletClosed() ) );
			}
		}

		if (m_wallet && m_wallet->isOpen())
		{
			m_wallet->setFolder("knetworkmanager");
			if (val)
			{
				// transfer secrets from wallet to config file
				QMap<QString, QMap<QString, QString> > allmaps;
				m_wallet->readMapList( "*", allmaps);

				if (!allmaps.isEmpty())
				{
					// write secrets to config file
					for(QMap<QString, QMap<QString, QString> >::Iterator it = allmaps.begin(); it != allmaps.end(); ++it)
					{
						KGlobal::config()->setGroup("Secret_" + it.key());
						for (QMap<QString, QString>::ConstIterator mapit = it.data().begin(); mapit != it.data().end(); ++mapit)
						{
							KGlobal::config()->writeEntry(mapit.key(), mapit.data());
						}
						// delete secret from kwallet as it is now stored in the config file
						m_wallet->removeEntry(it.key());
					}
				}
			}
			else
			{
				// transfer secrets from config file to wallet
				QStringList groups = KGlobal::config()->groupList();
				for ( QStringList::Iterator it = groups.begin(); it != groups.end(); ++it )
				{
					if ( !(*it).startsWith( "Secret_" ) )
						continue;
					QString id = (*it).right((*it).length() - QString("Secret:").length());
					QMap<QString, QString> secrets = KGlobal::config()->entryMap((*it));
					m_wallet->writeMap(id, secrets);

					KGlobal::config()->deleteGroup(*it);
				}
			}

			// write config key
			KGlobal::config()->setGroup("General");
			KGlobal::config()->writeEntry("StoreKeysUnencrypted", val);
		}
		else
		{
			retval = false;
		}
	}
	return retval;
}

#include "knetworkmanager-storage.moc"
