/***************************************************************************
  searchpanel.cpp  -  description
  -------------------
begin                : Tue Dec 28 1999
copyright            : (C) 1999-2000 by John Donoghue
email                : donoghue@chariot.net.au
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "knap_searchpanel.h"

#include <qpopupmenu.h>
#include <kapp.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qcombo.h>
#include <knap_listview.h>
#include <qlineedit.h>
#include <qpushbutton.h>

#include <klocale.h>
#include <qstrlist.h>
#include <kconfig.h>
#include <kglobal.h>

#include "songinfo.h"
#include "support_funcs.h"

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "speedvgood.xpm"
#include "speedgood.xpm"
#include "speedaverage.xpm"
#include "speedbad.xpm"

class SearchResultItem : public QListViewItem 
{
   public:
      SearchResultItem(QListView *parent,const char *s1,const char *s2,const char *s3,
      const char *s4,const char *s5,const char *s6,const char *s7, const char *s8)
	 : QListViewItem(parent,s1,s2,s3,s4,s5,s6,s7,s8)
	 {
	    int speed;

	    if(s4) 
	    {
	       speed = QString(s4).toInt();
	       /*
		  show a kind of speed indicator - crap really as it is assuming
		  that the user isn't lying about speed - this should be based on
		  ping time
		*/
	       if(speed<1) setPixmap(0,QPixmap((const char **)speedbad_xpm));
	       else if(speed<5) setPixmap(0,QPixmap((const char **)speedaverage_xpm));
	       else if(speed<8) setPixmap(0,QPixmap((const char **)speedgood_xpm));
	       else setPixmap(0,QPixmap((const char **)speedvgood_xpm));
	    }
	 }
      virtual QString text(int col) const
      {
	 static QString ret; // Copied on return from function so ok, even if ugly.
	 if(col==1) 
	 {
	    ret=convertSize(atol(QListViewItem::text(col)));
	    return ret;
	 }
	 else if (col==3) 
	 {
	    ret=convertSpeed(atoi(QListViewItem::text(col)));
	    return ret;
	 }
	 else
	    return QListViewItem::text(col);
      }
      virtual QString key(int col,bool asc) const
      {
	 static QString ret;
	 if(col==1) 
	 { // sort the size
	    ret.sprintf( "%012ld", QString(QListViewItem::text(col)).toLong() );
	    return ret;
	 }
	 else if(col==4) 
	 { // bitrate
	    ret.sprintf( "%06ld", QString(QListViewItem::text(col)).toLong() );
	    return ret;
	 }
	 else if(col==5)
	 {
	    int min,sec;
	    sscanf(QListViewItem::text(col),"%d:%d",&min,&sec);

	    ret.sprintf( "%09ld", (min * 60) + sec);
	    return ret;
	 }
	 return QListViewItem::text(col);
      }
};

   KNAP_SearchPanel::KNAP_SearchPanel(QWidget *parent, const char *name )
//: QWidget(parent,name)
  : QFrame(parent,name)
{

  setFrameStyle( QFrame::Panel | QFrame::Sunken );

   
   int maxw_sz;
   /* rows,cols , space,space*/
   QGridLayout *grid=new QGridLayout(this,6,7,5,5); //changed

   // add the first column of widths - they are all labels here - find the gigest and
   // set this is the max  width
   QLabel *l=new QLabel(i18n("Song title "),this);
   maxw_sz =  l->sizeHint().width();
   grid->addWidget(l,0,0);

   // the other search options
   l=new QLabel(i18n("Speed "),this);
   grid->addWidget(l,1,0);
   maxw_sz =  maxw_sz > l->sizeHint().width() ?
      maxw_sz : l->sizeHint().width();

   l=new QLabel(i18n("Frequency "),this);
   maxw_sz =  maxw_sz > l->sizeHint().width() ?
      maxw_sz : l->sizeHint().width();
   l->setMaximumWidth(maxw_sz);
   grid->addWidget(l,2,0);


   // the search test
   searchtext=new QLineEdit(this,"searchtext");

   // return caught to start search
   connect(searchtext,SIGNAL(returnPressed()),this,SLOT(calcSearch()));

   grid->addMultiCellWidget(searchtext,0,0,1,3);

   // now the prefs combo boxes fopr the 2nd column
   //
   // create the speed combo
   speedoption=createCompareValueCB("speedoption");
   speedoption->setMaximumSize( speedoption->sizeHint() );
   grid->addWidget(speedoption,1,1);
   connect(speedoption,SIGNAL(activated(int)),this,SLOT(optionComboActivated(int)));

   freqoption=createCompareValueCB("freqoption");
   freqoption->setMaximumHeight( freqoption->sizeHint().height() );
   grid->addWidget(freqoption,2,1);
   connect(freqoption,SIGNAL(activated(int)),this,SLOT(optionComboActivated(int)));

   // 3rd column - the freq and speed combos
   //

   speedtext=new QComboBox(false,this,"speedtext");
   for(int i=0;i<=10;i++) {
      speedtext->insertItem( convertSpeed(i) );
   }
   //speedtext->setMaximumWidth( basewidth );
   maxw_sz = speedtext->sizeHint().width();
   grid->addWidget(speedtext,1,2);

    // the freq combo
   freqtext=new QComboBox(false,this,"freqtext");
   freqtext->insertItem("11025");
   freqtext->insertItem("22050");
   freqtext->insertItem("44100");
   freqtext->insertItem("48800");
   grid->addWidget(freqtext,2,2);
   maxw_sz =  maxw_sz > freqtext->sizeHint().width() ?
            maxw_sz : freqtext->sizeHint().width(); 
   freqtext->setMaximumWidth( maxw_sz );

   // 4th column text for performer
   //
   grid->setColStretch(3,1000);
   // the search test
   grouptext=new QLineEdit(this,"searchtext");

   // return caught to start search
   //connect(searchtext,SIGNAL(returnPressed()),this,SLOT(calcSearch()));

   grid->addMultiCellWidget(grouptext,0,0,5,6);


   // 5th column text labels for bitrate and max results
   // no size set - coz we dont really care - since is alighed right
   //
   l=new QLabel(i18n("Group/Singer "),this);
   l->setAlignment(AlignRight);
   grid->addWidget(l,0,4);
   
   l=new QLabel(i18n("Bitrate "),this);
   l->setAlignment(AlignRight);
   grid->addWidget(l,1,4);

   l=new QLabel(i18n("Max results "),this);
   l->setAlignment(AlignRight);
   grid->addWidget(l,2,4);

   // 6th bitrate option combo and max results field
    bitrateoption=createCompareValueCB("bitrateoption");
   grid->addWidget(bitrateoption,1,5);
   maxw_sz = bitrateoption->sizeHint().width();
   connect(bitrateoption,SIGNAL(activated(int)),this,SLOT(optionComboActivated(int)));

   resultstext=new QLineEdit(this,"resultstext");
   // only allows 100 in napster anyway - more are ignored
   resultstext->setMaxLength(3);
   resultstext->setText("100");
   grid->addWidget(resultstext,2,5);
   resultstext->setMaximumWidth(maxw_sz);

   // 7th column

   // the bitrate combo
   bitratetext=new QComboBox(false,this,"bitratetext");
   bitratetext->insertItem("64");
   bitratetext->insertItem("74");
   bitratetext->insertItem("96");
   bitratetext->insertItem("112");
   bitratetext->insertItem("128");
   bitratetext->insertItem("160");
   bitratetext->insertItem("192");
   bitratetext->insertItem("256");
   bitratetext->insertItem("320");

   grid->addWidget(bitratetext,1,6);
   
   maxw_sz = bitratetext->sizeHint().width();


   // add search button
   searchbutton=new QPushButton(i18n("Search"),this,"searchbutton");
   searchbutton->setMaximumHeight(searchbutton->sizeHint().height()); 
   connect(searchbutton,SIGNAL(clicked()),this,SLOT(calcSearch()));
   grid->addWidget(searchbutton,2,6);
   maxw_sz =  maxw_sz > searchbutton->sizeHint().width() ?
            maxw_sz : searchbutton->sizeHint().width(); 

   searchbutton->setMaximumWidth( maxw_sz );

   // create and populate the list
   searchlistview=new KNAP_ListView(this,"searchview");
   searchlistview->setFrameStyle(QFrame::WinPanel|QFrame::Sunken);
   searchlistview->addColumn(i18n("file"),280);
   searchlistview->setColumnWidthMode(0,QListView::Manual);
   searchlistview->addColumn(i18n("size"),80);
   searchlistview->setColumnWidthMode(1,QListView::Manual);
   searchlistview->addColumn(i18n("frequency"),60);
   searchlistview->setColumnWidthMode(2,QListView::Manual);
   searchlistview->addColumn(i18n("speed"),80);
   searchlistview->setColumnWidthMode(3,QListView::Manual);
   searchlistview->addColumn(i18n("bitrate"));
   searchlistview->setColumnWidthMode(4,QListView::Manual);
   searchlistview->addColumn(i18n("time"));
   searchlistview->setColumnWidthMode(5,QListView::Manual);

   searchlistview->addColumn(i18n("user"),100);
   searchlistview->setColumnWidthMode(6,QListView::Manual);

   // sort on insertion  - on connection - ** was -1
   searchlistview->setSorting(3,false);

   // force a load of the saved list widths
   searchlistview->loadWidths();

   // default option is to allow multi selects
   multiselect =  true;
   searchlistview->setMultiSelection(true); // allow multi selects

   grid->addMultiCellWidget(searchlistview,3,3,0,6);

   // create the stats bar
   stats=new QLabel(i18n("No results"),this,"stats");
   stats->setFrameStyle(QFrame::Panel|QFrame::Sunken);
   stats->setMaximumHeight(searchbutton->sizeHint().height());
   grid->addMultiCellWidget(stats,4,4,0,6);


   // setup the menu popup

   searchpopupmenu=new QPopupMenu();
   if(searchpopupmenu) 
   {
      searchpopupmenu->insertItem(i18n("Download file"),this,SLOT(popupSearchDownload()));
      searchpopupmenu->insertSeparator();
      searchpopupmenu->insertItem(i18n("Ping User"),this,SLOT(popupPing()));
      searchpopupmenu->insertItem(i18n("Whois this User"),this,SLOT(popupWhois()));
      searchpopupmenu->insertItem(i18n("Add user to friend list"),this,SLOT(popupAddFriend()));

      searchpopupmenu->insertSeparator();
      searchpopupmenu->insertItem(i18n("Clear search list"),this,SLOT(clearSearch()));

      searchpopupmenu->insertSeparator();
      searchpopupmenu->insertItem(i18n("Toggle Show Path"),this,SLOT(popupShowPath()));
   }
   connect(searchlistview,SIGNAL(rightButtonPressed( QListViewItem *, const QPoint &, int )),
	 this,SLOT(popupSearch( QListViewItem *, const QPoint &, int )));
   connect(searchlistview,SIGNAL( dblClicked(QListViewItem *) ),
	 this,SLOT(dblclickSearchDownload( QListViewItem * ) ));

   // load in the last sessions search prefs
   loadSearchPrefs();
   // set the options on/off
   setExtraOptionsEnabled(true);

   // don't show paths as the default
   showsearchpath = false;

}
KNAP_SearchPanel::~KNAP_SearchPanel()
{
   cerr << "Saving search params\n";

   saveSearchPrefs(); 
}

/*
 * time to do a search - create the search string and send to
 * interested people
 */
void KNAP_SearchPanel::calcSearch()
{
   bool nosongname = false;
   const char *txt=searchtext->text(); // currentText();

   if(!txt || *txt=='\0') 
   {
      nosongname = true;
      txt = grouptext->text(); // no  song - use the group name only
   }

   if(!txt || *txt=='\0') return; // nothing given at all
   
   const char *m=resultstext->text();

   if(QString(m).toInt()<=0) m="100";

   QString tmp="FILENAME CONTAINS \"";

   tmp+=txt;
   tmp+="\" MAX_RESULTS ";
   tmp+=m;

   if(nosongname==false) // we might also have a groupname yet to process
   {
     txt=grouptext->text();
     if(txt && *txt!='\0')
      tmp += QString(" FILENAME CONTAINS \"") + txt + "\"";
   }
   if(speedtext->isEnabled()) 
   {
      tmp+=" LINESPEED \"";
      tmp+=getServerCompareString(speedoption->currentItem());
      tmp+="\" ";
      tmp+= QString().setNum(speedtext->currentItem());
   }
   if(bitratetext->isEnabled()) 
   {
      tmp+=" BITRATE \"";
      tmp+=getServerCompareString(bitrateoption->currentItem());
      tmp+="\" \"";
      tmp+=bitratetext->currentText();
      tmp+="\"";

   }
   if(freqtext->isEnabled()) 
   {
      tmp+=" FREQ \"";
      tmp+=getServerCompareString(freqoption->currentItem());
      tmp+="\" \"";
      tmp+=freqtext->currentText();
      tmp+="\"";
   }
#ifdef DEBUG_1
   cerr<<"SearchPanel::calcSearch(): "<<tmp<<endl;
#endif
// XX   emit search(tmp);
   emit sendcommand(QString("/search ") + tmp);

   return;
}

/*
 * add a song to the search list
 */
void KNAP_SearchPanel::addSearchItem(SongInfo *song)
{
   if(!song) return;

   //  QString s_filename;
   QString s_filename;
   if(!showsearchpath) s_filename=getFileName(song->filename);
   else s_filename=song->filename;


  cerr << "SearchPanel::addSearchItem(): " << s_filename << " sz: "
      << song->size << " freq: " << song->freq << " br: " << song->bitrate
      << endl;
   
   QListViewItem *i;


   // put the item in the listview
   // field 7 is invisible & holds the FULL path name of the song

   char buf[10];
   sprintf(buf,"%05d",song->speed);

   QString s_time = QString().setNum(floor(song->seconds / 60));
   s_time += ":";
   int sec = song->seconds % 60;
   if (sec < 10) s_time += "0";
   s_time += QString().setNum(sec);


   SongInfo* song2 = new SongInfo(*song);

   // results->append(song2);

   // lets show the results now for those with slow connections -> we'll get rid of
   // some later if optimise is on.
   i = new SearchResultItem(searchlistview,
	 s_filename,
	 QString().setNum(song->size),
	 QString().setNum(song->freq),
	 buf, // ->speed
	 QString().setNum(song->bitrate),
	 s_time,
	 song->user,song->filename);

  
   // display the number of matches in the stat bar
   // but only if it makes sense to do so now
   //
   int cnt = searchlistview->childCount();
   if(cnt!=0) stats->setText(QString().setNum(cnt) + i18n(" Matches found"));
   else stats->setText(i18n("No results"));

}

/*
 * time to popup the menu
 */
void KNAP_SearchPanel::popupSearch( QListViewItem *i, const QPoint &pos, int col)
{
   QListViewItem *selecteditem=searchlistview->currentItem();
   if(searchpopupmenu && selecteditem) {
      searchpopupmenu->exec(QCursor::pos());
   }
}

void KNAP_SearchPanel::clearSearch()
{
   searchlistview->clear();
   stats->setText(i18n("No results"));
}

/*
 * downlaod has been selected - request dload for each selected item
 */
void KNAP_SearchPanel::popupSearchDownload()
{
   cerr << "KNAP_SearchPanel::popupSearchDownload(): list" << endl;
   QListViewItem *item=searchlistview->firstChild();

   while(item) 
   {
      if(item->isSelected()) 
      {
	 dblclickSearchDownload(item);
	 searchlistview->setSelected(item,false);
      }
      item = item->nextSibling();
   }
}
void KNAP_SearchPanel::dblclickSearchDownload( QListViewItem *item )
{

        cerr << "KNAP_SearchPanel::dblclickSearchDownload(): " 
	   << item->text(0) << endl;

   SongInfo song(item->text(7),"",item->text(1).toLong(),
	 item->text(4).toInt(),
	 item->text(2).toInt(),
	 0,item->text(6),0,0,item->text(3).toInt());
   // XX emit download(&song); 
   emit sendcommand(QString("/download ") + item->text(6) 
	 + " \"" + item->text(7) + "\"");
	 

}
/*
 * menu selected for ping so ping all selected items
 */
void KNAP_SearchPanel::popupPing()
{
   QListViewItem *item=searchlistview->firstChild();

   while(item) {
      if(item->isSelected()) {
	 // selected so ping it
	 // XX emit ping(item->text(6));
	 emit sendcommand(QString("/ping ") + item->text(6));
      }
      item = item->nextSibling();
   }

}

/*
 * menu sel for who is - whois all selected
 */
void KNAP_SearchPanel::popupWhois()
{
   QListViewItem *item=searchlistview->firstChild();

   while(item) {
      if(item->isSelected()) {
	 // selected so whois it
	 // XX emit whois(item->text(6));
	 emit sendcommand(QString("/whois ") + item->text(6));
      }
      item = item->nextSibling();
   }
}

/*
 * menu sel for adding a friend
 */
void KNAP_SearchPanel::popupAddFriend()
{
   QListViewItem *item=searchlistview->firstChild();

   while(item) {
      if(item->isSelected()) {
	 // selected
	 // emit addFriend(item->text(6));

	 emit sendcommand(QString("/add ") + item->text(6));
      }
      item = item->nextSibling();
   }
}
/*
 * get setting for if we wanted to show full apths or not
 */
bool KNAP_SearchPanel::showSearchPath() const
{
   return showsearchpath;
}

/*
 * set whether to show full paths
 */
void KNAP_SearchPanel::setShowSearchPath(bool yes)
{
   QString s_filename;

   // if setting has changed - we will have to go thru each one and change it
   if(showsearchpath!=yes)
   {
      QListViewItem *it=searchlistview->firstChild();
      while(it) {
	 if(!yes) s_filename=getFileName(it->text(7));
	 else s_filename=it->text(7);

	 it->setText(0,s_filename);

	 it=it->nextSibling();
      }

      showsearchpath=yes;
   }
   return;
}

/*
 * set whether to use multiselect or not
 */
void KNAP_SearchPanel::setMultiSelect(bool yes)
{
   QString s_filename;

   if(multiselect!=yes)
   {
      // unselect all if not multi select????
      QListViewItem *item=searchlistview->firstChild();

      while(item) {
	 if(item->isSelected())
	    searchlistview->setSelected(item,false);
	 item = item->nextSibling();
      }

      searchlistview->setMultiSelection(yes); // allow multi selects

      multiselect=yes;
   }
   return;
}

// slot for when one of the other search option combos is cativated
// ( an item is selected/changed )
void KNAP_SearchPanel::optionComboActivated(int v)
{
   // don't really care about the value here
   // setExtraOptionsEnabled will handle it
   setExtraOptionsEnabled(true);
   // we only get this msg if an option was enabled
}

// enable/disable the extra search options in the search panel
// NOTE: that even if enable is true, some widgets will only
// be enabled if their sibling widget allows them to be
// ie if speed option is Don't care, then the speed combo will always
// be disabled
void KNAP_SearchPanel::setExtraOptionsEnabled(bool enable)
{
   // TODO FIX - dont care about 'enable' no more
   speedoption->setEnabled(enable);
   bitrateoption->setEnabled(enable);
   freqoption->setEnabled(enable);

   resultstext->setEnabled(enable);

   speedtext->setEnabled( enable && speedoption->currentItem()!=0 );
   bitratetext->setEnabled( enable && bitrateoption->currentItem()!=0 );
   freqtext->setEnabled( enable && freqoption->currentItem()!=0 );

   //  optimizeoption->setEnabled(enable);

   return;
}

// creates a combo box with the items
// DONT CARE
// AT LEAST
// AT BEST
// EQUAL TO
QComboBox * KNAP_SearchPanel::createCompareValueCB(const char *name)
{
   QComboBox *cb;

   cb=new QComboBox(false,this,name);
   cb->insertItem(i18n("DONT CARE"));
   cb->insertItem(i18n("AT LEAST"));
   cb->insertItem(i18n("AT BEST"));
   cb->insertItem(i18n("EQUAL TO"));

   return cb;
}

QString KNAP_SearchPanel::getServerCompareString(int pos)
{
   QString s;

   switch(pos)
   {
      case 1:
	 s = "AT LEAST";
	 break;
      case 2:
	 s="AT BEST";
	 break;
      case 3:
	 s="EQUAL TO";
	 break;
      default:
	 s = "DONT CARE";
   }
   return s;
}

/*
 * menu sel for show paths - toggle it
 */
void KNAP_SearchPanel::popupShowPath()
{
   setShowSearchPath(!showsearchpath);
}

/*
 * load in the search prefs from the config file
 */
void KNAP_SearchPanel::loadSearchPrefs()
{
   KConfig *config = KGlobal::config();

   setExtraOptionsEnabled(true);

   resultstext->setText(config->readEntry("search_results"));

   speedoption->setCurrentItem(config->readNumEntry("search_speedopt",0));
   bitrateoption->setCurrentItem(config->readNumEntry("search_bitrateopt",0));
   freqoption->setCurrentItem(config->readNumEntry("search_freqopt",0));

   freqtext->setCurrentItem(config->readNumEntry("search_freq",0));
   speedtext->setCurrentItem(config->readNumEntry("search_speed",0));
   bitratetext->setCurrentItem(config->readNumEntry("search_bitrate",0));

   return;
}
/*
 * save the prefs to the config for next time
 */
void KNAP_SearchPanel::saveSearchPrefs()
{
   KConfig *config = KGlobal::config();

   config->writeEntry("search_freq",freqtext->currentItem());
   config->writeEntry("search_speed",speedtext->currentItem());
   config->writeEntry("search_bitrate",bitratetext->currentItem());

   config->writeEntry("search_results",resultstext->text());

   config->writeEntry("search_speedopt",speedoption->currentItem());
   config->writeEntry("search_bitrateopt",bitrateoption->currentItem());
   config->writeEntry("search_freqopt",freqoption->currentItem());

   return;
}

