/* This is the start of a new garbage collector (replacing vm/garbage.c)
 *
 * That was a simple conservative mark and sweep collector (with compaction).
 *
 * I want to implement a generational collector with two heaps, called young heap and old heap.
 * Allocation from young heap is done for all objects < 50 words in size. All bigger objects automatically go to Old Heap.
 *
 * The Young Heap uses the notion of a Thread Local Heap (TLH) to avoid acquiring a lock on allocation of every object.
 *
 * tlh.h describes the structure of a TLH. The Young Heap has a list of free TLHs, a thread can acquire one of these for
 * itself by doing an atomic test & set on the lock word of the TLH.
 *
 * Only the owning thread may manipulate a TLH. Each TLH has 375 words for objects.
 *
 * If a thread can't acquire a new TLH, it has to do a conventional allocation from a free list of spaces.
 *
 * The young heap is collected more often that the old, and survivors of X collections are promoted to the old heap. The Old 
 * heap is compacted after collection.
 */

#include <config.h>

#ifdef ALLOCATOR

#ifdef KISSME_LINUX_USER
 #include <stdio.h>
#endif

 #include "assert.h"
#include "wrappers.h"
#include "cplist.h"


#include "../thread.h"
#include "jni.h"
#include "cplist.h"
#include "cptuplelist.h"

#include "public_methods.h"
#include "handles.h"

#include "youngheap.h"
#include "oldheap.h"

#include "garbage_defs.h"
#include "mark/mark.h"

#define ALLOCATOR_SIZE (10 * 1024 * 1024) //megabytes
#define ALLOCATOR_HANDLES_SIZE 200000
#define OLD_TO_YOUNG_RATIO 100   //the old heap is ratio times as large as the young heap

static int allocator_size = 0;

/* Must be called before ALLOCATOR_Init()
 * 
 *
 */

void ALLOCATOR_SetHeapSize( int nbytes )
{
  allocator_size = nbytes;
}

/* Allocates the bitfield, young heap and old heap in one large block. The handle block is in a separate area
 *
 */

tAllocatorHeap* ALLOCATOR_Init()
{
  tAllocatorHeap* heap = ( tAllocatorHeap*) sys_malloc( sizeof(tAllocatorHeap));
  if(heap == NULL)
    {
      return NULL;
    }

  heap->iDuringGC = 0;
  if(allocator_size == 0)
    allocator_size = ALLOCATOR_SIZE;

  assert(allocator_size > 0);

  heap->pi32Block = (int32*) sys_malloc( allocator_size );
  if(heap->pi32Block == NULL)
    {
      sys_free(heap);
      return NULL;
    }
  
  //Now allocate the handle block 
   heap->pi32HandleBlock = (int32*) sys_malloc( ALLOCATOR_HANDLES_SIZE * sizeof(int32*));
   if(heap->pi32HandleBlock == NULL)
     {
       sys_free(heap->pi32Block);
       sys_free(heap);
       return NULL;
     }

   heap->i32HandleBlockSize = (ALLOCATOR_HANDLES_SIZE);

   //Now initialize the handle block 
   if( HANDLES_Init(heap->pi32HandleBlock, heap->i32HandleBlockSize) != 0)
     {
       sys_free(heap->pi32Block);
       sys_free(heap->pi32HandleBlock);
       sys_free(heap);
       return NULL;
     }

   heap->hNextFreeHandle = HANDLES_FirstHandle(heap->pi32HandleBlock);

   heap->i32BlockSize = allocator_size / sizeof(int32);

   //Determine the size of the bitfield
   //We need 1 bit for every word left in the block


   //So the bitfield will represent i32BlockSize words - 1/32nd 
   //ie i32BlockSize words * 31/32
   heap->i32BFSize = ((heap->i32BlockSize * 31) / 32) / 32;

   //Determine the size of the young heap and old heap
   heap->i32OldHeapSize = (heap->i32BlockSize - heap->i32BFSize) / (OLD_TO_YOUNG_RATIO + 1); //temp val
   heap->i32YoungHeapSize = heap->i32OldHeapSize;
   heap->i32OldHeapSize = heap->i32BlockSize - heap->i32YoungHeapSize - heap->i32BFSize;

   heap->i32OldHeapBFSize = heap->i32OldHeapSize / 32; //might be one too little?
   heap->i32YoungHeapBFSize = heap->i32BFSize - heap->i32OldHeapBFSize;
							   
   //now set up the pointers
   heap->pi32YoungHeap = heap->pi32Block + heap->i32BFSize;
   heap->pi32OldHeap = heap->pi32YoungHeap + heap->i32YoungHeapSize;

   assert( (heap->pi32OldHeap + heap->i32OldHeapSize) == (heap->pi32Block + heap->i32BlockSize));

   //set the locks
   heap->youngMutex = sys_mutex_create(0);
   heap->oldMutex = sys_mutex_create(0);

   //Now initialize the young heap
   if(YOUNGHEAP_Init(heap) != 0)
     {
       //XXX we must free all that's been allocated 
	return NULL;
     }

   //Now initialize the old heap
   if(OLDHEAP_Init(heap) != 0)
     {
       //XXX we must free all that's been allocated 
	return NULL;
     }

#ifdef GARBAGE_VERBOSE
   eprintf("Young heap size is %i and old heap size %i\n", (int) heap->i32YoungHeapSize, (int) heap->i32OldHeapSize);
   eprintf("Young heap BF size is %i and old heap BF size %i\n", (int) heap->i32YoungHeapBFSize, (int) heap->i32OldHeapBFSize);
#endif
//   OLDHEAP_TestBitfield( heap );
   return heap;
}

//local methods
static int ALLOCATOR_MarkReachableFromMachine(tAllocatorHeap* heap);
static int ALLOCATOR_MarkReachableFromRPOT(tAllocatorHeap* heap);
static int ALLOCATOR_SweepAndCompact(tAllocatorHeap* heap);
int ALLOCATOR_ISHANDLE( tAllocatorHeap* heap, tOBREF h );

void doCorruptionCheck()
{

}

/* Does a garbage collection of the heaps 
 *
 *
 *
 */

int ALLOCATOR_gc(tAllocatorHeap* heap)
{
  if(heap->iDuringGC)
    {
      eprintf("Can't do recursive GC!\n");
      assert( 2==3);
      return -1;
    }

  heap->iDuringGC = 1;

  doCorruptionCheck();

  /* first mark all the items reachable from the machine */
  if(ALLOCATOR_MarkReachableFromMachine(heap) != 0)
    return -1;

#ifdef GARBAGE_VERBOSE
  eprintf("debug: _____ ALLOCATOR  did MarkReachableFromMachine _____\n");
  doCorruptionCheck();
#endif

  /* mark all the items reachable from the RPOT */
  if(ALLOCATOR_MarkReachableFromRPOT(heap) != 0)
    return -2;

  doCorruptionCheck();

  /* cunningly sweep and compact at the same time */
  if(ALLOCATOR_SweepAndCompact(heap) != 0)
    return -3;

#ifdef GARBAGE_VERBOSE
  eprintf("debug: _____ ALLOCATOR  did SweepAndCompact _____\n");
  doCorruptionCheck();
  eprintf("Leaving allocator gc\n");
#endif
  heap->iDuringGC = 0;

  return 0;
}


static int ALLOCATOR_MarkReachableFromMachine(tAllocatorHeap* heap)
{
  int threadIndex;
  int i = 0;
  tClassLoaderTuple*      pstCurrClass;
  tJNIData*    pstJNIData;

  if(ALLOCATOR_MARK_MarkThreadStacks(heap) != 0)
    {
      return -1;
    }

  /* go through the JNI local references for the current thread... in our
     single-processor environment it's the only one that can be executing
     native code at the moment... will be NULL if native code is not being
     executed 


     oh-oh, what happens with MT?

    Update for MT: Well this is the GC thread running, so it has NO native state
                   The other threads must never be in a native call, although there may be a native call higher up the chain

		   This means we must scan the pstJNIData for each thread
  */

  for(threadIndex = 0; threadIndex < 65536;threadIndex++)
    {
      pstJNIData = JNI_getJNIData( threadIndex ); 
            
      if (pstJNIData)
	{
	  for (i = 0; i < pstJNIData->u16NumLocalRefsUsed; i++)
	    {
	      if (pstJNIData->ppstLocalRefs[i])
		{
		  ALLOCATOR_MARK_RecursivelyMark(heap, (tOBREF) pstJNIData->ppstLocalRefs[i], 0, GARBAGE_REACHABLE, NULL);
		}
	    }
	  
	}
    }


  /* go through the JNI global references */
  for (i = 0; i < JNI_u16NumGlobalRefsUsed; i++)
    {
      if (JNI_ppstGlobalRefs[i])
        {
          ALLOCATOR_MARK_RecursivelyMark(heap, (tOBREF) JNI_ppstGlobalRefs[i], 0, GARBAGE_REACHABLE, NULL);
        }
    }

  /* go through C stack looking for more objects */
  
  ALLOCATOR_MARK_MarkCStacks(heap);
  
  /* go through all classes looking for relevant static fields */
  /* also look to see if a java.lang.Class object has been created for the class */
  for(CPTUPLELIST_GoTop(); CPTUPLELIST_IsMore(); CPTUPLELIST_Skip())
    {
      pstCurrClass = CPTUPLELIST_GetCurrentClass();
      /* go through size, not count, because we need every field - those that are
         inherited as well... count only gives the current class's number of
         fields... and don't really want to go through all the parent classes
         as well to find the fields */
      for (i = 0; i < pstCurrClass->pstClass->u16StatSize; i++)
        {
          if (ALLOCATOR_ISHANDLE(heap, (tOBREF) pstCurrClass->pi32StatVars[i]))
            {
              ALLOCATOR_MARK_RecursivelyMark(heap, (tOBREF) pstCurrClass->pi32StatVars[i], 1, GARBAGE_REACHABLE, (tOBREF) pstCurrClass);
            }
        }
      if (pstCurrClass->classObject)
        {
          ALLOCATOR_MARK_RecursivelyMark(heap, (tOBREF) pstCurrClass->classObject, 0, GARBAGE_REACHABLE, (tOBREF) pstCurrClass);
        }
    }
  return 0;
}

static int ALLOCATOR_MarkReachableFromRPOT(tAllocatorHeap* heap)
{
  //empty until we do persistence
  return 0;
}
static int ALLOCATOR_SweepAndCompact(tAllocatorHeap* heap)
{
  if(OLDHEAP_SweepAndCompact(heap) != 0)
    return -1;
  return 0;
}

int ALLOCATOR_ISHANDLE( tAllocatorHeap* heap, tOBREF h )
{
  if( OLDHEAP_ISHANDLE( heap, h ))
    return 1;

  if( YOUNGHEAP_ISHANDLE( heap, h ))
    return 1;

  return 0;
}

int ALLOCATOR_getNumTotalWords(tAllocatorHeap* heap)
{
  return heap->i32OldHeapSize;
}
int ALLOCATOR_getNumFreeWords(tAllocatorHeap* heap)
{
  int ret = 0;

  int32* freePointer = heap->pi32OldFirstFree;
  while(freePointer)
    {
      ret += *freePointer;
      freePointer = (int32*) *(freePointer + 1);
    }
  return ret;
}

#endif
