#ifdef USE_DISTRIBUTED_STORE

/* gamma.c by John Leuner

   See gamma.h for a list of functions and a description of the gamma interface.
 */


//#define GAMMA_DEBUG
#include "storemanager.h" 
#include "gamma.h"
#include "store_communication.h"
#include "alpha.h"
#include <assert.h>
#include <pthread.h>


/* List of dirty pages that need to be flushed to ALPHA */

struct dirty_page_entry* dirty_list = NULL;

/* We should close whatever sockets we are using, shutdown the threads etc */

void GAMMA_shutdown()
{

    while(dirty_list != NULL)
	{
	    GAMMA_flush(0x0L, (long long) 0x700000000000L);
	    printf("GAMMA is waiting for dirty pages to be cleared, 5 secs %x\n", dirty_list);
	    sleep(5);
	}

}
void GAMMA_initialize()
{
    /*   pthread_t clean_tid;
   if(pthread_create(&clean_tid, NULL, & (GAMMA_cleanup), NULL) == -1)
      {
	  perror("Could not start cleanup thread in GAMMA\n");
	  exit(88);
	  }*/
}


/*
 Read is reponsible for find the page that contains the data being requested. The caller is only allowed to request 4k of data at a time.

The first step is to get a page_entry for the page representing this address. We do this with ETA_get_page_for_address. If ETA doesn't have an entry for this address, we tell it to create one, ETA_create_page_for_address. Since this is a fresh page, we read the relevant data from the Virtual Disk Server using the Alpha interface ALPHA_read_blocking. (In the event that Alpha returned an error, we return an error code too).

We then continue by copying the data from this page to the buffer (locking the page before we do so). Note the ETA_get_page_for_address will automatically locate and fetch the page, even if it is on another machine.

We return the number of bytes copied to the buffer.
*/


int GAMMA_read(long long address, void* buf, long long offset, int buf_len)
{
  struct page_entry* stpeCurrentPage; //The page for address

  int success = 0;
  //The caller must give a page-aligned address
  if((address % BLOCK_SIZE) != 0)
    {
      printf("GAMMA_read address is not BLOCK aligned %x %x\n", (int) (address >> 32), (int)((address << 32) >> 32));
      exit(227);
    }

#ifdef GAMMA_DEBUG
  printf("GAMMA_read about to get page for address %x %x, offset %x\n", (int) (address >> 32), (int)((address << 32) >> 32), (int) offset);
#endif 
  
  // We ask ETA for the page, ETA guarantees that we have an up-to-date copy of the page
  stpeCurrentPage = ETA_get_page_for_address(address, ETA_GET_PAGE_FOR_READ);

  if(stpeCurrentPage == NULL)
    {
#ifdef GAMMA_DEBUG
  printf("GAMMA_read creating page for address %x %x\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif
      //This page entry does not exist, we have to create it
       success = ETA_create_page_for_address(address);
      if(success == -1)
	{
	  printf("GAMMA_read failed to create page for address\n");
	  exit(228);
	}

      //We try again
       stpeCurrentPage = ETA_get_page_for_address(address, ETA_GET_PAGE_FOR_READ);
       assert(stpeCurrentPage != NULL);


#ifdef GAMMA_DEBUG
  printf("GAMMA_read reading from ALPHA for address %x %x\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif      
  //Now we read the data from Alpha
      success = ALPHA_read_blocking(address, stpeCurrentPage->page_data, BLOCK_SIZE);
      if(success < 0)
	{
	  //This page didn't exist in ALPHA
	  printf("ALPHA_read_blocking failed for address %x %x\n",  (int) (address >> 32), (int)((address << 32) >> 32));
	  return -1;
	}
      assert(success == 0);
    }


  //Why do we do this twice?
  /*
#ifdef GAMMA_DEBUG
  printf("GAMMA_read about to get page for address (second time) %x %x\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif 

  stpeCurrentPage = ETA_get_page_for_address(address, ETA_GET_PAGE_FOR_READ);

  if(stpeCurrentPage == NULL)
    {
      printf("GAMMA_read failed to get page even after create\n");
      exit(229);
    }
  */

#ifdef GAMMA_DEBUG
  printf("GAMMA_read about to lock page entry %x %x\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif 

  //Lock the page before we access it (to make sure noone can write to it while we're reading)
  success = ETA_lock_page_entry( stpeCurrentPage );
  assert(success == 0);

  //Read the data from the page to the buffer
  memcpy(buf, stpeCurrentPage->page_data + offset, buf_len);

#ifdef GAMMA_DEBUG
  printf("GAMMA_read about to unlock page entry %x %x\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif 

  //Unlock the page
  success = ETA_unlock_page_entry( stpeCurrentPage );
  assert(success == 0);

#ifdef GAMMA_DEBUG
  printf("GAMMA_read finished %i\n", buf_len);
#endif

  //Return how much data was copied
  return buf_len;
}



/*

Write also looks for a page entry with ETA_get_page_for_address. If there is no such page it creates it.

If we are writing less than a complete page, we have to read the rest of the page from ALPHA. So we call ALPHA_read_blocking to get the page.

Then we simply lock the page and write to it.

Before unlocking, we add this page entry to the dirty_list of pages that need to be sent to Alpha at some stage.

Write will return -1 whenever an error occurs, if the page is locked by another thread, it will block until it has access. If exiting normally, write returns the number of bytes written to the page.
*/


int GAMMA_write(long long address, void* buf, long long offset, int buf_len)
{
  struct page_entry* stpeCurrentPage;
  int success = 0;


  if((address % BLOCK_SIZE) != 0)
    {
      printf("GAMMA_write address is not BLOCK aligned %x %x\n", (int) (address >> 32), (int)((address << 32) >> 32));
      exit(227);
    }
  assert(offset >= 0);
  assert(offset + buf_len <= BLOCK_SIZE);
#ifdef GAMMA_DEBUG
  printf("GAMMA_write about to get page for address %x %x\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif 
  stpeCurrentPage = ETA_get_page_for_address(address, ETA_GET_PAGE_FOR_WRITE);

  if(stpeCurrentPage == NULL)
    {
      //This page does not exist, we have to create it
#ifdef GAMMA_DEBUG
  printf("GAMMA_write about to create page for address %x %x number\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif 
      success = ETA_create_page_for_address(address);
      if(success != 0)
	{
	  printf("GAMMA_write failed to create page for address\n");
	  exit(228);
	}
      
      if(buf_len < BLOCK_SIZE)
	{
#ifdef GAMMA_DEBUG
  printf("GAMMA_write about to get page for address %x %x number 3\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif 
	  stpeCurrentPage = ETA_get_page_for_address(address, ETA_GET_PAGE_FOR_WRITE);
	  assert(stpeCurrentPage != NULL);
#ifdef GAMMA_DEBUG
  printf("GAMMA_write about to lock page for address %x %x number\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif 
          success = ETA_lock_page_entry( stpeCurrentPage );
	  assert(success == 0);
	  //We are writing less than the whole page, so we load it from ALPHA first
	  success = ALPHA_read_blocking(address, stpeCurrentPage->page_data, BLOCK_SIZE);
	  if(success != 0) 
	    {
	      printf("GAMMA_write could not read from ALPHA, address %x %x\n", (int) (address >> 32), (int)((address << 32) >> 32));
	    }
	  success = ETA_unlock_page_entry( stpeCurrentPage );
	  assert(success == 0);
	}
    }

#ifdef GAMMA_DEBUG
  printf("GAMMA_write about to get page for address %x %x number 2\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif 

  /* Why do we do this twice?*/
  stpeCurrentPage = ETA_get_page_for_address(address, ETA_GET_PAGE_FOR_WRITE);

  if(stpeCurrentPage == NULL)
    {
      printf("GAMMA_write failed to get page even after create\n");
      exit(229);
      }

#ifdef GAMMA_DEBUG
  printf("GAMMA_write about to lock page for address 2 %x %x number\n", (int) (address >> 32), (int)((address << 32) >> 32));
#endif 
  success = ETA_lock_page_entry( stpeCurrentPage );
  assert(success == 0);

//Copy the data to the page
  memcpy(stpeCurrentPage->page_data + offset, buf, buf_len);

//mark page as dirty for flushing to alpha
//We add it to the dirty_list
   if(stpeCurrentPage->flushed == 0)
       {
	   stpeCurrentPage->flushed = 1;
      if(dirty_list == NULL)
	{
	    #ifdef GAMMA_DEBUG
	  printf("GAMMA_write adding %x as new head\n", stpeCurrentPage);
	  #endif

	  dirty_list = (struct dirty_page_entry*) malloc(sizeof(struct dirty_page_entry));
	  dirty_list->magic = 0xdeadbeef;
	  dirty_list->stpePage = stpeCurrentPage;
	  dirty_list->next = dirty_list->prev = NULL;
	  
	}
      else
	{
	    struct dirty_page_entry* newhead = (struct dirty_page_entry*) malloc(sizeof(struct dirty_page_entry));
	    #ifdef GAMMA_DEBUG
	    printf("GAMMA_write adding %x to head\n", stpeCurrentPage);
	    #endif
	    newhead->next = dirty_list;
	    newhead->prev = NULL;
	    newhead->next->prev = newhead;
	    dirty_list = newhead;
    	  dirty_list->magic = 0xdeadbeef;
	    newhead->stpePage = stpeCurrentPage;	    
	}
       }
      
  success = ETA_unlock_page_entry( stpeCurrentPage );
  assert(success == 0);
#ifdef GAMMA_DEBUG
  printf("GAMMA_write finished %i\n", buf_len);
#endif
  return buf_len;
}


/*
  flush: Forces all unwritten pages in the range address_start - address_end to be written to ALPHA immediately.
*/

int GAMMA_flush(long long start_address, long long end_address)
{
  struct dirty_page_entry *iter = dirty_list;
  int success;
#ifdef GAMMA_DEBUG
  printf("GAMMA_flush start\n");
#endif
    while(iter != NULL)
    {
	assert(iter->magic == 0xdeadbeef);
	//	printf("GAMMA_flush checking %x\n", iter);
      if((iter->stpePage->address >= start_address) && (iter->stpePage->address <= end_address))
	{

#ifdef GAMMA_DEBUG
  printf("GAMMA_flush writing page for address %x %x\n", (int) (iter->stpePage->address >> 32), (int)((iter->stpePage->address << 32) >> 32));
#endif 
  ETA_lock_page_entry(iter->stpePage);
	  success = ALPHA_write_non_blocking(iter->stpePage);
	  assert(success == 0);

	  iter->stpePage->flushed = 0;
	  ETA_unlock_page_entry(iter->stpePage);
	  
	  //remove this item from the dirty list
	  if(iter->prev == NULL)
	      {
		  dirty_list = iter->next;
	  if(dirty_list != NULL)
		  dirty_list->prev = NULL;
		  free(iter);
	      }
	  else
	      {
		  iter->prev->next = iter->next;
	  if(iter->next != NULL)
		  iter->next->prev = iter->prev;
		  free(iter);
	      }
	}
	iter = iter->next;
	}
#ifdef GAMMA_DEBUG
  printf("GAMMA_flush end\n");
#endif
}
























































#endif











